/* config.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

private abstract class Sugar.Config {
    public static Config instance () {
        if (_instance == null) {
            _instance = new GConfBackend ();
            if (_instance.is_valid ())
                debug ("Detected GConf configuration backend");
            else {
                _instance = new FileBackend ();
                debug ("Detected file configuration backend");
            }
        }
        return _instance;
    }

    public abstract bool is_valid ();
    public abstract string get_color () throws Error;
    public abstract string get_nickname () throws Error;

    private class FileBackend : Config {
        public FileBackend () {
            _conf = new KeyFile ();
            var config_path = Path.build_filename (Profile.path, "config");

            try {
                _conf.load_from_file (config_path, KeyFileFlags.NONE);
            } catch (Error error) {
                warning ("Cannot open profile config file %s: %s",
                        config_path, error.message);
            }
        }

        public override bool is_valid () {
            return true;
        }

        public override string get_color () throws Error {
            return _conf.get_value ("Buddy", "color");
        }

        public override string get_nickname () throws Error {
            return _conf.get_value ("Buddy", "nickname");
        }

        KeyFile _conf;
    }

    private class GConfBackend : Config {
        public GConfBackend () {
            _conf = GConf.Client.get_default ();
        }

        public override bool is_valid () {
            try {
                return _conf.dir_exists ("/desktop/sugar/desktop");
            } catch (Error error) {
                return false;
            }
        }

        public override string get_color () throws Error {
            return _conf.get_string ("/desktop/sugar/user/color");
        }

        public override string get_nickname () throws Error {
            return _conf.get_string ("/desktop/sugar/user/nick");
        }

        private GConf.Client _conf;
    }

    private static Config _instance;
}
