# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from os.path import exists, expanduser
from gettext import gettext as _

import gtk

from sugar.graphics import style

from jarabe.controlpanel.sectionview import SectionView


# Only for XO laptops
_XSESSION_PATH = expanduser('~/.xsession')
_XSESSION_FILE = """\
exec sweets-sugar
"""


class Sweets(SectionView):

    def __init__(self, model, alerts):
        SectionView.__init__(self)
        self._sn_url_sid = 0

        self.set_border_width(style.DEFAULT_SPACING * 2)
        self.set_spacing(style.DEFAULT_SPACING)

        def section_new(title, container, spacing=style.DEFAULT_SPACING):
            self.pack_start(gtk.HSeparator(), expand=False)
            label = gtk.Label(title)
            label.set_alignment(0, 0)
            self.pack_start(label, expand=False)
            box = container()
            box.props.border_width = style.DEFAULT_SPACING
            box.props.spacing = spacing
            self.pack_start(box, expand=False)
            return box

        section = section_new(_('Sweets Distribution features'), gtk.VBox, 0)
        self._xsession = gtk.CheckButton()
        self._xsession.props.label = \
                _('Sweets Desktop is default Sugar session')
        self._xsession.connect('toggled',
                lambda button: _set_xsession(button.props.active))
        if _is_xo():
            section.pack_start(self._xsession, expand=False)

        self.show_all()

        self._orig_xsession = _xsession_enabled()
        self.setup()

    def setup(self):
        self._xsession.props.active = _xsession_enabled()

    def undo(self):
        if self._orig_xsession != _xsession_enabled():
            _set_xsession(self._orig_xsession)
        self.setup()

    @property
    def needs_restart(self):
        return self._xsession.props.active != self._orig_xsession

    @needs_restart.setter
    def needs_restart(self, value):
        # needs_restart is fully calculated
        pass


def _xsession_enabled():
    if not exists(_XSESSION_PATH):
        return False
    with file(_XSESSION_PATH) as f:
        return f.read() == _XSESSION_FILE


def _set_xsession(enable):
    if enable:
        with file(_XSESSION_PATH, 'w') as f:
            f.write(_XSESSION_FILE)
    else:
        if exists(_XSESSION_PATH):
            os.unlink(_XSESSION_PATH)

def _is_xo():
    return exists('/ofw')
