# Copyright (C) 2013 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import dgettext

import dbus
from sugar_network.toolkit import Option
from jarabe import plugins


_ = lambda x: dgettext('sugar-plugin-upgrade', x)

ORDER = 100
TITLE = _('Process unattended system upgrades periodically')

upgrade_timeout = Option(
        'call the system upgrade in specified number of seconds',
        default=60 * 60 * 24, type_cast=int, name='upgrade-timeout')

_logger = logging.getLogger('plugins.upgrade')
_schedule = None


def init():
    pass


def start():
    global _schedule

    _schedule = plugins.Schedule('upgrade', upgrade_timeout.value, _upgrade)

    sn = plugins.get('sn')
    if sn is None:
        _logger.error(
                'sugar-plugin-sn is disabled, schedule upgrade right now')
        _schedule.start()
    else:
        _logger.warning('Monitor Sugar Network connection to start upgrade')
        sn.connection().connect('event', __event_cb)


def __event_cb(sender, event, data):
    if event != 'inline':
        return
    if data['state'] == 'online':
        _schedule.start()
    elif data['state'] == 'offline':
        _schedule.stop()


def _upgrade():
    _logger.info('Upgrade the system')
    _PKTransaction().UpdateSystem(False)


class _PKTransaction(object):

    def __init__(self, finished_cb=None):
        self._finished_cb = finished_cb
        self.error_code = None
        self.error_details = None

        bus = dbus.SystemBus()
        pk = dbus.Interface(
                bus.get_object('org.freedesktop.PackageKit',
                    '/org/freedesktop/PackageKit', False),
                'org.freedesktop.PackageKit')
        self._proxy = dbus.Interface(
                bus.get_object('org.freedesktop.PackageKit',
                    pk.GetTid(), False),
                'org.freedesktop.PackageKit.Transaction')

        self._signals = []
        for signal, cb in [
                ('Finished', self.__finished_cb),
                ('ErrorCode', self.__error_code_cb),
                ]:
            self._signals.append(self._proxy.connect_to_signal(signal, cb))

    def __getattr__(self, attr):
        return getattr(self._proxy, attr)

    def __finished_cb(self, status, runtime):
        _logger.debug('Transaction finished: %s', status)
        if self._finished_cb is not None:
            self._finished_cb()
        for i in self._signals:
            i.remove()
        self._proxy = None

    def __error_code_cb(self, code, details):
        self.error_code = code
        self.error_details = details
        _logger.error('Transaction failed: (%s) %s', code, details)


if __name__ == '__main__':
    import gobject
    from dbus.mainloop.glib import threads_init, DBusGMainLoop

    logging.basicConfig(level=logging.DEBUG)

    gobject.threads_init()
    threads_init()
    DBusGMainLoop(set_as_default=True)

    mainloop = gobject.MainLoop()
    _PKTransaction(mainloop.quit).UpdateSystem(False)
    mainloop.run()
