# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import json
import shutil
import logging
from os.path import exists
from gettext import dgettext

import gtk
import gobject

from sugar import util

from sugar_network import client
from sugar_network.client import clones
from sugar_network.toolkit.pylru import lrucache
from sugar.bundle.activitybundle import ActivityBundle
from sugar.bundle.contentbundle import ContentBundle
from sugar.bundle.bundleversion import NormalizedVersion
from sugar.bundle.bundle import AlreadyInstalledException
from jarabe.model import mimeregistry
from jarabe.plugins.sn import SN_BROWSER_NAME, get_client
from jarabe.journal.journalentrybundle import JournalEntryBundle


_ = lambda x: dgettext('sugar-plugin-sn', x)

_logger = logging.getLogger('plugins.sn.bundleregistry')
_stub_icon_path = None
_online_cache = lrucache(64)


def stub_icon():
    global _stub_icon_path
    if not _stub_icon_path:
        theme = gtk.icon_theme_get_default()
        _stub_icon_path = theme.lookup_icon('empty', 0, 0).get_filename()
    return _stub_icon_path


class BundleRegistry(gobject.GObject):
    """Tracks the available activity bundles"""

    __gsignals__ = {
        'bundle-added': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                         ([gobject.TYPE_PYOBJECT])),
        'bundle-removed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                           ([gobject.TYPE_PYOBJECT])),
        'bundle-changed': (gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE,
                           ([gobject.TYPE_PYOBJECT])),
    }

    def __init__(self):
        gobject.GObject.__init__(self)
        self._bundles = {}
        self._favorites = {}

        favorite_activities = client.profile_path('favorite_activities')
        if exists(favorite_activities):
            try:
                with file(favorite_activities) as f:
                    self._favorites = json.load(f)
                    favorites = self._favorites.setdefault('favorites', {})
                    for key, bundle_info in favorites.items():
                        if not bundle_info:
                            bundle_info = favorites[key] = {}
                        bundle_info.setdefault('position', [-1, -1])
            except Exception:
                _logger.exception('Cannot process favorite_activities file')
        self._favorites.setdefault('favorites', {})

        get_client().connect('event', self.__Event_cb)
        self._populate()

    def __iter__(self):
        return self._bundles.itervalues()

    def __len__(self):
        return len(self._bundles)

    def get_bundle(self, context_guid):
        if context_guid == 'org.laptop.JournalActivity':
            return None
        elif context_guid == SN_BROWSER_NAME:
            return _BrowserInfo()

        if context_guid in self._bundles:
            return self._bundles[context_guid]

        if context_guid in _online_cache:
            return _online_cache[context_guid]

        try:
            props = get_client().get(['context', context_guid],
                    reply=['guid', 'favorite', 'clone', 'title'])
            bundle = _ContextInfo(props)
        except Exception:
            _logger.warning('Cannot fetch %r activity metadata', context_guid)
            bundle = None

        # Keep even None budles to avoid needless retrying
        _online_cache[context_guid] = bundle

        return bundle

    def get_activities_for_type(self, mime_type):
        result = []

        mime = mimeregistry.get_registry()
        default_bundle_id = mime.get_default_activity(mime_type)
        default_bundle = None

        for bundle in self._bundles.values():
            if mime_type in (bundle.get_mime_types() or []):
                if bundle.get_bundle_id() == default_bundle_id:
                    default_bundle = bundle
                else:
                    result.append(bundle)

        if default_bundle is not None:
            result.insert(0, default_bundle)

        return result

    def is_bundle_favorite(self, bundle_id, version):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return False
        return bundle.props['favorite']

    def set_bundle_favorite(self, bundle_id, version, favorite):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return
        get_client().put(['context', bundle_id], favorite, cmd='favorite')
        self.emit('bundle-changed', bundle)

    def get_bundle_position(self, bundle_id, version):
        key = ' '.join([bundle_id, version])
        if key in self._favorites['favorites']:
            return self._favorites['favorites'][key]['position']
        else:
            return -1, -1

    def set_bundle_position(self, bundle_id, version, x, y):
        key = ' '.join([bundle_id, version])
        bundle_info = self._favorites['favorites'].setdefault(key, {})
        bundle_info['position'] = [int(x), int(y)]
        with file(client.profile_path('favorite_activities'), 'w') as f:
            json.dump(self._favorites, f, indent=2)
        bundle = self._bundles.get(bundle_id)
        if bundle is not None:
            self.emit('bundle-changed', bundle)

    def is_activity_protected(self, bundle_id):
        # No need, is_user_activity() is enough
        return False

    def is_installed(self, bundle):
        if isinstance(bundle, ContentBundle) or \
                isinstance(bundle, JournalEntryBundle):
            return bundle.is_installed()

        installed = self._bundles.get(bundle.get_bundle_id())
        return installed is not None and \
                NormalizedVersion(bundle.get_activity_version()) == \
                NormalizedVersion(installed.get_activity_version())

    def install(self, bundle, uid=None, force_downgrade=False):
        if isinstance(bundle, JournalEntryBundle):
            bundle.install(uid)
            return
        elif isinstance(bundle, ContentBundle):
            bundle.install()
            return

        installed = self._bundles.get(bundle.get_bundle_id())
        if installed is not None:
            if not force_downgrade and \
                    NormalizedVersion(bundle.get_activity_version()) <= \
                    NormalizedVersion(installed.get_activity_version()):
                raise AlreadyInstalledException
            installed.uninstall()

        self._add_bundle(bundle.get_bundle_id(),
                {'favorite': False, 'clone': 2},
                ActivityBundle(bundle.install()))

    def uninstall(self, bundle, force=False, delete_profile=False):
        if isinstance(bundle, ContentBundle) or \
                isinstance(bundle, JournalEntryBundle):
            if bundle.is_installed():
                bundle.uninstall()
        else:
            bundle = self._bundles.get(bundle.get_bundle_id())
            if bundle is not None:
                bundle.uninstall()

    def upgrade(self, bundle):
        self.install(bundle)

    def _populate(self):
        response = get_client().get(['context'],
                reply=['guid', 'favorite', 'clone'],
                # TODO process result by portions less than 1024
                limit=1024, clone=2)
        for props in response['result']:
            if props['guid'] in self._bundles:
                continue
            self._add_bundle(props['guid'], props)

    def _add_bundle(self, bundle_id, props, activity_bundle=None):
        if activity_bundle is not None:
            bundle = _BundleInfo(activity_bundle)
        else:
            bundle = None
            for path in client.clones.walk(bundle_id):
                try:
                    bundle = _BundleInfo(ActivityBundle(path))
                    break
                except Exception:
                    _logger.exception('Cannot load %r bundle from %r',
                            bundle_id, path)
            if bundle is None:
                _logger.info('No bundles for %r', bundle_id)
                return None

        bundle.props = props
        self._bundles[bundle.get_bundle_id()] = bundle
        _logger.info('Add %r bundle', bundle.get_bundle_id())
        self.emit('bundle-added', bundle)

    def _remove_bundle(self, bundle_id):
        if bundle_id not in self._bundles:
            return
        _logger.info('Remove %r bundle', bundle_id)
        bundle = self._bundles.pop(bundle_id)
        self.emit('bundle-removed', bundle)

    def _set_favorite(self, bundle_id, favorite):
        bundle = self._bundles.get(bundle_id)
        if bundle is None:
            return
        bundle.props['favorite'] = favorite
        self.emit('bundle-changed', bundle)

    def _set_clone(self, bundle_id, clone):
        if clone:
            props = get_client().get(['context', bundle_id],
                    reply=['guid', 'favorite', 'clone'])
            bundle = self._bundles.get(bundle_id)
            if bundle is None:
                bundle = self._add_bundle(bundle_id, props)
            else:
                bundle.props = props
                self.emit('bundle-changed', bundle)
        else:
            self._remove_bundle(bundle_id)

    def __Event_cb(self, sender, event, data):
        if data.get('document') != 'context':
            return
        if event in ('create', 'update'):
            bundle_id = data['guid']
            props = data['props']
            if props.get('clone') in (0, 2):
                self._set_clone(bundle_id, props['clone'])
            if 'favorite' in props:
                self._set_favorite(bundle_id, props['favorite'])
        elif event == 'delete':
            bundle_id = data['guid']
            self._set_clone(bundle_id, 0)
        elif event == 'populate':
            self._populate()


class _ContextInfo(object):

    def __init__(self, props):
        self.props = props
        self._tmp_icon = None

    def get_name(self):
        return self.props['title']

    def get_bundle_id(self):
        return self.props['guid']

    def get_icon(self):
        if self._tmp_icon is None:
            blob = None
            try:
                blob = get_client().get(
                        ['context', self.get_bundle_id(), 'artifact_icon'])
            except Exception, error:
                _logger.debug('Fail to get icon for %r: %s',
                        self.get_bundle_id(), error)
            if not blob:
                self._tmp_icon = stub_icon()
            else:
                self._tmp_icon = util.TempFilePath()
                with file(self._tmp_icon, 'w') as f:
                    f.write(blob)

        # Cast to `str to avoid spreading `util.TempFilePath`
        return str(self._tmp_icon)

    def get_tags(self):
        # Doesn't matter with Sweets' features enabled
        return []

    def get_activity_version(self):
        return ''

    def get_installation_time(self):
        return 0

    def get_command(self):
        # Doesn't matter with Sweets' features enabled
        return ''

    def is_user_activity(self):
        return True

    def get_path(self):
        return '/'

    def get_mime_types(self):
        return []


class _BundleInfo(object):

    def __init__(self, bundle):
        self.props = {
                'guid': bundle.get_bundle_id,
                'favorite': False,
                }
        self._bundle = bundle

    def uninstall(self):
        _logger.debug('Uninstall %r', self._bundle.get_bundle_id())
        shutil.rmtree(self._bundle.get_path(), ignore_errors=True)

    def __getattr__(self, name):
        return getattr(self._bundle, name)


class _BrowserInfo(object):

    def __init__(self):
        icon = gtk.icon_theme_get_default().lookup_icon('sugar-network', 0, 0)
        self._icon_path = icon.get_filename() if icon is not None else stub_icon()

    def get_name(self):
        return _('Sugar Network')

    def get_bundle_id(self):
        return SN_BROWSER_NAME

    def get_icon(self):
        return self._icon_path

    def get_tags(self):
        return []

    def get_activity_version(self):
        return '0'

    def get_installation_time(self):
        return 0

    def get_command(self):
        return ''

    def is_user_activity(self):
        return False

    def get_path(self):
        return '/'
