# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import time
import gettext
import logging
from datetime import datetime
from os.path import dirname, join, isdir, isfile, exists

import dbus
import gobject

from sugar.env import get_profile_path
from sugar_network.toolkit import Option


plugins = Option(
        'space separated list of plugins to enable',
        type_cast=Option.list_cast, type_repr=Option.list_repr,
        default=[], name='plugins')

modules = []

_BWLIST_DIR = '/etc/sugar'

_SCHEDULE_DELAY = 180
_NM_STATE_CONNECTED = [3, 70]
_NM_STATE_DISCONNECTED = [4]

_logger = logging.getLogger('plugins')
_enabled_modules = []
_schedules = []
_plugins_srcroot = dirname(__file__)
_blacklist = None


def init():
    Option.seek('shell', [plugins])
    all_modules = []

    for filename in os.listdir(_plugins_srcroot):
        path = join(_plugins_srcroot, filename)
        if not isdir(path):
            continue
        locale_path = join(path, 'locale')
        if exists(locale_path):
            gettext.bindtextdomain('sugar-plugin-%s' % filename, locale_path)
        try:
            mod = __import__('jarabe.plugins.' + filename,
                    globals(), locals(), [filename])
        except ImportError, error:
            _logger.warning('Cannot import %r plugin: %s', filename, error)
            continue
        mod.name = mod.__name__.split('.')[-1]
        if set(['ORDER', 'init', 'start']) - set(dir(mod)):
            _logger.warning('Skip misconfigured %r plugin', mod.name)
            continue
        if hasattr(mod, 'TITLE'):
            modules.append(mod)
        if hasattr(mod, 'online_schedule'):
            _schedules.append(_Schedule(mod))
        all_modules.append(mod)

    # Load options only after importing all plugins to let them expose options
    Option.load([
        '/etc/sweets.conf',
        '~/.config/sweets/config',
        join(get_profile_path(), 'sweets.conf'),
        ])

    for mod in all_modules:
        if not hasattr(mod, 'TITLE') or mod.name in plugins.value:
            _enabled_modules.append(mod)
        else:
            _logger.info('Plugin %r does not contain CP sub-section', mod.name)

    modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))
    _enabled_modules.sort(lambda x, y: cmp(x.ORDER, y.ORDER))

    binding = []
    for mod in _enabled_modules:
        if hasattr(mod, 'binding'):
            binding.extend(mod.binding())
    binding = '\n'.join(binding)

    binding_path = join(get_profile_path(), 'plugins.binding')
    if exists(binding_path) != bool(binding) or \
            binding and file(binding_path).read() != binding:
        if binding:
            with file(binding_path, 'w') as f:
                f.write(binding)
        elif exists(binding_path):
            os.unlink(binding_path)
        _logger.info('Restart shell to source newly create bindings')
        file(join(get_profile_path(), 'restart'), 'w').close()
        exit(0)

    for mod in _enabled_modules:
        _logger.info('Initialize %r plugin', mod.name)
        mod.init()

    if _schedules:
        # Do not overload system on startup by running all schedules
        gobject.timeout_add_seconds(_SCHEDULE_DELAY, _connection_manager)


def start():
    for mod in _enabled_modules:
        _logger.info('Start %r plugin', mod.name)
        mod.start()


def blacklisted(category, component):
    _init_bw_lists()
    return component in _blacklist.get(category, [])


def blacklist(category, component):
    _init_bw_lists()
    _blacklist.setdefault(category, set())
    _blacklist[category].add(component)


def _init_bw_lists():
    global _blacklist

    if _blacklist is not None:
        return

    _blacklist = {}
    whitelist = _populate_list('whitelist')
    for category_, components_ in _populate_list('blacklist').items():
        components_ -= whitelist.get(category_, set())
        if components_:
            _blacklist[category_] = components_


def _populate_list(name):
    result = {}

    root = join(_BWLIST_DIR, name)
    if not exists(root):
        return result

    for filename in os.listdir(root):
        path = join(root, filename)
        if not isfile(path):
            continue

        with file(path) as f:
            for line in f:
                parts = line.split('#', 1)[0].split()
                if not parts:
                    continue
                if len(parts) == 2:
                    category, component = parts
                    result.setdefault(category, set())
                    result[category].add(component)
                else:
                    _logger.warning('Wrong formed %rs line: %r', name, line)

    return result


def _connection_manager():
    current_state = [False]

    def connected():
        if current_state[0]:
            return
        current_state[0] = True
        for i in _schedules:
            i.start()

    def disconnected():
        if not current_state[0]:
            return
        current_state[0] = False
        for i in _schedules:
            i.stop()

    def NetworkManagerStateChanged_cb(state):
        if state in _NM_STATE_CONNECTED:
            connected()
        elif state in _NM_STATE_DISCONNECTED:
            disconnected()

    try:
        bus = dbus.SystemBus()
        obj = bus.get_object('org.freedesktop.NetworkManager',
                '/org/freedesktop/NetworkManager')
        props = dbus.Interface(obj, dbus.PROPERTIES_IFACE)
        state = props.Get('org.freedesktop.NetworkManager', 'State')
        if state in _NM_STATE_CONNECTED:
            connected()
        bus.add_signal_receiver(NetworkManagerStateChanged_cb,
                'StateChanged', 'org.freedesktop.NetworkManager')
    except dbus.DBusException:
        _logger.exception('Cannot connect to NetworkManager')
        if _is_online():
            connected()


def _is_online():
    with file('/proc/net/route') as route:
        for line in route.readlines():
            try:
                if int(line.split('\t')[1], 16) == 0:
                    return True
            except ValueError:
                pass
    return False


class _Schedule(object):

    def __init__(self, mod, fallback_delay=0):
        self._mod = mod
        self._started = None

        self._checkpoint_path = \
                join(get_profile_path(), 'plugins-%s.checkpoint' % mod.name)
        if not exists(self._checkpoint_path):
            file(self._checkpoint_path, 'w').close()

        _logger.debug('Register %r schedule', mod.name)

    def start(self):
        if self._started:
            return
        _logger.debug('Start scheduling %r', self._mod.name)
        self._started = True
        gobject.idle_add(self._schedule)

    def stop(self):
        if not self._started:
            return
        _logger.debug('Stop scheduling %r', self._mod.name)
        if self._started is not True:
            gobject.source_remove(self._started)
        self._started = None

    def _schedule(self):
        checkpoint = int(os.stat(self._checkpoint_path).st_mtime)
        ts = int(time.time())

        if checkpoint > ts:
            timeout = max(1, checkpoint - ts)
            _logger.debug('Checkpoint for %r schedule expires in %s second(s)',
                    self._mod.name, timeout)
            gobject.timeout_add_seconds(timeout, self._schedule)
            return

        _logger.debug('Trigger %r schedule', self._mod.name)
        try:
            timeout = self._mod.online_schedule()
        except Exception:
            _logger.exception('Failed to trigger %r, abort schedule',
                    self._mod.name)
            return

        if not timeout:
            _logger.exception('Finish scheduling %r', self._mod.name)
            return

        checkpoint = ts + timeout
        os.utime(self._checkpoint_path, (checkpoint, checkpoint))
        _logger.debug('New checkpoint for %r schedule will be at %s',
                self._mod.name, datetime.fromtimestamp(checkpoint))
        timeout = max(1, checkpoint - int(time.time()))
        gobject.timeout_add_seconds(timeout, self._schedule)


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG)

    blacklist = _populate_list('blacklist')
    print 'blacklist:', blacklist
    for category, components in blacklist.items():
        for i in components:
            print '\t', category, i, blacklisted(category, i)

    whitelist = _populate_list('whitelist')
    print 'whitelist:', whitelist
    for category, components in whitelist.items():
        for i in components:
            print '\t', category, i, blacklisted(category, i)
