# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

import gtk
import wnck
import gconf

from sugar import wm
from sugar.graphics.xocolor import XoColor
from sugar.activity.activityfactory import create_activity_id

from jarabe.view.launcher import LaunchWindow
from jarabe.journal import model
from jarabe.journal.misc import launch as _launch
from jarabe.model import shell, bundleregistry
from jarabe.plugins.sn import connection, get_browser, report
from jarabe.plugins.sn.bundleregistry import stub_icon, ContextInfo


_logger = logging.getLogger('plugins.sn.launcher')


class Launcher(object):

    def __init__(self):
        self._launches = {}
        self._screen = wnck.screen_get_default()
        self._screen.connect('window-opened', self.__window_opened_cb)
        connection().connect('event', self.__Event_cb)

    def launch(self, bundle, activity_id=None, **kwargs):
        if not isinstance(bundle, ContextInfo):
            return _launch(bundle, activity_id, **kwargs)

        if activity_id:
            activity = shell.get_model().get_activity_by_id(activity_id)
            if activity is not None:
                _logger.debug('Resume %r activity', activity_id)
                activity.get_window().activate(gtk.get_current_event_time())
                return

        def found_jobjects(jobjects, total):
            if jobjects:
                props = jobjects[0]
                kwargs['object_id'] = props['uid']
                if 'icon-color' in props:
                    kwargs['color'] = XoColor(props['icon-color'])
            self._launch(bundle, activity_id, **kwargs)

        def not_found_jobjects(error):
            _logger.exception('Failed to launch %r: %s',
                    bundle.get_bundle_id(), error)

        if activity_id and 'object_id' not in kwargs:
            _logger.debug('Look for jobject for %r activity_id', activity_id)
            # pylint: disable-msg=W0212
            model._get_datastore().find({'activity_id': activity_id},
                    ['uid', 'icon-color'],
                    reply_handler=found_jobjects,
                    error_handler=not_found_jobjects,
                    byte_arrays=True)
        else:
            self._launch(bundle, activity_id, **kwargs)

    def _launch(self, bundle, activity_id, color=None, **kwargs):
        if not activity_id:
            activity_id = create_activity_id()
        _logger.info('Starting %r: activity_id=%r %r',
                bundle.get_bundle_id(), activity_id, kwargs)
        self._start_launcher(bundle.get_bundle_id(), bundle.get_icon(),
                activity_id, color)
        if color:
            kwargs['color'] = color.to_string()
        connection().get(['context', bundle.get_bundle_id()], cmd='launch',
                activity_id=activity_id, spawn=True, **kwargs)

    def _start_launcher(self, bundle_id, icon, activity_id, color):
        _logger.debug('Start %r launcher', activity_id)
        if color is None:
            gc = gconf.client_get_default()
            color = XoColor(gc.get_string('/desktop/sugar/user/color'))
        window = LaunchWindow(activity_id, icon, color)
        window.connect('realize', self.__window_realize_cb,
                bundle_id, activity_id)
        window.show()
        self._launches[activity_id] = window

    def _hide_launcher(self, activity_id):
        if activity_id not in self._launches:
            return None
        _logger.debug('Hide %r launcher', activity_id)
        self._launches[activity_id].hide()

    def __window_opened_cb(self, screen, window):
        if window.get_window_type() == wnck.WINDOW_NORMAL and \
                wm.get_sugar_window_type(window) != 'launcher':
            self._hide_launcher(wm.get_activity_id(window))

    def __window_realize_cb(self, widget, bundle_id, activity_id):
        wm.set_activity_id(widget.window, str(activity_id))
        widget.window.property_change('_SUGAR_WINDOW_TYPE', 'STRING', 8,
                gtk.gdk.PROP_MODE_REPLACE, 'launcher')
        wm.set_bundle_id(widget.window, str(bundle_id))

    def __Event_cb(self, sender, event, data):
        if data.get('cmd') != 'launch':
            return

        if event == 'launch':
            if data['activity_id'] not in self._launches:
                bundle = bundleregistry.get_registry().get_bundle(data['guid'])
                if bundle is None:
                    icon = stub_icon()
                else:
                    icon = bundle.get_icon()
                color = data.get('color')
                if color:
                    color = XoColor(color)
                self._start_launcher(data['guid'], icon,
                        data['activity_id'], color)

        elif event == 'exit':
            _logger.debug('Destroy %r launcher', data['activity_id'])
            self._launches.pop(data['activity_id']).destroy()

        elif event == 'failure':
            _logger.warning('Activity %r failed', data['activity_id'])
            window = self._launches.pop(data['activity_id'])
            report.ReportFrame(window, data)
            _reveal_window(window)


def _reveal_window(gtk_window):
    """Make window active

    In contrast with present(), brings window to the top
    even after invoking on response on non-gtk events.
    See #1423.

    """
    if gtk_window.window is None:
        gtk_window.show()
        return
    timestamp = gtk.get_current_event_time()
    if not timestamp:
        timestamp = gtk.gdk.x11_get_server_time(gtk_window.window)
    gtk_window.window.focus(timestamp)
