
/* Copyright (c) Miguel Angel Sepulveda, 1998. */

/* This program is freely distributable without licensing fees 
   and is provided without guarantee or warrantee expressed or 
   implied. This program is -not- in the public domain. */

#include "Gd.h" 
 
/* System Specifications */ 
#define TEMPERATURE    1.0  
#define FRICTION       1.0  
#define POLYMERLENGTH  14 
 
/* System state variables */ 
static double Time = 0.0;  
static double Angle = 0.0; 
static double Average_temperature = 0.0; 
static double Temperature = 0.0;  
static double KE = 0.0;  
 
const  int    nr_time_steps = 10; 
const  double time_step = 0.012;  
const  double time_increment = nr_time_steps * time_step; 
 
/* Physical System */ 
GdPolymer polymer(POLYMERLENGTH); 
GdPolymerDriver driver(polymer); 
  
/* Windows Environment Variables */ 
#define WIDTH    600              
#define HEIGHT   600  
#define WHITE    1.0, 1.0, 1.0, 1.0 
#define BLACK    0.0, 0.0, 0.0, 0.0 
 
#define wnRIGHT  1.0              // projection clipping planes    
#define wnLEFT  -1.0              // which define viewing volume   
#define wnTOP    1.0   
#define wnBOT   -1.0   
#define wnFAR   20.0   
#define wnNEAR   1.5   
 
/* These are windows handles */   
static int winIdMain; 
static int winIdSub;  
  
/* Light sources */  
GLfloat light_diffuse[] = {1.0F, 1.00F, 1.00F, 1.0F};    
GLfloat light_ambient[] = {1.0F, 1.0F, 1.0F, 1.0F};    
GLfloat light_position[] = {1.0F, 1.0F, 1.0F, 0.0F};    
GLUquadricObj *qobj;    
   
 
void drawString(char *s){  
  for (unsigned int i = 0; i < strlen(s); i++)  
	glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, s[i]);  
};  
 
 
void subDisplay(){ 
  char label[100];  
  
  glutSetWindow(winIdSub);	   
  glClearColor(0.25, 0.25, 0.25, 0.0); 
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); 
  glColor3f(0.0, 1.0, 0.0);  
  
  // Draw Box  
  glBegin(GL_LINE_LOOP);  
  glVertex2f(0.0F, 0.0F);  
  glVertex2f(0.0F, 0.99F);  
  glVertex2f(0.999F, 0.99F);  
  glVertex2f(0.999F, 0.0F);  
  glEnd();  
   
  // Draw Text info  
  glColor3f(1.0F, 1.0F, 1.0F);  
  sprintf(label, "Time = %8.3f ", Time);  
  glRasterPos2f(0.05F, 0.75F);  
  drawString(label);  
  
  sprintf(label, "Temperature  = %8.3f ", Temperature);  
  glRasterPos2f(0.05F, 0.55F);  
  drawString(label);  
  
  sprintf(label, "Average Temperature  = %8.3f ", Average_temperature);  
  glRasterPos2f(0.05F, 0.35F);  
  drawString(label);  
 
  sprintf(label, "Solution Temperature  = %8.3f ", TEMPERATURE);  
  glRasterPos2f(0.05F, 0.15F);  
  drawString(label);  
  
  sprintf(label, "Kinetic Energy = %8.3f ", KE);  
  glRasterPos2f(0.35F, 0.55F);  
  drawString(label);  
 
  sprintf(label, "Angle = %8.3f ", Angle);  
  glRasterPos2f(0.35F, 0.35F);  
  drawString(label);  
 
  sprintf(label, "Solution Friction = %8.3f ", FRICTION);  
  glRasterPos2f(0.35F, 0.15F);  
  drawString(label);  
  
  glutSwapBuffers(); 
 
}; 
   
 
void scene(){  
 
  GdRenderer::draw(polymer);  
  
};  
 
 
void mainDisplay(){  
  glutSetWindow(winIdMain); 


  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);  
      // Clear the color and depth buffers    
      // This is like cleaning up the blackboard    
        
  // CONTINUE MODELVIEW TRANSFORMATION: Set and orient the camera    
  glLoadIdentity();                // Load unity on current matrix
  glTranslatef(0.0, 0.0, -4.0);    // Move the camera 4 steps back    
      // we leave the camera pointing in the -z direction. Actually    
      // this operates by moving the following scene 4 steps in the -z    
      // direction. Only the elements of the scene that fall inside the    
      // viewing volume (see projection transformation later) will appear     
      //on screen.    
      
  // Render polymer  
  glScalef(0.5, 0.5, 0.5); 
  glRotatef(Angle, 0, 1, 0); 
  scene();  
 
  glutSwapBuffers(); 
}; 
 
  
void subReshape(int w, int h){  
  glViewport(0, 0, w, h); 
  glMatrixMode(GL_PROJECTION);   
  glLoadIdentity();     
  gluOrtho2D(0.0F, 1.0F, 0.0F, 1.0F); 
};  
 
   
void mainReshape(int w, int h){  
 
  // VIEWPORT TRANSFORMATION
  glViewport(0, 0, w, h);  

  // PROJECTION TRANSFORMATION
  glMatrixMode(GL_PROJECTION);   
  glLoadIdentity();     
  glFrustum(wnLEFT, wnRIGHT, wnBOT, wnTOP, wnNEAR, wnFAR);   

  // MODELVIEW TRANSFORMATION
  glMatrixMode(GL_MODELVIEW);   
  
  glutSetWindow(winIdSub);  
  glutReshapeWindow(w-10, h/10);  
  glutPositionWindow(5, 5);  
  glutSetWindow(winIdMain);  
};  
   
void idle(){  
  static int count = 0; 
  static double sum_temp = 0.0; 
 
  // Update time  
  Time += time_increment;  
  count++; 
 
  // Polymer Dynamics 
  driver.propagate(nr_time_steps); 
  polymer.moveTo(0.,0.,0.); 
  KE = GdProperties::kineticEnergy(polymer); 
  Temperature = GdProperties::temperature(polymer); 
  sum_temp +=  Temperature; 
  Average_temperature = sum_temp/count; 
  Angle += 1.0; 
 
  // Update both windows  
  glutSetWindow(winIdMain);  
  glutPostRedisplay();  
  glutSetWindow(winIdSub);  
  glutPostRedisplay();  
  
};  
  
  
void fxInit(){  
  GdRenderer::initialize(); 
  qobj = gluNewQuadric();  
  gluQuadricDrawStyle(qobj, (enum GLenum) GLU_FILL);    
     
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);    
  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);    
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);    
  glEnable(GL_LIGHTING);    
  glEnable(GL_LIGHT0);    
  glEnable(GL_DEPTH_TEST);    
  glEnable(GL_NORMALIZE); 
};  
 
 
 
 
void keyboard(unsigned char key, int, int){ 
  static bool background_f = true; 
  static bool info_f = true; 
 
  switch (key){ 
  case 'b':  
    if(background_f){ 
      glClearColor(WHITE); 
    }else{ 
      glClearColor(BLACK); 
    }; 
    background_f = !background_f; 
    break; 
 
  case 'q': case 'Q':  
    exit(0); 
 
  case 'i': 
    if(info_f){ 
      glutSetWindow(winIdSub); 
      glutHideWindow(); 
    } else { 
      glutSetWindow(winIdSub); 
      glutShowWindow(); 
    }; 
    info_f = !info_f; 
    break; 
  }; 
}; 
 
 
 
static const double epsilon = 1.0; 
static const double sigma = 1.0; 
 
double LennardJonesForce(double r2){ 
  static double rc = sigma * pow(2.0, 0.16666666); 
  static double rc2 = rc * rc; 
  static double e24 = 24.0 * epsilon; 
  static double s2 = sigma * sigma; 
 
  if(r2 > rc2) return 0.0; 
 
  double r1 = s2/r2; 
  double c6 = pow(r1, 3.0); 
  double c12 = c6 * c6; 
  double f; 
  f = c6 - 2.0 * c12; 
  return e24 * f/r2; 
}; 
 
 
double FeneForce(double r2){ 
  static double R0 = 1.5 * sigma; 
  static double R02 = R0 * R0; 
  static double k = 30.0 * epsilon; 
 
 
  if(r2 > R02) return 0.0; 
 
  double ratio2 = r2/R02; 
  double f; 
  f = k /(1.0 - ratio2); 
  return f; 
}; 
 
 
int main (int argc, char *argv[]){ 
  // polymer initializations 
  polymer.linear(0.99); 
 
  driver.initRepelFunc(LennardJonesForce, 200, 0.2, 1.5); 
  driver.initAttrcFunc(FeneForce, 200, 0.01, 1.498); 
  driver.setCutoff(1.5, 2.5); 
  driver.setTemperature(TEMPERATURE); 
  driver.setFriction(FRICTION); 
  driver.setTimeStep(time_step);  
   
  /* Glut initializations */ 
  glutInit(&argc, argv); 
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH); 
  glutInitWindowPosition(5, 5); 
  glutInitWindowSize(WIDTH, HEIGHT); 
     
  winIdMain = glutCreateWindow("Polymer");  
  glutDisplayFunc(mainDisplay); 
  glutReshapeFunc(mainReshape);  
  glutKeyboardFunc(keyboard); 
  glutIdleFunc(idle);  
  fxInit(); 
 
  winIdSub = glutCreateSubWindow(winIdMain, 5, 5, WIDTH-10, HEIGHT/10); 
  glutDisplayFunc(subDisplay); 
  glutReshapeFunc(subReshape);  
 
  glutMainLoop();  
 
  return 1; 
}; 
