/*****************************************************************************
  Module: mpgrecorder.cpp - Implements the MPEGrecorder class.
  Copyright (C) 1999  Andrew L. Sandoval

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/
/* Module: mpgrecorder.cpp                                                   */
/* Developer: Andrew L. Sandoval                                             */
/* Purpose: Implements the MPEGrecorder class.  MPEGrecorder is invoked to   */
/* record MPEG content from an MPEG capture card (i.e. Antex, Snazzi, etc.)  */
/* Depending on the Constructor selected output can be sent directly to an   */
/* open file HANDLE for network streaming, etc., or a file will be opened.   */
/* By default C:\temp\date-time.mp3 will be used.                            */
/*****************************************************************************/
#include "mpgrecorder.h"
#include "mpgformat.h"
#include <sys/types.h>
#include <sys/stat.h>

//
// Supporting utility functions:
//
HWAVEIN createWaveHandle( DWORD bitrate, DWORD sampFreq, UINT deviceId,
						  MMRESULT* lastResult, HANDLE event,
						  int *normalWave, WORD layer)
{
 const MPEG3waveformat format(bitrate, sampFreq, layer);
 HWAVEIN hdev = NULL;
 MMRESULT mmr;

 //See if there is a device that supports MP3 already
 //If not, use a NORMAL Wave and SPECIFY AN ENCODER
 if(normalWave) *normalWave = 0;
 mmr = waveInOpen(&hdev, deviceId, format.waveformat(), 
				 reinterpret_cast<DWORD>(event),
		         0, CALLBACK_EVENT);
 if(lastResult) *lastResult = mmr;
 if(mmr !=MMSYSERR_NOERROR) 
 {
	 cerr << "There are no devices which support WAVE_FORMAT_MPEG.LAYER3 (" << mmr << ")" << endl;
	 cerr << "Looking for a device which supports WAVE_FORMAT_PCM..." << endl;
	 const NORMALwaveformat normalFormat(bitrate, sampFreq);
	 mmr = waveInOpen(&hdev, deviceId, normalFormat.waveformat(),
					  reinterpret_cast<DWORD>(event),
					  0, CALLBACK_EVENT);
	 if(lastResult) *lastResult = mmr;
	 if(mmr!=MMSYSERR_NOERROR)
	 {
		 cerr << "There are no devices which support WAVE_FORMAT_PCM (" << mmr << ")" << endl;
		 return NULL;
	 }
	 if(normalWave) *normalWave = 1;
 }
 return hdev;
}

void addDateFilename(string &addTo)
{
 struct _stat st;
 char *tempdir = "";

 if(_stat("C:\\temp", &st)!=-1) tempdir = "C:\\temp";
 else
 {
	tempdir = getenv("TEMP");
	if(!tempdir)
	{
		tempdir = "";  //Current Directory
	}
 }
 char *dslash = "\\";	
 if(strlen(tempdir) && tempdir[strlen(tempdir)-1] == '\\') dslash = "";
	
 SYSTEMTIME now;
 GetLocalTime(&now);

 ostringstream fg;	
 fg << tempdir << dslash << now.wMonth << "-" << now.wDay << "-"
    << now.wYear << "-" << now.wHour << "." << now.wMinute 
	<< ".mp3" << ends;
 addTo += fg.str();
}

HANDLE createEncoderPipe(int normalWave, DWORD bitrate, DWORD sampFreq,
						 HANDLE *procReadPipe, PROCESS_INFORMATION *cpi,
						 const char *filename)
{
 if(!normalWave) return NULL;
 string parms( "lame.exe -x -r -ms -b " );  

 unsigned long lameBitrate = bitrate / 1000;
 ostringstream brate;
 
 brate << lameBitrate;
 parms += string(brate.str());
 parms += " - ";  //stdin
 if(filename) parms += filename;
 else addDateFilename( parms );
  
 cerr << endl << "Streaming WAV output to:" << endl << parms << endl;

 HANDLE readPipe;
 HANDLE writePipe;

 PSECURITY_DESCRIPTOR pSD;
 SECURITY_ATTRIBUTES  sa;

 pSD = (PSECURITY_DESCRIPTOR)LocalAlloc(LPTR, 
	                                    SECURITY_DESCRIPTOR_MIN_LENGTH);
 InitializeSecurityDescriptor(pSD, SECURITY_DESCRIPTOR_REVISION);
 SetSecurityDescriptorDacl(pSD, TRUE, (PACL)NULL, FALSE);
 
 sa.nLength = sizeof(sa);
 sa.lpSecurityDescriptor = pSD;
 sa.bInheritHandle = TRUE;

 CreatePipe( &readPipe, &writePipe, &sa, 0 );
 if(procReadPipe) *procReadPipe = readPipe;

 STARTUPINFO si;
 PROCESS_INFORMATION mypi;

 memset( reinterpret_cast<void*>(&si), 0, sizeof(si) );
 memset( reinterpret_cast<void*>(&mypi), 0, sizeof(mypi) );
 si.cb = sizeof(si);
 si.hStdInput = readPipe;
 si.hStdOutput = GetStdHandle(STD_OUTPUT_HANDLE); //outputHandle;
 si.hStdError = GetStdHandle(STD_ERROR_HANDLE);
 si.dwFlags = STARTF_USESTDHANDLES;
 
 CreateProcess( NULL, const_cast<char*>(parms.c_str()), NULL, NULL,
	            TRUE, CREATE_NEW_PROCESS_GROUP, 
				NULL, NULL, &si, &mypi);

 if(cpi) memcpy(reinterpret_cast<void*>(cpi), 
	           reinterpret_cast<void*>(&mypi), sizeof(mypi));

 WaitForInputIdle(mypi.hProcess, 2000);
 //
 // Write Wav Header:
 //
 //struct WaveHeader genericWaveHeader = 
 //	{ WAV_ID_RIFF, 0xFFFFFFFF, WAV_ID_WAVE,
 //   WAV_ID_FMT, 16, WAV_ID_PCM, 2, sampFreq, 
 //	  (4 * sampFreq), 4, 16, WAV_ID_DATA, 0xFFFFFFFF };

 //DWORD written = 0;
 //WriteFile( writePipe, &genericWaveHeader, sizeof(genericWaveHeader), 
 //	        &written, NULL);
 return writePipe;
}


HANDLE createUnnamedFile(int normalWave)
{
 if(normalWave) return NULL;
 string filename;
 
 addDateFilename( filename );
 HANDLE unnamed = CreateFile(filename.c_str(),
		              GENERIC_WRITE, 0, NULL,
					  CREATE_ALWAYS, 
					  FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN, 
					  NULL);
 return unnamed;
}

HANDLE createNamedFile(const char *filename, int normalWave)
{
	if(normalWave) return NULL;
	return CreateFile(filename,
		              GENERIC_WRITE, 0, NULL,
					  CREATE_ALWAYS, 
					  FILE_ATTRIBUTE_NORMAL|FILE_FLAG_SEQUENTIAL_SCAN, 
					  NULL);
}


WAVEHDR * addBuffer(MPEGrecorder *mpgRec, DWORD blockSize, bool add = FALSE)
{
	MMRESULT mmr;

	if(!mpgRec) return NULL;

	WAVEHDR * prepared = new WAVEHDR;
	memset(prepared, 0, sizeof(WAVEHDR));
	prepared->dwBufferLength = blockSize;
	prepared->lpData = new char[ blockSize +1 ];

	mmr = waveInPrepareHeader(mpgRec->getDeviceHandle(), prepared, sizeof(WAVEHDR));
	mpgRec->SetLastResult(mmr);
	
	if(add)
	{
		mmr = waveInAddBuffer(mpgRec->getDeviceHandle(), prepared, sizeof(WAVEHDR));
		mpgRec->SetLastResult(mmr);
	}
	return prepared;
}

unsigned long processFullBuffer(MPEGrecorder *mpgRec)
{
 if(!mpgRec) return 0;
 WAVEHDR *process = NULL;
 MMRESULT mmr;
 unsigned long count = 0;

 process = mpgRec->GetFullBuffer(&count);

 if(process)
 {
	 mmr = waveInUnprepareHeader(mpgRec->getDeviceHandle(), process, sizeof(WAVEHDR));
	 mpgRec->SetLastResult(mmr);

	 if(mmr == MMSYSERR_NOERROR && process->dwBytesRecorded)
	 {
		 //
		 // Save buffer  OR pipe to the MP3 encoder
		 //
		 DWORD written = 0;
		 HANDLE writeHandle = (mpgRec->isNormalWave() ? 
			                     mpgRec->GetEncoderPipe() :
		                         mpgRec->GetOutputHandle());

		 WriteFile(writeHandle, process->lpData, 
		           process->dwBytesRecorded, &written, NULL);
	 }
	 if(process->lpData) delete[] process->lpData;
	 delete process;
 }

 return count;
}

//////////
//Thread to ADD buffers:
/////////
DWORD WINAPI handleFullBuffers(LPVOID arg)
{
 MPEGrecorder * mpgRec = reinterpret_cast<MPEGrecorder*>(arg);
 unsigned long count = 0;
 
 while(mpgRec->Recording())
 {
	 count = processFullBuffer(mpgRec);
	 if(!count) Sleep(20); //Yield this thread
 }
 count = processFullBuffer(mpgRec);
 return count;
}

//////////
//Thread to ADD buffers:
/////////
DWORD WINAPI manageBuffers(LPVOID arg)
{
	MPEGrecorder * mpgRec = reinterpret_cast<MPEGrecorder*>(arg);

	WAVEHDR *prepared = NULL;
	WAVEHDR *process = NULL;
	DWORD blockSize = mpgRec->getBlockSize();
	MMRESULT mmr;
	int recordingStarted = 0;

	while(mpgRec->Recording())
	{
		prepared = addBuffer(mpgRec, blockSize);
		
		WaitForSingleObject(mpgRec->GetBufferEvent(), INFINITE);
		
		if(!recordingStarted)
		{
			process = addBuffer(mpgRec, blockSize, TRUE);
			recordingStarted++;
			mmr = waveInStart(mpgRec->getDeviceHandle());
			mpgRec->SetLastResult(mmr);
			continue;  //The event is signalled on OPEN
		}
		//
		//Otherwise the device is open and a new buffer is ready: 
		// Add prepared buffer
		// Save process buffer
		// Next process buffer = prepared
		//
		mmr = waveInAddBuffer(mpgRec->getDeviceHandle(), prepared, sizeof(WAVEHDR));
		mpgRec->SetLastResult(mmr);

		mpgRec->AddFullBuffer( process );  //Hand this buffer off

		process = prepared;
		mpgRec->RunUserCallback();
	}
	waveInStop(mpgRec->getDeviceHandle());
	Sleep( 250 );
	while(processFullBuffer(mpgRec)) cerr << "*";  //Get last buffer(s)
	if(mpgRec->isNormalWave()) 
	{
		cerr << endl << "Done processing... Cleaning up..." << endl;
		CloseHandle( mpgRec->GetEncoderPipe());
		mpgRec->shutdownChild();
	}
	SetEvent(mpgRec->GetRecordEndEvent());
	return NULL;
}

//
// MPEGrecorder members:
//
MPEGrecorder::~MPEGrecorder()
{
	if(this->recording) 
	{
		this->recording = FALSE;
		waveInStop(this->hdev);
		Sleep(200);
	}
	if(this->closeOnDestroy) CloseHandle(this->writeToHandle);
	if(this->hdev) waveInClose(this->hdev);
	SetEvent(this->notifyUser);
	CloseHandle(this->bufferFullEvent);
	CloseHandle(this->notifyUser);
}

WAVEHDR * MPEGrecorder::GetFullBuffer(unsigned long *cnt)
{
 WAVEHDR *process = NULL;
 
 WaitForSingleObject(dequeMutex, INFINITE);
 unsigned long count = fullBuffers.size();
 if(count)
 {
	 process = fullBuffers[0];
	 fullBuffers.pop_front();
 }
 ReleaseMutex(dequeMutex);
 if(cnt) *cnt = count;
 return process;
}

MPEGrecorder& MPEGrecorder::AddFullBuffer(WAVEHDR *in)
{
	WaitForSingleObject(dequeMutex, INFINITE);
	fullBuffers.push_back( in );
	ReleaseMutex(dequeMutex);
	return *this;
}

MPEGrecorder& MPEGrecorder::Start()
{
	if(this->recording) return *this;
	if(!this->hdev)
	{
		cerr << "The selected device does not support MPEG encoding." << endl;
		this->recording = FALSE;
		SetEvent(this->notifyUser);  //Tell the user we are bailing...
		return *this;
	}	
	this->recording = TRUE;

	DWORD handleFullBuffer_tid;	
	CreateThread(NULL, 0, handleFullBuffers, 
		         reinterpret_cast<LPVOID>(this), 0, &handleFullBuffer_tid);
	
	DWORD addBuffer_tid;	
	CreateThread(NULL, 0, manageBuffers, 
		         reinterpret_cast<LPVOID>(this), 0, &addBuffer_tid);
	
	return *this;
}

MPEGrecorder& MPEGrecorder::Stop()
{
	if(!this->recording) return *this;
	this->recording = FALSE;
	return *this;
}

//
// Run the UserCallback in a new thread.  We do not want bad
// code in the callback to hang the buffering process.
//

DWORD WINAPI UserCallbackThreadMPG(LPVOID arg)
{
	UserCallBackDataBlockMPG *cb = reinterpret_cast<UserCallBackDataBlockMPG*>(arg);
	if(cb!=NULL && cb->callback!=NULL && cb->recorder!=NULL)
	{
		MPEGrecorder *mpgRec = cb->recorder;
		(cb->callback)(*mpgRec);
	}
	delete cb;
	return NULL;
}

MPEGrecorder& MPEGrecorder::SaferUserCallback()
{
	if(callback)
	{
		UserCallBackDataBlockMPG *cb = new UserCallBackDataBlockMPG;
		if(cb)
		{
			cb->callback = callback;
			cb->recorder = this;
			DWORD tid;
			CreateThread(NULL, 0, UserCallbackThreadMPG, 
						 reinterpret_cast<LPVOID>(cb), 0, &tid);
		}	
	}
	return *this;
}

MPEGrecorder& MPEGrecorder::shutdownChild()
{
	if(this->normalWave)
	{
		CloseHandle( this->childReadPipe );
		TerminateThread( this->child_pi.hThread, 0);
		CloseHandle( this->child_pi.hThread );
		TerminateProcess( this->child_pi.hProcess, 0);
		CloseHandle( this->child_pi.hProcess );
		CloseHandle( GetStdHandle(STD_ERROR_HANDLE) );  //?
		CloseHandle( GetStdHandle(STD_OUTPUT_HANDLE) ); //?
	}
	return *this;
}
