/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef __DJGPP__
#  define _USE_LIBM_MATH_H 1
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "kalc.h"
#include "realfunctions.h"
#include "mem.h"


void exitAbort(void)
{
  /*
   * This function is called by the user command `abort'. It simply
   * exits, without doing finalization.
   */

  printf("Warning: status file not saved\n");
  exit(3);
}


void f_ip(void)
{
  /*
   * This function calls the _f_ip function through the wrapper.
   */

  run1_1_Function(_f_ip, "ip");
}


Object _f_ip(Object n, int *err)
{
  /*
   * This function returns the integer part of its argument.
   */
  
  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_ip(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_fp(void)
{
  /*
   * This funciton calls the _f_fp function through the wrapper.
   */
  
  run1_1_Function(_f_fp, "fp");
}


Object _f_fp(Object n, int *err)
{
  /*
   * This function returns the fractional part of its argument.
   */
  
  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_fp(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_rand(void)
{
  /*
   * This function inserts a pseudo-random real number between 0 and 1
   * in the stack.
   */

  insertReal((double)rand() / RAND_MAX);
}


void f_rdz(void)
{
  /*
   * This function calls the _f_rdz function through the wrapper.
   */
  
  run1_0_Function(_f_rdz, "rdz");
}


int _f_rdz(Object n)
{
  /*
   * This function sets the random number generator seed to its argument,
   * or to a value based on the time if 0 is given as argument.
   */
  
  if (n.type == TYPE_REAL) {
    /* If argument is zero, use time as seed */
    if (n.value.real == 0)
      srand(time(NULL));
    else
      srand((unsigned int)(n.value.real * RAND_MAX));
    return ERR_NOERR;
  } else
    return ERR_BADARGUMENTTYPE;
}


void f_gamma(void)
{
  /*
   * This function calls the _f_gammap1 function through the wrapper.
   */

  run1_1_Function(_f_gammap1, "!");
}


Object _f_gammap1(Object n, int *err)
{
  /*
   * This function returns gamma(x + 1).
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_gammap1(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_lgamma(void)
{
  /*
   * This function calls the _f_lgamma function through the wrapper.
   */
  
  run1_1_Function(_f_lgamma, "lgamma");
}


Object _f_lgamma(Object n, int *err)
{
  /*
   * This function returns the logarithm of the gamma function of its
   * argument.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_lgamma(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_perm(void)
{
  /*
   * This function calls the _f_perm function through the wrapper.
   */
  
  run2_1_Function(_f_perm, "perm");
}


Object _f_perm(Object n, Object p, int *err)
{
  /*
   * This function returns the number of permutations of n objects, taken
   * p by p.
   *
   * Definition:
   *    perm(n, p) = n! / (n - p)!
   *
   * Errors for non-integer arguments
   */
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = (re_fp(n.value.real) || re_fp(n.value.real))
      ? ERR_BADARGUMENTVALUE : ERR_NOERR;
    n.value.real = re_perm(n.value.real, p.value.real);
  } else 
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_comb(void)
{
  /*
   * This function calls the _f_comb function through the wrapper.
   */
  
  run2_1_Function(_f_comb, "comb");
}


Object _f_comb(Object n, Object p, int *err)
{
  /*
   * This function returns the number of combinations of n elements taken
   * p by p.
   *
   * Definition:
   *    comb(n, p) = n! / (p!(n - p)!)
   *
   * Errors for non-integer arguments
   */
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = (re_fp(n.value.real) || re_fp(n.value.real))
      ? ERR_BADARGUMENTVALUE : ERR_NOERR;
    n.value.real = re_comb(n.value.real, p.value.real);
  } else 
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_min(void)
{
  /*
   * This function calls the _f_min function through the wrapper.
   */

  run2_1_Function(_f_min, "min");
}


Object _f_min(Object n, Object p, int *err)
{
  /*
   * This function returns the smallest of its arguments.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_min(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_max(void)
{
  /*
   * This function calls the _f_max function through the wrapper.
   */

  run2_1_Function(_f_max, "max");
}


Object _f_max(Object n, Object p, int *err)
{
  /*
   * This function returns the gratest of its arguments.
   */
  
  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_max(n.value.real, p.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_sign(void)
{
  /*
   * This function calls the _f_sign function through the wrapper.
   */

  run1_1_Function(_f_sign, "sign");
}


Object _f_sign(Object n, int *err)
{
  /*
   * This function returns the sign function of its argument.
   *
   * Definition:
   *    sign(x) = x/|x|
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_sign(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sign(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_psign(void)
{
  /*
   * This function calls the _f_psign function through the wrapper.
   */
  
  run1_1_Function(_f_psign, "psign");
}


Object _f_psign(Object n, int *err)
{
  /*
   * This function returns -1 for negative arguments, or 1 for non-negative
   * arguments.
   */
   
  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_psign(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_mant(void)
{
  /*
   * This function calls the _f_mant function through the wrapper.
   */
  
  run1_1_Function(_f_mant, "mant");
}


Object _f_mant(Object n, int *err)
{
  /*
   * This function returns the mantissa of its argument.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_mant(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_xpon(void)
{
  /*
   * This function calls the _f_xpon function through the wrapper.
   */

  run1_1_Function(_f_xpon, "xpon");
}


Object _f_xpon(Object n, int *err)
{
  /*
   * This function returns the exponent of its argument.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_xpon(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_rnd(void)
{
  /*
   * This function calls the _f_rnd function through the wrapper.
   */

  run2_1_Function(_f_rnd, "rnd");
}


Object _f_rnd(Object n, Object p, int *err)
{
  /*
   * This function round n to p decimal places.
   * Complex numbers have their real and imaginary places rounded.
   */

  if (type(p) == TYPE_REAL) {
    /* Adjust p if necessary. */
    if (p.value.real < 0)
      p.value.real = 0;
    
    switch (type(n)) {
    case TYPE_REAL:
      *err = ERR_NOERR;
      n.value.real = re_rnd(n.value.real, p.value.real);
      break;

    case TYPE_CMP:
      *err = ERR_NOERR;
      n.value.cmp.re = re_rnd(n.value.cmp.re, p.value.real);
      n.value.cmp.im = re_rnd(n.value.cmp.im, p.value.real);
      break;

    default:
      *err = ERR_BADARGUMENTTYPE;
      break;
    }
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_type(void)
{
  /*
   * This function calls the _f_type function through the wrapper.
   */

  run1_1_Function(_f_type, "type");
}


Object _f_type(Object n, int *err)
{
  /*
   * This function returns the type of the element.
   */

  *err = ERR_NOERR;
  n.value.real = (double) n.type;
  n.type = TYPE_REAL;
  return n;
}


void
f_vtype(void)
{
  /*
   * This functions calls the _f_vtype function through the wrapper.
   */

  run1_1_Function(_f_vtype, "vtype");
}


Object
_f_vtype(Object n, int *err)
{
  /*
   * Returns the type of the object in the given variable name, or -1
   * if that variable doesn't exist.
   */

  Object *ob;
  Object retVal;

  if (type(n) != TYPE_ID) {
    *err = ERR_BADARGUMENTTYPE;
    return n;
  }

  ob = recallObject(n.value.str);

  retVal.type = TYPE_REAL;
  retVal.value.real = ob ? type(*ob) : -1;

  *err = ERR_NOERR;
  return retVal;
}



void doShell(void)
{
  /*
   * Executes the rest of the command line, or an interactive shell if
   * there is nothing else.
   * Does not show stack.
   */
  
  char *cmdline = getTillEOL();
  int exitValue;
  char wait[10];

  linesToDisplay = 0;

  /* Remove initial space from command line */
  ltrim(cmdline);

  /*
   * NOTE: The DJGPP compiler's library function system does exactly what
   * I want: when there is a command line, the command is executed, and
   * if the command line is empty, the command processor is executed
   * intereactively. (Actually, the function is even more smart than that.)
   *
   * However, other libraries may not do this (my Linux library doesn't),
   * so I use a dirty trick: change the command line to "/bin/sh" (actually
   * the value of the SHELL environment variable, so that the full command
   * line is something like "sh -c /bin/sh". Ugly, but works.
   *
   * You might have to change the code below depending on the behavior of
   * your library. If you use any system other than MSDOS, and either
   * the SHELL environment variable is set or /bin/sh is a shell, this will
   * work. For other DOS compiler's libraries, you will have to check
   * the documentation, and change bellow accordingly.
   */
#ifndef MSDOS
  if (!*cmdline) {
    cmdline = getenv("SHELL");
    if (!cmdline)
      cmdline = "/bin/sh";
  }
#endif

  exitValue = system(cmdline);

  printf("-- Exited with code %d. Press ENTER to continue.", exitValue);
  fgets(wait, 9, stdin);
}


void f_nan(void) 
{
  /*
   * This function inserts NaN (Not-A-Number) in the stack.
   */

  insertReal(0./0.);
}


void f_inf(void) 
{
  /*
   * This function inserts Inf (positive infinity) in the stack.
   */

  insertReal(HUGE_VAL);
}


void f_minusInf(void) 
{
  /*
   * This function inserts -Inf (negative infinity) in the stack.
   */

  insertReal(-HUGE_VAL);
}


void f_eval(void)
{
  /*
   * This function evaluates the given object
   */

  if (!enoughArgs(1))
    doError("eval", ERR_TOOFEWARGUMENTS);

  if (type(**tos) == TYPE_ID) {
    char *idName = strdup((**tos).value.str);
    _f_drop();
    doUnquoted(idName);
  } else if (type(**tos) == TYPE_FUNC) {
    void (*f)(void) = (**tos).value.func;
    _f_drop();
    (*f)();
  } else if (type(**tos) == TYPE_TAGGED) {
    dtag(tos);
    f_eval();
  } else if (type(**tos) == TYPE_PROG) {
    Object *obj = *tos--;
    evalProgram(obj->value.comp);
    freeObjectSpecials(*obj);
    free(obj);
  }
}


void f_lastarg(void) 
{
  /*
   * This function inserts the objects saved in the last args area in
   * the stack.
   */

  if (_f_depth() < currStackCap - lastArgN) {
    register int i;
    for (i = lastArgN - 1; i >= 0; i--)
      insertObject(*lastArgs[i]);
  }
}
