/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "kalc.h"
#include "realfunctions.h"


void f_pi(void)
{
  /*
   * This command inserts the constant pi in the stack.
   */

  insertReal(PI);
}


void f_sin(void)
{
  /*
   * This command calls the _f_sin command through the wrapper.
   */
  
  run1_1_Function(_f_sin, "sin");
}


Object _f_sin(Object n, int *err)
{
  /*
   * This function returns the sine of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_sin(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sin(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_cos(void)
{
  /*
   * This function calls the _f_cos function through the wrapper.
   */

  run1_1_Function(_f_cos, "cos");
}


Object _f_cos(Object n, int *err)
{
  /*
   * This function returns the co-sine of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_cos(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_cos(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_tan(void)
{
  /*
   * This function calls the _f_tan function through the wrapper.
   */

  run1_1_Function(_f_tan, "tan");
}


Object _f_tan(Object n, int *err)
{
  /*
   * This function returns the tangent of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_tan(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_tan(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_sec(void)
{
  /*
   * This function calls the _f_sec function through the wrapper.
   */

  run1_1_Function(_f_sec, "sec");
}


Object _f_sec(Object n, int *err)
{
  /*
   * This function returns the secant of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_sec(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_sec(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_csc(void)
{
  /*
   * This function calls the _f_csc function through the wrapper.
   */

  run1_1_Function(_f_csc, "csc");
}


Object _f_csc(Object n, int *err)
{
  /*
   * This function returns the co-secant of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_csc(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_csc(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_cot(void)
{
  /*
   * This function calls the _f_cot function through the wrapper.
   */

  run1_1_Function(_f_cot, "csc");
}


Object _f_cot(Object n, int *err)
{
  /*
   * This function returns the co-tangent of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_cot(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_cot(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_asin(void)
{
  /*
   * This function calls the _f_asin function through the wrapper.
   */
  
  run1_1_Function(_f_asin, "asin");
}


Object _f_asin(Object n, int *err)
{
  /*
   * This function returns the arc sine of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If number is outside the range -1 <= n <= 1 (|x| <= 1), the
      result is a complex number. */
    if (fabs(n.value.real) <= 1) {
      *err = ERR_NOERR;
      n.value.real = re_asin(n.value.real);
      if (userOptions.angleMode == ANGLE_DEG)
	n.value.real *= 180/PI;
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_asin(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acos(void)
{
  /*
   * This function calls the _f_acos function through the wrapper.
   */

  run1_1_Function(_f_acos, "acos");
}


Object _f_acos(Object n, int *err)
{
  /*
   * This function returns the arc co-sine of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If number is outside the range -1 <= n <= 1 (|x| <= 1), the
      result is a complex number. */
    if (fabs(n.value.real) <= 1) {
      *err = ERR_NOERR;
      n.value.real = re_acos(n.value.real);
      if (userOptions.angleMode == ANGLE_DEG)
	n.value.real *= 180/PI;
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acos(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_atan(void)
{
  /*
   * This function calls the _f_atan function through the wrapper.
   */

  run1_1_Function(_f_atan, "atan");
}


Object _f_atan(Object n, int *err)
{
  /*
   * This function returns the arc tangent of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_atan(n.value.real);
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= 180/PI;
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_atan(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_atan2(void)
{
  /*
   * This function calls the _f_atan2 function through the wrapper.
   */

  run2_1_Function(_f_atan2, "atan2");
}


Object _f_atan2(Object n, Object p, int *err)
{
  /*
   * This function returns the arc tangent of p/n, using the signs of
   * both arguments to determine the sign of the result.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;

    /* 0/0 is not defined. */
    if (n.value.real || p.value.real) {
      n.value.real = re_atan2(n.value.real, p.value.real);
      if (userOptions.angleMode == ANGLE_DEG)
	n.value.real *= 180/PI;
    } else
      n.value.real = 0./0.;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_asec(void)
{
  /*
   * This function calls the _f_asec function through the wrapper.
   */

  run1_1_Function(_f_asec, "asec");
}


Object _f_asec(Object n, int *err)
{
  /*
   * This function returns the arc secant of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the result is in the range -1 < x < 1, the result is a complex
       number. */
    if (fabs(n.value.real) >= 1) {
      *err = ERR_NOERR;
      n.value.real = re_asec(n.value.real);
      if (userOptions.angleMode == ANGLE_DEG)
	n.value.real *= 180/PI;
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_asec(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acsc(void)
{
  /*
   * This function calls the _f_acsc function through the wrapper.
   */

  run1_1_Function(_f_acsc, "acsc");
}


Object _f_acsc(Object n, int *err)
{
  /*
   * This function returns the arc co-secant of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    /* If the result is in the range -1 < x < 1, the result is a complex
       number. */
    if (fabs(n.value.real) >= 1) {
      *err = ERR_NOERR;
      n.value.real = re_acsc(n.value.real);
      if (userOptions.angleMode == ANGLE_DEG)
	n.value.real *= 180/PI;
      break;
    }
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acsc(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_acot(void)
{
  /*
   * This function calls the _f_acot function through the wrapper.
   */

  run1_1_Function(_f_acot, "acot");
}


Object _f_acot(Object n, int *err)
{
  /*
   * This function returns the arc co-tangent of its argument.
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    n.value.real = re_acot(n.value.real);
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= 180/PI;
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_acot(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_vers(void)
{
  /*
   * This function calls the _f_vers function through the wrapper.
   */

  run1_1_Function(_f_vers, "vers");
}


Object _f_vers(Object n, int *err)
{
  /*
   * This function returns the versine of its argument.
   *     vers(x) = 1 - cos(x)
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_vers(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_vers(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_hav(void)
{
  /*
   * This function calls the _f_hav function through the wrapper.
   */

  run1_1_Function(_f_hav, "hav");
}


Object _f_hav(Object n, int *err)
{
  /*
   * This function returns the haversine of its argument.
   *     hav(x) = vers(x)/2
   */

  switch (type(n)) {
  case TYPE_REAL:
    *err = ERR_NOERR;
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= PI/180;
    n.value.real = re_hav(n.value.real);
    break;

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.value.cmp = cmp_hav(n.value.cmp);
    break;
    
  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


void f_dTOr(void)
{
  /*
   * This function calls the _f_dTOr function through the wrapper.
   */

  run1_1_Function(_f_dTOr, "d>r");
}


Object _f_dTOr(Object n, int *err)
{
  /*
   * This function converts from degrees to radians.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_dTOr(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void f_rTOd(void)
{
  /*
   * This function calls the _f_rTOd function through the wrapper.
   */

  run1_1_Function(_f_rTOd, "r>d");
}  


Object _f_rTOd(Object n, int *err)
{
  /*
   * This function converts from radians to degress.
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = re_rTOd(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}
