/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>

#include "cmp.h"


/*********************************************************************
 * NOTE:                                                             *
 *   In the function definitions in this file, z is a complex number *
 * so that z = x + y*i, and that its magnitude is r and its argument *
 * is theta.                                                         *
 *********************************************************************/



/*********************
 * General functions *
 *********************/
Complex cmp_conj(Complex n)
{
  /*
   * Complex conjugate.
   *
   * Definition:
   *    conj(z) = x - y*i
   */

  n.im = -n.im;
  return n;
}


Complex cmp_sign(Complex n)
{
  /*
   * Returns the unit vector in the direction of a complex number.
   *
   * Definition:
   *                     x                  y
   *     sign(z) = --------------- + i*---------------
   *               sqrt(x^2 + y^2)     sqrt(x^2 + y^2)
   */
  
  double absval = cmp_abs(n);
  n.re /= absval;
  n.im /= absval;
  return n;
}


/************************
 * Arithmetic functions *
 ************************/
Complex cmp_add(Complex n, Complex p)
{
  /*
   * Sum of two complex numbers.
   *
   * Definition:
   *    (a + b*i) + (c + d*i) = (a + c) + i*(b + d)
   */

  n.re += p.re;
  n.im += p.im;
  return n;
}


Complex cmp_sub(Complex n, Complex p)
{
  /*
   * Difference of two complex numbers.
   *
   * Definition:
   *    (a + b*i) - (c + d*i) = (a - c) + i*(b - d)
   */

  n.re -= p.re;
  n.im -= p.im;
  return n;
}


Complex cmp_mul(Complex n, Complex p)
{
  /*
   * Product of two complex numbers.
   *
   * Definition:
   *    (a + bi)*(c + di) = (ac - bd) + i*(ad + bc)
   */
  
  Complex r;

  r.re = n.re * p.re - n.im * p.im;
  r.im = n.re * p.im + n.im * p.re;
  return r;
}


Complex cmp_inv(Complex n)
{
  /*
   * Inverse of a complex number.
   *
   * Definition:
   *
   *    (a + bi)^-1 = a/(a^2 + b^2) - b*i/(a^2 + b^2)
   */
  
  double d = n.re*n.re + n.im*n.im;

  n.re /= d;
  n.im /= -d;
  return n;
}


Complex cmp_chs(Complex n)
{
  /*
   * Changes the sign of a complex number.
   *
   * Definition:
   *    -(x + y*i) = -x -i*y
   */

  n.re = -n.re;
  n.im = -n.im;
  return n;
}


/*****************************************
 * Exponential and logarithmic functions *
 *****************************************/
Complex cmp_exp(Complex n)
{
  /*
   * Raises e to a complex number.
   *
   * Definition:
   *    exp(z) = e^x * (cos(y) + i*sin(y))
   */

  Complex result;
  double expX = exp(n.re);

  result.re = expX * cos(n.im);
  result.im = expX * sin(n.im);

  return result;
}


Complex cmp_ln(Complex n)
{
  /*
   * Natural logarithm of a complex number.
   *
   * Definition:
   *    ln(z) = ln(r) + i*theta
   */
  
  Complex result;

  result.re = log(cmp_abs(n));
  result.im = cmp_arg(n);

  return result;
}


Complex cmp_pow(Complex n, Complex p)
{
  /*
   * Raises n to pth power.
   *
   * Definition:  (c is a complex number)
   *    z^c = exp(c*ln(z))  (As for real numbers)
   *    or, if z = 0, 0
   */
  
  if (n.re == 0 && n.im == 0) {
    Complex cmp0 = {0, 0};
    return cmp0;
  }

  return cmp_exp(cmp_mul(p, cmp_ln(n)));
}


Complex cmp_sqrt(Complex n)
{
  /*
   * Square root of a complex number.
   *
   * Definition:
   *    sqrt(n) = n^.5   (As for real numbers)
   */
  
  Complex cmp_onehalf = {0.5, 0};

  return cmp_pow(n, cmp_onehalf);
}


Complex cmp_log(Complex n)
{
  /*
   * Decimal logarithm of a complex number.
   *
   * Definition:
   *    log(z) = ln(z)/ln(10)    (As for real numbers)
   *
   * Here, 0.434294481903252 is 1/ln(10)
   */
  
  Complex c_1_ln10 = {0.434294481903252, 0};

  return cmp_mul(cmp_ln(n), c_1_ln10);
}


Complex cmp_alog(Complex n)
{
  /*
   * Returns 10 raised to n.
   *
   * Definition:
   *    alog(z) = exp(z*ln(10))    (As for real numbers)
   */
  
  Complex c_ln10 = {2.30258509299405, 0};

  return cmp_exp(cmp_mul(n, c_ln10));
}


Complex cmp_cis(double n) 
{
  /*
   * Returns exp(i*n).
   */

  Complex z;
  z.re = 0;
  z.im = n;

  return cmp_exp(z);
}


/***************************
 * Trigonometric functions *
 ***************************/
Complex cmp_sin(Complex n)
{
  /*
   * Sine of a complex number.
   *
   * Definition:
   *    sin(z) = sin(x)*cosh(y) + i*cos(x)*sinh(y)
   */
  
  Complex result;

  result.re = sin(n.re) * cosh(n.im);
  result.im = cos(n.re) * sinh(n.im);

  return result;
}


Complex cmp_cos(Complex n)
{
  /*
   * Co-sine of a complex number.
   *
   * Definition:
   *    cos(z) = cos(x)*cosh(y) - i*sin(x)*sinh(y)
   */
  
  Complex result;

  result.re = cos(n.re) * cosh(n.im);
  result.im = -sin(n.re) * sinh(n.im);

  return result;
}


Complex cmp_asin(Complex n)
{
  /*
   * Arc sine of a complex number.
   *
   * Definition:
   *    asin(z) = -i*ln(i*z + sqrt(1 - z^2))
   */
  
  Complex cmpi = {0, 1};
  Complex cmpineg = {0, -1};
  Complex cmp1 = {1, 0};

  return cmp_mul(cmpineg,
		 cmp_ln(cmp_add(cmp_mul(cmpi, n),
				cmp_sqrt(cmp_sub(cmp1,
						 cmp_sq(n))))));
}


Complex cmp_acos(Complex n)
{
  /*
   * Arc co-sine of a complex number.
   *
   * Definition:
   *    acos(z) = -i*ln(z + sqrt(z^2 - 1))
   *    (which means acos(z) = -i*acosh(z))
   */
  
  Complex cmpineg = {0, -1};

  return cmp_mul(cmpineg, cmp_acosh(n));
}


Complex cmp_atan(Complex n)
{
  /*
   * Arc tangent of a complex number.
   *
   * Definition:
   *               i    i + z
   *     atan(z) = - ln -----
   *               2    i - z
   */
  
  Complex cmphalfi = {0, 0.5};
  Complex cmpi = {0, 1};

  return cmp_mul(cmphalfi,
		 cmp_ln(cmp_div(cmp_add(cmpi, n),
				cmp_sub(cmpi, n))));
}


Complex cmp_vers(Complex n) 
{
  /*
   * Versine of a complex number.
   *
   * Definition:
   *    vers(z) = 1 - cos(z)     (As for real numbers)
   */

  Complex cmp1 = {1, 0};

  return cmp_sub(cmp1, n);
}


Complex cmp_hav(Complex n)
{
  /*
   * Haversine of a complex number.
   *
   * Definition:
   *    hav(z) = vers(z)/2       (As for real numbers)
   */

  Complex cmp2 = {2, 0};

  return cmp_div(cmp_vers(n), cmp2);
}

  


/************************
 * Hyperbolic functions *
 ***********************/
Complex cmp_sinh(Complex n)
{
  /*
   * Hyperbolic sine of a complex number.
   *
   * Definition:
   *    sinh(z) = (exp(x) - exp(-x)) / 2  (As for real numbers)
   */
  
  Complex cmp2 = {2, 0};

  return cmp_div(cmp_sub(cmp_exp(n), cmp_exp(cmp_chs(n))), cmp2);

}


Complex cmp_cosh(Complex n)
{
  /*
   * Hyperbolic co-sine of a complex number.
   *
   * Definition:
   *    cosh(z) = (exp(x) + exp(-x)) / 2  (As for real numbers)
   */
  
  Complex cmp2 = {2, 0};

  return cmp_div(cmp_add(cmp_exp(n), cmp_exp(cmp_chs(n))), cmp2);
}


Complex cmp_asinh(Complex n)
{
  /*
   * Hyperbolic arc-sine of a complex number.
   *
   * Definition:
   *    asinh(z) = ln(z + sqrt(z^2 + 1))  (As for real numbers)
   */
  
  Complex cmp1 = {1, 0};

  return cmp_ln(cmp_add(n,
			cmp_sqrt(cmp_add(cmp_sq(n),
					 cmp1))));
}


Complex cmp_acosh(Complex n)
{
  /*
   * Hyperbolic arc co-sine of a complex number.
   *
   * Definition:
   *    acosh(z) = ln(z + sqrt(z^2 - 1))
   */
  
  Complex cmp1 = {1, 0};

  return cmp_ln(cmp_add(n,
			cmp_sqrt(cmp_sub(cmp_sq(n),
					 cmp1))));
}


Complex cmp_atanh(Complex n)
{
  /*
   * Hyperbolic arc tangent of a complex number.
   *
   * Definition:
   *                1    1 + z
   *     atanh(z) = - ln -----
   *                2    1 - z
   */
  
  Complex cmp_onehalf = {1.5707963267949, 0};
  Complex cmp1 = {1, 0};

  return cmp_mul(cmp_onehalf,
		 cmp_ln(cmp_div(cmp_add(cmp1, n),
				cmp_sub(cmp1, n))));
}


Complex cmp_gd(Complex n) 
{
  /*
   * Gudermannian function of a complex number.
   *
   * Definition:
   *    gd(z) = 2atan(e^z) - pi/2
   */

  Complex cmp_halfpi = {1.5707963267949, 0};
  Complex cmp_two = {2, 0};
  

  return cmp_sub(cmp_mul(cmp_two, cmp_atan(cmp_exp(n))), cmp_halfpi);
}
