/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "kalc.h"


hxs normalize(hxs h) 
{
  /*
   * This function normalizes a hxs, that is, only the wordSize
   * least significant bits are kept, the other are zeroed.
   */
  
  register int i;
  hxs mask = 0;

  for (i = 0; i < userOptions.wordSize; i++)
    mask |= ((hxs)1 << i);

  return (h & mask);
}


void f_rTOb(void) 
{
  /*
   * This function calls the _f_rTOb function through the wrapper.
   */

  run1_1_Function(_f_rTOb, "r>b");
}


void f_bTOr(void)
{
  /*
   * This function calls the _f_bTOr function through the wrapper.
   */

  run1_1_Function(_f_bTOr, "b>r");
}


void f_sl(void)
{
  /*
   * This function calls the _f_sl function through the wrapper.
   */

  run1_1_Function(_f_sl, "sl");
}


void f_slb(void)
{
  /*
   * This function calls the _f_slb function through the wrapper.
   */

  run1_1_Function(_f_slb, "slb");
}


void f_sr(void)
{
  /*
   * This function calls the _f_sr function through the wrapper.
   */

  run1_1_Function(_f_sr, "sr");
}


void f_srb(void)
{
  /*
   * This function calls the _f_srb function through the wrapper.
   */

  run1_1_Function(_f_srb, "srb");
}


void f_rl(void)
{
  /*
   * This function calls the _f_rl function through the wrapper.
   */

  run1_1_Function(_f_rl, "rl");
}


void f_rlb(void)
{
  /*
   * This function calls the _f_rlb function through the wrapper.
   */

  run1_1_Function(_f_rlb, "rlb");
}


void f_rr(void)
{
  /*
   * This function calls the _f_rr function through the wrapper.
   */

  run1_1_Function(_f_rr, "rr");
}


void f_rrb(void)
{
  /*
   * This function calls the _f_rrb function through the wrapper.
   */

  run1_1_Function(_f_rrb, "rrb");
}


void f_asr(void)
{
  /*
   * This function calls the _f_asr function through the wrapper.
   */

  run1_1_Function(_f_asr, "asr");
}


Object _f_rTOb(Object n, int *err)
{
  /*
   * This function converts a real number into an hex string.
   */

  if (type(n) == TYPE_REAL) {
    __f_rTOb(&n);
    *err = ERR_NOERR;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_bTOr(Object n, int *err)
{
  /*
   * This function converts an hex string into a real number.
   */

  if (type(n) == TYPE_HXS) {
    n.type = TYPE_REAL;
    n.value.real = (double) normalize(n.value.h);
    *err = ERR_NOERR;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_sl(Object n, int *err)
{
  /*
   * This function shifts the hxs one bit left.
   */

  if (type(n) == TYPE_HXS) {
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h << 1);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_slb(Object n, int *err)
{
  /*
   * This function shifts the hxs one byte left.
   */

  if (type(n) == TYPE_HXS) {
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h << 8);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_sr(Object n, int *err)
{
  /*
   * This function shifts the hxs one bit right.
   */

  if (type(n) == TYPE_HXS) {
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h >> 1);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_srb(Object n, int *err)
{
  /*
   * This function shifts the hxs one byte right.
   */

  if (type(n) == TYPE_HXS) {
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h >> 8);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_rl(Object n, int *err)
{
  /*
   * This function rotates the hxs one bit left.
   */

  if (type(n) == TYPE_HXS) {
    unsigned char outBit = ((n.value.h & (1 << (userOptions.wordSize - 1)))
			    >> (userOptions.wordSize - 1));
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h << 1) | outBit;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_rlb(Object n, int *err)
{
  /*
   * This function rotates the hxs one byte left.
   */

  if (type(n) == TYPE_HXS) {
    unsigned char outByte = ((n.value.h & (0xFF << (userOptions.wordSize - 8)))
			     >> (userOptions.wordSize - 8));
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h << 8) | outByte;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_rr(Object n, int *err)
{
  /*
   * This function rotates the hxs one bit right.
   */

  if (type(n) == TYPE_HXS) {
    unsigned char outBit = n.value.h & 1;
    *err = ERR_NOERR;
    n.value.h = (normalize(n.value.h >> 1)
		 | ((hxs)outBit << (userOptions.wordSize - 1)));
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_rrb(Object n, int *err)
{
  /*
   * This function rotates the hxs one byte right.
   */

  if (type(n) == TYPE_HXS) {
    unsigned char outBit = n.value.h & 0xFF;
    *err = ERR_NOERR;
    n.value.h = (normalize(n.value.h >> 8)
		 | ((hxs)outBit << (userOptions.wordSize - 8)));
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_asr(Object n, int *err)
{
  /*
   * This function performs an arithmetic shift to the right: the
   * most significant bit is preserved.
   */

  if (type(n) == TYPE_HXS) {
    hxs msb = n.value.h & ((hxs)1 << (userOptions.wordSize - 1));
    *err = ERR_NOERR;
    n.value.h = normalize(n.value.h >> 1) | msb;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void __f_rTOb(Object *n)
{
  /*
   * This function converts the given object into an hex string,
   * assuming it is a real number.
   */

  n->type = TYPE_HXS;
  n->value.h = (n->value.real) ? (hxs) n->value.real : 0;
}
