/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <setjmp.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#else
  extern int optopt, opterr, optind;
  char *optarg;
#endif
#ifdef HAVE_LIBREADLINE
#  include <readline/readline.h>
#  include <readline/history.h>
#else
#  define RDLINESIZE 255
char *readline(char *prompt);
#endif

#include "cmp.h"
#include "realfunctions.h"
#include "kalc.h"


Object **s;
Object **tos;
int currStackCap;
jmp_buf jmpError;
/*
 * linesToDisplay contains the number of lines to display the next
 * time the stack is shown. The user-settable nLines is the number
 * of lines requested by the user (default 4).
 * The reason for this is to allow the "=" command to display an
 * specific number of lines if needed.
 */
int linesToDisplay = -9876;
Options userOptions;
char currStatusFile[PATHSIZE];


int main(int argc, char *argv[])
{
  /*
   * This is the entry point of the program. Basically, it parses command
   * line options, performs initialization of the calculator and starts
   * a loop that reads commands, executes them, and displays the stack.
   */

  register int i;
  char opt;
  int batch = 0, loadStack = 1, loadMem = 1;

  /* Initialize the stack */
  currStackCap = STACKINCREMENT;
  s = (Object **) calloc(currStackCap + 1, sizeof(Object *));
  tos = s;

  /* Process command line arguments */
  opterr = 0;   /* Display own error message */
  while ((opt = getopt(argc, argv, ":bd:sm-:")) != -1)
    switch (opt) {
    case 'b':
      batch = 1;
      break;

    case 'd': {
      char *rest;

      linesToDisplay = (int) strtol(optarg, &rest, 10);
      if (!strcasecmp(rest, optarg)) {
        fprintf(stderr, "kalc: Invalid argument for -d: %s\n", optarg);
        exit(1);
      }
    }
    break;

    case 's':
      loadStack = 0;
      break;

    case 'm':
      loadMem = 0;
      break;

    case '-':
      if (!strcmp(optarg, "help"))
        printf(HELPMSG);
      else if (!strcmp(optarg, "version"))
        printf(COPYRIGHT);
      else {
        fprintf(stderr, "kalc: Unknown option: --%s\n", optarg);
        exit(1);
      }
      
      exit(0);

    case ':':
      fprintf(stderr, "kalc: Option -%c requires an argument\n", optopt);
      exit(1);

    case '?':
      fprintf(stderr, "kalc: Unknown option: -%c\n", optopt);
      exit(1);

    default:
      abort();
    }

  /* Perform initialization of calculator */
  initialize(loadStack, loadMem);

  /* linesToDisplay is initialized to -9876. If the user changed it with
     the -d option, update the user setting. If not, the update
     linesToDisplay with the setting from the configuration file (or the
     default. */
  if (linesToDisplay != -9876)
    userOptions.nLines = linesToDisplay;
  else
    linesToDisplay = userOptions.nLines;

  /* Display copyright message */
  fputs(COPYRIGHT, stderr);

  /* If any commands were given on command line, process them */
  if (!setjmp(jmpError)) {
    for (i = optind; i < argc; i++)
      parseLine(argv[i]);
  }

  showStack(linesToDisplay);

  /* If in batch mode, exit now */
  if (batch)
    finalize();

  /* If any error is found, control returns here */
  if (setjmp(jmpError)) {
    showStack(linesToDisplay);
    linesToDisplay = userOptions.nLines;
  }

  /* Processes commands until "q" or "quit" is found */
  while (1) {
    readCommands();
    showStack(linesToDisplay);
    linesToDisplay = userOptions.nLines;
  }
}


void initialize(int loadStack, int loadMem)
{
  /*
   * This function performs initialization of the calculator. It initializes
   * the memory, user options, Readline library, random number seed and
   * reads the init file, if present.
   */

  char rcFilePath[PATHSIZE];
  FILE *fp;

#ifdef MSDOS   /* On DOS, rc file is in current directory */
  strcpy(rcFilePath, "kalcrc.ini");
#else          /* On other systems, it is in the home directory */
  strcpy(rcFilePath, getenv("HOME"));
  strcat(rcFilePath, "/.kalcrc");
#endif

  /* Try to read rc file to get last used status file */
  if ((fp = fopen(rcFilePath, "r")) != NULL) {
    fgets(currStatusFile, PATHSIZE - 1, fp);
    *(currStatusFile + strlen(currStatusFile) - 1) = '\0';
    fclose(fp);
  } else
    /* If not possible, use default */
#ifdef MSDOS   /* On DOS, the file is in current directory */
    strcpy(currStatusFile, "kalc.ini");
#else          /* On other systems, it is in the home directory */
  {
    strcpy(currStatusFile, getenv("HOME"));
    strcat(currStatusFile, "/.kalc");
  }
#endif

  /* First, set default values for all options */
  setDefaults();

  /* Initialize the memory */
  initNextMemoryLevel();

  /* Try to read status file */
  if (!readStatusFromFile(currStatusFile, loadStack, loadMem))
    puts("Status file invalid/inexistent -- using default settings\n");
  
  /* Initialize random number seed */
  srand(time(NULL));

  /* Initialize ReadLine library */
#ifdef HAVE_LIBREADLINE
  rl_readline_name = "kalc";
  rl_bind_key('\t', insertStackElement);
#endif
}


void finalize(void)
{
  /*
   * This functions stores the options, stack and memory in the init file.
   */

  char rcFilePath[PATHSIZE];
  FILE *fp;
  
  if (!saveStatusToFile(currStatusFile)) {
    fputs("Could not write status file\n", stderr);
    exit(2);
  }

#ifdef MSDOS   /* On DOS, rc file is in current directory */
  strcpy(rcFilePath, "kalcrc.ini");
#else          /* On other systems, it is in the home directory */
  strcpy(rcFilePath, getenv("HOME"));
  strcat(rcFilePath, "/.kalcrc");
#endif

  /* Save last used status file in rc file */
  if ((fp = fopen(rcFilePath, "w")) != NULL) {
    fprintf(fp, "%s\n", currStatusFile);
    fclose(fp);
  } else {
    fputs("Could not write configuration file\n", stderr);
    exit(2);
  }
  
  exit(0);
}


void showStack(int n)
{
  /*
   * This function displays n lines from the stack.
   */

  register int i;
  /* register int padding = getPadding(); */

  /* If negative, show all items */
  /* If less stack items than desired number, display all present */
  if ((n > _f_depth()) || (n < 0))
    n = _f_depth();

  for (i = n; i > 0; i--) {
    char *strObject = stackPrepare(**(tos - i + 1), i);
    printf("%s\n", strObject);
    free(strObject);
  }
}


void readCommands(void)
{
  /*
   * This function uses the Readline library to read a line of input,
   * and parses that line.
   */

  char *l;

  l = readline(userOptions.prompt);
  if (l) {
#ifdef HAVE_LIBREADLINE
    if (*l)
      add_history(l);
#endif

    parseLine(l);
    free(l);
  } else
    finalize();
}


#ifdef HAVE_LIBREADLINE
int insertStackElement(int level, int key)
{
  /*
   * This function, which is called when the TAB key is pressed during
   * input, inserts the object at the levelth stack level in the Readline
   * buffer, adding spaces at either end if necessary.
   *
   * level is the universal argument given to the command. key is the key
   * used to invoke the command, and is unused.
   *
   * This function returns 1 if there is not that level, or zero otherwise.
   */

  if (level <= 0 || !enoughArgs(level)) {
    ding();
    return 1;
  }

  /* Begin undo group */
  rl_begin_undo_group();

  /* Insert a space before if necessary */
  if (rl_point > 0 && !isspace(*(rl_line_buffer + rl_point - 1)))
    rl_insert_text(" ");

  /* Insert the number */
  rl_insert_text(editDecomp(**(tos - level + 1)));

  /* Insert a space after, if necessary */
  if (rl_end != rl_point && !isspace(*(rl_line_buffer + rl_point)))
    rl_insert_text(" ");

  /* End undo group */
  rl_end_undo_group();

  return 0;
}
#endif


void freeObjectSpecials(Object obj)
{
  /* This function free()'s space allocated for some object types, but not
   * the space allocated by the object itself.
   *
   * Specifically, it free()'s the allocated space for the actual values
   * of strings and identifiers. For tagged objects, it calls itself
   * recursively and then free()'s the name. For composites, it calls
   * itself on each object.
   *
   * The object itself is not free()'d.
   */

  if (type(obj) == TYPE_STR || type(obj) == TYPE_ID
      || type(obj) == TYPE_UNQUOTEDID)
    free(obj.value.str);
  else if (type(obj) == TYPE_TAGGED) {
    free(obj.value.tagged.tag);
    /* The only case when the object of a tagged object should be NULL is
       when the user enters a tag without object. This will generate an error,
       later. */
    if (obj.value.tagged.obj) {
      freeObjectSpecials(*obj.value.tagged.obj);
      free(obj.value.tagged.obj);
    }
  } else if (type(obj) == TYPE_PROG) {
    Composite *comp = obj.value.comp, *toFree;

    while (comp) {
      freeObjectSpecials(*comp->obj);
      toFree = comp;
      comp = comp->next;
      free(toFree);
    } 
  }
}


/*
 * The code below, specific for DJGPP, replaces some of the start-up code
 * with empty-bodied functions. This makes the program smaller by removing
 * things which aren't necessary such as loading the environment from a
 * file and doing shell metacharacter globbing. As an added bonus, by
 * removing the globbing, you can enter something such as `2 5 *' in the
 * command line safely, unless your DOS shell does globbing.
 */
#ifdef __DJGPP__
#include <crt0.h>

void   __crt0_load_environment_file(char *progname) { }
char **__crt0_glob_function(char *arg) { return 0; }
#endif
