/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>
#include <math.h>

#include "cmp.h"
#include "kalc.h"


/********************
 * Global variables *
 ********************/
int stackMarks[COMPLEVELS];
int nMarks = 0;


void parseLine(char *l)
{
  /*
   * This function parses a line with zero or more tokens.
   */

  char *tok;
  Object obj;

  initializeTokenizer(l);
  while ((tok = getNextToken()) != NULL) {
    obj = processToken(tok);
    evaluateObj(obj, 0);
    freeObjectSpecials(obj);
    free(tok);
  }

  if (nMarks != 0) {
    /* Unfinished composite object found */
    while (_f_depth() != stackMarks[0])
      f_drop(); /* Why does _f_drop() fail? */
    nMarks = 0;
    doError("Syntax", ERR_UNFINISHEDCOMP);
  }
}


Object processToken(char *tok)
{
  /*
   * This function processes a single token.
   */

  int parsed;
  double real;
  Object returnObj;

  switch (*tok) {
  case '@':
    /* A comment. Ignore rest of line. */
    getTillEOL();
    returnObj.type = TYPE_NOTANOBJECT;
    return returnObj;
    
  case '(': {
    /* If the token starts with a '(', try to parse a complex number */
    Complex cmp;

    parsed = parseComplex(tok, &cmp);
    if (parsed) {
      returnObj.type = TYPE_CMP;
      returnObj.value.cmp = cmp;
      return returnObj;
    }

    /* Not parsed, so there is an error */
    printf("Syntax Error in %s\n", tok);
    longjmp(jmpError, 3);
  }

  case '#': {
    hxs h;
    parsed = parseHxs(tok+1, &h);
    if (parsed) {
      returnObj.type = TYPE_HXS;
      returnObj.value.h = h;
      return returnObj;
    }
    
    printf("Syntax Error in %s\n", tok);
    longjmp(jmpError, 3);
  } 

  case '\"':
    /* It is a string */
    returnObj.type = TYPE_STR;
    returnObj.value.str = strdup(tok + 1);
    return returnObj;

  case '\'':
    /* It is an identifier */
    returnObj.type = TYPE_ID;
    returnObj.value.str = strdup(tok + 1);
    return returnObj;
    

  case ':': 
    /* Its a tagged object */
    {
      Object realOb;

      returnObj.type = TYPE_TAGGED;
      returnObj.value.tagged.tag = strdup(tok + 1);
      free(tok);
      tok = getNextToken();
      if (tok == NULL) {
	returnObj.value.tagged.obj = NULL;
	freeObjectSpecials(returnObj);
	doError("Syntax", ERR_MISSINGOBJECT);
      }
      
      realOb = processToken(tok);
      returnObj.value.tagged.obj = objdup(&realOb);
      if (type(*returnObj.value.tagged.obj) == TYPE_UNQUOTEDID)
	returnObj.value.tagged.obj->type = TYPE_ID;
      return returnObj;
    }
    

  case '=':
    /* Set number of lines to display, unless followed by another equal
     sign, in this case it is the == function (equality test) */
    if (*(tok + 1) != '=') {
      linesToDisplay = atoi(tok + 1);
      if (linesToDisplay == 0)
        linesToDisplay = -1;
      returnObj.type = TYPE_NOTANOBJECT;
      return returnObj;
    }

  default:
    {
      void (*f)(void);
      
      /* Try to parse as a real number */
      parsed = parseReal(tok, &real);
      if (parsed) {
	returnObj.type = TYPE_REAL;
	returnObj.value.real = real;
	return returnObj;
      }

      /* If it is not a number, see if it is a function name. */
      f = getFuncAddress(tok);
      if (f) {
	returnObj.type = TYPE_FUNC;
	returnObj.value.func = f;
	return returnObj;
      }
      
      /* If it is not a function, then it is an identifer, without the
         quotes. */
      returnObj.value.str = strdup(tok);
      returnObj.type = TYPE_UNQUOTEDID;
      return returnObj;
    }
  }
}


void evaluateObj(Object obj, int evalIDs) 
{
  /*
   * This funcion evaluates objects. If the object is the end of program
   * maker, >>, it is evaluated.
   * If evalIDs is true, then identifiers and programs are always evaluated.
   */

  static void (*startProgFuncAddr)(void) = &startProgram;
  static void (*endProgFuncAddr)(void) = &endProgram;

  if (type(obj) == TYPE_FUNC
      && (nMarks == 0
	  || obj.value.func == endProgFuncAddr
	  || obj.value.func == startProgFuncAddr)) {
    (*obj.value.func)();
    return;
  }

  if (type(obj) == TYPE_UNQUOTEDID) {
    if (strcmp(obj.value.str, ">>") == 0) {
      endProgram();
      return;
    } else if (strcmp(obj.value.str, "<<") == 0) {
      startProgram();
      return;
    } else if (nMarks == 0) {
      doUnquoted(obj.value.str);
      return;
    }
  } else if (type(obj) == TYPE_ID && evalIDs) {
    doUnquoted(obj.value.str);
    return;
  } else if (type(obj) == TYPE_PROG && evalIDs) {
    evalProgram(obj.value.comp);
    return;
  }

  if (type(obj) == TYPE_PROG
      && nArrows != -1 && lastArrowLevel[nArrows] >= progNesting) {
    double n = _f_depth() - lastArrowDepth[nArrows];
    insertReal(n);
    f_dobind();
    evalProgram(obj.value.comp);
    f_abnd();
    --nArrows;
    return;
  }
    
  if (type(obj) != TYPE_NOTANOBJECT)
    insertObject(obj);
}


int parseReal(char *tok, double *real)
{
  /*
   * This function tries to parse a real number in tok.
   *
   * If possible, stores the number in *real, and returns 1, otherwise
   * returns 0.
   */

  char *rest;

  *real = strtod(tok, &rest);
  if (*rest)
    return 0;

  return 1;
}


int parseComplex(char *tok, Complex *cmp)
{
  /*
   * This function tries to parse a complex number in tok.
   *
   * If possible, stores the complex number in *cmp and returns 1,
   * otherwise returns 0.
   */

  char *lastChar = tok + strlen(tok) - 1;
  char *rest;
  int polar = 0;

  /* Complex numbers must end with a ')' */
  if (*lastChar != ')')
    return 0;
  /* If there was one, remove it */
  *lastChar = '\0';

  /* Now the real part is parsed, and rest points to the rest (space, comma,
     space and imaginary part */
  cmp->re = strtod(tok+1, &rest);
  if (!*rest)   /* No rest? Error */
    return 0;

  /* Skip white space */
  ltrim(rest);
  /* If the rest points to a comma, skip it */
  if (*rest == ',')
    rest++;
  /* Skip whitespace */
  ltrim(rest);
  /* If the first character is a '<', flag polar form */
  if (*rest == '<') {
    polar = 1;
    rest++;
  }

  /* Parse imaginary part.
     NOTE: I use lastChar here to save a variable. */
  cmp->im = strtod(rest, &lastChar);

  /* Remove whitespace from the rest (lastChar) */
  ltrim(lastChar);

  if (*lastChar)   /* There should not be anything else. */
    return 0;      /* If there is, error */

  /* If the number was in polar mode, convert it to rectangular */
  if (polar) {
    double re, im;

    if (userOptions.angleMode == ANGLE_DEG)
      cmp->im *= PI/180;

    re = cmp_pTOr_re(*cmp);
    im = cmp_pTOr_im(*cmp);

    cmp->re = re;
    cmp->im = im;
  }

  return 1;
}


int parseHxs(char *str, hxs *h) 
{
  /*
   * This function tries to parse an hexadecimal string.
   * Retruns 1 if successful, storing the hxs in h, or 0 if there was
   * a syntax error.
   */

  int baseToParse = userOptions.base;
  char *iter = str;
  char *rest;

  while (*iter != '@' && *iter)
    iter++;

  if (*iter) {
    *iter = '\0';
    iter++;
    baseToParse = (int) strtol(iter, &rest, 10);
    if (*rest || baseToParse < 2 || baseToParse > 36)
      return 0;
  } else {
    iter--;
    switch (*iter) {
    case 'b': baseToParse = 2;   *iter = '\0'; break;
    case 'o': baseToParse = 8;   *iter = '\0'; break;
    case 'd': baseToParse = 10;  *iter = '\0'; break;
    case 'h': baseToParse = 16;  *iter = '\0'; break;
    }
  }

  *h = hxs_parse(str, &rest, baseToParse);
  if (*rest)
    return 0;

  return 1;
}



/***********************
 * Tokenizer functions *
 ***********************/
static char *t_pos;

void initializeTokenizer(char *line)
{
  /*
   * This function initalizes the tokenizer, setting the t_pos varible
   * (which contains the current position in the string) to the beginning.
   */

  t_pos = line;
}


char *getNextToken()
{
  /*
   * This function returns the next token in the string, or NULL if there
   * are no more tokens.
   *
   * The return vallue is malloc()'ed, so you should free() it after using.
   */

  /* Character that will end the token -- default space */
  char rmatch = ' ';
  char *token;
  size_t curmaxsize = 30;
  register int i = 0;

  if (!(token = (char *) malloc(30 * sizeof(char))))
    doError("Fatal", ERR_NOTENOUGHMEMORY);

  /* No more tokens? */
  if (*t_pos == '\0')
    return NULL;

  /* Skip trailing whitespace */
  ltrim(t_pos);

  /* If first character is '(' or '"' or ''' or ':', adjust ending character */
  if (*t_pos == '(')
    rmatch = ')';
  else if (*t_pos == '\"' || *t_pos == '\'' || *t_pos == ':') {
    rmatch = *t_pos;
    *(token + i++) = *t_pos++;
  } else if (*t_pos == '#') {
    /* For the input of hxs, it is allowed to have spaces just after the
     * '#' sign, so we skip them here. */
    *(token + i++) = *t_pos++;
    ltrim(t_pos);
  }
  
  /* Find characters until we get the ending character */
  for (; *t_pos != rmatch && *t_pos != '\0'; i++) {
    /* If there is no space, double size of buffer */
    if (i >= curmaxsize) {
      curmaxsize *= 2;
      if (!(token = (char *) realloc(token, curmaxsize * sizeof(char))))
        doError("Fatal", ERR_NOTENOUGHMEMORY);
    }

    /* Copy character */
    if (*t_pos != '\\')
      *(token + i) = *t_pos++;
    else {
      switch (*(++t_pos)) {
      case 'n':  *(token + i) = '\n'; break;
      case '\"': *(token + i) = '\"'; break;
      case '\'': *(token + i) = '\''; break;
      case '\\': *(token + i) = '\\'; break;
      default:   *(token + i) = '\\'; *(token + ++i) = *t_pos; break;
      }
      t_pos++;
    }

    /* Assure there is the terminating character, unless it is space */
    if (*t_pos == '\0' && *(token + i) != rmatch && rmatch != ' ') {
      printf("Syntax Error: %c expected\n", rmatch);
      longjmp(jmpError, 3);
    }
  }

  /* Add closing parenthesis, if necessary */
  if (rmatch == ')')
    *(token + i++) = ')';

  /* Add null character to terminate token */
  *(token + i) = '\0';

  /* Change last seen character of string (if there is one) to space, so that
     it is not seen next time. (This is necessary for tokens inside quotes
     and such.) */
  if (*t_pos != '\0')
    *t_pos = ' ';

  /* No token? Return NULL */
  if (!*token)
    return NULL;

  return token;
}


char *getTillEOL(void)
{
  /*
   * This function returns the rest of the line as a single token.
   */

  char *returnValue = t_pos;

  /* Make t_pos point to the end of the line. */
  while (*++t_pos)
    ;

  return returnValue;
}
