/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.HostBuilder;
import org.ovirt.engine.sdk4.types.Host;

/**
 * A service that manages hosts.
 */
public interface HostsService extends Service {
    /**
     * Creates a new host.
     * 
     * The host is created based on the attributes of the `host` parameter. The `name`, `address` and `root_password`
     * properties are required.
     * 
     * For example, to add a host send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <host>
     *   <name>myhost</name>
     *   <address>myhost.example.com</address>
     *   <root_password>myrootpassword</root_password>
     * </host>
     * ----
     * 
     * NOTE: The `root_password` element is only included in the client-provided initial representation and is not
     * exposed in the representations returned from subsequent requests.
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is newly added we override the host firewall
     * definitions by default.
     * 
     * To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        /**
         * When set to `true` it means this host should deploy also hosted engine components. Missing value is treated
         * as `true` i.e deploy. Omitting this parameter means `false` and will perform no operation in hosted engine
         * area.
         */
        AddRequest deployHostedEngine(Boolean deployHostedEngine);
        AddRequest host(Host host);
        /**
         * The host definition from which to create the new host is passed as parameter, and the newly created host
         * is returned.
         */
        AddRequest host(HostBuilder host);
        /**
         * When set to `true` it means this host should un-deploy hosted engine components and this host will not
         * function as part of the High Availability cluster. Missing value is treated as `true` i.e un-deploy.
         * Omitting this parameter means `false` and will perform no operation in hosted engine area.
         */
        AddRequest undeployHostedEngine(Boolean undeployHostedEngine);
    }
    
    /**
     * Creates a new host.
     * 
     * The host is created based on the attributes of the `host` parameter. The `name`, `address` and `root_password`
     * properties are required.
     * 
     * For example, to add a host send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <host>
     *   <name>myhost</name>
     *   <address>myhost.example.com</address>
     *   <root_password>myrootpassword</root_password>
     * </host>
     * ----
     * 
     * NOTE: The `root_password` element is only included in the client-provided initial representation and is not
     * exposed in the representations returned from subsequent requests.
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is newly added we override the host firewall
     * definitions by default.
     * 
     * To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
     * ----
     */
    public interface AddResponse extends Response {
        /**
         * The host definition from which to create the new host is passed as parameter, and the newly created host
         * is returned.
         */
        Host host();
    }
    
    /**
     * Creates a new host.
     * 
     * The host is created based on the attributes of the `host` parameter. The `name`, `address` and `root_password`
     * properties are required.
     * 
     * For example, to add a host send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts
     * ----
     * 
     * With the following request body:
     * 
     * [source,xml]
     * ----
     * <host>
     *   <name>myhost</name>
     *   <address>myhost.example.com</address>
     *   <root_password>myrootpassword</root_password>
     * </host>
     * ----
     * 
     * NOTE: The `root_password` element is only included in the client-provided initial representation and is not
     * exposed in the representations returned from subsequent requests.
     * 
     * IMPORTANT: Since version 4.1.2 of the engine when a host is newly added we override the host firewall
     * definitions by default.
     * 
     * To add a hosted engine host, use the optional `deploy_hosted_engine` parameter:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/hosts?deploy_hosted_engine=true
     * ----
     */
    AddRequest add();
    
    /**
     * Get a list of all available hosts.
     * 
     * For example, to list the hosts send the following request:
     * 
     * ....
     * GET /ovirt-engine/api/hosts
     * ....
     * 
     * The response body will be something like this:
     * 
     * [source,xml]
     * ----
     * <hosts>
     *   <host href="/ovirt-engine/api/hosts/123" id="123">
     *     ...
     *   </host>
     *   <host href="/ovirt-engine/api/hosts/456" id="456">
     *     ...
     *   </host>
     *   ...
     * </host>
     * ----
     * 
     * The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
     * the `search` parameter.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Indicates if all of the attributes of the hosts should be included in the response.
         * 
         * By default the following host attributes are excluded:
         * 
         * - `hosted_engine`
         * 
         * For example, to retrieve the complete representation of the hosts:
         * 
         * ....
         * GET /ovirt-engine/api/hosts?all_content=true
         * ....
         * 
         * NOTE: These attributes are not included by default because retrieving them impacts performance. They are
         * seldom used and require additional queries to the database. Use this parameter with caution and only when
         * specifically required.
         */
        ListRequest allContent(Boolean allContent);
        /**
         * Indicates if the search performed using the `search` parameter should be performed taking case into
         * account. The default value is `true`, which means that case is taken into account. If you want to search
         * ignoring case set it to `false`.
         */
        ListRequest caseSensitive(Boolean caseSensitive);
        /**
         * Indicates if the results should be filtered according to the permissions of the user.
         */
        ListRequest filter(Boolean filter);
        /**
         * Sets the maximum number of hosts to return. If not specified all the hosts are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of hosts to return. If not specified all the hosts are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of hosts to return. If not specified all the hosts are returned.
         */
        ListRequest max(BigInteger max);
        /**
         * A query string used to restrict the returned hosts.
         */
        ListRequest search(String search);
    }
    
    /**
     * Get a list of all available hosts.
     * 
     * For example, to list the hosts send the following request:
     * 
     * ....
     * GET /ovirt-engine/api/hosts
     * ....
     * 
     * The response body will be something like this:
     * 
     * [source,xml]
     * ----
     * <hosts>
     *   <host href="/ovirt-engine/api/hosts/123" id="123">
     *     ...
     *   </host>
     *   <host href="/ovirt-engine/api/hosts/456" id="456">
     *     ...
     *   </host>
     *   ...
     * </host>
     * ----
     * 
     * The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
     * the `search` parameter.
     */
    public interface ListResponse extends Response {
        List<Host> hosts();
    }
    
    /**
     * Get a list of all available hosts.
     * 
     * For example, to list the hosts send the following request:
     * 
     * ....
     * GET /ovirt-engine/api/hosts
     * ....
     * 
     * The response body will be something like this:
     * 
     * [source,xml]
     * ----
     * <hosts>
     *   <host href="/ovirt-engine/api/hosts/123" id="123">
     *     ...
     *   </host>
     *   <host href="/ovirt-engine/api/hosts/456" id="456">
     *     ...
     *   </host>
     *   ...
     * </host>
     * ----
     * 
     * The order of the returned list of hosts is guaranteed only if the `sortby` clause is included in
     * the `search` parameter.
     */
    ListRequest list();
    
    /**
     * A Reference to service managing a specific host.
     */
    HostService hostService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

