/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.types.LinkLayerDiscoveryProtocolElement;

/**
 * A service to fetch information elements received by Link Layer Discovery Protocol (LLDP).
 */
public interface LinkLayerDiscoveryProtocolService extends Service {
    /**
     * Fetches information elements received by LLDP.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
    }
    
    /**
     * Fetches information elements received by LLDP.
     */
    public interface ListResponse extends Response {
        /**
         * Retrieves a list of information elements received by LLDP.
         * 
         * For example, to retrieve the information elements received on the NIC `321` on host `123`,
         * send a request like this:
         * 
         * ....
         * GET ovirt-engine/api/hosts/123/nics/321/linklayerdiscoveryprotocolelements
         * ....
         * 
         * It will return a response like this:
         * 
         * [source,xml]
         * ----
         * <link_layer_discovery_protocol_elements>
         *     ...
         *   <link_layer_discovery_protocol_element>
         *     <name>Port Description</name>
         *     <properties>
         *       <property>
         *         <name>port description</name>
         *         <value>Summit300-48-Port 1001</value>
         *       </property>
         *     </properties>
         *     <type>4</type>
         *   </link_layer_discovery_protocol_element>
         *   ...
         * <link_layer_discovery_protocol_elements>
         * ----
         */
        List<LinkLayerDiscoveryProtocolElement> elements();
    }
    
    /**
     * Fetches information elements received by LLDP.
     */
    ListRequest list();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

