/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.ClusterBuilder;
import org.ovirt.engine.sdk4.builders.StorageDomainBuilder;
import org.ovirt.engine.sdk4.builders.TemplateBuilder;
import org.ovirt.engine.sdk4.builders.VmBuilder;
import org.ovirt.engine.sdk4.types.Cluster;
import org.ovirt.engine.sdk4.types.StorageDomain;
import org.ovirt.engine.sdk4.types.Template;
import org.ovirt.engine.sdk4.types.Vm;

public interface StorageDomainTemplateService extends Service {
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    public interface GetResponse extends Response {
        Template template();
    }
    
    GetRequest get();
    
    /**
     * Action to import a template from an export storage domain.
     * 
     * For example, to import the template `456` from the storage domain `123` send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/storagedomains/123/templates/456/import
     * ----
     * 
     * With the following request body:
     * 
     * [source, xml]
     * ----
     * <action>
     *   <storage_domain>
     *     <name>myexport</name>
     *   </storage_domain>
     *   <cluster>
     *     <name>mycluster</name>
     *   </cluster>
     * </action>
     * ----
     */
    public interface ImportRequest extends Request<ImportRequest, ImportResponse> {
        /**
         * Indicates if the import should be performed asynchronously.
         */
        ImportRequest async(Boolean async);
        /**
         * Use the optional `clone` parameter to generate new UUIDs for the imported template and its entities.
         * 
         * The user might want to import a template with the `clone` parameter set to `false` when importing a template
         * from an export domain, with templates that was exported by a different {product-name} environment.
         */
        ImportRequest clone_(Boolean clone_);
        ImportRequest cluster(Cluster cluster);
        ImportRequest cluster(ClusterBuilder cluster);
        ImportRequest exclusive(Boolean exclusive);
        ImportRequest storageDomain(StorageDomain storageDomain);
        ImportRequest storageDomain(StorageDomainBuilder storageDomain);
        ImportRequest template(Template template);
        ImportRequest template(TemplateBuilder template);
        ImportRequest vm(Vm vm);
        ImportRequest vm(VmBuilder vm);
    }
    
    /**
     * Action to import a template from an export storage domain.
     * 
     * For example, to import the template `456` from the storage domain `123` send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/storagedomains/123/templates/456/import
     * ----
     * 
     * With the following request body:
     * 
     * [source, xml]
     * ----
     * <action>
     *   <storage_domain>
     *     <name>myexport</name>
     *   </storage_domain>
     *   <cluster>
     *     <name>mycluster</name>
     *   </cluster>
     * </action>
     * ----
     */
    public interface ImportResponse extends Response {
    }
    
    /**
     * Action to import a template from an export storage domain.
     * 
     * For example, to import the template `456` from the storage domain `123` send the following request:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/storagedomains/123/templates/456/import
     * ----
     * 
     * With the following request body:
     * 
     * [source, xml]
     * ----
     * <action>
     *   <storage_domain>
     *     <name>myexport</name>
     *   </storage_domain>
     *   <cluster>
     *     <name>mycluster</name>
     *   </cluster>
     * </action>
     * ----
     */
    ImportRequest import_();
    
    /**
     * Register the Template means importing the Template from the data domain, by inserting the configuration of the
     * Template and disks into the DB without the copy process.
     */
    public interface RegisterRequest extends Request<RegisterRequest, RegisterResponse> {
        /**
         * Indicates whether a template is allowed to be registered with only some of its disks.
         * 
         * If this flag is `true`, the engine will not fail in the validation process if an image is not found, but
         * instead it will allow the template to be registered without the missing disks. This is mainly used during
         * registration of a template when some of the storage domains are not available. The default value is `false`.
         */
        RegisterRequest allowPartialImport(Boolean allowPartialImport);
        /**
         * Indicates if the registration should be performed asynchronously.
         */
        RegisterRequest async(Boolean async);
        RegisterRequest clone_(Boolean clone_);
        RegisterRequest cluster(Cluster cluster);
        RegisterRequest cluster(ClusterBuilder cluster);
        RegisterRequest exclusive(Boolean exclusive);
        RegisterRequest template(Template template);
        RegisterRequest template(TemplateBuilder template);
    }
    
    /**
     * Register the Template means importing the Template from the data domain, by inserting the configuration of the
     * Template and disks into the DB without the copy process.
     */
    public interface RegisterResponse extends Response {
    }
    
    /**
     * Register the Template means importing the Template from the data domain, by inserting the configuration of the
     * Template and disks into the DB without the copy process.
     */
    RegisterRequest register();
    
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    public interface RemoveResponse extends Response {
    }
    
    RemoveRequest remove();
    
    StorageDomainContentDisksService disksService();
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

