'use strict';

const common = require('../common');
const assert = require('assert');
const path = require('path');
const fs = require('fs');
const os = require('os');
const URL = require('url').URL;
const Buffer = require('buffer').Buffer;

function pathToFileURL(p) {
  if (!path.isAbsolute(p))
    throw new Error('Path must be absolute');
  if (common.isWindows && p.startsWith('\\\\'))
    p = p.slice(2);
  return new URL(`file://${p}`);
}

const p = path.resolve(common.fixturesDir, 'a.js');
const url = pathToFileURL(p);

assert(url instanceof URL);

// Check that we can pass in a URL object successfully
fs.readFile(url, common.mustCall((err, data) => {
  assert.ifError(err);
  assert(Buffer.isBuffer(data));
}));

// Check that using a non file:// URL reports an error
const httpUrl = new URL('http://example.org');
fs.readFile(httpUrl, common.mustCall((err) => {
  common.expectsError({
    code: 'ERR_INVALID_URL_SCHEME',
    type: TypeError,
    message: 'The URL must be of scheme file'
  })(err);
}));

// pct-encoded characters in the path will be decoded and checked
fs.readFile(new URL('file:///c:/tmp/%00test'), common.mustCall((err) => {
  assert(err);
  assert.strictEqual(err.message,
                     'Path must be a string without null bytes');
}));

if (common.isWindows) {
  // encoded back and forward slashes are not permitted on windows
  ['%2f', '%2F', '%5c', '%5C'].forEach((i) => {
    fs.readFile(new URL(`file:///c:/tmp/${i}`), common.mustCall((err) => {
      common.expectsError({
        code: 'ERR_INVALID_FILE_URL_PATH',
        type: TypeError,
        message: 'File URL path must not include encoded \\ or / characters'
      })(err);
    }));
  });
} else {
  // encoded forward slashes are not permitted on other platforms
  ['%2f', '%2F'].forEach((i) => {
    fs.readFile(new URL(`file:///c:/tmp/${i}`), common.mustCall((err) => {
      common.expectsError({
        code: 'ERR_INVALID_FILE_URL_PATH',
        type: TypeError,
        message: 'File URL path must not include encoded / characters'
      })(err);
    }));
  });

  fs.readFile(new URL('file://hostname/a/b/c'), common.mustCall((err) => {
    common.expectsError({
      code: 'ERR_INVALID_FILE_URL_HOST',
      type: TypeError,
      message: `File URL host must be "localhost" or empty on ${os.platform()}`
    })(err);
  }));
}
