//
// Copyright (c) 2014 Red Hat, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package org.ovirt.engine.sdk.generator.python;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.ovirt.engine.sdk.generator.XsdData;
import org.ovirt.engine.sdk.generator.python.templates.BasestringHackTemplate;
import org.ovirt.engine.sdk.generator.python.templates.BrokersImportsTemplate;
import org.ovirt.engine.sdk.generator.python.templates.FindRootClassTemplate;
import org.ovirt.engine.sdk.generator.python.templates.ParseClassTemplate;
import org.ovirt.engine.sdk.generator.python.templates.SuperAttributesTemplate;
import org.ovirt.engine.sdk.generator.templates.AbstractTemplate;

public class XsdCodegen {
    private static final String XSD_PARAMS_FILE = "../src/ovirtsdk/xml/params.py";

    // Marks for parts of the code that aren't generated by the generateDS.py code:
    private static final String BEGIN_NOT_GENERATED = "# Begin NOT_GENERATED";
    private static final String END_NOT_GENERATED = "# End NOT_GENERATED";

    // The required version of generateDS.py:
    private static final String GENERATE_DS_VERSION = "2.12a";

    /**
     * The lines of the generated source code.
     */
    private List<String> source = new ArrayList<>();

    /**
     * Generates parameter classes.
     */
    public void generate() throws IOException {
        // Check that the version of generateDS.py is correct:
        String version = runCommand("generateDS.py", "--version");
        if (!version.equals("generateDS.py version " + GENERATE_DS_VERSION)) {
            throw new IOException(
                "The version of generateDS.py isn't correct, it should be " + GENERATE_DS_VERSION + "."
            );
        }

        // Get the location of the XML schemma file:
        File xsdFile = XsdData.getInstance().getFile();

        // Run the generateDS.py program to generate the params.py file:
        runCommand("generateDS.py", "-f", "-o", XSD_PARAMS_FILE, xsdFile.getAbsolutePath());

        // Load all the lines of the params.py file in memory so that we can modify them easily:
        try (BufferedReader in = new BufferedReader(new FileReader(XSD_PARAMS_FILE))) {
            String line;
            while ((line = in.readLine()) != null) {
                source.add(line);
            }
        }

        // Modify the code:
        addImports();
        fixExternalEncoding();
        addBasestringHack();
        addSuperAttributes();
        renameExportMethod();

        // Generate the class map:
        source.add("");
        source.add(BEGIN_NOT_GENERATED);
        source.add("");
        generateClassMap();
        source.add("");
        source.add("");
        generateTagsMap();
        source.add("");
        source.add("");
        appendFunctions();
        source.add("");
        source.add(END_NOT_GENERATED);

        // Write the modified params.py file:
        try (PrintWriter out = new PrintWriter(XSD_PARAMS_FILE)) {
            for (String line : source) {
                out.println(line);
            }
        }
    }

    private void addImports() throws IOException {
        String text = new BrokersImportsTemplate().evaluate();
        String[] lines = text.split("\n");
        addLines(findLastImport() + 1, 0, lines);
    }

    /**
     * The code generated by {@code generateDS.py} uses the Python 2 {@code basestring} builtin, but this has been
     * removed in Python 3. In order to support both Python 2 and Python 3 this method adds code that creates
     * {@code basestring} assigning {@code str} to it if it doesn't exist:
     *
     * <pre>
     * try:
     *     basestring = basestring
     * except NameError:
     *     basestring = str
     * </pre>
     */
    private void addBasestringHack() throws IOException {
        String text = new BasestringHackTemplate().evaluate();
        String[] lines = text.split("\n");
        addLines(findLastImport() + 1, 0, lines);
    }

    /**
     * Returns the index of the last {@code import} line.
     */
    private int findLastImport() throws IOException {
        // Find the last line containing an import statement:
        int index = 0;
        for (int i = 0; i < source.size(); i++) {
            String line = source.get(i);
            if (line.startsWith("import ")) {
                index = i;
            }
        }
        if (index == 0) {
            throw new IOException("Can't find position to add imports.");
        }
        return index;
    }

    /**
     * By default {@code generateDS.py} assumes that the XML text is encoded using the default Python encoding, as
     * returned by {@code sys.getdefaultencoding()}, but this changes from system to system, and may be different if
     * running from an IDE. We need to make sure that it is always UTF-8, as that is what the RESTAPI returns.
     */
    private void fixExternalEncoding() throws IOException {
        for (int i = 0; i < source.size(); i++) {
            String line = source.get(i);
            if (line.matches("^ExternalEncoding = '[^']*'$")) {
                line = "ExternalEncoding = 'UTF-8'";
                source.set(i, line);
            }
        }
    }

    private void addSuperAttributes() throws IOException {
        // Find the last line of the base class:
        int firstIndex = findClassFirstLine("GeneratedsSuper");
        if (firstIndex == -1) {
            throw new IOException("Can't find base class to add methods.");
        }
        String firstLine = source.get(firstIndex);
        int firstIndent = getIndent(firstLine);

        String text = new SuperAttributesTemplate().evaluate();
        String[] lines = text.split("\n");

        // Insert the methods after the last line of the class:
        int lastIndex = findBlockLastLine(firstIndex);
        addLines(lastIndex + 1, firstIndent + 4, lines);
    }

    /**
     * The classes generated by {@code generateDS.py} contains an {@code export} method that clashes with the
     * {@code export} method generated for some brokers (the {@code export} action of a virtual machine, for example).
     * To avoid that clash we need to rename it.
     */
    private void renameExportMethod() throws IOException {
        for (int i = 0; i < source.size(); i++) {
            String line = source.get(i);
            line = line.replaceAll("def export\\(", "def export_(");
            line = line.replaceAll("\\.export\\(", ".export_(");
            source.set(i, line);
        }
    }

    private void generateClassMap() throws IOException {
        // Sort the names of the elements:
        Map<String, String> map = XsdData.getInstance().getTypesByTag();
        List<String> names = new ArrayList<>(map.keySet());
        Collections.sort(names);

        // Generate the map:
        addLines(source.size(), 0, "_rootClassMap = {");
        for (String name : names) {
            String type = map.get(name);
            String line = String.format("\"%-31s: %s,", name + "\"", type);
            addLines(source.size(), 20, line);
        }
        addLines(source.size(), 16, "}");
    }

    private void generateTagsMap() throws IOException {
        // Sort the type names:
        Map<String, String> map = XsdData.getInstance().getTagsByType();
        List<String> types = new ArrayList<>(map.keySet());
        Collections.sort(types);

        // Generate a list of pairs, each containing the type and the corresponding XML tag:
        addLines(source.size(), 0, "_tag_for_type = {");
        for (String type : types) {
            String tag = map.get(type);
            String line = String.format("%s: \"%s\",", type, tag);
            addLines(source.size(), 4, line);
        }
        addLines(source.size(), 0, "}");
    }

    private void appendFunctions() throws IOException {
        appendFunction(new FindRootClassTemplate());
        appendFunction(new ParseClassTemplate());
    }

    private void appendFunction(AbstractTemplate template) throws IOException {
        String text = template.evaluate();
        String[] lines = text.split("\n");
        addLines(source.size(), 0, lines);
        source.add("");
    }

    /**
     * Runs system command
     *
     * @param command
     *            command to run
     *
     * @return command output
     *
     * @throws IOException
     */
    private String runCommand(String... command) throws IOException {
        String stdout = "";
        String stderr = "";
        String s;

        ProcessBuilder builder = new ProcessBuilder();
        builder.command(command);
        Process process = builder.start();

        BufferedReader stdInput = new BufferedReader(new
            InputStreamReader(process.getInputStream()));

        BufferedReader stdError = new BufferedReader(new
            InputStreamReader(process.getErrorStream()));

        while ((s = stdInput.readLine()) != null) {
            stdout += s;
        }

        while ((s = stdError.readLine()) != null) {
            stderr += s;
        }

        if (!stderr.equals(""))
            throw new RuntimeException(stderr);

        return stdout;
    }

    /**
     * Adds a set of lines to the generated source code at the given position.
     *
     * @param position the position where to add the new lines
     * @param indent the indentation level of the added lines
     * @param lines the lines to add
     */
    private void addLines(int position, int indent, String... lines) {
        for (String line : lines) {
            if (line.length() > 0) {
                for (int i = 0; i < indent; i++) {
                    line = " " + line;
                }
            }
            source.add(position, line);
            position++;
        }
    }

    /**
     * Calculates the level of indentation of a given Python source line. Comments aren't taken into account.
     *
     * @param line the Python source line
     * @return the indentation of the line or -1 if the line is empty
     */
    private int getIndent(String line) {
        // Strip comments:
        int position = line.indexOf('#');
        if (position != -1) {
            line = line.substring(0, position);
        }

        // Find the first non blank character:
        int indent = 0;
        while (indent < line.length() && line.charAt(indent) == ' ') {
            indent++;
        }

        // If the indent is equal to the length of the line then it is an empty line, and indentation for empty lines
        // doesn't make sense:
        if (indent == line.length()) {
            indent = -1;
        }

        return indent;
    }

    /**
     * Finds the index of the line where the given class starts.
     *
     * @param name the name of the class
     * @return the index of the line where the class starts or -1 if no such class exists
     */
    private int findClassFirstLine(String name) {
        Pattern pattern = Pattern.compile(" *class ([0-9a-zA-Z_]+)\\([^\\)]*\\):");
        for (int i = 0; i < source.size(); i++) {
            Matcher matcher = pattern.matcher(source.get(i));
            if (matcher.matches() && name.equals(matcher.group(1))) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Finds the index of the last line of the block that starts in the given line.
     *
     * @param first the index of the first line of the block
     * @return the index of the last line of the block or -1 if no such class exists.
     */
    private int findBlockLastLine(int first) {
        String firstLine = source.get(first);
        int firstIndent = getIndent(firstLine);
        int last = first;
        for (int i = first + 1; i < source.size(); i++) {
            String line = source.get(i);
            int indent = getIndent(line);
            if (indent >= 0) {
                if (indent <= firstIndent) {
                    break;
                }
                last = i;
            }
        }
        return last;
    }
}
