/*
	 $Header: /usr/local/sources/CVS/postgres/src/backend/port/next/dynloader.c,v 2.1 1994/08/04 18:20:58 tom Exp $

	Author: Tom Hageman.
*/

#ifndef TEST1

#include "port-protos.h"

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <mach/mach.h>
#include <nlist.h>
#include <mach-o/rld.h>
#include <mach-o/fat.h>
#include <mach-o/arch.h>

#include <objc/zone.h>

/*
 * Dynamically loaded function list. 
 * {{Postgres95 changed the dynamic function loader interface (again!)
 *   This typedef used to be in 4.2 dynamic_loader.h...}}
 */

typedef struct df_list {
    char *funcname;			/* Name of function */
    func_ptr func;			/* Function address */
    struct df_list *next;
} DynamicFunctionList;


static DynamicFunctionList *dynamic_file_load( char **err, char *filename,
					       char **address, long *size );
static void free_function_list( DynamicFunctionList *flist );


/* postgres95 dynamic loader interface. */

typedef struct {
	char *loadAddress;
	long loadSize;
	DynamicFunctionList *functionList;
} _dl_handle;

static char *_dl_error;

void *
pg_dlopen( char *filename )
{
	_dl_handle *handle = (_dl_handle *) malloc(sizeof(*handle));

	_dl_error = NULL;

	if (!handle) {
		_dl_error = "cannot allocate handle (out of memory)";
	}
	else if (!(handle->functionList = dynamic_file_load(&_dl_error, filename,
							    &handle->loadAddress,
							    &handle->loadSize))) {
		free(handle);
		handle = NULL;
	}
	return handle;
}

func_ptr
pg_dlsym( void *handle, char *funcname )
{
	DynamicFunctionList *flist = ((_dl_handle *) handle)->functionList;

	_dl_error = NULL;
	if (flist) do {
		if (funcname[0] == flist->funcname[0] &&
		    strcmp(funcname, flist->funcname) == 0) {
			return flist->func;
		}
	} while (flist = flist->next);

	_dl_error = "symbol name not found.";
	/* not found. */
	return NULL;
}

void
pg_dlclose( void * handle )
{
	_dl_error = NULL;
	free_function_list(((_dl_handle *) handle)->functionList);
	free(((_dl_handle *) handle)->loadAddress);
	free(handle);
}

char *
pg_dlerror()
{
	return (_dl_error ? _dl_error : "");
}

/* Error message handling. */

static struct {
	char	*addr;
	int	size;
} loadErrorBuf;

static NXStream *
openErrorStream( void )
{
	/* Free error buffer if left over from last time. */
	if (loadErrorBuf.addr) {
		vm_deallocate(task_self(), (vm_address_t) loadErrorBuf.addr,
			      (vm_size_t) loadErrorBuf.size);
		loadErrorBuf.addr = 0;
		loadErrorBuf.size = 0;
	}
	return NXOpenMemory(NULL, 0, NX_WRITEONLY);
}

static char *
errorMessage( NXStream *errorStream )
{
	if (errorStream) {
		int dummy;
		NXPutc(errorStream, '\0');
		NXGetMemoryBuffer(errorStream, &loadErrorBuf.addr,
				  &loadErrorBuf.size, &dummy);
		NXClose(errorStream);

		return loadErrorBuf.addr;
	}
	return "dynamic_file_load failed.";
}

static void
closeErrorStream( NXStream *errorStream )
{
	if (errorStream)
		NXCloseMemory(errorStream, NX_FREEBUFFER);
}


/* Address callback function for rld(3).
   (This seems to be the only way to obtain the load adress.)
   Note that this gives the address AT WHICH TO RELOCATE TO; it does NOT
   direct rld to copy the actual module to that address, so we have
   to do that by hand later on...
 */

static char *loadAddress;
static unsigned long loadSize, loadHeaderSize;

static NXZone *loadZone;

static unsigned long
dynamic_load_address( unsigned long size, unsigned long headers_size )
{
	if (!loadZone) {
		loadZone = NXCreateZone(vm_page_size, vm_page_size, 1);
		NXNameZone(loadZone, "dynload");
	}
	loadSize = size -= headers_size;
	loadHeaderSize = headers_size;
	loadAddress = NXZoneMalloc(loadZone, size);

	return (unsigned long) loadAddress - headers_size;
}


static DynamicFunctionList *load_symbols( const char *filename, char *objAddr,
					  NXStream *loadErrorStream );
static int handle_fat_binary( char **objAddrPtr, int *objSizePtr,
			      const char *filename, NXStream *errorStream );

static
DynamicFunctionList *
dynamic_file_load( char **err, char *filename, char **address, long *size )
{
	DynamicFunctionList *result = NULL;
	NXStream *loadStream;
	NXStream *loadErrorStream;
	char *objAddr;
	int objSize;
	int dummy;
	struct mach_header *header;

	rld_address_func(dynamic_load_address);

	if ((loadStream = NXMapFile(filename, NX_READONLY)) == NULL) {
		*err = "error opening file";
		return result;	/* i.e., NULL */
	}

	loadErrorStream = openErrorStream();

	NXGetMemoryBuffer(loadStream, &objAddr, &objSize, &dummy);

	if (!handle_fat_binary(&objAddr, &objSize, filename, loadErrorStream) ||
	    !rld_load_from_memory(loadErrorStream, &header, filename,
				  objAddr, objSize, NULL) ||
	    (!(result = load_symbols(filename, objAddr, loadErrorStream)) &&
	     (rld_unload(NULL), TRUE))) {

		*err = errorMessage(loadErrorStream);

		NXCloseMemory(loadStream, NX_FREEBUFFER);
		return result;	/* i.e., NULL */
	}
	/* Now, copy the module to our private buffer, and unload it. */
	memcpy(loadAddress, (char *) header + loadHeaderSize, loadSize);
	rld_unload(NULL);
	(*address) = loadAddress;
	(*size) = loadSize;
	closeErrorStream(loadErrorStream);
	NXCloseMemory(loadStream, NX_FREEBUFFER);

	return result;
}

static DynamicFunctionList *
load_symbols( const char *filename, char *objAddr, NXStream *loadErrorStream )
{
	DynamicFunctionList *funcList = NULL, **nextFuncInsert = &funcList;
	struct load_command *lp;
	struct symtab_command *sp;
	unsigned long i;
	struct nlist *sym_table;
	char *name_table;
	int text_section = 1;
	/* TODO: find actual __text section from load commands.
	   in practice, the text section is always Number One. */

#if DEBUG
	fprintf(stderr, "DEBUG load_symbols(%s @%p):\n", filename, objAddr);
#endif
	/* Find the symbol table load command. */
	for (i = 0,
	     lp = (struct load_command *)(objAddr + sizeof(struct mach_header));
	     lp->cmd != LC_SYMTAB;
	     i++,
	     lp = (struct load_command *)((char *) lp + lp->cmdsize)) {

		if (i == ((struct mach_header *) objAddr)->ncmds) {
			NXPrintf(loadErrorStream,
				 "Symbol table missing in file %s.", filename);
			return NULL;
		}
	}

	/* Get pointers to symbol-table and symbol-nametable. */
	sp = (struct symtab_command *) lp;
	sym_table = (struct nlist *)(objAddr + sp->symoff);
	name_table = (char *)(objAddr + sp->stroff);

	/* Now scan the table for global __TEXT symbols, and create a
	   DynamicFunctionList entry for each of them. */

	for (i = 0;  i < sp->nsyms;  i++, sym_table++) {
		char *sym_name;
		unsigned long sym_value;

		if (!(sym_table->n_type & N_EXT))		/* static. */
			continue;

		if ((sym_table->n_type & N_TYPE) == N_UNDF)	/* external. */
			continue;

		sym_name = name_table + sym_table->n_un.n_strx;

		if (!rld_lookup(NULL, sym_name, &sym_value))
			continue;

		if ((sym_table->n_type & N_TYPE) == N_SECT &&
		    (sym_table->n_sect == text_section)) {
			DynamicFunctionList *newEntry;

			if ((newEntry = malloc(sizeof(*newEntry))) == NULL ||
			    /* allocate space for the function's name,
			       skip leading `_'.
			       (strlen(sym_name + 1) + 1 == strlen(sym_name)) */
			    ((newEntry->funcname = malloc(strlen(sym_name))) == NULL &&
			     (free(newEntry), TRUE))) {
				free_function_list(funcList);
				NXPrintf(loadErrorStream,
					 "Out of memory in dynamic loader.");
				return NULL;
			}

			strcpy(newEntry->funcname, sym_name + 1);
			newEntry->func = (func_ptr) sym_value;

			/* Add the new entry to the end of the list. */
			(*nextFuncInsert) = newEntry;
			nextFuncInsert = &newEntry->next;
#if DEBUG
			fprintf(stderr, "  %-*.*s @%p\n", sizeof(newEntry->funcname), sizeof(newEntry->funcname), newEntry->funcname, newEntry->func);
#endif
		}
	}
	if (funcList == NULL) {
		NXPrintf(loadErrorStream, "No global symbols.");
	}
#if DEBUG
	fflush(stderr);
#endif
	return funcList;
}


static void
free_function_list( DynamicFunctionList *flist )
{
	DynamicFunctionList *next;

	if (flist) do {
		next = flist->next;
		free(flist->funcname);
		free(flist);
	} while (flist = next);
}

/* Examine the memory-mapped file given by `*objAddrPtr' and `*objSizePtr'.
   If it is a fat binary, adjust `*objAddrPtr' and `*objSizePtr' to point
   to the object that matches the current architecture and return 1; if no
   match is found, return 0.  If it is not a fat binary, don't do
   anything and return -1. */

static int
handle_fat_binary( char **objAddrPtr, int *objSizePtr, const char *filename,
		   NXStream *errorStream)
{
	struct fat_header *fat_header = (struct fat_header *) *objAddrPtr;
	struct fat_arch *fat_arch = (struct fat_arch *) &fat_header[1];
	const NXArchInfo *hostArch;
	unsigned long i;

	if (fat_header->magic == FAT_MAGIC) {
		/* A fat file, in host order */
	}
	else if (fat_header->magic == FAT_CIGAM) {
		/* A fat file, in non-host order; must be swapped. */
		unsigned long *lp = (long *) fat_header;

		i = (NXSwapLong(fat_header->nfat_arch) * sizeof(*fat_arch) +
		     sizeof(*fat_header)) / sizeof(unsigned long);

		do {
			*lp = NXSwapLong(*lp);
			lp++;
		} while (--i);
	}
	else {
		/* Not a fat file. */
		return -1;
	}

	/* Now that we have a fat file, find a member that matches the
	   host architecture. */
	if ((hostArch = NXGetLocalArchInfo()) == NULL) {
		NXPrintf(errorStream, "cannot determine architecture of host!");
		return 0;
	}
	if ((fat_arch = NXFindBestFatArch(hostArch->cputype,
					  hostArch->cpusubtype, fat_arch,
					  fat_header->nfat_arch)) == NULL) {
		NXPrintf(errorStream, "cannot find matching architecture"
			 " for %s in fat file %s.", hostArch->name, filename);
		return 0;
	}

	/* Adjust object address and -size to point to the matching member. */
	(*objAddrPtr) += fat_arch->offset;
	(*objSizePtr) = fat_arch->size;

	return 1;
}

#ifdef TEST
main(argc, argv)
int	argc;
char	*argv[];
{
	int i;

	if (argc <= 1) {
		fprintf(stderr, "Usage: %s <object-files>\n", argv[0]);
		exit(1);
	}
	for (i = 1;  i < argc;  i++) {
		char *address;
		long size;
		char *err;
		DynamicFunctionList *flist;

		if ((flist = dynamic_file_load(&err, argv[i], &address, &size)) == NULL) {
			fprintf(stderr, "%s: %s\n", argv[i], err);
			continue;
		}
		printf("%s: size=%ld [%#p - %#p]\n",
		       argv[i], size, address, address + size);
		do {
			printf("    %-*.*s %p (result = %d)\n",
			       sizeof(flist->funcname), sizeof(flist->funcname),
			       flist->funcname, flist->func, (*flist->func)());
		} while (flist = flist->next);
		printf("\n");
	}
	exit(0);
}
#endif /* TEST */

#else /* TEST1 */

/*  A module for testing dynamic loading with. */
int global_variable = 1;

int three()		{ return 3; }
int minus_five()	{ return -5; }
static int static_function() { return 0; }
int longnamed_1234567890() { return 1; }

#endif /* TEST1 */
