/*
 *
 *  FILENAME: calltree.c
 *
 *  PURPOSE: Build a data structure based on the call tree of specified
 *           files.
 *
 *  AUTHOR: Doug Hellmann
 *
 *  DATE: 26 August 1995
  *
 *  COMMENTS: 
 *
*/

#include <stdio.h>
#include <stdlib.h>

#include "client.h"

#include "calltreP.h"
#include "calltree.h"
#include <string.h>
#include "list.h"
#include "stack.h"

#include "Tree.h"

extern Widget callTreeDlg;


/*
 *
 * PURPOSE: Create the memory for a new CallTree.
 *
 * PARAMETERS: none
 *
 * RETURNS: User managed memory.
 *
*/
static CallTree * CallTree_New(List * all_fcns)
{
  CallTree * Return;
  
  Return = (CallTree*) malloc (sizeof(CallTree));
  
  Return->fcn_name = NULL;
  Return->file_name = NULL;
  Return->line_num = -1;
  Return->calls = ListCreate();
  Return->called_by = ListCreate();
  
  Return->search_result = FALSE;
  Return->search_path = FALSE;
  Return->search_subnode = FALSE;

  if (all_fcns)
  {
    Return->root_node = FALSE;
    Return->all_fcns = all_fcns;
  }
  else
  {
    Return->all_fcns = ListCreate();
    Return->root_node = TRUE;
  }
  
  return Return;
}

/*
 *
 * PURPOSE: duplicate a string in new memory
 *
 * PARAMETERS: null terminated string
 *
 * RETURNS: new memory containing copy of the string
 *
*/
#ifndef IBM
char * strdup(const char * str)
{
  char * Return;
  
  Return = (char*) malloc ( sizeof(char) * (strlen(str) + 1) );
  strcpy(Return, str);
  
  return Return;
}
#endif

/*
 *
 * PURPOSE: recursively deletes memory used by call tree
 *
 * PARAMETERS: call tree to erase
 *
 * RETURNS: none
 *
*/
void CallTree_Delete(CallTree * ct)
{
  if (!ct)
    return;
  
  ListDestroy(ct->calls, FALSE);
  ListDestroy(ct->called_by, FALSE);
  
  if (ct->fcn_name)
    free(ct->fcn_name);
  if (ct->root_node)
	ListDestroy(ct->all_fcns, TRUE);
  
  free(ct);
}

/*
 *
 * PURPOSE: Compare the two names of the functions in the specified
 *          CallTrees.
 *
 * PARAMETERS: two pointers to CallTrees
 *
 * RETURNS: <0 if left is less than right
 *           0 if left is equal to right
 *          >0 if left is greater than right
 *
*/
static int CallTree_Compare(CallTree * left, CallTree * right)
{
  if ( (!left) && (!right) )
    return 0;
    
  if (!left)
    return -1;
  if (!right)
    return 1;
  
  if ( (!left->fcn_name) && (!right->fcn_name) )
    return 0;
    
  if (!left->fcn_name)
    return -1;
  if (!right->fcn_name)
    return 1;
    
  return strcmp(left->fcn_name, right->fcn_name);
}



/*
 *
 * PURPOSE: Add a node in the calls list of the specified call tree.
 *
 * PARAMETERS: CallTree, name of function
 *
 * RETURNS: Added node.
 *
*/
CallTree * CallTree_AddCall(CallTree * ct, char * fcn_name)
{
  CallTree * newNode;
  int position;
  CallTree * searchNode = CallTree_New(NULL);
  
  /*
   * Should look for the fcn_name in  the all_fcns list here.  If it
   * is found, use that node as the subnode of ct.  If it is not found,
   * create a new node and put it in both lists.
   *
  */
  searchNode->fcn_name = strdup(fcn_name);
  position = ListFindItem(ct->all_fcns, searchNode, CallTree_Compare);
  
  if (position < 0)
  {
    newNode = CallTree_New(ct->all_fcns);
    
    if (!newNode)
      return NULL;

    newNode->fcn_name = strdup(fcn_name);
    ListAddItem(ct->all_fcns, newNode, CallTree_Delete, -1);
  }
  else
  {
    newNode = ListPos(ct->all_fcns, position);
  }
  
  ListAddItem(ct->calls, newNode, CallTree_Delete, -1);
/*
  ListSortAsc(ct->calls, CallTree_Compare);
*/

  
#ifdef DEBUG_Calltree_parse
  if (ct->fcn_name)
    fprintf(stderr, "adding %s as call of %s\n", fcn_name, ct->fcn_name);
  else
    fprintf(stderr, "adding %s as global of Return\n", fcn_name);
#endif

  CallTree_Delete(searchNode);
  
  return newNode;
}

/*
 *
 * PURPOSE: Create a CallTree from the output of a command.
 *
 * PARAMETERS: String to execute as shell command.
 *
 * RETURNS: CallTree created.
 *
*/
CallTree * CallTree_BuildFromCommand(char * command)
{
  char * command_output;
  int success;
  char * out_file_name = tmpnam(NULL);
  char buf[1024];
  CallTree * Return;
  
  sprintf(buf, "%s > %s", command, out_file_name);
  
  command_output = issueCommandToText(callTreeDlg, buf, "", 
  			      ERROR_DIALOGS, NULL, 0, 0, 
  			      &success);
  
  if (!success)
    return NULL;
  
  Return = CallTree_ReadFromFile(out_file_name);
      
  unlink(out_file_name);

  if (out_file_name)
    free(out_file_name);
  
  return Return;  
}


/*
 *
 * PURPOSE: Create a file from which a call tree can be created using 
 *          CallTree_BuildReadFromFile.
 *
 * PARAMETERS: Pattern for file names to use with cflow.
 *
 * RETURNS: Root of CallTree built.
 *
*/
CallTree * CallTree_BuildFromFiles(char * filenames)
{
  CallTree * Return;
  char buf[1024];
  
  sprintf(buf, "cflow %s", filenames);
  
  Return = CallTree_BuildFromCommand(buf);
  
  return Return;
}


/*
 *
 * PURPOSE: Read a file containing cflow output and build the call tree from
 *          it.
 *
 * PARAMETERS: File name pattern to use with cflow.
 *
 * RETURNS: Root of CallTree built.
 *
*/
CallTree * CallTree_ReadFromFile(char * filename)
{
  CallTree * Return;
  CallTree * lastAdded;
  CallTree * currentNode;
  FILE * input;
  char buf[1024];
  int current_depth = 0;
  Stack * lastAddedStack = Stack_New();
  
  if (!filename)
    return NULL;
  if (!*filename)
    return NULL;
    
  input = fopen(filename, "r");
  
  if (!input)
    return NULL;
    
  Return = CallTree_New(NULL);
  currentNode = Return;
  Stack_Push(lastAddedStack, Return, NULL);
    
  while ( fgets(buf, sizeof(buf), input) != NULL )
  {
    char srcname[128];
    char def_filename[128];
    int def_linenum;
    int level; /* number of tabs - 1 */
    int i;
    int name_starts, passed_number;
    char * find_filename;
    
    /* count the tabs (nesting level) and pick out the name of the function */
      
    level = 0;
    passed_number = FALSE;
    for (i = 0; i < strlen(buf); i++)
    {
      if (isspace(buf[i]))
      {
        if (!passed_number)
          passed_number = TRUE;
        level++;
      }
      else
      {
        if (passed_number)
          break;
      }      
    }
    
    name_starts = i;
    
    if (buf[name_starts] == '.')
      name_starts++;
    
    for (; i < strlen(buf); i++)
    {
      if (buf[i] == ':')
        break;
    }
    
    level--; /* get rid of first tab, they all have at least one */
       
    strncpy(srcname, buf + name_starts, i - name_starts);
    srcname[i - name_starts] = '\0';
    
#ifdef DEBUG_Calltree_parse
    fprintf(stderr, "level is %d, current_depth is %d\n", level, current_depth);
#endif
    
    if (level == current_depth)
    {
      lastAdded = CallTree_AddCall(currentNode, srcname);
    }
    else if (level > current_depth)
    {
#ifdef DEBUG_Calltree_parse
      fprintf(stderr, "going down a level\n");
#endif
      
      Stack_Push(lastAddedStack, lastAdded, NULL);

      currentNode = lastAdded;

      lastAdded = CallTree_AddCall(currentNode, srcname);
      current_depth++;
    }
    else
    {      
      while (level < current_depth)
      {
        currentNode = Stack_Pop(lastAddedStack);
        currentNode = Stack_Peek(lastAddedStack);

#ifdef DEBUG_Calltree_parse
	if (currentNode)
	{
	  if (currentNode->fcn_name)
            fprintf(stderr, "\ngoing up a level to %s\n", currentNode->fcn_name);
          else
            fprintf(stderr, "\ngoing up a level to Return\n");
        }
	else
          fprintf(stderr, "going up a level to global\n");
#endif

	current_depth --;
      }

      lastAdded = CallTree_AddCall(currentNode, srcname);
      
    }
    
    /*
    ** Scan for filename and linenumber of definition
    */
    find_filename = strchr(buf, '<');
    
    if (find_filename && (find_filename[1] != '>') )
    {
      find_filename++; /* skip '<' */
      sscanf(find_filename, "%s %d", &def_filename, &def_linenum);
      
#ifdef  DEBUG_Calltree_parse
      fprintf(stderr, "Found %s in %s at %d\n", 
        srcname, def_filename, def_linenum);
#endif

      lastAdded->file_name = strdup(def_filename);
      lastAdded->line_num = def_linenum - 1; /* corrects for actual definition */
    }
    
  }
  
  fclose(input);
  
  Stack_Delete(lastAddedStack, FALSE);
  
  for (lastAdded = ListReset(Return->all_fcns);
       lastAdded;
       lastAdded = ListNext(Return->all_fcns))
  {
     CallTree * callee;
     
     for (callee = ListReset(lastAdded->calls); 
     	  callee; 
     	  callee = ListNext(lastAdded->calls))
     {
        ListAddItem(callee->called_by, lastAdded, NULL, -1);
        ListSortAsc(callee->called_by, CallTree_Compare);
     }
  }

  ListSortAsc(Return->all_fcns, CallTree_Compare);
  
  return Return;
}

/*
 *
 * PURPOSE: Dump one node of a call tree (recursive).
 *
 * PARAMETERS: CallTree, depth, File pointer open for output.
 *
 * RETURNS: None.
 *
*/
static void CallTree_nodeDump(CallTree * tree, int depth, FILE * output, int recurse)
{
  int i;
  CallTree * subnode;
  
  if (!tree)
    return;
  if (!tree->fcn_name)
    return;
    
  if (!recurse)
    return;
  
  if (tree->fcn_name)
  {
    for (i = 0; i < depth + 1; i++)
    {
      fprintf(output, ".");
    }
    fprintf(output, "%s", tree->fcn_name);
    if (tree->file_name)
      fprintf(output, "(%s", tree->file_name);
    if (tree->line_num >= 0)
      fprintf(output, "%d", tree->line_num);
    fprintf(output, ")\n");
  }
  
  for (subnode = ListReset(tree->calls);
       subnode != NULL;
       subnode = ListNext(tree->calls))
  {
    CallTree_nodeDump(subnode, depth + 1, output, recurse - 1);
  }

}


/*
 *
 * PURPOSE: Print the call tree to output file specified.
 *
 * PARAMETERS: CallTree, Open file pointer for output.
 *
 * RETURNS: None.
 *
*/
void CallTree_Dump(CallTree * tree, FILE * output)
{
  CallTree_nodeDump(tree, -1, output, 1);
}

/*
 *
 * PURPOSE: Locate a node in the CallTree with the name given.
 *
 * PARAMETERS: CallTree (to get to the fcn list), nodeName (fcn name)
 *
 * RETURNS: CallTree* to node with nodeName, or NULL if not found
 *
*/
CallTree * CallTree_FindNode(CallTree * tree, char * nodeName)
{
	CallTree * Return = NULL;
	
	if (!nodeName)
		return Return;
	if (!*nodeName)
		return Return;
	if (!tree)
		return Return;
		
	for (Return = ListReset(tree->all_fcns); 
		 Return != NULL; 
		 Return = ListNext(tree->all_fcns))
	{
	  if (!strcmp(Return->fcn_name, nodeName))
	  	return Return;
	}
}


/*
 *
 * PURPOSE: Count the number of local functions considered "children" of
 *          the tree, using <direction> to determine which way the relationship
 *          points to choose the correct list.
 *
 * PARAMETERS: CallTree*, int direction
 *
 * RETURNS: count
 *
*/
int CallTree_CountLocalChildren(CallTree*tree, int direction)
{
  List * sub_list;
  int Return;
  CallTree * sub_node;
  
  if (direction == SHOW_FUNCTIONS_CALLED)
    sub_list = tree->calls;
  else
  if (direction == SHOW_FUNCTIONS_WHICH_CALL)
    sub_list = tree->called_by;
  else
    sub_list = NULL;
    
  Return = 0;
  
  for ( sub_node = ListReset(sub_list); 
  	sub_node != NULL; 
  	sub_node = ListNext(sub_list))
  {
    char *dummy1, *dummy2;
    
    if (sub_node->line_num >= 0)
      Return++;
  }
  
  return Return;
}

/*
 *
 * PURPOSE: Mark all nodes clear to reset before performing a search.
 *
 * PARAMETERS: Any node in the call tree, to get the all_fcns list.
 *
 * RETURNS: None.
 *
*/
void CallTree_ClearPath(CallTree * root_node)
{
  CallTree * node;
  
  if (!root_node)
    return;
  
  for ( node = ListReset(root_node->all_fcns);
  	node != NULL;
  	node = ListNext(root_node->all_fcns))
  {
    node->search_result = 0;
    node->search_path = 0;
    node->search_subnode = 0;
  }
}


/*
 *
 * PURPOSE: Beginning at the specified root node, search downwards to
 *          identify all paths to the specified destination function.
 *
 * PARAMETERS: CallTree* (root node), char* (target function), int (mode of search)
 *
 * RETURNS: True if found, False if not found
 *
*/
int CallTree_MarkPath(CallTree * start_node, char * target_name, int mode)
{
  CallTree * sub_node;
  List * sub_list;
  
  if (!start_node)
  {
    fprintf(stderr, "Can not search from NULL\n");
    return 0;
  }
  if (!target_name)
  {
    fprintf(stderr, "Can not search for NULL\n");
    return 0;
  }
  if (!*target_name)
  {
    fprintf(stderr, "Can not search for NULL string\n");
    return 0;
  }
  
#ifdef DEBUG_Calltree_search
  fprintf(stderr, "Searching at %s\n", start_node->fcn_name);
#endif
    
	/*
	** If we have seen this node before, than we know that
	** it is possible that this node, or a sub node of this node,
	** is a match.  Either possibility means that the parent of
	** this node needs to set its subnode value to True.
	*/
  if (start_node->search_path)
  {
    if (start_node->search_result)
	{
		/* If we found the target in this node before, it will still be here. */

#ifdef DEBUG_Calltree_search
       fprintf(stderr, "Subnode '%s' already known to match\n", 
      	 start_node->fcn_name);
#endif

      return 1;
	}
    else if (!start_node->search_subnode)
	{
    	/* If we never found anything along this way, we still won't. */

#ifdef DEBUG_Calltree_search
       fprintf(stderr, "Subnode '%s' already known to not match.\n", 
      	 start_node->fcn_name);
#endif

      return 0;
	}
    else
    {
      	/* If we found the target under this node, it is still there. */
      	
      start_node->search_subnode++;
      
#ifdef DEBUG_Calltree_search
       fprintf(stderr, "Subnode '%s' has %d hits\n", 
      	 start_node->fcn_name, 
      	 start_node->search_subnode);
#endif

      return start_node->search_subnode;
    }

  }
    
  start_node->search_result = 0;	/* this node does not match */
  start_node->search_path = 1; 		/* mark this node searched */
  start_node->search_subnode = 0;	/* haven't looked at subnodes */
  
  /*
  ** Check this node.
  */
  if (!strcmp(start_node->fcn_name, target_name))
  {
    /*
    ** Note:  By skipping the subnodes when we find the call,
    **        we are going to avoid recursive calls which would
    **        put us in a bad loop.
    */
    start_node->search_result = 1;
#ifdef DEBUG_Calltree_search
    fprintf(stderr, "*** Found.\n");
#endif
    return 1;
  }
  else
  {
    /*
    ** Check subnodes
    */
    
    switch (mode)
    {
      case SHOW_FUNCTIONS_CALLED:
		sub_list = start_node->calls;
      break;

      case SHOW_FUNCTIONS_WHICH_CALL:
		sub_list = start_node->called_by;
      break;

      default:
        fprintf(stderr, "Do not know how to search with mode %d\n", mode);
		return 0;
      break;
    } 
    
#ifdef DEBUG_Calltree_search
    if (ListSize(sub_list))
      fprintf(stderr, "Checking subnodes\n");
#endif

    for (sub_node = ListReset(sub_list);
	 sub_node != NULL;
	 sub_node = ListNext(sub_list))
    {
    	int new_hits;
    	
       new_hits = CallTree_MarkPath(sub_node, target_name, mode);
       start_node->search_subnode += new_hits;
       		
#ifdef DEBUG_Calltree_search
       if (new_hits > 0)
         fprintf(stderr, "Subtree %s has %d hits\n",
         	sub_node->fcn_name, new_hits);
#endif
    }
  }

#ifdef DEBUG_Calltree_search
       if (start_node->search_subnode > 0)
         fprintf(stderr, "Subtree %s has %d hits\n",
         	start_node->fcn_name, start_node->search_subnode);
#endif
  
  return start_node->search_subnode;
}
