/*
 *
 *  FILENAME: client.c
 *
 *  COPYRIGHT: January 1996
 *
 *  PURPOSE: Provides a stub main procedure for use with clients of the
 *           Nedit system.
 *
 *  AUTHOR: Doug Hellmann
 *
 *  DATE: 18 January 1996
 *
 *  COMMENTS: 
 *
*/

#include "client.h"
#include "shell.h"
#include <sys/wait.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef IBM
#define NBBY 8
#include <sys/select.h>
#endif
#include <errno.h>
#include <signal.h>
#include <misc.h>
#include <stdlib.h>

Display *TheDisplay;
WindowInfo * WindowList;

int ServerMode = TRUE;

extern char * GetPrefEditorCommand(void);

/*
** Build preferences database for local program.
*/
XrmDatabase CreateLocalPrefDB(
		char * pref_file_name, 
		char * app_name,
		XrmOptionDescRec * op_table,
		int op_table_count, 
		int *argcInOut, 
		char **argvInOut)
{
    return CreatePreferencesDatabase(pref_file_name, app_name, 
	    op_table, op_table_count, 
	    (unsigned int *)argcInOut, argvInOut);
}

void RestoreLocalPrefs(char * app_name, 
			PrefDescripRec *pref_descrip, int pref_count,
			XrmDatabase prefDB, XrmDatabase appDB)
{
    XFontStruct *font;
#ifndef VMS
    RestorePreferences(prefDB, appDB, app_name,
    	    APP_CLASS, pref_descrip, pref_count);
#else
    RestorePreferences(prefDB, appDB, app_name,
    	    APP_CLASS, pref_descrip, pref_count);
#endif /* VMS */

}

/*
 *
 * PURPOSE: Exit a client program.
 * 
 * CALLED AS RESULT OF: Button or menu entry for exiting the program.
 *
*/
void ClientExitCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  exit(0);
}

/*
 * Separate text into lines to be added to the
 * list widget as separate items.
 *
*/
void TextToListAsItems(char * text, Widget list)
{
  int i;
  char * line;
  char * copy;
  
  if (!list)
    return;
  
  if (!text)
    return;
    
  copy = (char*) XtMalloc ( sizeof(char) * (strlen(text) + 1) );
  strcpy(copy, text);
    
  for ( line = strtok(copy, "\n");
        line != NULL;
        line = strtok(NULL, "\n"))
  {
    XmString xstr;
    
    xstr = XmStringCreateLtoR(line, XmSTRING_DEFAULT_CHARSET);
    XmListAddItemUnselected(list, xstr, 0);
    XmStringFree(xstr);    
  }
  
  XmListSetBottomPos(list, 0);
  	  
  XtFree(copy);
}

static void cancelCB(Widget w, int *abortFlag, caddr_t call_data)
{
    *abortFlag = True;
}

static Widget createWorkingDialog(Widget parent, int *abortFlag)
{
    Widget dlg;
    XmString st;
    
    dlg = XmCreateWorkingDialog(parent, "working", NULL, 0);
    XtUnmanageChild(XmMessageBoxGetChild(dlg, XmDIALOG_OK_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(dlg, XmDIALOG_HELP_BUTTON));
    XtAddCallback(XmMessageBoxGetChild(dlg, XmDIALOG_CANCEL_BUTTON),
    	    XmNactivateCallback, (XtCallbackProc)cancelCB, (XtPointer) abortFlag);
    XtVaSetValues(XmMessageBoxGetChild(dlg, XmDIALOG_MESSAGE_LABEL),
    	    XmNlabelString,
    	    st=XmStringCreateSimple("Waiting for command to complete"), 0);
    XtVaSetValues(XtParent(dlg), XmNtitle, " ", 0);
    XmStringFree(st);
    XtManageChild(dlg);
    
    return dlg;
}

/*
** Issue a a command, feed it the string input, and either return or
** store its output in a list widget (if listW is NULL, return the output
** from the command as a string to be freed by the caller, otherwise store the
** output in the list widget listW).  Flags may
** be set to ACCUMULATE, and/or ERROR_DIALOGS.  ACCUMULATE causes output
** from the command to be saved up until the command completes.  ERROR_DIALOGS
** presents stderr output separately in popup a dialog, and also reports
** failed exit status as a popup dialog including the command output.
** ERROR_DIALOGS should only be used along with ACCUMULATE.
*/
char *issueCommandToList(Widget dlogParent, char *command, char *input,
	int flags, Widget listW, int *success)
{
    int status, stdinFD, stdoutFD, stderrFD, maxFD;
    int len;
    pid_t childPid;
    int nWritten, nRead;
    buffer *buf, *outBufs = NULL, *errBufs = NULL;
    char *outText, *errText, *inPtr = input;
    int resp, inLength = strlen(input);
    fd_set readfds, writefds;
    struct timeval timeout;
    Widget workingDlg = NULL;
    int failure, errorReport, cancel = False;
    int outEOF = False, errEOF = (flags & ERROR_DIALOGS) ? False : True;
    int abortFlag = False;
    XtAppContext context = XtDisplayToApplicationContext(TheDisplay);
    time_t startTime = time(NULL);
    time_t lastIOTime = time(NULL);
    
    *success = False;
    
    /* verify consistency of input parameters */
    if ((flags & ERROR_DIALOGS) && !ACCUMULATE)
    	return NULL;
    
    /* put up a watch cursor over the waiting window */
    BeginWait(dlogParent);

    /* fork the subprocess and issue the command */
    childPid = forkCommand(dlogParent, command, &stdinFD, &stdoutFD, 
    	    (flags & ERROR_DIALOGS) ? &stderrFD: NULL);
    
    /* set the pipes connected to the process for non-blocking i/o */
    if (fcntl(stdinFD, F_SETFL, O_NONBLOCK) < 0)
    	perror("NEdit: Internal error (fcntl)");
    if (fcntl(stdoutFD, F_SETFL, O_NONBLOCK) < 0)
    	perror("NEdit: Internal error (fcntl1)");
    if (flags & ERROR_DIALOGS) {
	if (fcntl(stderrFD, F_SETFL, O_NONBLOCK) < 0)
    	    perror("NEdit: Internal error (fcntl2)");
    }
    
    /* if there's nothing to write to the process' stdin, close it now */
    if (inLength == 0)
    	close(stdinFD);
    	
    /*
    ** Loop writing input to process and reading input from it until
    ** end-of-file is reached on both stdout and stderr pipes.
    */
    while (!(outEOF && errEOF)) {
    
	/* Process all pending X events, regardless of whether
	   select says there are any. */
	while (XtAppPending(context)) {
	    XtAppProcessEvent(context, XtIMAll);
	}
   
	/* If the process is taking too long, put up a working dialog */
	if (workingDlg == NULL &&  time(NULL) >= startTime + 6)
	    workingDlg = createWorkingDialog(dlogParent, &abortFlag);
	
	/* Check the abort flag set by the working dialog when the user
	   presses the cancel button in the dialog */
	if (abortFlag) {
	    kill(-childPid, SIGTERM);
	    
	    /*
	    ** Don't know if or why we might need this.
	    */
	    waitpid(childPid, &status, WUNTRACED);
	    
	    freeBufList(&outBufs);
	    freeBufList(&errBufs);
    	    close(stdoutFD);
    	    if (flags & ERROR_DIALOGS)
    	    	close(stderrFD);
    	    EndWait(dlogParent);
	    return NULL;
	}
	
	/* Block and wait for something to happen, but wakeup every second
	   to check abort flag and waiting dialog and output timers */
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_SET(ConnectionNumber(TheDisplay), &readfds);
	maxFD = ConnectionNumber(TheDisplay);
	if (!outEOF) {
	    FD_SET(stdoutFD, &readfds);
	    maxFD = stdoutFD > maxFD ? stdoutFD : maxFD;
	}
	if (!errEOF) {
	    FD_SET(stderrFD, &readfds);
	    maxFD = stderrFD > maxFD ? stderrFD : maxFD;
	}
	if (inLength > 0) {
	    FD_SET(stdinFD, &writefds);
	    maxFD = stdinFD > maxFD ? stdinFD : maxFD;
	}
	if (select(maxFD+1, &readfds, &writefds, NULL, &timeout) == -1) {
	    if (EINTR != errno)
		perror("NEdit: select");
	}
	
	/* Dump intermediate output to window if the process is taking a long
	   time.  If there's read data pending, hold off so that the output
	   is in bigger chunks (each is an undo operation) */
	if (!(flags & ACCUMULATE) && listW!=NULL &&
		!FD_ISSET(stdoutFD, &readfds) && time(NULL) >= lastIOTime + 3) {
    	    outText = coalesceOutput(&outBufs);
	    len = strlen(outText);
	    if (len != 0) {
	        TextToListAsItems(outText, listW);
	    }
		XtFree(outText);
	}
	
	/* write input to the sub-process stdin, close stdin when finished */
	if (FD_ISSET(stdinFD, &writefds) && inLength > 0) {
	    nWritten = write(stdinFD, inPtr, inLength);
	    if (nWritten == -1) {
    		if (errno != EWOULDBLOCK) {
    		    perror("NEdit: Write to filter command failed");
    		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
    		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
    		    return NULL;
    		}
	    } else {
		inPtr += nWritten;
		inLength -= nWritten;
		if (inLength <= 0)
    		    close(stdinFD);
    	    }
    	}

    	/* read the output from stdout and create a linked list of buffers */
    	if (FD_ISSET(stdoutFD, &readfds)) {
    	    buf = (buffer *)XtMalloc(sizeof(buffer));
    	    nRead = read(stdoutFD, buf->contents, FILTER_BUF_SIZE);

    	    if (nRead == -1) { /* error */
		if (errno != EWOULDBLOCK) {
		    perror("NEdit: Error reading filter output");
		    XtFree((char *)buf);
		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
		    return NULL;
		}
    	    } else if (nRead == 0) { /* eof */
    		outEOF = True;
    		XtFree((char *)buf);
    	    } else { /* characters read */
    		buf->length = nRead;
    		addOutput(&outBufs, buf);
   	    }
    	}

    	/* read the output from stderr and create a linked list of buffers */
    	if ((flags & ERROR_DIALOGS) && FD_ISSET(stderrFD, &readfds)) {
    	    buf = (buffer *)XtMalloc(sizeof(buffer));
    	    nRead = read(stderrFD, buf->contents, FILTER_BUF_SIZE);
    	    if (nRead == -1) { /* error */
		if (errno != EWOULDBLOCK && errno != EAGAIN) {
		    perror("NEdit: Error reading filter error stream");
		    XtFree((char *)buf);
		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
		    return NULL;
		}
    	    } else if (nRead == 0) { /* eof */
    		errEOF = True;
    		XtFree((char *)buf);
    	    } else { /* chars read */
    		buf->length = nRead;
    		addOutput(&errBufs, buf);
   	    }
    	}
    }
    close(stdoutFD);
    if (flags & ERROR_DIALOGS)
    	close(stderrFD);
    
    /* assemble the output from the process' stderr and stdout streams into
       null terminated strings, and free the buffer lists used to collect it */
    outText = coalesceOutput(&outBufs);
    if (flags & ERROR_DIALOGS)
    	errText = coalesceOutput(&errBufs);

    /* pop down the working dialog if it's up */
    if (workingDlg != NULL)
    	XtUnmanageChild(workingDlg);
    
    /* Wait for the child process to complete and get its return status */
    waitpid(childPid, &status, 0);
    
    EndWait(dlogParent);
    
    XBell(XtDisplay(dlogParent), 50);
    
    /* Present error and stderr-information dialogs.  If a command returned
       error output, or if the process' exit status indicated failure,
       present the information to the user. */
    if (flags & ERROR_DIALOGS) {
	failure = WIFEXITED(status) && WEXITSTATUS(status) != 0;
	errorReport = *errText != '\0';
	if (failure && errorReport) {
    	    removeTrailingNewlines(errText);
    	    truncateString(errText, DF_MAX_MSG_LENGTH);
    	    resp = DialogF(DF_WARN, dlogParent, 2, "%s",
    	    	    "Cancel", "Proceed", errText);
    	    cancel = resp == 1;
	} else if (failure) {
    	    truncateString(outText, DF_MAX_MSG_LENGTH-70);
    	    resp = DialogF(DF_WARN, dlogParent, 2,
    	       "Command reported failed exit status.\nOutput from command:\n%s",
    		    "Cancel", "Proceed", outText);
    	    cancel = resp == 1;
	} else if (errorReport) {
    	    removeTrailingNewlines(errText);
    	    truncateString(errText, DF_MAX_MSG_LENGTH);
    	    resp = DialogF(DF_INF, dlogParent, 2, "%s",
    	    	    "Proceed", "Cancel", errText);
    	    cancel = resp == 2;
	}
	XtFree(errText);
	if (cancel) {
	    XtFree(outText);
    	    return NULL;
	}
    }
    
    *success = True;
    
    /* insert the remaining output, and move the insert point to the end */
    if (listW != NULL) {

       TextToListAsItems(outText, listW);
        
	XtFree(outText);
	return NULL;
    } else
    	return outText;
}

/*
 *
 * PURPOSE: Call a remote editor on a file and jump to the specified line.
 *
 * PARAMETERS: path, filename, line number
 *
 * RETURNS: none
 *
*/
void RemoteEditFile(char * path, char * filename, int line)
{
  char * editor = (char*) GetPrefEditorCommand();
  char * p;
  char buf[MAXPATHLEN];
  char mini_buf[32];
  int i;
  
  if (!path)
    return;
  if (!filename)
    return;
  if (line < 0)
    return;
    
  buf[0] = '\0';
    
  for ( p = editor, i = 0;
  	*p != '\0';
  	p++)
  {
    if (*p == '%')
    {
    	/* handle parts */
    	p++;
    	switch (*p)
    	{
    	  case 'l':	/* line number */
    	    sprintf(mini_buf, "%d", line);
    	    strcat(buf, mini_buf);
    	    i+=strlen(mini_buf);
    	  break;
    	  
    	  case 'f':	/* file name */
    	    strcat(buf, path);
    	    i+=strlen(path);
    	    strcat(buf, filename);
    	    i+=strlen(filename);
    	  break;
    	  
    	  case 't':	/* tag name */
    	  break;
    	  
    	  default:
    	    buf[i] = '%';
    	    buf[i+1] = *p;
    	    buf[i+2] = '\0';
    	    i+=2;
    	  break;
    	}
    }
    else
    {
      buf[i] = *p;
      buf[i+1] = '\0';
      i++;
    }
  }
  
#ifdef  DEBUG_Client
  fprintf(stderr, "Editor buf is '%s'\n", buf);
#endif
  
  system(buf);
}

/*
 * Add an item to a list in alphabetical order.
 *
 * This is adapted from O'Reilly V6A
 *
*/
void XmListAddItemAlpha(Widget list_w, char * newtext)
{
   char * text;
   XmString str, *strlist;
   int u_bound, l_bound = 0;
   
   if (!newtext)
     return;
   if (!*newtext)
     return;
     
   XtVaGetValues(list_w,
   	XmNitemCount, &u_bound,
   	XmNitems, &strlist,
   	NULL);
   u_bound--;
   
   /* perform binary search */
   while (u_bound >= l_bound)
   {
     int i = l_bound + (u_bound - l_bound) / 2;
     
     if (!XmStringGetLtoR(strlist[i], XmSTRING_DEFAULT_CHARSET, &text))
       break;
     if (strcmp(text, newtext) > 0)
       u_bound = i - 1;
     else
       l_bound = i + 1;
     XtFree(text);
   }
   
   str = XmStringCreateSimple(newtext);
   XmListAddItem(list_w, str, l_bound+1);
   XmStringFree(str);
}

/*
 * FileToString
 *
 * PURPOSE : Return a file's contents as a pointer to a string.
 *
 * PARAMS  : filename - file to be read in as a string.
 *
*/
char * 
FileToString(filename)
  char * filename;
{
  /*int n;*/
  off_t n;
  FILE *fileptr;
  char * buf;
  struct stat filebuf;

  /* check for bad filenames */

  if (filename == NULL)
    return NULL;
  if (filename[0] == '\0')
    return NULL;

  /* handle directories */

  if (filename[strlen(filename) - 1] == '/')
    return NULL;

  /* 
     Find out how big the file is, if there is an error,
     return NULL.
  */

  if (stat(filename, &filebuf) == -1)
    return NULL;

  n = filebuf.st_size;
  if( (buf = (char *) malloc(sizeof(char) * (n+1)) ) == NULL)
  {
    fprintf(stderr, "cannot allocate %d bytes in FileToString\n", n);
    return NULL;
  }
  fileptr = fopen(filename, "r");

  if (fileptr == NULL)
  {
    free(buf);
    return NULL;
  }

  fread(buf, sizeof(char), n, fileptr);
  buf[n] = '\0';

  fclose(fileptr);

  return buf;
} /* FileToString */



/*
** Issue a a command, feed it the string input, and either return or
** store its output in a text widget (if textW is NULL, return the output
** from the command as a string to be freed by the caller, otherwise store the
** output between leftPos and rightPos in the text widget textW).  Flags may
** be set to ACCUMULATE, and/or ERROR_DIALOGS.  ACCUMULATE causes output
** from the command to be saved up until the command completes.  ERROR_DIALOGS
** presents stderr output separately in popup a dialog, and also reports
** failed exit status as a popup dialog including the command output.
** ERROR_DIALOGS should only be used along with ACCUMULATE.
*/
/*static*/ 
char *issueCommandToText(Widget dlogParent, char *command, char *input,
	int flags, Widget textW, int replaceLeft, int replaceRight,
	int *success)
{
    int status, stdinFD, stdoutFD, stderrFD, maxFD;
    int len, leftPos = replaceLeft, rightPos = replaceRight;
    pid_t childPid;
    int nWritten, nRead;
    buffer *buf, *outBufs = NULL, *errBufs = NULL;
    char *outText, *errText, *inPtr = input;
    int resp, inLength = strlen(input);
    fd_set readfds, writefds;
    struct timeval timeout;
    Widget workingDlg = NULL;
    int failure, errorReport, cancel = False;
    int outEOF = False, errEOF = (flags & ERROR_DIALOGS) ? False : True;
    int abortFlag = False;
    XtAppContext context = XtDisplayToApplicationContext(TheDisplay);
    time_t startTime = time(NULL);
    time_t lastIOTime = time(NULL);
    
    *success = False;
    
    /* verify consistency of input parameters */
    if ((flags & ERROR_DIALOGS) && !ACCUMULATE)
    	return NULL;
    
    /* put up a watch cursor over the waiting window */
    BeginWait(dlogParent);

    /* fork the subprocess and issue the command */
    childPid = forkCommand(dlogParent, command, &stdinFD, &stdoutFD, 
    	    (flags & ERROR_DIALOGS) ? &stderrFD: NULL);
    
    /* set the pipes connected to the process for non-blocking i/o */
    if (fcntl(stdinFD, F_SETFL, O_NONBLOCK) < 0)
    	perror("NEdit: Internal error (fcntl)");
    if (fcntl(stdoutFD, F_SETFL, O_NONBLOCK) < 0)
    	perror("NEdit: Internal error (fcntl1)");
    if (flags & ERROR_DIALOGS) {
	if (fcntl(stderrFD, F_SETFL, O_NONBLOCK) < 0)
    	    perror("NEdit: Internal error (fcntl2)");
    }
    
    /* if there's nothing to write to the process' stdin, close it now */
    if (inLength == 0)
    	close(stdinFD);
    	
    /*
    ** Loop writing input to process and reading input from it until
    ** end-of-file is reached on both stdout and stderr pipes.
    */
    while (!(outEOF && errEOF)) {
    
	/* Process all pending X events, regardless of whether
	   select says there are any. */
	while (XtAppPending(context)) {
	    XtAppProcessEvent(context, XtIMAll);
	}
   
	/* If the process is taking too long, put up a working dialog */
	if (workingDlg == NULL &&  time(NULL) >= startTime + 6)
	    workingDlg = createWorkingDialog(dlogParent, &abortFlag);
	
	/* Check the abort flag set by the working dialog when the user
	   presses the cancel button in the dialog */
	if (abortFlag) {
	    kill(-childPid, SIGTERM);
	    freeBufList(&outBufs);
	    freeBufList(&errBufs);
    	    close(stdoutFD);
    	    if (flags & ERROR_DIALOGS)
    	    	close(stderrFD);
    	    EndWait(dlogParent);
	    return NULL;
	}
	
	/* Block and wait for something to happen, but wakeup every second
	   to check abort flag and waiting dialog and output timers */
	timeout.tv_sec = 1;
	timeout.tv_usec = 0;
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_SET(ConnectionNumber(TheDisplay), &readfds);
	maxFD = ConnectionNumber(TheDisplay);
	if (!outEOF) {
	    FD_SET(stdoutFD, &readfds);
	    maxFD = stdoutFD > maxFD ? stdoutFD : maxFD;
	}
	if (!errEOF) {
	    FD_SET(stderrFD, &readfds);
	    maxFD = stderrFD > maxFD ? stderrFD : maxFD;
	}
	if (inLength > 0) {
	    FD_SET(stdinFD, &writefds);
	    maxFD = stdinFD > maxFD ? stdinFD : maxFD;
	}
	if (select(maxFD+1, &readfds, &writefds, NULL, &timeout) == -1) {
	    if (EINTR != errno)
		perror("NEdit: select");
	}
	
	/* Dump intermediate output to window if the process is taking a long
	   time.  If there's read data pending, hold off so that the output
	   is in bigger chunks (each is an undo operation) */
	if (!(flags & ACCUMULATE) && textW!=NULL &&
		!FD_ISSET(stdoutFD, &readfds) && time(NULL) >= lastIOTime + 3) {
    	    outText = coalesceOutput(&outBufs);
	    len = strlen(outText);
	    if (len != 0) {
	    	XmTextReplace(textW, leftPos, rightPos, outText);
		XtFree(outText);
		leftPos += len;
		rightPos = leftPos;
	    }
	}
	
	/* write input to the sub-process stdin, close stdin when finished */
	if (FD_ISSET(stdinFD, &writefds) && inLength > 0) {
	    nWritten = write(stdinFD, inPtr, inLength);
	    if (nWritten == -1) {
    		if (errno != EWOULDBLOCK) {
    		    perror("NEdit: Write to filter command failed");
    		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
    		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
    		    return NULL;
    		}
	    } else {
		inPtr += nWritten;
		inLength -= nWritten;
		if (inLength <= 0)
    		    close(stdinFD);
    	    }
    	}

    	/* read the output from stdout and create a linked list of buffers */
    	if (FD_ISSET(stdoutFD, &readfds)) {
    	    buf = (buffer *)XtMalloc(sizeof(buffer));
    	    nRead = read(stdoutFD, buf->contents, FILTER_BUF_SIZE);
    	    if (nRead == -1) { /* error */
		if (errno != EWOULDBLOCK) {
		    perror("NEdit: Error reading filter output");
		    XtFree((char *)buf);
		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
		    return NULL;
		}
    	    } else if (nRead == 0) { /* eof */
    		outEOF = True;
    		XtFree((char *)buf);
    	    } else { /* characters read */
    		buf->length = nRead;
    		addOutput(&outBufs, buf);
   	    }
    	}

    	/* read the output from stderr and create a linked list of buffers */
    	if ((flags & ERROR_DIALOGS) && FD_ISSET(stderrFD, &readfds)) {
    	    buf = (buffer *)XtMalloc(sizeof(buffer));
    	    nRead = read(stderrFD, buf->contents, FILTER_BUF_SIZE);
    	    if (nRead == -1) { /* error */
		if (errno != EWOULDBLOCK && errno != EAGAIN) {
		    perror("NEdit: Error reading filter error stream");
		    XtFree((char *)buf);
		    freeBufList(&outBufs);
	    	    freeBufList(&errBufs);
		    if (workingDlg != NULL)
    	    		XtUnmanageChild(workingDlg);
    	    	    EndWait(dlogParent);
		    return NULL;
		}
    	    } else if (nRead == 0) { /* eof */
    		errEOF = True;
    		XtFree((char *)buf);
    	    } else { /* chars read */
    		buf->length = nRead;
    		addOutput(&errBufs, buf);
   	    }
    	}
    }
    close(stdoutFD);
    if (flags & ERROR_DIALOGS)
    	close(stderrFD);
    
    /* assemble the output from the process' stderr and stdout streams into
       null terminated strings, and free the buffer lists used to collect it */
    outText = coalesceOutput(&outBufs);
    if (flags & ERROR_DIALOGS)
    	errText = coalesceOutput(&errBufs);

    /* pop down the working dialog if it's up */
    if (workingDlg != NULL)
    	XtUnmanageChild(workingDlg);
    
    /* Wait for the child process to complete and get its return status */
    waitpid(childPid, &status, 0);
    
    EndWait(dlogParent);
    
    /* Present error and stderr-information dialogs.  If a command returned
       error output, or if the process' exit status indicated failure,
       present the information to the user. */
    if (flags & ERROR_DIALOGS) {
	failure = WIFEXITED(status) && WEXITSTATUS(status) != 0;
	errorReport = *errText != '\0';
	if (failure && errorReport) {
    	    removeTrailingNewlines(errText);
    	    truncateString(errText, DF_MAX_MSG_LENGTH);
    	    resp = DialogF(DF_WARN, dlogParent, 2, "%s",
    	    	    "Cancel", "Proceed", errText);
    	    cancel = resp == 1;
	} else if (failure) {
    	    truncateString(outText, DF_MAX_MSG_LENGTH-70);
    	    resp = DialogF(DF_WARN, dlogParent, 2,
    	       "Command reported failed exit status.\nOutput from command:\n%s",
    		    "Cancel", "Proceed", outText);
    	    cancel = resp == 1;
	} else if (errorReport) {
    	    removeTrailingNewlines(errText);
    	    truncateString(errText, DF_MAX_MSG_LENGTH);
    	    resp = DialogF(DF_INF, dlogParent, 2, "%s",
    	    	    "Proceed", "Cancel", errText);
    	    cancel = resp == 2;
	}
	XtFree(errText);
	if (cancel) {
	    XtFree(outText);
    	    return NULL;
	}
    }
    
    *success = True;
    
    /* insert the remaining output, and move the insert point to the end */
    if (textW != NULL) {
	XmTextReplace(textW, leftPos, rightPos, outText);
	XmTextSetInsertionPosition(textW, leftPos + strlen(outText));
	XtFree(outText);
	return NULL;
    } else
    	return outText;
}
