/*
 *
 *  FILENAME: findfile.c
 *
 *  PURPOSE: Present the user with a window in which they can enter
 *           a filename pattern and a root directory.  These will be
 *           used as arguments to find, and the results will be displayed.
 *
 *  AUTHOR: Doug Hellmann
 *
 *  DATE: Thu Aug 24 13:57:05 EDT 1995
 *
 *  COMMENTS: 
 *
*/

#include "client.h"

extern WindowInfo *WindowList;
extern Display *TheDisplay;
Widget findFileDialog = NULL;
XtAppContext context;

Widget CreateFindFileDialog();
extern Widget NeditCreateToplevelDialog(char*, char*, char*);

static Widget findFileList, 	/* list for find output */
              findFileTextSearch, 	/* for user to tell us what to look for */
              findFileTextFiles,    /* for user to tell us which files to search */
              findFileOk, 		/* ok button (executes search) */
              findFileCancel;	/* cancel button (dismisses dialog) */

#ifdef LOCAL_PREF_FILE_NAME
#undef LOCAL_PREF_FILE_NAME
#endif
#define LOCAL_PREF_FILE_NAME ".findfile"

#ifdef APP_NAME
#undef APP_NAME
#endif
#define APP_NAME "findfile" /* application name for loading resources */


char HeaderText[] = "\
# Preferences file for FindFile\n\
#\n\
# This file is created automatically by FindFile.\n\
#\n\
# Codes to be embedded in editor option: \n\
#\n\
#    %f - filename\n\
#    %t - tag name\n\
#    %l - line number\n\
#\n";

char *fallbackResources[] = {
    "NEdit*menuBar.marginHeight: 1",
    "NEdit*pane.sashHeight: 11",
    "NEdit*pane.sashWidth: 11",
    "NEdit*text.selectionArrayCount: 3",
    
    "NEdit*XmText.background: white",
    "NEdit*XmText.foreground: black",
    "NEdit*XmList.background: white",
    "NEdit*XmList.foreground: black",
    "NEdit*background: grey",
    "NEdit*foreground: black",

    "NEdit*fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*find_file_list.fontList: -adobe-courier-medium-r-normal-*-12-120-75-75-m-70-iso8859-1",
    "NEdit*XmList.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*XmText.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",
    "NEdit*XmTextField.fontList:-adobe-courier-medium-r-normal-*-12-*-*-*-*-*-*-*",

    "NEdit*background: #b3b3b3",
    "NEdit*foreground: black",
    "NEdit*statisticsLine.background: #b3b3b3",
    "NEdit*text.background: #e5e5e5",
    "NEdit*text.foreground: black",
    "NEdit*XmText*foreground: black",
    "NEdit*XmText*background: #cccccc",
    "NEdit*XmList*foreground: black",
    "NEdit*XmList*background: #cccccc",
    "NEdit*XmTextField*background: #cccccc",
    "NEdit*XmTextField*foreground: black",
    
    "NEdit*XmToggleButton.selectorColor: red",

    NULL
};

char cmdLineHelp[] =
#ifndef VMS
"Usage:  findfile  [-editor editor_name] [-font font]\n\
               [-display [host]:server[.screen]\n\
               [-geometry geometry] [-xrm resourcestring] [file...]\n";
#else
"";
#endif /*VMS*/

XrmOptionDescRec LocalOpTable[] = {
    {"-editor", ".editor", XrmoptionSepArg, (caddr_t)NULL},
};

struct localPrefData {
    char editor[MAXPATHLEN];		/* name of editor command under X */
} LocalPrefData;

PrefDescripRec LocalPrefDescrip[] = {
    {"editor", "editor", PREF_STRING,
    	"xterm -title %f -exec vi +%l %f &",
    	LocalPrefData.editor, (void *)sizeof(LocalPrefData.editor), True},
};

/*
** Functions to return preferences values.
*/
char * GetPrefEditorCommand(void)
{
  return LocalPrefData.editor;
}

/*
** Main function
*/

int main(int argc, char **argv)
{
    int i, fileSpecified;
    XrmDatabase prefDB, localPrefDB;
    
    /* Initialize toolkit and open display. */
    XtToolkitInitialize();
    context = XtCreateApplicationContext();

    /* Set up a warning handler to trap obnoxious Xt grab warnings */
    SuppressPassiveGrabWarnings();

    /* Set up default resources if no app-defaults file is found */
    XtAppSetFallbackResources(context, fallbackResources);
    
#if XmVersion >= 1002
    /* Allow users to change tear off menus with X resources */
    XmRepTypeInstallTearOffModelConverter();
#endif
    
#ifdef VMS
    /* Convert the command line to Unix style (This is not an ideal solution) */
    ConvertVMSCommandLine(&argc, &argv);
#endif /*VMS*/
    
    /* Read the preferences file and command line into a database */
    prefDB = CreateNEditPrefDB(&argc, argv);
    localPrefDB = CreateLocalPrefDB(LOCAL_PREF_FILE_NAME, APP_NAME, 
    			LocalOpTable, XtNumber(LocalOpTable), &argc, argv);

    /* Open the display and read X database and remaining command line args */
    TheDisplay = XtOpenDisplay (context, NULL, APP_NAME, APP_CLASS, NULL,
    	    0, &argc, argv);
    if (!TheDisplay) {
	XtWarning ("findfile: Can't open display\n");
	exit(0);
    }

    /* Store preferences from the command line and .findfile file, 
       and set the appropriate preferences */
    RestoreNEditPrefs(prefDB, XtDatabase(TheDisplay));
    RestoreLocalPrefs(APP_NAME, LocalPrefDescrip, XtNumber(LocalPrefDescrip),
    			localPrefDB, XtDatabase(TheDisplay));

    SetPointerCenteredDialogs(GetPrefRepositionDialogs());
    
/* 
    SavePreferences(TheDisplay, LOCAL_PREF_FILE_NAME, HeaderText,
    	    LocalPrefDescrip, XtNumber(LocalPrefDescrip)); 
 */

    	    
    /* Process any command line arguments (-tags, and files to edit) not
       already processed by RestoreNEditPrefs. */
    fileSpecified = FALSE;

    if (!findFileDialog)
      findFileDialog = CreateFindFileDialog();
    XtPopup(findFileDialog, XtGrabNone);

    /* Process events. */
    XtAppMainLoop(context);

    return 0;
}


/*
 *
 * PURPOSE: Perform search.
 *
 * CALLED AS RESULT OF: Activate callback on text widget or Ok button.
 *
*/
void findFileSearchCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  char * pattern;
  char * root_dir;
  FILE * search;
  char cmd[1024];
  char buf[128];
  int success;
  
  pattern = XmTextGetString(findFileTextFiles);
  if (!pattern)
    return;
  if (!*pattern)
    return;
    
  root_dir = XmTextGetString(findFileTextSearch);
  if (!root_dir)
  {
    XtFree(pattern);
    return;
  }
  if (!*root_dir)
  {
    XtFree(pattern);
    return;
  }
  
  XmListDeleteAllItems(findFileList);
  
  /*
   * Always pass /dev/null as a parameter so that find has more than one
   * file name on the command line, and it will always print the file name.
   * It sure would be nice if there were a command line argument to do that.
  */
  cmd[0] = '\0';
  strcat(cmd, "find '");
  strcat(cmd, root_dir);
  if (root_dir[strlen(root_dir) - 1] != '/')
    strcat(cmd, "/");
  strcat(cmd, "' -name '");
  strcat(cmd, pattern);
#ifdef IBM
  strcat(cmd, "' -ls");
#else
  strcat(cmd, "' -print");
#endif

#ifdef DEBUG_Findfile
  fprintf(stderr, "Executing '%s'\n", cmd);
#endif

  issueCommandToList(findFileDialog, cmd, "", 0, findFileList, &success);   
  
  XtFree(pattern);
  XtFree(root_dir);
}

/*
 *
 * PURPOSE: When the user selects an item, we take them to the source file
 *          at that point.
 *
 * CALLED AS RESULT OF: User selection in list (single select or default 
 *                      action).
 *
*/
void findFileGotoCB(Widget w, XtPointer xtp1, XtPointer xtp2)
{
  char * buf, *file_buf = NULL;
  int i, l;
  XmListCallbackStruct * cbs = (XmListCallbackStruct*)xtp2;
  char filename[MAXPATHLEN], pathname[MAXPATHLEN];

  XmStringGetLtoR(cbs->item, XmSTRING_DEFAULT_CHARSET, &buf);
    
  for (file_buf = buf + strlen(buf) - 2;
       (*file_buf != ' ') && 
         (*file_buf != '\0') &&
         (file_buf != buf);
       file_buf--);
  file_buf++;
  
  for (i = 0, l = strlen(file_buf); i < l; i++)
  {
    if ( (file_buf[i] == '\n') || (file_buf[i] == ' ') )
      file_buf[i] = '\0';
  }

  ParseFilename(file_buf, filename, pathname);
  
#ifdef  DEBUG_Findfile
    fprintf(stderr, "'%s' gives:\n", file_buf);
    fprintf(stderr, "filename: '%s'\n", filename);
    fprintf(stderr, "pathname: '%s'\n", pathname);
#endif
  
  RemoteEditFile(pathname, filename, 0);
  
  XtFree(buf);
}


/*
 *
 * PURPOSE: Creates the window for the find interface.
 *
 * PARAMETERS: None.
 *
 * RETURNS: Dialog (child) created.
 *
*/

Widget CreateFindFileDialog()
{
  Widget Return;
  Pixmap iconPixmap = 0, maskPixmap = 0;
  Widget form, sep;
  Widget tags_list_window;
  Widget label;
  Widget frame1, frame2, frame3;
  Dimension width, height;
  char cwd[MAXPATHLEN];
  
  Return = NeditCreateToplevelDialog("NEdit_find_file_list", 
  		"NEdit Find File", "Find File");
  		
      /* the form is the root widget */

  form = XtVaCreateWidget("find_file_list_form", xmFormWidgetClass, 
    			      Return,
    			      XmNtopOffset, 2,
    			      XmNbottomOffset, 2,
    			      XmNleftOffset, 2,
    			      XmNrightOffset, 2,
    			      NULL);

      /* create the action area buttons */

  findFileCancel = XtVaCreateManagedWidget("Exit", xmPushButtonWidgetClass,
    			      form,
    			      XmNrightAttachment, XmATTACH_FORM,
    			      XmNbottomAttachment, XmATTACH_FORM,
    			      XmNbottomOffset, 4,
    			      XmNrightOffset, 4,
    			      NULL);
  XtAddCallback(findFileCancel, XmNactivateCallback, 
		      ClientExitCB, (XtPointer) findFileDialog);

  findFileOk = XtVaCreateManagedWidget("Find", xmPushButtonWidgetClass,
    			      form,
    			      XmNleftAttachment, XmATTACH_FORM,
    			      XmNbottomAttachment, XmATTACH_FORM,
    			      XmNbottomOffset, 4,
    			      XmNleftOffset, 4,
    			      XmNshowAsDefault, True,
    			      NULL);
  XtAddCallback(findFileOk, XmNactivateCallback, 
		      (XtCallbackProc) findFileSearchCB, NULL);

  sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, form,
    			      XmNleftAttachment, XmATTACH_FORM,
    			      XmNrightAttachment, XmATTACH_FORM,
    			      XmNbottomAttachment, XmATTACH_WIDGET,
    			      XmNbottomWidget, findFileOk,
    			      NULL);
    			      
  label = XtVaCreateManagedWidget("File Name Pattern: ", xmLabelWidgetClass, form,
  			      XmNleftAttachment, XmATTACH_FORM,
  			      XmNbottomAttachment, XmATTACH_WIDGET,
  			      XmNbottomWidget, sep,
  			      XmNbottomOffset, 4,
  			      NULL);
  findFileTextFiles = XtVaCreateManagedWidget("search_files_text", xmTextWidgetClass, form,
  			      XmNleftAttachment, XmATTACH_WIDGET,
  			      XmNleftWidget, label,
  			      XmNrightAttachment, XmATTACH_FORM,
  			      XmNbottomAttachment, XmATTACH_WIDGET,
  			      XmNbottomWidget, sep,
  			      NULL);
  XtAddCallback(findFileTextFiles, XmNactivateCallback, 
		      (XtCallbackProc) findFileSearchCB, NULL);
  			      
  label = XtVaCreateManagedWidget("Search From: ", xmLabelWidgetClass, form,
  			      XmNleftAttachment, XmATTACH_FORM,
  			      XmNbottomAttachment, XmATTACH_WIDGET,
  			      XmNbottomWidget, findFileTextFiles,
  			      XmNbottomOffset, 4,
  			      NULL);
  findFileTextSearch = XtVaCreateManagedWidget("search_pattern_text", xmTextWidgetClass, form,
  			      XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET,
  			      XmNleftWidget, findFileTextFiles,
  			      XmNrightAttachment, XmATTACH_FORM,
  			      XmNbottomAttachment, XmATTACH_WIDGET,
  			      XmNbottomWidget, findFileTextFiles,
  			      NULL);  			      
  XtAddCallback(findFileTextSearch, XmNactivateCallback, 
		      (XtCallbackProc) findFileSearchCB, NULL);
  getcwd(cwd, MAXPATHLEN);
  XmTextSetString(findFileTextSearch, cwd);

  sep = XtVaCreateManagedWidget("sep", xmSeparatorWidgetClass, form,
    			      XmNleftAttachment, XmATTACH_FORM,
    			      XmNrightAttachment, XmATTACH_FORM,
    			      XmNbottomAttachment, XmATTACH_WIDGET,
    			      XmNbottomWidget, findFileTextSearch,
    			      NULL);

  findFileList = XmCreateScrolledList(form, "find_file_list", NULL, 0);
  XtVaSetValues(XtParent(findFileList),
  		XmNleftAttachment, XmATTACH_FORM,
  		XmNrightAttachment, XmATTACH_FORM,
  		XmNtopAttachment, XmATTACH_FORM,
  		XmNbottomAttachment, XmATTACH_WIDGET,
  		XmNbottomWidget, sep,
  		XmNwidth, 800,
  		NULL);
  XtVaSetValues(findFileList, 
  		XmNvisibleItemCount, 20, 
  		XmNwidth, 800,
  		XmNselectionPolicy, XmSINGLE_SELECT,
  		XmNscrollingPolicy, XmAUTOMATIC,
  		XmNscrollBarDisplayPolicy, XmSTATIC,
  		NULL);
  XtAddCallback(findFileList, XmNsingleSelectionCallback, findFileGotoCB, NULL);
  XtAddCallback(findFileList, XmNdefaultActionCallback, findFileGotoCB, NULL);

  XtManageChild(findFileList);
  XtManageChild(form);
  
  return Return;
}

