/**************************************************************
 *
 * list.c
 *
 * This is a module which provides a "linked list" for the
 * programmer.
 *
 * See list.h for a description of the programming interface.
 *
 * Started on 2/17/93 hellmann
 *
 *
 * $Log:	list.c,v $
** Revision 1.4  96/06/10  14:45:55  hellmann
** Fixed some memory problems found with Insight.
** 
** Revision 1.3  96/05/17  12:47:50  hellmann
** Fixed prototypes.
** 
** Revision 1.2  96/02/20  15:49:25  hellmann
** Changed #include directives for local files.
** 
** Revision 1.1  96/01/31  15:26:33  hellmann
** Initial revision
** 
** Revision 1.1  95/12/26  09:10:05  hellmann
** Initial revision
** 
 * Revision 1.6  1995/07/02  23:32:21  hellmann
 * Special case for finding position 0 in empty list.
 *
 * Revision 1.5  1995/07/01  20:05:32  hellmann
 * ListFindItem() will compare the data pointers it uses to store values
 * in a List if the comparison function is NULL.
 *
 * Revision 1.4  1995/05/25  18:42:13  hellmann
 * changed Position to position to fix name clash
 *
 * Revision 1.3  1995/01/12  21:47:03  hellmann
 * Modified and ifdefed some debugging messages which came up in error
 * conditions.
 *
 * Revision 1.2  1994/11/09  18:58:05  wolff
 *  Added log file.
 *
 *
 *
 **************************************************************
*/

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#include <stdlib.h>
#include <stdio.h>
#include "list.h"

struct NodeStruct;
typedef struct NodeStruct Node;

struct NodeStruct
{
   Node * Next;
   void * Data;
   void (* Destroy) ();
};

#define MagicValue 1993

struct ListStruct
{
   int  MagicNumber;   /* used to verify that pointer is a linked list */
   Node * Head;        /* head of the actual list of nodes */
   Node * Tail;        /* tail of the actual list of nodes */
   Node * Cursor;      /* current node (used in iteration, reset, next) */
   int Count;          /* number of items in list */
   int CursorValid;    /* whether the cursor is valid */
   int IsSorted;       /* whether the list has been sorted */
};



/*
 * ListCreate
 *
 * PURPOSE : Create one occurance of a List.
 *
 * PARAMS  : None.
 *
 * RETURNS : Pointer to a new List.
 *
*/
List * ListCreate()
{
   List * tmp;

   if ((tmp = (List *) malloc (sizeof(List))) == NULL)
      return NULL;

   tmp->MagicNumber = MagicValue;
   tmp->Head = NULL;
   tmp->Tail = NULL;
   tmp->Count = 0;
   tmp->CursorValid = FALSE;
   tmp->IsSorted = TRUE;

   return tmp;
}

/*
 * ListDestroy
 *
 * PURPOSE : To free all of the memory used by a list and it's contents.
 *
 * PARAMS  : TheList - the list to erase
 *           FreeSpace - if true, free space used by elements of list
 *                       if false, don't
 *
*/
int ListDestroy ( TheList, FreeSpace )
   List * TheList;
   int FreeSpace;
{
   Node * p;

   if (TheList == NULL)
      return -1;

   if (TheList->MagicNumber != MagicValue)
      return -1;

   for (p = TheList->Head; p != NULL; p = TheList->Head)
   {
      TheList->Head = p->Next;

      if (FreeSpace)
        (p->Destroy)(p->Data);
      free(p);
   }
   
   free(TheList);

   return 0;

} /* ListDestroy */

/**************************************************************
 *
 * ListSize
 *
 * PURPOSE : Return the number of data items in the list.
 *
 * PARAMS  : TheList
 *
*/
int ListSize ( TheList )
   List * TheList;
{
  if (TheList == NULL)
  {
#ifdef DEBUG_LIST
     fprintf(stderr, "Invalid list in ListSize\n");
     fflush(stderr);
#endif
     return -1;
  }

  if (TheList->MagicNumber != MagicValue)
  {
#ifdef DEBUG_LIST
     fprintf(stderr, "List pointer is not really a list in ListSize\n");
     fflush(stderr);
#endif
     return -1;
  }

  return TheList->Count;
} /* ListSize */

/**************************************************************
 *
 * ListPos
 *
 * PURPOSE : Return the Data item from the list at a given position
 *
 * PARAMS  : TheList
 *           position - what position to return
 *                (-1 indicates last item in list)
*/
void * ListPos ( List * TheList, int position )
{
  Node * p;
  int i;

  if (TheList == NULL)
     return NULL;

  if (TheList->MagicNumber != MagicValue)
     return NULL;

  if (position > TheList->Count)
     return NULL;

  if (position == -1)
     if (TheList->Tail != NULL)
        return TheList->Tail->Data;
     else
	return NULL;

  p = TheList->Head;

  for (i = 0; i < position; i++)
     p = p->Next;

  if (!p)
    return NULL;
  else
    return p->Data;

} /* ListPos */


/**************************************************************
 *
 * ListAddItem
 *
 * PURPOSE : Add a data element to the given list.
 *
 * PARAMS  : TheList - list to add item to
 *           TheData - data item to add, cast to void *
 *           Destroy - function to erase TheData for
 *                     when List is destroyed
 *           position - where in the list to add it
 *                (-1 indicates end, anywhere else is in list)
 *
*/
int ListAddItem( TheList, TheData, Destroy, position )
   List * TheList;
   void * TheData;
   void (* Destroy) ();
   int position;
{
  Node * NewNode;
  Node * p;
  Node * q;
  int i;

  if (TheList == NULL)
     return -1;

  if (TheList->MagicNumber != MagicValue)
     return -1;

  if ((NewNode = (Node *) malloc (sizeof(Node))) == NULL)
     return -1;

  TheList->IsSorted = FALSE;

  NewNode->Destroy = Destroy;
  NewNode->Data = TheData;

/* if no elements in list, put it at the beginning regardless */

  if (TheList->Count == 0)
  {
    TheList->Head = NewNode;
    TheList->Tail = NewNode;
    NewNode->Next = NULL;
  }

/* other wise do different things */

  else
  {

/* when position < 0 or position > number of elements, put at end */

    if ((position < 0) || (position > TheList->Count))
    {
      TheList->Tail->Next = NewNode;
      NewNode->Next = NULL;
      TheList->Tail = NewNode;
    }

/* otherwise, put it in the position-th place in the list */

    else
    {
      q = NULL;
      p = TheList->Head;

      for (i = 0; i < position; i++)
      {
	q = p;
	p = p->Next;
      }

      if (q != NULL)
        q->Next = NewNode;
      else
	TheList->Head = NewNode;

      NewNode->Next = p;

    }

  }

  (TheList->Count)++;

  return 0;

} /* ListAddItem */

/**************************************************************
 *
 * ListCompareInternalPointers
 *
 * PURPOSE : Compare two pointers for equality
 *
 * PARAMS  : two pointers
*/
static int ListCompareInternalPointers(void * p1, void * p2)
{
  if (p1 < p2)
    return -1;
  if (p1 > p2)
    return 1;
  return 0;
}

/**************************************************************
 *
 * ListFindItem
 *
 * PURPOSE : Find the item if it is in the list
 *
 * PARAMS  : TheList
 *           TheItem - pointer to object of same type
 *           CompareFcn - comparison function
*/
int ListFindItem( TheList, TheItem, CompareFcn )
   List * TheList;
   void * TheItem;
   int (* CompareFcn) ();

{
   Node * p;
   Node * q;
   int i;
   int (* CompareWith)();

   if (TheList == NULL)
      return -1;

   if (TheList->MagicNumber != MagicValue)
      return -1;
      
   if (CompareFcn == NULL)
     CompareWith = ListCompareInternalPointers;
   else
     CompareWith = CompareFcn;

   for ( p = TheList->Head, i = 0;
         (p != NULL) && (CompareWith(p->Data, TheItem) != 0);
	 q = p, p = p->Next, i++);

   if (p != NULL)
   if (CompareWith(p->Data, TheItem) == 0)
     return i;

   return -1;

} /* ListFindItem */

/**************************************************************
 *
 * ListRemovePos
 *
 * PURPOSE : To remove the item at the given position in the list.
 *
 * PARAMS  : TheList
 *           position
 *           DeleteIt   - if true, delete item, else don't
 *
*/
int ListRemovePos ( TheList, position, DeleteIt )
   List * TheList;
   int position;
   int DeleteIt;
{
  Node * p;
  Node * q;
  int i;

  if (TheList == NULL)
     return -1;

  if (TheList->MagicNumber != MagicValue)
     return -1;

  if (position >= TheList->Count)
     return -1;

  if (position < 0)
     return -1;

  TheList->IsSorted = FALSE;

  q = NULL;
  p = TheList->Head;

  for (i = 0; i < position; i++)
  {
    q = p;
    p = p->Next;
  }

  if (q == NULL)
    TheList->Head = p->Next;
  else
    q->Next = p->Next;

  if (TheList->Tail == p)
    TheList->Tail = q;

  if (DeleteIt)
  {
    (p->Destroy)(p->Data);
  }

  free(p);

  (TheList->Count)--;

  return 0;

} /* ListRemovePos */

/**************************************************************
 *
 * ListRemoveItem
 *
 * PURPOSE : To take an item out of the list.
 *
 * PARAMS  : TheList
 *           TheItem - pointer to object of same type
 *           CompareFcn - comparison function
 *           DeleteIt   - if true, delete item, else don't
 *
*/
int ListRemoveItem ( TheList, TheItem, CompareFcn, DeleteIt )
   List * TheList;
   void * TheItem;
   int (* CompareFcn) ();
   int DeleteIt;
{
  return ListRemovePos(TheList, 
		       ListFindItem(TheList, TheItem, CompareFcn), DeleteIt);
} /* ListRemoveItem */

/**************************************************************
 *
 * ListIterate
 *
 * PURPOSE : Perform the specified function on each of the node's
 *           data areas.
 *
 * PARAMS  : TheList - list to iterate over
 *           Action  - function to perform on the item.
 *
*/
int ListIterate ( List * TheList, int (* Action) (void*) )
{
  int erc;

  if (TheList == NULL)
     return -1;

  if (TheList->MagicNumber != MagicValue)
     return -1;

  if (TheList->Count == 0)
     return -1;

  for (TheList->Cursor = TheList->Head;
       TheList->Cursor != NULL;
       TheList->Cursor = TheList->Cursor->Next)
  {
    erc = (* Action)(TheList->Cursor->Data);
    if (erc != 0)
       return erc;
  }

  TheList->IsSorted = FALSE;

  return 0;

} /* ListIterate */

/**************************************************************
 *
 * ListReset
 *
 * PURPOSE : Return the First element of the list, reset the list
 *           cursor to point to the second element.
 *
 * PARAMS  : TheList - list to look at
 *
 * RETURNS : pointer to data area of the first item in the list
 *
*/
void * ListReset ( TheList )
   List * TheList;
{
   if (TheList == NULL)
      return NULL;

   if (TheList->MagicNumber != MagicValue)
      return NULL;

   if (TheList->Count == 0)
      return NULL;

   TheList->CursorValid = TRUE;
   TheList->Cursor = TheList->Head->Next;
   TheList->IsSorted = FALSE;

   return TheList->Head->Data;

} /* ListReset */


/**************************************************************
 *
 * ListNext
 *
 * PURPOSE : Return the next item in the list.
 *
 * PARAMS  : TheList - the list to look at
 *
 * RETURNS : the next item in the list, the first item if it hasn't been
 *           called before
 *
*/
void * ListNext ( TheList )
   List * TheList;
{
   Node * tmp;

   if (TheList == NULL)
      return NULL;

   if (TheList->MagicNumber != MagicValue)
      return NULL;

   if (TheList->Count == 0)
      return NULL;

   if (!TheList->CursorValid)
      /*return ListReset(TheList);*/
      return NULL;

   if ((TheList->Cursor == NULL) && (TheList->CursorValid))
   {
      TheList->CursorValid = FALSE;
      return NULL;
   }

   tmp = TheList->Cursor;
   TheList->Cursor = TheList->Cursor->Next;

   TheList->IsSorted = FALSE;

   if (tmp != NULL)
      return tmp->Data;
   else return NULL;

} /* ListNext */

/**************************************************************
 *
 * ListSortAsc
 *
 * PURPOSE : To sort the list using the specified comparison
 *           function.  It is assumed that the function takes
 *           two parameters, returns 0 if they are equal, < 0
 *           if the first one is smaller than the second, and > 0
 *           if the first one is larger than the second.
 *
 * PARAMS  : TheList - list to sort
 *           CompareFcn - function to compare two list items
 *
*/
int ListSortAsc( TheList, CompareFcn )
   List * TheList;
   int (* CompareFcn) ();
{
  Node * SortedList;
  Node * SortedListEnd;
  Node * p, 
       * q, * bq,
       * r;

  if (TheList == NULL)
     return -1;

  if (TheList->MagicNumber != MagicValue)
     return -1;

  if ((TheList->IsSorted) == TRUE)
     return 0;

  SortedList = NULL;
  SortedListEnd = NULL;

  for (; TheList->Head != NULL; )
  {

    /* find the smallest element in the current list */
    bq = NULL;
    q = TheList->Head;
    r = NULL;

    for (p = TheList->Head; p != NULL; p = p->Next)
    {
      if (CompareFcn(p->Data, q->Data) < 0)
      {
	bq = r;
	q = p;
      }
      /* r = q; you dummy */
      r = p;
    }

    if (SortedList == NULL)
    {
      SortedList = q;
      SortedListEnd = q;
    }
    else
    {
      if (SortedListEnd != NULL)
         SortedListEnd->Next = q;
      else
      {
	 fprintf(stderr, "ListSort: SortedListEnd is NULL");
	 break;
      }
      SortedListEnd = SortedListEnd->Next;
    }


    if (bq == NULL)
      TheList->Head = q->Next;
    else
      bq->Next = q->Next;

    q->Next = NULL;

  }

  TheList->Head = SortedList;
  TheList->Tail = SortedListEnd;

  TheList->IsSorted = TRUE;

  return 0;

} /* ListSort */
