/*
** 
**  $Header: /usr/home/hellmann/nedit_add_ons/RCS/commandOptions.c,v 1.5 96/02/22 17:55:34 hellmann Exp $
** 
**  FILENAME: commandOptions.c
** 
**   PURPOSE: Define platform specific options for different versions of
**			  different programs.
** 
**    AUTHOR: Doug Hellmann
** 
**  COMMENTS: 
** 
**    RCSLOG:   $Log:	commandOptions.c,v $
** Revision 1.5  96/02/22  17:55:34  hellmann
** Make sure the first character of options is always set to null
** 
** Revision 1.4  96/02/22  17:31:30  hellmann
** Added ... for push button options.
** 
** Revision 1.3  96/02/22  15:09:20  hellmann
** Build menus and handle boolean and string options.
** Still no string default value.
** 
** Revision 1.2  96/02/21  17:59:07  hellmann
** Fixed header file name
** 
** Revision 1.1  96/02/21  17:56:14  hellmann
** Initial revision
** 
** Revision 1.1  96/02/21  17:30:11  hellmann
** Initial revision
** 
** 
** 
*/
#include "client.h"
#include "commandOptions.h"
#include "stringobj.h"
#include <DialogF.h>

/* 
**   PURPOSE: Change the stored value of a string option.
** 
** ARGUMENTS: standard push button callback
** 
**   RETURNS: none
** 
**  COMMENTS: 
*/ 
static void string_option_change(Widget w, XtPointer xtp1, XtPointer xtp2)
{
	CmdOption * opt = (CmdOption*)xtp1;
	XmPushButtonCallbackStruct * cbs = (XmPushButtonCallbackStruct*) xtp2;
	char new_value[DF_MAX_PROMPT_LENGTH];
	int response;
	
	sprintf(new_value, "%s", opt->value.s_value);
	
    response = DialogF(DF_PROMPT, XtParent(w), 2,
    	    opt->option_description, new_value, "OK", "Cancel");
    if (response == 2)
    	return;
	
#ifdef DEBUG_Commandoptions
	fprintf(stderr, "New value of %s is '%s'\n", opt->option_description, new_value);
#endif
	
	sprintf(opt->value.s_value, "%s", new_value);
}

/* 
**   PURPOSE: Change the stored value of a boolean option.
** 
** ARGUMENTS: standard toggle button callback
** 
**   RETURNS: none
** 
**  COMMENTS: 
*/ 
static void bool_option_changed(Widget w, XtPointer xtp1, XtPointer xtp2)
{
	CmdOption * opt = (CmdOption*)xtp1;
	XmToggleButtonCallbackStruct * cbs = (XmToggleButtonCallbackStruct*) xtp2;
	Widget menu;
	WidgetList siblings;
	Cardinal num_siblings;
	CmdOption * sibling_opt;
	int i;
	
		/* store the value, no matter what */
		
	opt->value.b_value = cbs->set;
	
		/* if turned off, we are done */
		
	if (!cbs->set)
		return;
		
	if (opt->ignore_siblings)
		return;

		/* any siblings which want siblings ignored should be turned off
		   if this is turned on */
		   
	menu = XtParent(w);
	XtVaGetValues(menu, XmNchildren, &siblings, XmNnumChildren, &num_siblings, NULL);

	for (i = 0; i < num_siblings; i++)
	{
		if (siblings[i] == w)
			continue;
		
		XtVaGetValues(siblings[i], XmNuserData, &sibling_opt, NULL);
		
		if (sibling_opt->ignore_siblings)
			XmToggleButtonSetState(siblings[i], False, True);
	}	
}


/* 
**   PURPOSE: If an option is set to override or ignore its siblings, we
**			  want to set their values to FALSE when it is set to true.
** 
** ARGUMENTS: standard toggle button callback
** 
**   RETURNS: none
** 
**  COMMENTS: Assumes that all siblings are going to be toggle buttons.
*/ 
static void bool_option_ignore_siblings(Widget w, XtPointer xtp1, XtPointer xtp2)
{
	Widget menu = (Widget) xtp1;
	XmToggleButtonCallbackStruct * cbs = (XmToggleButtonCallbackStruct*) xtp2;
	WidgetList siblings;
	Cardinal num_siblings;
	int i;

	if (!cbs->set)
		return;
		
	XtVaGetValues(menu, XmNchildren, &siblings, XmNnumChildren, &num_siblings, NULL);

	for (i = 0; i < num_siblings; i++)
	{
		if (siblings[i] == w)
			continue;
		
		XmToggleButtonSetState(siblings[i], False, True);
	}	
}

/* 
**   PURPOSE: Build the Option menu using the platform specific info above.
** 
** ARGUMENTS: Parent widget (menu)
**			  Pointer to array of CmdOption structures defining the menu
**			  	layout
** 
**   RETURNS: TRUE if builds a menu, FALSE otherwise.
** 
**  COMMENTS: For every UNIX program there is a potential to have different
**            command line options on different platforms or for different
**            versions.  This is an attempt to handle that.
*/ 
int BuildOptionMenu(Widget parent, CmdOption * menuOptions)
{
	int i;
	
#ifdef DEBUG_Commandoptions
	fprintf(stderr, "Entering BuildOptionMenu\n\tparent widget is %s\n", 
			XtName(parent));
#endif
	
	i = 0;
	while (menuOptions[i].option_name || menuOptions[i].sub_menu)
	{
#ifdef DEBUG_Commandoptions
		fprintf(stderr, "%d: ", i);
#endif
		
		if (menuOptions[i].sub_menu)
		{
			/*
			** Create the sub menu from this point.
			*/
			Widget cascade, subMenu;
			
#ifdef DEBUG_Commandoptions
			fprintf(stderr, "building sub menu\n");
#endif
			
			subMenu = XmCreatePulldownMenu(parent, "sub_menu", NULL, 0);
			
			cascade = XtVaCreateManagedWidget(menuOptions[i].sub_menu_name,
							xmCascadeButtonWidgetClass,
							parent,
							XmNsubMenuId, subMenu,
							NULL);
							
			BuildOptionMenu(subMenu, menuOptions[i].sub_menu);
		}
		else
		{
			/*
			** Create contents of this menu.
			*/
			Widget button;
			XmString label;
			char label_str[MAX_STRING_VALUE_LENGTH+4];
						
#ifdef DEBUG_Commandoptions
			fprintf(stderr, "adding toggle button\n");
#endif


			
			switch (menuOptions[i].option_type)
			{
			
				case CMD_OPTION_BOOLEAN:
					sprintf(label_str, "%s", menuOptions[i].option_description);
					label = XmStringCreateLtoR(label_str, XmSTRING_DEFAULT_CHARSET);

					button = XtVaCreateManagedWidget(menuOptions[i].option_name,
									xmToggleButtonWidgetClass,
									parent,
									XmNlabelString, label,
									XmNset, menuOptions[i].value.b_value,
									XmNuserData, &(menuOptions[i]),
									NULL);
					XtAddCallback(	button, 
									XmNvalueChangedCallback, 
									bool_option_changed, 
									(XtPointer) &(menuOptions[i]) );
									
					if (menuOptions[i].ignore_siblings)
					{
						XtAddCallback(button, XmNvalueChangedCallback,
								bool_option_ignore_siblings,
								(XtPointer) parent);
					}
				break;
				
				case CMD_OPTION_STRING:
					sprintf(label_str, "%s ...", menuOptions[i].option_description);
					label = XmStringCreateLtoR(label_str, XmSTRING_DEFAULT_CHARSET);
				
					button = XtVaCreateManagedWidget(menuOptions[i].option_name,
									xmPushButtonWidgetClass,
									parent,
									XmNlabelString, label,
									XmNuserData, &(menuOptions[i]),
									NULL);
									
					XtAddCallback(  button,
									XmNactivateCallback,
									string_option_change,
									(XtPointer) &(menuOptions[i]) );
					
					menuOptions[i].value.s_value[0] = '\0';
				break;
				
				default:
					fprintf(stderr, "Unknown menu option type: '%s' %d\n",
						menuOptions[i].option_description,
						menuOptions[i].option_type);
				break;
			}
							
			XmStringFree(label);
		}
		
		i++;
	}
	
	return FALSE;
}


/* 
**   PURPOSE: Build a string representing the values of all of the options
**			  in the hierarchy passed in.
** 
** ARGUMENTS: Pointer to array of CmdOption structures.
** 
**   RETURNS: Character string which contains all of the command line arguments
**			  based on values of options and information the user specifies.
** 
**  COMMENTS: 
*/ 
char * BuildOptionString(CmdOption * menuOptions)
{
	int i;
	char * Return;
	StringObj * buf = StringObj_Create("");
	
#ifdef DEBUG_Commandoptions
	fprintf(stderr, "Entering BuildOptionString\n");
#endif
	
	i = 0;
	while (menuOptions[i].option_name || menuOptions[i].sub_menu)
	{
		if (menuOptions[i].option_name)
		{
			switch (menuOptions[i].option_type)
			{
				case CMD_OPTION_BOOLEAN:
					if (menuOptions[i].value.b_value)
					{
						StringObj_Concat(buf, " ");
						StringObj_Concat(buf, menuOptions[i].option_flag);
						StringObj_Concat(buf, " ");
					}
				break;
				
				case CMD_OPTION_STRING:
					if (menuOptions[i].value.s_value[0])
					{
						StringObj_Concat(buf, " ");
						StringObj_Concat(buf, menuOptions[i].option_flag);
						StringObj_Concat(buf, menuOptions[i].value.s_value);
						StringObj_Concat(buf, " ");
					}
				break;
				
				default:
				break;
			}
		}
		else
		{
			char * sub_string;
			
			sub_string = BuildOptionString(menuOptions[i].sub_menu);
			
			StringObj_Concat(buf, sub_string);
			
			free(sub_string);
		}
		
		i++;
	}
	
	Return = strdup(StringObj_AsString(buf));
	
	StringObj_Destroy(buf);
	
	return Return;
}
