/****  cbdialog.c  ****  a simple dialog box ****/

/*
The dialog box is built on top of the simplewin system, given
in simplewin.[ch].  This implementation provides a dialog box
header and optional secondary message, but supports button-
activated responses only.  If a callback is associated with
the selected button, it will be executed and TRUE will be
returned; otherwise FALSE is returned.
*/

#include <stdio.h>
#include <string.h>
#include "cbdialog.h"


/*
Private functions:
*/

static int allocate_and_center_button_text();


/*
make_dialog_buttons() attaches a set of buttons to an existing
window derived from logical_window_frame; see simplewin.[ch].
Buttons are specified with the "button" structure; see dialog.h.
Buttons are spaced evenly in the dialog box, and button labels
are centered in equal-sized buttons.  All button-related internal
structures are allocated from the heap and attached to the
"extension" pointer in logical_window_frame.
*/

int make_dialog_buttons(lwin, buttons)
logical_window_frame *lwin;
button *buttons;
{
	int btn_x, btn_y, i, len, num_btn, most_chars;
	unsigned btn_width, btn_height;
	lbutton *btn_win;

	for (most_chars = num_btn = 0; buttons[num_btn].label;
			num_btn++)
		if ((len = strlen(buttons[num_btn].label)) > most_chars)
			most_chars = len;
	btn_win = (lbutton *)
		malloc((unsigned) ((num_btn + 1) * sizeof(lbutton)));
	if (btn_win == NULL)
		return FALSE;
	lwin->extension = (lbutton *) btn_win;	/* attach to window */
	btn_y = lwin->height - lwin->font_height -
		lwin->font->max_bounds.descent - (BUTTON_BORDER_WIDTH * 5);
	btn_height = lwin->font_height +
		lwin->font->max_bounds.descent;
	btn_width = lwin->font_width * most_chars;
	for (i = 0; i < num_btn; i++) {
		btn_x = (lwin->width / (num_btn * 2)) -
			(btn_width / 2) + (i * (lwin->width / num_btn));
		btn_win[i].bid = XCreateSimpleWindow(display, lwin->xwin,
			btn_x, btn_y, btn_width, btn_height,
			BUTTON_BORDER_WIDTH,
			BlackPixel(display, screen),
			WhitePixel(display, screen));
		XDefineCursor(display, btn_win[i].bid, popup_cursor);
		XSelectInput(display, btn_win[i].bid, ExposureMask |
			EnterWindowMask | LeaveWindowMask |
			ButtonReleaseMask | ButtonPressMask |
			OwnerGrabButtonMask);
		if (!allocate_and_center_button_text(&btn_win[i],
				&buttons[i], most_chars)) {
			free(btn_win);
			for ( ; --i >=0; )
				free(btn_win[i].label);
			return FALSE;
		}
		btn_win[i].callback = *buttons[i].callback;
		btn_win[i].parent = lwin->xwin;
		btn_win[i].x = btn_x;
		btn_win[i].y = btn_y;
		btn_win[i].width = btn_width;
		btn_win[i].height = btn_height;
	}	/* for */
	btn_win[i].bid = EOB;	/* logically terminate	*/
							/* the set of buttons	*/
	XMapSubwindows(display, lwin->xwin);
	return TRUE;
}	/* make_dialog_buttons */


/*
allocate_and_center_button_text() creates private (safe)
copies of the client's button text strings.
*/

static int allocate_and_center_button_text(btn_win,
	btn_data, max_len)
lbutton *btn_win;
button *btn_data;
int max_len;
{
	int j, pad_len;

	j = strlen(btn_data->label);		/* center each button */
	pad_len = (max_len - j) / 2;
	btn_win->label = (char *) malloc((unsigned) (j + pad_len + 1));
	if (btn_win->label == NULL)
		return FALSE;
	for ( ; j > -1; j--)
		btn_win->label[j + pad_len] = btn_data->label[j];
	for (j = 0; j < pad_len; j++)
		btn_win->label[j] = ' ';
	return TRUE;
}	/* allocate_and_center_button_text */


/*
dialog_loop() loops over the input; it attempts to execute
the callback associated with the activated button.
*/

int dialog_loop(lwin, header, message)
logical_window_frame *lwin;
char *header, *message;
{
	lbutton *btn_win = (lbutton *) lwin->extension;
	XEvent event;
	int i;

	if (btn_win == NULL)
		return FALSE;
	activate_window(lwin, "");
	while (TRUE) {
		XNextEvent(display, &event);
		switch (event.type) {
			case Expose:
				while (XCheckTypedWindowEvent(display, lwin->xwin,
						Expose, &event))
					;
				if (event.xexpose.window == lwin->xwin) {
					window_display_header(lwin, header);
					window_go_cr(lwin, 1, 2);
					window_puts(lwin, message, gc);
					display_buttons(lwin, gc);
				}
				break;
			case EnterNotify:
				for (i = 0; btn_win[i].bid != EOB; i++)
					if (event.xcrossing.window == btn_win[i].bid) {
						highlight_button(&btn_win[i]);
						break;
					}
				break;
			case LeaveNotify:
				for (i = 0; btn_win[i].bid != EOB; i++)
					if (event.xcrossing.window == btn_win[i].bid) {
						unhighlight_button(&btn_win[i]);
						break;
					}
				break;
			case ButtonRelease:
				for (i = 0; btn_win[i].bid != EOB; i++) {
					if (event.xbutton.window == btn_win[i].bid) {
						unhighlight_button(&btn_win[i]);
						deactivate_window(lwin);
						if (*btn_win[i].callback == NULL)
							return FALSE;
						else {
							(*btn_win[i].callback)();
							return TRUE;
						}
					}
				}
				break;
		}
	}
}   /* dialog_loop */


/*
cleanup_dialog_button_structures() first frees the heap space
used for labels, and then frees the heap-based array.
*/

void cleanup_dialog_button_structures(lwin)
logical_window_frame *lwin;
{
	int i;
	lbutton *btn_win = (lbutton *) lwin->extension;

	if (btn_win == NULL)
		return;
	for (i = 0; btn_win[i].bid != EOB; i++)
		free(btn_win[i].label);
	free(lwin->extension);
	XDestroySubwindows(display, lwin->xwin);
}	/* cleanup_dialog_button_structures */


/*
display_buttons() is a private routine used by display_loop()
to display the buttons' text.  X displays the buttons' borders.
*/

void display_buttons(lwin, gc)
logical_window_frame *lwin;
GC gc;
{
	int i;
	lbutton *btn_win = (lbutton *) lwin->extension;

	if (btn_win == NULL)
		return;
	for (i = 0; btn_win[i].bid != EOB; i++)
		XDrawImageString(display, btn_win[i].bid, gc,
			0, lwin->font_height - 1, btn_win[i].label,
			strlen(btn_win[i].label));
}	/* display_buttons */


/*
highlight_buttons() is a private routine that draws a one
pixel wide rectangle on the inside of the window.
*/

void highlight_button(bw)
lbutton *bw;
{
	XDrawRectangle(display, bw->bid, hgc, 0, 0,
		bw->width - 1, bw->height - 1);
}	/* highlight_button */


/*
unhighlight_button() is a private routine that erases the
one pixel wide rectangle inside the button border.
*/

void unhighlight_button(bw)
lbutton *bw;
{
	XDrawRectangle(display, bw->bid, hgc, 0, 0,
		bw->width - 1, bw->height - 1);
}	/* unhighlight_button */

