/****  response.c  ****  a simple char. input response box  ****/

/*
The response box is built on top of the simplewin and dialog
modules.  It provides routines to read a string response from
the user, in addition to the button support provided by the
dialog module.  See simplewin.[ch] and dialog.[ch].
*/

#include <X11/keysym.h>
#include "dialog.h"
#include "response.h"

#include "point.bit"
#include "savepoint.bit"



/*
Globals:
*/

static Pixmap point, save_point;


/*
Private functions:
*/

static void make_point();
static void backspace_char();
static void echo_char();
static void window_no_point();
static void window_put_point();
static void window_puts_point();
static int window_gets_filter();
static int window_getche_event_loop();
static int window_getche();
static void window_display_char_with_point();



/*
initialize_responsebox_structures() sets up the
data structures needed to support the response box
in a window.
*/

void initialize_responsebox_structures(font)
XFontStruct *font;
{
	initialize_window_structures(font);
	make_point();
}	/* initialize_responsebox_structures */


/*
cleanup_responsebox_structures() frees dynamic
data structures.
*/

void cleanup_responsebox_structures()
{
	cleanup_window_structures();
	XFreePixmap(display, point);
	XFreePixmap(display, save_point);
}	/* cleanup_responsebox_structures */


/*
responsebox_loop() processed key and button input.
It returns the value associated with the selected button.
*/

int responsebox_loop(lwin, header, message, response)
logical_window_frame *lwin;
char *header, *message, *response;
{
	lbutton *btn_win = (lbutton *) lwin->extension;
	XEvent event;
	int btn_result = FALSE, key_result = FALSE;
	int i, terminate = FALSE;

	activate_window(lwin, "");
	while (!terminate) {
		XNextEvent(display, &event);
		switch (event.type) {
			case Expose:
				while (XCheckTypedWindowEvent(display, lwin->xwin,
						Expose, &event))
					;
				if (event.xexpose.window == lwin->xwin) {
					window_display_header(lwin, header);
					window_go_cr(lwin, 1, 2);
					window_clr_eol(lwin);
					window_puts_point(lwin, message, gc);
					*response = EOS;
					display_buttons(lwin, gc);
				}
				break;
			case KeyPress:
				XPutBackEvent(display, &event);
				key_result =
					window_gets_filter(lwin, response, MAX_STR,
						header, message);
				terminate = TRUE;
				break;
			case EnterNotify:
				if (btn_win == NULL)
					break;
				for (i = 0; btn_win[i].bid != EOB; i++)
					if (event.xcrossing.window == btn_win[i].bid) {
						highlight_button(&btn_win[i]);
						break;
					}
				break;
			case LeaveNotify:
				if (btn_win == NULL)
					break;
				for (i = 0; btn_win[i].bid != EOB; i++)
					if (event.xcrossing.window == btn_win[i].bid) {
						unhighlight_button(&btn_win[i]);
						break;
					}
				break;
			case ButtonRelease:
				if (btn_win == NULL)
					break;
				for (i = 0; btn_win[i].bid != EOB; i++) {
					if (event.xbutton.window == btn_win[i].bid) {
						unhighlight_button(&btn_win[i]);
						btn_result = btn_win[i].value;
						terminate = TRUE;
					}
				}
				break;
		}	/* switch */
	}	/* while */
	deactivate_window(lwin);
	if (btn_result)
		return btn_result;
	else
		return (!key_result) ? ESC_KEY : key_result;
}   /* responsebox_loop */


/*
make_point() makes a window point/cursor
that can be used for keyboard processing.
*/

static void make_point()
{
	if (display_depth == 1) {
		point = XCreateBitmapFromData(display,
			RootWindow(display, screen),
			point_bits, point_width, point_height);
		save_point = XCreateBitmapFromData(display,
			RootWindow(display, screen), savepoint_bits,
			savepoint_width, savepoint_height);
	}
	else {
		point = XCreatePixmapFromBitmapData(display,
			RootWindow(display, screen),
			point_bits, point_width, point_height,
			BlackPixel(display, screen), WhitePixel(display, screen),
			display_depth);
		save_point = XCreatePixmapFromBitmapData(display,
			RootWindow(display, screen), savepoint_bits,
			savepoint_width, savepoint_height,
			BlackPixel(display, screen), WhitePixel(display, screen),
			display_depth);
	}
}    /* make_point */


/*
The following two functions are used to display and "un-display"
the software-maintained cursor.
*/

static void window_no_point(lwin)
logical_window_frame *lwin;
{
	XCopyArea(display, save_point, lwin->xwin, gc, 0, 0,
		point_width, point_height, lwin->cursor_x - 2,
		lwin->cursor_y + 2);
}    /* window_no_point */


static void window_put_point(lwin)
logical_window_frame *lwin;
{
	XCopyArea(display, lwin->xwin, save_point, gc,
		lwin->cursor_x - 2, lwin->cursor_y + 2,
		point_width, point_height, 0, 0);
	XCopyArea(display, point, lwin->xwin, gc, 0, 0,
		point_width, point_height, lwin->cursor_x - 2,
		lwin->cursor_y + 2);
}    /* window_put_point */


/*
window_puts_point() prints a string, followed by the point,
in a window w/o interpreting each character.
*/

static void window_puts_point(lwin, str, gc)
logical_window_frame *lwin;
char *str;
GC gc;			/* allow normal or reverse video */
{
	int text_width;

	window_puts(lwin, str, gc);
	text_width = XTextWidth(lwin->font, str, strlen(str));
	lwin->cursor_x += text_width;
	window_go_xy(lwin, lwin->cursor_x, lwin->cursor_y);
	window_put_point(lwin);
}   /* window_puts_point */


/*
window_gets_filter() reads a string from a window, up to some maximum
limit, as long as printable ASCII characters are entered.  If <Esc>
is pressed, it returns FALSE.
*/

static int window_gets_filter(lwin, str, limit, header, message)
logical_window_frame *lwin;
char *str;
int limit; /* the max. number of chars. that can be read. */
char *header;
char *message;
{
	lbutton *btn_win = (lbutton *) lwin->extension;
	XEvent event;
	KeySym keysym;
	char ch, *temp, *start;
	int i, j, row, col;

	temp = start = str;
	for (i = 0, row = r_pos(lwin), col = c_pos(lwin), limit--;
			i < limit; ) {
		XNextEvent(display, &event);
		switch (event.type) {
			case Expose:
				while (XCheckTypedWindowEvent(display,
						lwin->xwin, Expose, &event))
					;
				if (event.xexpose.window == lwin->xwin) {
					*str = EOS;
					window_display_header(lwin, header);
					window_go_cr(lwin, 1, 2);
					window_clr_eol(lwin);
					window_puts(lwin, message, gc);
					window_go_cr(lwin, col, row);
					window_puts_point(lwin, start, gc);
					display_buttons(lwin, gc);
				}
				break;
			case ButtonPress:
				break;
			case ButtonRelease:
				if (btn_win == NULL) {
					*start = EOS;
					return FALSE;
				}
				else {
					*str = EOS;
					for (j = 0; btn_win[j].bid != EOB; j++)
						if (event.xbutton.window == btn_win[j].bid)
							return btn_win[j].value;
				}
				break;
			case KeyPress:
				ch = (char) window_getche(lwin, &event, &keysym);
				if (keysym == XK_Escape) {
					*start = EOS;		/* return a null string */
					return FALSE;
				}
				else if (keysym == XK_BackSpace ||
						keysym == XK_Delete) {
					if (str > temp) {
						str--;
						i--;
						backspace_char(lwin);
					}
				}
				else if (keysym == XK_Return) {
					*str = EOS;
					return TRUE;
				}
				else if (keysym >= XK_space &&
						keysym <= XK_asciitilde) {
					*str = ch;
					str++;
					i++;
				}    /* end if-then-else */
				break;
			default:
				break;
		}    /* end switch */
	}    /* end for loop */
	if (i == limit)     /* reached limit, terminate string */
		*str = EOS;
	return TRUE;
}   /* window_gets_filter */


/*
backspace_char() attempts to remove the previously entered
character from the "buffer."
*/

static void backspace_char(lwin)
logical_window_frame *lwin;
{
	window_no_point(lwin);
	lwin->cursor_x -= lwin->font_width;
	if (lwin->cursor_x < 0)
		lwin->cursor_x = 0;
	display_char(display, lwin->xwin, gc,
		lwin->cursor_x, lwin->cursor_y, BLANK);
	window_put_point(lwin);
}   /* backspace_char */


/*
window_getche() reads a character from a window, with echo.
*/

static int window_getche(lwin, event, keysym)
logical_window_frame *lwin;
XEvent *event;
KeySym *keysym;
{
	int key, return_key;

	window_go_xy(lwin, lwin->cursor_x, lwin->cursor_y);
	key = get_keystroke(event, keysym);
	if (*keysym == XK_Escape)
		return_key = ESC;
	else if (*keysym == XK_Return)
		return_key = LF;
	else if (*keysym == XK_BackSpace || *keysym == XK_Delete)
		return_key = BKSP;
	else if (*keysym == XK_Tab)
		return_key = TAB;
	else if (*keysym >= XK_space && *keysym <= XK_asciitilde) {
		echo_char(lwin, (char) key);
		return_key = key;
	}
	else
		return_key = FALSE;
	if (lwin->cursor_y < 0)
		lwin->cursor_y = 0;
	if (lwin->cursor_y > lwin->height)
		lwin->cursor_y -= lwin->font_height;
	window_go_xy(lwin, lwin->cursor_x, lwin->cursor_y);
	return return_key;
}   /* window_getche */


/*
window_getche_event_loop() reads a character from a window, with echo
in an event loop.
*/

static int window_getche_event_loop(lwin)
logical_window_frame *lwin;
{
	XEvent event;
	KeySym keysym;
	int key, kontinue = TRUE;

	while (kontinue) {
		XNextEvent(display, &event);
		switch (event.type) {
			case KeyPress:
				key = window_getche(lwin, &event, &keysym);
				kontinue = FALSE;
				break;
			default:
				break;
		}
	}    /* end event loop */
	return key;
}   /* window_getche_event_loop */


/*
echo_char() echos a character to the screen.
*/

static void echo_char(lwin, ch)
logical_window_frame *lwin;
char ch;
{
	if (lwin->cursor_x < lwin->width)
	    window_display_char_with_point(lwin, ch);
}   /* echo_char */


/*
window_display_char_with_point() implements the display of a software-
based keyboard cursor to accompany each displayed character.
*/

static void window_display_char_with_point(lwin, ch)
logical_window_frame *lwin;
char ch;
{
	window_no_point(lwin);
	display_char(display, lwin->xwin, gc,
		lwin->cursor_x, lwin->cursor_y, ch);
	lwin->cursor_x += lwin->font_width;
	window_put_point(lwin);
}    /* window_display_char_with_point */

