/****    simplewin.c    ****/

/********************************************************************
*  Copyright (c) 1990 Iris Computing Laboratories.
*
*  This software is provided for demonstration purposes only.  As
*  freely-distributed, modifiable source code, this software carries
*  absolutely no warranty.
********************************************************************/


#include "simplewin.h"


/*
Globals:
*/

GC gc, rgc, hgc;					/* used by all windows */
Cursor popup_cursor;



/*
initialize_window_structures() initializes data structures that
are shared by multiple windows -- module initialization.
*/

void initialize_window_structures(font)
XFontStruct *font;
{
	set_up_GC(&gc, font, NORMAL);
	set_up_GC(&rgc, font, REVERSE);
	set_up_GC(&hgc, font, NORMAL);
	XSetFunction(display, hgc, GXinvert);
	make_popup_cursor();
}	/* initialize_window_structures */


/*
cleanup_window_structures() frees/deletes module-level
window data structures.
*/

void cleanup_window_structures()
{
	XFreeCursor(display, popup_cursor);
	XFreeGC(display, gc);
	XFreeGC(display, rgc);
	XFreeGC(display, hgc);
}	/* cleanup_window_structures */


/*
make_window() creates an unmapped window.  It sets certain
components of the size_hints structure for that window, as a
convenience to the calling module.  It stores the X window
id as a member of the logical window data structure, lwin.
The window font is set by store_window_font_info().
*/

void make_window(parent, lwin, x, y, width, height,
	border_width, size_hints, font)
Window parent;
logical_window_frame *lwin;
int x, y;					/* row/column metrics */
unsigned int width, height;
unsigned int border_width;
XSizeHints *size_hints;
XFontStruct *font;
{
	lwin->font_height =
		font->max_bounds.ascent + font->max_bounds.descent;
	width *= font->max_bounds.width;	/* convert to pixels */
	height *= lwin->font_height;
	height += font->max_bounds.descent;	/* bottom padding */
	lwin->xwin = XCreateSimpleWindow(display, parent, x, y,
		width, height, border_width, BlackPixel(display, screen),
		WhitePixel(display, screen));

	size_hints->flags = PPosition | PSize | PMinSize | PMaxSize;
	size_hints->x = x;
	size_hints->y = y;
	size_hints->width = width;
	size_hints->height = height;
	lwin->size_hints = size_hints;	/* keep a reference to this */
	lwin->font = font;				/* keep a reference to this */

	lwin->width = width;			/* convenience variables */
	lwin->height = height;
	lwin->font_width = font->max_bounds.width;
	lwin->trunc_height =			/* without the padding */
		(lwin->height / lwin->font_height) * lwin->font_height;

	lwin->cursor_x = 0;
	lwin->cursor_y = lwin->font_height;	/* a little extra at top */
	lwin->is_active = FALSE;
	lwin->extension = NULL;
	XDefineCursor(display, lwin->xwin, popup_cursor);
	XSelectInput(display, lwin->xwin, ExposureMask |
		KeyPressMask | KeyReleaseMask |
		ButtonPressMask | ButtonReleaseMask | OwnerGrabButtonMask |
		EnterWindowMask | LeaveWindowMask | StructureNotifyMask);
}    /* make_window */


/*
cleanup_window() performs window-specific clean-up operations.
At present, the only structure is the window itself.
*/

void cleanup_window(lwin)
logical_window_frame *lwin;
{
	XDestroyWindow(display, lwin->xwin);
}	/* cleanup_window */


/*
set_up_GC() creates a GC that can be used by a system of
similar windows.  In addition, it initializes the current
font for the GC.
*/

void set_up_GC(gc, font, mode)
GC *gc;
XFontStruct *font;
int mode;
{
	*gc = XCreateGC(display, RootWindow(display, screen), 0, NULL);
	XSetFont(display, *gc, font->fid);
	if (mode == REVERSE) {
	    XSetForeground(display, *gc, WhitePixel(display, screen));
		XSetBackground(display, *gc, BlackPixel(display, screen));
	}
	else {
		XSetForeground(display, *gc, BlackPixel(display, screen));
		XSetBackground(display, *gc, WhitePixel(display, screen));
	}
}   /* set_up_GC */


/*
make_popup_cursor() makes an arrow cursor.
*/

void make_popup_cursor()
{
	popup_cursor = XCreateFontCursor(display, XC_left_ptr);
}    /* make_popup_cursor */


/*
activate_window() maps the window to the top.  If there
is a message, it is dispalyed as a header.
*/

void activate_window(lwin, message)
logical_window_frame *lwin;
char *message;
{
	if (lwin->is_active)
		return;
	lwin->is_active = TRUE;
	XMapRaised(display, lwin->xwin);
	if (*message)
		window_display_header(lwin, message);
}    /* activate_window */


/*
deactivate_window() unmaps the window and resets
the cursor coordinates.
*/

void deactivate_window(lwin)
logical_window_frame *lwin;
{
	if (!lwin->is_active)
		return;
	lwin->is_active = FALSE;
	XUnmapWindow(display, lwin->xwin);
	lwin->cursor_y = lwin->font_height;
	lwin->cursor_x = 0;
}   /* deactivate_window */


/*
window_puts() prints a string in a window w/o
interpreting each character.
*/

void window_puts(lwin, str, gc)
logical_window_frame *lwin;
char *str;
GC gc;		/* allow normal or reverse video */
{
	display_string(display, lwin->xwin, gc,
   	    lwin->cursor_x, lwin->cursor_y, str);
}   /* window_puts */


/*
window_puts_center() centers a string in a window w/o
interpreting each character.
*/

void window_puts_center(lwin, str, gc)
logical_window_frame *lwin;
char *str;
GC gc;		/* allow normal or reverse video */
{
	int pad = (last_col(lwin) - strlen(str)) / 2;

	if (pad < 0)
		pad = 0;
	display_string(display, lwin->xwin, gc,
   	    lwin->cursor_x + (pad * lwin->font_width),
   	    lwin->cursor_y, str);
}   /* window_puts_center */


/*
window_puts_by_char() draws a string on a window by making
repeated calls to window_putchar().
*/

int window_puts_by_char(lwin, str)
logical_window_frame *lwin;
char *str;
{
	for ( ; *str; str++)
		window_putchar(lwin, *str);
	return TRUE;
}    /* window_puts_by_char */


/*
window_puts_lf() prints a string in a window on a
line-by-line basis, based on linefeed characters.
NOTE:  This function temporarily modifies the string as
it displays each line.  Normally, strings should be treated
as constants.  This function has the potential to cause problems
with compilers that treat strings as read-only data structures.
With gcc, for example, you must use the `-fwritable-strings' option
to prevent core dumps, if you use this function.
*/

void window_puts_lf(lwin, str, gc)
logical_window_frame *lwin;
register char *str;
GC gc;
{
	char *sub_string;

	sub_string = "";	/* null str ==> first loop not entered */
	while (*str) {
		sub_string = str;
		while (*str && *str != LF)
			str++;
		if (*str == LF) {
			*str = EOS;
			display_string(display, lwin->xwin, gc,
	   	    	lwin->cursor_x, lwin->cursor_y,
				sub_string);
			lwin->cursor_y += lwin->font_height;
			*str++ = LF;
		}
	}
	display_string(display, lwin->xwin, gc,
		lwin->cursor_x, lwin->cursor_y,
		sub_string);
}   /* window_puts_lf */


/*
window_putchar() prints a character in a window.
*/

int window_putchar(lwin, ch)
logical_window_frame *lwin;
char ch;
{
	if (ch == LF) {
		lwin->cursor_y += lwin->font_height;
		lwin->cursor_x = 0;
	}
	else {
		display_char(display, lwin->xwin, gc,
			lwin->cursor_x, lwin->cursor_y, ch);
		lwin->cursor_x += lwin->font_width;
	}
	return TRUE;
}   /* window_putchar */


/*
window_clr_eol() clears/overwrites the current line of a
window from the current position to the end of the line.
*/

void window_clr_eol(lwin)
logical_window_frame *lwin;
{
	XClearArea(display, lwin->xwin, lwin->cursor_x,
		lwin->cursor_y - lwin->font->max_bounds.ascent,
		lwin->width - lwin->cursor_x,
		lwin->font_height, FALSE);
}   /* window_clr_eol */


/*
window_display_header() can be used to display a left-justified
header on the top line of a window.
*/

void window_display_header(lwin, str)
logical_window_frame *lwin;
char *str;
{
	XSetFunction(display, rgc, GXset);
	XFillRectangle(display, lwin->xwin, rgc, 0, 0, lwin->width,
		lwin->font_height + lwin->font->max_bounds.descent);
	XSetFunction(display, rgc, GXcopy);
	display_string(display, lwin->xwin, rgc,
   	    lwin->font_width,
   	    lwin->font_height - lwin->font->max_bounds.descent, str);
}	/* window_display_header */


/*
window_reverse_row() displays a row in reverse video.
*/

void window_reverse_row(lwin)
logical_window_frame *lwin;
{
	XSetFunction(display, rgc, GXset);
	XFillRectangle(display, lwin->xwin, rgc, 0,
		lwin->cursor_y - lwin->font->max_bounds.ascent, lwin->width,
		lwin->font_height + lwin->font->max_bounds.descent);
	XSetFunction(display, rgc, GXcopy);
}	/* window_reverse_row */


/*
window_go_xy() sets the coordinates within a window in pixels.
*/

int window_go_xy(lwin, x, y)
logical_window_frame *lwin;
int x, y;
{
	if (x < 0 || x > lwin->width)
		return FALSE;
	if (y < 0 || y > lwin->height)
		return FALSE;
	lwin->cursor_x = x;
	lwin->cursor_y = y;
	return TRUE;
}   /* window_go_xy */


/*
window_go_cr() sets the coordinates within a window
in rows and columns.
*/

int window_go_cr(lwin, col, row)
logical_window_frame *lwin;
int col, row;
{
	if (col < 0 || (col * lwin->font_width) >
				(lwin->width - lwin->font_width))
		return FALSE;
	if (row < 0 || ((row + 1) * lwin->font_height) >
			lwin->trunc_height)
		return FALSE;
	window_go_xy(lwin, col * lwin->font_width,
		(row + 1) * lwin->font_height);
	return TRUE;
}    /* window_go_cr */


/*
window_clr_rest() clears the remainder of a window, relative
to the cursor, that is, current row-column coordinates.
*/

void window_clr_rest(lwin)
logical_window_frame *lwin;
{
	XClearArea(display, lwin->xwin, lwin->cursor_x,
		lwin->cursor_y - lwin->font->max_bounds.ascent,
		lwin->width - lwin->cursor_x,
		lwin->font_height, FALSE);
	XClearArea(display, lwin->xwin, 0,
		lwin->cursor_y + lwin->font->max_bounds.descent,
		0, 0, FALSE);
}   /* window_clr_rest */


/*
window_cursor_bottom() takes the cursor to the bottom line of
the window in font-based metrics.
*/

void window_cursor_bottom(lwin)
logical_window_frame *lwin;
{
	window_go_xy(lwin, lwin->cursor_x, (int) lwin->trunc_height);
}   /* window_cursor_bottom */


/*
r_pos() returns the current row (y-coordinate) within
a given window.
*/

int r_pos(lwin)
logical_window_frame *lwin;
{
	return (lwin->cursor_y / lwin->font_height) - 1;
}   /* r_pos */


/*
c_pos() returns the current column (x-coordinate) within
a given window.
*/

int c_pos(lwin)
logical_window_frame *lwin;
{
	return (lwin->cursor_x / lwin->font_width);
}   /* c_pos */


/*
last_row() returns the number of the last row in the
window, zero-based.
*/

int last_row(lwin)
logical_window_frame *lwin;
{
	return (lwin->trunc_height / lwin->font_height) - 1;
}    /* last_row */


/*
last_col() returns the number of the last column in the
window, zero-based.
*/

int last_col(lwin)
logical_window_frame *lwin;
{
	return (lwin->width / lwin->font_width) - 1;
}    /* last_col */


/*
get_keystroke() reads and returns a keystroke. 
*/

int get_keystroke(event, keysym)
XEvent *event;
KeySym *keysym;		/* return the keysym */
{
	char text[20];
	int len;

	len = XLookupString(event, text, 20, keysym, 0);
	if (len == 1)
		return text[0];
	else
		return FALSE;
}   /* get_keystroke */

