/****  xclk.c  ****/

/********************************************************************
*  Copyright (c) 1990 Iris Computing Laboratories.
*
*  This software is provided for demonstration purposes only.  As
*  freely-distributed, modifiable source code, this software carries
*  absolutely no warranty.
********************************************************************/

/*
xclk displays the time in a small window, but only when
"prodded" to do so.  When the pointer enters the window,
the time is displayed; when the pointer exits the window,
the time is cleared.  A optional header is available for
use with uwm.
*/

#include "xclk.h"



/*
main() sets up the data structures, calls the display
routines, and then waits for the application to finish.
*/

void main(argc, argv)
int argc;
char *argv[];
{
	Window clk_win;
	XFontStruct *clk_font, *hdr_font;
	Cursor clk_cursor;
	GC gc, rgc;
	XSizeHints cw_size_hints;
	int has_hdr = FALSE, hdr_height = 0;	/* no header by default */
	int clk_width, clk_height;
	char hdr_font_name[MAX_FONT_NAME], clk_font_name[MAX_FONT_NAME];

	process_font_options(argc, argv, &has_hdr, hdr_font_name,
		clk_font_name);
	init_display(*argv);
	if (has_hdr)
		load_font(&hdr_font, *argv, hdr_font_name);
	else
		hdr_font = NULL;
	load_font(&clk_font, *argv, clk_font_name);

	if (has_hdr) {
		hdr_height = hdr_font->max_bounds.ascent +
			hdr_font->max_bounds.descent;
		hdr_height += (hdr_height / 4);			/* add 25% */
	}
	set_up_size_hints(clk_font, &clk_width,
		&clk_height, hdr_height, &cw_size_hints);
	clk_win = XCreateSimpleWindow(display,
		RootWindow(display, screen), 0, 0,
		clk_width, clk_height, BORDER_WIDTH,
		BlackPixel(display, screen), WhitePixel(display, screen));
	XSetStandardProperties(display, clk_win, *argv, *argv,
		None, argv, argc, &cw_size_hints);
	XSelectInput(display, clk_win, ExposureMask | ButtonPressMask |
		EnterWindowMask | LeaveWindowMask);

	clk_cursor = XCreateFontCursor(display, XC_left_ptr);
	XDefineCursor(display, clk_win, clk_cursor);
	set_up_GC(&gc, clk_font, NORMAL);
	if (has_hdr)
		set_up_GC(&rgc, hdr_font, REVERSE);
	display_clock(clk_win, clk_font, hdr_font, gc, rgc,
		clk_width, clk_height, hdr_height);
	cleanup_resources(clk_win, clk_font, hdr_font, clk_cursor);
	exit(0);
}	/* main */


/*
process_font_options() parses the command line and
initializes font-related variables.
*/

void process_font_options(argc, argv, has_hdr, hdr_font_name,
	clk_font_name)
int argc;
char *argv[];
int *has_hdr;
char *hdr_font_name, *clk_font_name;
{
	int syntax_error = FALSE;
	char *app_name = argv[0];

	if (argc > 4) {
		fprintf(stderr,
		"usage: %s [-h] [-hf<header_font>] [-cf<clock_font>]\n",
			app_name);
 		exit(-1);
	}
	strcpy(hdr_font_name, HDR_FONT);		/* the defaults */
	strcpy(clk_font_name, CLK_FONT);
	while (--argc > 0) {
		argv++;
		if (strlen(*argv) == 1)
			syntax_error = TRUE;
		if (strlen(*argv) == 2 && strcmp(*argv, "-h"))
			syntax_error = TRUE;
		if (strlen(*argv) > 2 &&
				(strncmp(*argv, "-hf", 3) &&
					strncmp(*argv, "-cf", 3)))
			syntax_error = TRUE;

		if (strcmp(*argv, "-h") == 0)
			*has_hdr = TRUE;
		else if (strncmp(*argv, "-hf", 3) == 0) {
			strncpy(hdr_font_name, &(*argv)[3], MAX_FONT_NAME);
			*has_hdr = TRUE;
		}
		else if (strncmp(*argv, "-cf", 3) == 0)
			strncpy(clk_font_name, &(*argv)[3], MAX_FONT_NAME);
		else
			/* do nothing */;
	}
	if (syntax_error) {
		fprintf(stderr,
		"usage: %s [-h] [-hf<header_font>] [-cf<clock_font>]\n",
			app_name);
		exit(-1);
	}
}	/* process_font_options */


/*
display_clock() determines the time and displays it.
A button click terminates the event loop.
*/

void display_clock(clk_win, clk_font, hdr_font, gc, rgc,
	clk_width, clk_height, hdr_height)
Window clk_win;
XFontStruct *clk_font, *hdr_font;
GC gc, rgc;
int clk_width, clk_height, hdr_height;
{
    XEvent event;

	XMapRaised(display, clk_win);
	while (TRUE) {
		XNextEvent(display, &event);
		switch (event.type) {
			case Expose:
				display_header(clk_win, rgc, clk_width,
					hdr_height, hdr_font);
				break;
			case EnterNotify:
				update_clock(clk_win, clk_font, gc, get_the_time(),
					clk_width, clk_height, hdr_height);
				break;
			case LeaveNotify:
				update_clock(clk_win, clk_font, gc, NULL,
					clk_width, clk_height, hdr_height);
				break;
			case ButtonPress:
				return;			/* terminate the program */
		}
	}
}	/* display_clock */


/*
display_header() displays a reverse video header with a
"neutral" title centered in the header.  Some window managers
will add a title bar with the application name.
*/

void display_header(clk_win, rgc, clk_width, hdr_height, hdr_font)
Window clk_win;
GC rgc;
int clk_width, hdr_height;
XFontStruct *hdr_font;
{
	char *label = "time";

	if (hdr_height == 0)
		return;
	XSetFunction(display, rgc, GXset);
	XFillRectangle(display, clk_win, rgc,
		0, 0, clk_width, hdr_height);
	XSetFunction(display, rgc, GXcopy);
	XDrawImageString(display, clk_win, rgc,
		(clk_width - XTextWidth(hdr_font, label, strlen(label))) / 2,
		hdr_height - (hdr_height / 4),		/* 25% */
		label, strlen(label));
}	/* display_header */


/*
update_clock() displays the time in the center of the non-
header window area.  Proportional fonts are OK.
*/

void update_clock(clk_win, clk_font, gc, time,
	clk_width, clk_height, hdr_height)
Window clk_win;
XFontStruct *clk_font;
GC gc;
char *time;
int clk_width, clk_height, hdr_height;
{
	if (time == NULL)
		XClearArea(display, clk_win, 0, hdr_height, 0, 0, FALSE);
	else {
		int x_pos, y_pos;

		x_pos =
			(clk_width - XTextWidth(clk_font, time, strlen(time))) /
			2;
		y_pos = clk_height - clk_font->max_bounds.ascent;
		XDrawImageString(display, clk_win, gc,
			x_pos, y_pos, time, strlen(time));
	}
}	/* update_clock */


/*
get_the_time() performs the time calculations,
and returns the result, stored in a local buffer.
*/

char *get_the_time()
{
    long tyme = time(0);
    struct tm *local_time = localtime(&tyme);
    int hour;
    static char char_time[20];

	hour = (int) local_time->tm_hour;
	if (hour == 0)
		hour = 12;
	else if (hour > 12)
		hour -= 12;
	sprintf(char_time, "%02d:%02d", hour, local_time->tm_min);
	return char_time;
}	/* get_the_time */


/*
set_up_GC() creates a GC that can be used by a system of
similar windows.  In addition, it initializes the current
font for the GC.
*/

void set_up_GC(gc, font, mode)
GC *gc;
XFontStruct *font;
int mode;
{
	*gc = XCreateGC(display, RootWindow(display, screen), 0, NULL);
	XSetFont(display, *gc, font->fid);
	if (mode == REVERSE) {
	    XSetForeground(display, *gc, WhitePixel(display, screen));
		XSetBackground(display, *gc, BlackPixel(display, screen));
	}
	else {
		XSetForeground(display, *gc, BlackPixel(display, screen));
		XSetBackground(display, *gc, WhitePixel(display, screen));
	}
}   /* set_up_GC */


/*
cleanup_resources() performs an orderly clean-up
form the fonts, display, etc.
*/

void cleanup_resources(clk_win, clk_font, hdr_font, clk_cursor)
Window clk_win;
XFontStruct *clk_font, *hdr_font;
Cursor clk_cursor;
{
	XFreeCursor(display, clk_cursor);
	XDestroyWindow(display, clk_win);
	XUnloadFont(display, clk_font->fid);
	if (hdr_font)
		XUnloadFont(display, hdr_font->fid);
	XCloseDisplay(display);
}	/* cleanup_resources */


/*
set_up_size_hints() fills in the window dimensions.
xclk does not support resizing.
*/

void set_up_size_hints(clk_font, clk_width, clk_height,
	hdr_height, size_hints)
XFontStruct *clk_font;
int *clk_width;
int *clk_height;
int hdr_height;
XSizeHints *size_hints;
{
	char *time = get_the_time();

	size_hints->flags = PPosition | PSize | PMinSize | PMaxSize;
	size_hints->x = 0;
	size_hints->y = 0;
	*clk_width = XTextWidth(clk_font, time, strlen(time)) * 2;
	*clk_height = clk_font->max_bounds.ascent +
		clk_font->max_bounds.descent;
	*clk_height *= 2;
	*clk_height += hdr_height;
	size_hints->width = size_hints->min_width =
		size_hints->max_width = *clk_width;
	size_hints->height = size_hints->min_height =
		size_hints->max_height = *clk_height;
}	/* set_up_size_hints */

