/**  xdump.c  **  a simple hex-dump window w/o resize capability  **/

/********************************************************************
*  Copyright (c) 1990 Iris Computing Laboratories.
*
*  This software is provided for demonstration purposes only.  As
*  freely-distributed, modifiable source code, this software carries
*  absolutely no warranty.
********************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "simplewin.h"


#define BORDER_WIDTH	2
#define NUM_ROWS		30
#define NUM_COLUMNS		78
#define FONT_SIZE		"8x13"




/*
Functions:
*/

void main(), hex_dump(), cleanup_resources();
int display_next_page(), build_next_page();
char *build_next_line();



/*
main() tests for command line errors, sets up the
data structures, calls hex_dump(), and then waits for
the application to finish.
*/

void main(argc, argv)
int argc;
char *argv[];
{
	logical_window_frame lwindow, *lwin = &lwindow;
	XFontStruct *font;
	XSizeHints dw_size_hints;
	FILE *dump_file;
	char *app_name = "xdump";

	if (argc != 2) {
		fprintf(stderr, "usage:  xdump <filename>\n");
 		exit(-1);
	}
	if ((dump_file = fopen(argv[1], "rb")) == NULL) {
		fprintf(stderr, "Can't open input file %s\n", argv[1]);
		exit(-1);
	}
	init_display(app_name);
	load_font(&font, app_name, FONT_SIZE);
	initialize_window_structures(font);
	/* no support for resize */
	dw_size_hints.min_width = dw_size_hints.max_width =
		font->max_bounds.width * NUM_COLUMNS;
	make_window(RootWindow(display, screen), lwin, 0, 0,
		NUM_COLUMNS, NUM_ROWS, BORDER_WIDTH, &dw_size_hints, font);
	XSetStandardProperties(display, lwin->xwin, app_name, app_name,
		None, argv, argc, &dw_size_hints);
	hex_dump(dump_file, lwin, argv[1]);
	cleanup_resources(lwin, font);
	exit(0);
}	/* main */


/*
cleanup_resources() performs an orderly clean-up.
*/

void cleanup_resources(lwin, font)
logical_window_frame *lwin;
XFontStruct *font;
{
	cleanup_window_structures();
	cleanup_window(lwin);
	XUnloadFont(display, font->fid);
	XCloseDisplay(display);
}	/* cleanup_resources */


/*
hex_dump() manages the initial exposure, events, etc.
*/

void hex_dump(dump_file, dump_win, filename)
FILE *dump_file;
logical_window_frame *dump_win;
char *filename;
{
	char header[NUM_COLUMNS + 1];
	char next_page[NUM_ROWS - 2][NUM_COLUMNS + 1];
	int eof = FALSE, stop = FALSE;
	XEvent event;

	strcpy(header, "xdump:  ");
	strncat(header, filename, NUM_COLUMNS - 9);
	eof = build_next_page(dump_file, next_page);
	activate_window(dump_win, "");
  	while (!stop) {
  		XNextEvent(display, &event);
  		switch (event.type) {
  			case Expose:
  				window_go_cr(dump_win, 1, 0);
  				window_display_header(dump_win, header);
				window_go_cr(dump_win, 0, 1);
  				display_next_page(dump_win, next_page, eof);
  				break;
			case ButtonPress:
				if (!eof && event.xbutton.button == Button1) {
					eof = build_next_page(dump_file, next_page);
					display_next_page(dump_win, next_page, eof);
				}
				if (event.xbutton.button == Button3)
					stop = TRUE;
				break;
		}
	}
	deactivate_window(dump_win);
}	/* hex_dump */


/*
display_next_page() clears the window and dumps the next page
from the file.  In particular, it manages overall file read
operations and builds 16-byte buffers/chunks of raw data for
processing by build_next_line().  If the last chunk is less
that 16 bytes, it must be padded with 0s.
*/

int display_next_page(dump_win, next_page, last_page)
logical_window_frame *dump_win;
char next_page[][NUM_COLUMNS + 1];
int last_page;
{
	int last = NUM_ROWS - 2, row;

  	window_go_cr(dump_win, 0, 1);
  	window_clr_rest(dump_win);
	for (row = 0; row < last; row++) {
		window_go_cr(dump_win, 1, row + 1);
		window_puts(dump_win, next_page[row], gc);
	}
	window_cursor_bottom(dump_win);
	if (!last_page) {
		window_reverse_row(dump_win);
		window_puts_center(dump_win,
			" <ClickLeft> for next page -- <ClickRight> to quit...",
			rgc);
	}
	else {
		window_clr_eol(dump_win);
		window_reverse_row(dump_win);
		window_puts_center(dump_win,
			" <ClickRight> to finish...", rgc);
	}
	return last_page;
}	/* display_next_page */


/*
build_next_page() builds a page of formatted output.
*/

int build_next_page(dump_file, next_page)
FILE *dump_file;
char next_page[][NUM_COLUMNS + 1];
{
	static int count = 0, last = NUM_ROWS - 2;
	int ch, i, last_page = FALSE, row;
	char char_buffer[16];
	
	for (row = 0; !last_page && row < last; row++) {
		for (i = 0; i < 16; i++) {				/* read file */
			if ((ch = getc(dump_file)) == EOF) {
				while (i < 16)
					char_buffer[i++] = EOS;		/* pad buffer */
				last_page = TRUE;
				fclose(dump_file);
				break;
			}
			char_buffer[i] = (char) ch;	/* just copy next byte */
		}
		strcpy(next_page[row], build_next_line(char_buffer, count));
		count++;
	}
	for ( ; last_page && row < last; row++)
		next_page[row][0] = EOS;
	return last_page;
}	/* build_next_page */


/*
build_next_line() builds the next line of hex-dumped output,
returning a pointer to the output line.  The output has
an assembler dump format.
*/

char *build_next_line(char_buffer, count)
char *char_buffer;
int count;
{
	static char dump_buffer[NUM_COLUMNS + 1];
	int i, j;

	sprintf(dump_buffer, "%05x ", count * 16);	/* change to + */
	for (i = 0, j = 6; i < 8; i++, j += 3)
		sprintf(&dump_buffer[j], " %02x", char_buffer[i]);
	sprintf(&dump_buffer[j], "--%02x", char_buffer[i++]);
	for (j += 4; i < 16; i++, j += 3)
		sprintf(&dump_buffer[j], " %02x", char_buffer[i]);
	sprintf(&dump_buffer[j], "  |");
	for (i = 0, j += 3; i < 8; i++, j++)
		sprintf(&dump_buffer[j], "%c",
			isprint(char_buffer[i]) ? char_buffer[i] : '.');
	sprintf(&dump_buffer[j], "-");
	for (j++; i < 16; i++, j++)
		sprintf(&dump_buffer[j], "%c",
			isprint(char_buffer[i]) ? char_buffer[i] : '.');
	sprintf(&dump_buffer[j], "|");
	dump_buffer[NUM_COLUMNS] = EOS;
	return dump_buffer;
}	/* build_next_line */

