/****    xwaste.c    ****/

/********************************************************************
*  Copyright (c) 1990 Iris Computing Laboratories.
*
*  This software is provided for demonstration purposes only.  As
*  freely-distributed, modifiable source code, this software carries
*  absolutely no warranty.
********************************************************************/


/********************************************************/
/* XWASTE implement a two-stage file deletion system.	*/
/* See the complementary module delete.c.				*/
/********************************************************/


#include "xwaste.h"


/*
Private globals:
*/

static XtResource resources[] = {
	{XtNbrowserRows, XtCBrowserRows, XtRInt, sizeof(int),
		XtOffset(ApplicationDataPtr, browser_rows), XtRImmediate,
		(caddr_t) DEFAULT_BROWSER_ROWS},
	{XtNbrowserColumns, XtCBrowserColumns, XtRInt, sizeof(int),
		XtOffset(ApplicationDataPtr, browser_columns), XtRImmediate,
		(caddr_t) DEFAULT_BROWSER_COLUMNS},
	{XtNbrowserFont, XtCBrowserFont, XtRString, sizeof(char *),
		XtOffset(ApplicationDataPtr, browser_font_name),
		XtRString, DEFAULT_BROWSER_FONT_NAME},
	{XtNbrowserForeground, XtCBrowserForeground, XtRPixel, sizeof(Pixel),
		XtOffset(ApplicationDataPtr, browser_foreground),
		XtRString, XtDefaultForeground},
	{XtNbrowserBackground, XtCBrowserBackground, XtRPixel, sizeof(Pixel),
		XtOffset(ApplicationDataPtr, browser_background),
		XtRString, XtDefaultBackground},
};


static XtActionsRec actionsTable[] = {
	{"selectFile", (XtActionProc) SelectFile},
	{"fillBasket", (XtActionProc) FillBasket},
};


static Widget basketShell, topLevel, fileSpecForm,
	directoryLabel, filenameLabel,
	directoryBox, filenameBox, buttonBoxMain, vPort,
	vPane, vPaneBasket, wasteBasket, buttonBoxBasket,
	buttonQuit, buttonDelete, buttonBasket, buttonDoneBasket;


static Pixmap icon_pixmap;		/* freed by Quit() */
static XFontStruct *font;		/* freed by Quit() */


/*
main() determines the home directory and basket dimensions,
creates the main widgets, sets up the alert boxes, initializes
data structures, etc.
*/

void main(argc, argv)
int argc;
char *argv[];
{
	unsigned int basket_width, basket_height;
	Arg args[5];
	int i;
	ApplicationData browser_data;

	if (!get_home_directory(home_directory))
		exit(-1);

	topLevel = XtInitialize(argv[0], "XWasteMotif", NULL, 0,
		&argc, argv);
	process_resources(&browser_data, &font);
	i = 0;
	XtSetArg(args[i], XmNinput, (XtArgVal) TRUE); i++;
	XtSetValues(topLevel, args, i);

	create_main_panel_widgets();
	basket_create_offscreen_pixmap(topLevel, font,
		&browser_data, &basket_width, &basket_height);
	create_basket_widgets(basket_width, basket_height);
	max_basket_entries = get_max_files();
	XtAddActions(actionsTable, XtNumber(actionsTable));

	menu_create(basketShell);
	alert_create_ync(topLevel);
	alert_create_continue(topLevel);

	initialize_shell_structures();

	XtRealizeWidget(topLevel);
	XtRealizeWidget(basketShell);

	XtMainLoop();
}	/* main */


/*
process_resources() reads dimensions for the browser (off-screen
pixmap of the wastebasket), checks their bounds, and exits with
warning message(s), if there are problems.
*/

void process_resources(browser_data, font)
ApplicationData *browser_data;
XFontStruct **font;
{
	XtGetApplicationResources(topLevel, browser_data, resources,
		XtNumber(resources), NULL, 0);
	if (browser_data->browser_rows < 1 ||
			browser_data->browser_rows > MAX_BROWSER_ROWS) {
		fprintf(stderr, "xwaste: Illegal row dimension: %d\n",
			browser_data->browser_rows);
		fprintf(stderr,
			"Number of rows must be in the interval: [1,%d]\n",
			MAX_BROWSER_ROWS);
		exit(-1);
	}
	if (browser_data->browser_columns < 1 ||
			browser_data->browser_columns > MAX_BROWSER_COLUMNS) {
		fprintf(stderr, "xwaste: Illegal column dimension: %d\n",
			browser_data->browser_columns);
		fprintf(stderr,
			"Number of columns must be in the interval: [1,%d]\n",
			MAX_BROWSER_COLUMNS);
		exit(-1);
	}
	if ((*font = XLoadQueryFont(XtDisplay(topLevel),
			browser_data->browser_font_name)) == NULL) {
		fprintf(stderr,
			"Couldn't load font: %s; using default browser font.\n",
			browser_data->browser_font_name);
		if ((*font = XLoadQueryFont(XtDisplay(topLevel),
				DEFAULT_BROWSER_FONT_NAME)) == NULL) {
			fprintf(stderr,
		"Couldn't load default browser font either, exiting...\n");
			exit(-1);
		}
	}
}	/* process_resources */


/*
create_main_panel_widgets() creates/initializes the widgets that
constitute the top-level, main panel.
*/

void create_main_panel_widgets()
{
	Arg args[10];
	int i;

	vPane = XtCreateManagedWidget("vPane",
		xmPanedWindowWidgetClass, topLevel, NULL, 0);
	i = 0;
	XtSetArg(args[i], XmNorientation, (XtArgVal) XmHORIZONTAL); i++;
	buttonBoxMain = XtCreateManagedWidget("buttonBoxMain",
		xmRowColumnWidgetClass, vPane, args, i);
	i = 0;
	XtSetArg(args[i], XmNpacking, (XtArgVal) XmPACK_COLUMN); i++;
	XtSetArg(args[i], XmNnumColumns, (XtArgVal) 2); i++;
	XtSetArg(args[i], XmNisAligned, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XmNentryAlignment,
		(XtArgVal) XmALIGNMENT_CENTER); i++;
	fileSpecForm = XtCreateManagedWidget("fileSpecForm",
		xmRowColumnWidgetClass, vPane, args, i);
	i = 0;
	XtSetArg(args[i], XmNlabelType, (XtArgVal) XmSTRING); i++;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Directory:",
			XmSTRING_DEFAULT_CHARSET)); i++;
	XtSetArg(args[i], XmNborderWidth, (XtArgVal) 0); i++;
	directoryLabel = XtCreateManagedWidget("directoryLabel",
		xmLabelWidgetClass, fileSpecForm, args, i);
	i = 0;
	XtSetArg(args[i], XmNeditMode,
		(XtArgVal) XmSINGLE_LINE_EDIT); i++;
	directoryBox= XtCreateManagedWidget("directoryBox",
		xmTextWidgetClass, fileSpecForm, args, i);
	i = 0;
	XtSetArg(args[i], XmNlabelType, (XtArgVal) XmSTRING); i++;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Filename:",
			XmSTRING_DEFAULT_CHARSET)); i++;
	XtSetArg(args[i], XmNborderWidth, (XtArgVal) 0); i++;
	filenameLabel = XtCreateManagedWidget("filenameLabel",
		xmLabelWidgetClass, fileSpecForm, args, i);
	i = 0;
	XtSetArg(args[i], XmNeditMode,
		(XtArgVal) XmSINGLE_LINE_EDIT); i++;
	filenameBox = XtCreateManagedWidget("filenameBox",
		xmTextWidgetClass, fileSpecForm, args, i);
	i = 0;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Quit", XmSTRING_DEFAULT_CHARSET)); i++;
	buttonQuit = XtCreateManagedWidget("buttonQuit",
		xmPushButtonWidgetClass, buttonBoxMain, args, i);
	XtAddCallback(buttonQuit, XmNactivateCallback, Quit, NULL);
	i = 0;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Delete",
			XmSTRING_DEFAULT_CHARSET)); i++;
	buttonDelete = XtCreateManagedWidget("buttonDelete",
		xmPushButtonWidgetClass, buttonBoxMain, args, i);
	XtAddCallback(buttonDelete, XmNactivateCallback, Delete, NULL);
	i = 0;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Basket",
			XmSTRING_DEFAULT_CHARSET)); i++;
	buttonBasket = XtCreateManagedWidget("buttonBasket",
		xmPushButtonWidgetClass, buttonBoxMain, args, i);
	XtAddCallback(buttonBasket, XmNactivateCallback, Basket, NULL);
}	/* create_main_panel_widgets */


/*
create_basket_widgets() creates/initializes the widgets
in the pop-up wastebasket frame.
*/

void create_basket_widgets(basket_width, basket_height)
unsigned int basket_width, basket_height;
{
	Arg args[10];
	int i;

	i = 0;
	XtSetArg(args[i], XmNallowShellResize, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XmNmappedWhenManaged, (XtArgVal) FALSE); i++;
	basketShell = XtCreateApplicationShell("basketShell",
		topLevelShellWidgetClass, args, i);
	vPaneBasket = XtCreateManagedWidget("vPaneBasket",
		xmPanedWindowWidgetClass, basketShell, NULL, 0);
	i = 0;
	XtSetArg(args[i], XmNorientation, (XtArgVal) XmHORIZONTAL); i++;
	buttonBoxBasket = XtCreateManagedWidget("buttonBoxBasket",
		xmRowColumnWidgetClass, vPaneBasket, args, i);
	i = 0;
	XtSetArg(args[i], XmNwidth,
		(XtArgVal) DEFAULT_BASKET_WIDTH); i++;
	XtSetArg(args[i], XmNheight,
		(XtArgVal) DEFAULT_BASKET_HEIGHT); i++;
	XtSetArg(args[i], XmNscrollingPolicy,
		(XtArgVal) XmAUTOMATIC); i++;
	XtSetArg(args[i], XmNscrollBarDisplayPolicy,
		(XtArgVal) XmSTATIC); i++;
	XtSetArg(args[i], XmNscrollBarPlacement,
		(XtArgVal) XmTOP_LEFT); i++;
	vPort = XtCreateManagedWidget("vPort",
		xmScrolledWindowWidgetClass, vPaneBasket, args, i);
	i = 0;
	XtSetArg(args[i], XmNlabelString,
		XmStringCreateLtoR("Done", XmSTRING_DEFAULT_CHARSET)); i++;
	buttonDoneBasket = XtCreateManagedWidget("buttonDoneBasket",
		xmPushButtonWidgetClass, buttonBoxBasket, args, i);
	XtAddCallback(buttonDoneBasket,
		XmNactivateCallback, BasketDone, NULL);
	i = 0;
	XtSetArg(args[i], XmNwidth, (XtArgVal) basket_width); i++;
	XtSetArg(args[i], XmNheight, (XtArgVal) basket_height); i++;
	wasteBasket = XtCreateManagedWidget("wasteBasket", widgetClass,
		vPort, args, i);
	basket_set_reference_widget(wasteBasket);
}	/* create_basket_widgets */


/*
initialize_shell_structures sets up miscellaneous structures
for the top-level widgets (currently, just the icon).
*/

void initialize_shell_structures()
{

#include "xwaste.icon"

	Arg args[3];

	icon_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
		RootWindowOfScreen(XtScreen(topLevel)), xwaste_bits,
		xwaste_width, xwaste_height);
	XtSetArg(args[0], XmNiconPixmap, (XtArgVal) icon_pixmap);
	XtSetValues(topLevel, args, 1);
	XtSetArg(args[0], XmNiconPixmap, (XtArgVal) icon_pixmap);
	XtSetValues(basketShell, args, 1);
}	/* initialize_shell_structures */


/*
Action procedures for the wastebasket menu -- each must be
associated with a translation table entry.
*/

/*ARGSUSED*/
static void SelectFile(w, event)	/* highlight the filename */
Widget w;
XEvent *event;
{
	basket_select_file(event);
}	/* SelectFile */


/*ARGSUSED*/
static void FillBasket(w, event)
Widget w;
XEvent *event;
{
	basket_reset_current_file();
	basket_fill_with_filenames(NO_HIGHLIGHT);
}	/* FillBasket */


/*
Callback procedures:
*/

/*
Quit() terminates the application.
*/
/*ARGSUSED*/
void Quit(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
	XFreePixmap(XtDisplay(w), icon_pixmap);
	XFreeFont(XtDisplay(w), font);
	exit(0);
}	/* Quit */


/*
Delete() is the starting point for file deletions.
*/
/*ARGSUSED*/
void Delete(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
	char *filename;

	if ((filename = get_filename()) != NULL)
		expand_delete_filename(filename);
}	/* Delete */


/*
Basket() is the starting point for
wastebasket recovery/browser operations.
*/
/*ARGSUSED*/
void Basket(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
	if (wastebasket_empty()) {
		alert("The wastebasket is empty.");
		return;
	}
	XtMapWidget(basketShell);
	basket_reset_current_file();/* no highlighted file at		*/
								/* start-up, or after exposure	*/
	if (!basket_fill_with_filenames(NO_HIGHLIGHT))
		return;
	if (current_num_files > max_basket_entries) {
		alert_beep(topLevel, 8);
		alert("Please read the message in the console window!");
		fprintf(stderr,
"The wastebasket contains more entries than\n\
specified by the environment variable\n\
WASTEBASKET_SIZE (currently:  %d).\n\
If you make modifications, your wastebasket\n\
size will be reduced accordingly.\n", max_basket_entries);
	}
}	/* Basket */


/*
BasketDone() removes the "basket" window.
*/
/*ARGSUSED*/
void BasketDone(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
	XtUnmapWidget(basketShell);
}	/* BasketDone */


/*
General-purpose routines:
*/

/*
get_filename() gets a filename.
*/

char *get_filename()
{
	char *filename;

	filename = XmTextGetString(filenameBox);
	if (!strlen(filename)) {
		alert("Please select a file.");
		return NULL;		/* return if no selection */
	}
	else
		return filename;
}	/* get_filename */


/*
get_updated_directory() manages and returns
the "current" directory.
*/

char *get_updated_directory()
{
	static char previous_dir[MAX_PATH_LEN];
	char *current_dir;

	current_dir = XmTextGetString(directoryBox);
	if (strlen(current_dir) == 0) {
		if ((current_dir = (char *)
				getcwd(previous_dir, MAX_PATH_LEN)) == NULL) {
			fprintf(stderr,
			"Unable to determine current working directory!!\n");
			exit(-1);
		}
	}
	if (strcmp(current_dir, previous_dir))
		strcpy(previous_dir, current_dir);
	return previous_dir;	/*  now, same as current_dir */
}	/* get_updated_directory */


/*
expand_delete_filename() expands wildcards and then applies
delete_file() to each filename in the expansion.
*/

void expand_delete_filename(filename)
char *filename;
{
	char cmd_string[MAX_PATH_LEN + MAX_STR];
	char file_spec[MAX_PATH_LEN + MAX_STR];
	char *filename_buffer, *next;
	char temp_file[MAX_PATH_LEN + MAX_STR];
	int continue_deletions = TRUE;

	sprintf(temp_file, "%s.%d",
		"iris.wastebasket.delete.temp.filenames", getpid());
	unlink(temp_file);		/* if it exists */
	sprintf(file_spec, "%s/%s", get_updated_directory(), filename);
	sprintf(cmd_string, "echo %s > %s", file_spec, temp_file);
	system(cmd_string);
	if (file_size(temp_file) == -1) {
		alert("Unknown error during creation of a temporary file!");
		return;
	}
	if ((filename_buffer =
			(char *) malloc((unsigned) (file_size(temp_file) + 1)))
				== NULL) {
		alert("Memory allocation error!");
		unlink(temp_file);
		return;
	}
	if (!load_buffer(temp_file, filename_buffer)) {
		alert("Unable to access a temporary file!");
		free(filename_buffer);
		unlink(temp_file);
		return;
	}
	next = strtok(filename_buffer, PARSE_TOKENS);
	while (next != NULL && continue_deletions) {
		continue_deletions = delete_file(next);
		next = strtok(NULL, PARSE_TOKENS);
	}
	unlink(temp_file);
	free(filename_buffer);
}	/* expand_delete_filename */

	
/*
delete_file() is used to delete a file. An alert
box is used to make sure of the deletion.
*/

int delete_file(file_spec)
char *file_spec;
{
	char msg[MAX_FILE_SPEC + 25];
	int result;

	if (file_size(file_spec) == -1) {
		sprintf(msg, "No matching file(s):  %s", file_spec);
		alert(msg);
		return TRUE;
	}
	sprintf(msg, "Delete file:  %s", file_spec);
	alert_query(msg);
	if ((result = get_alert_result()) == ALERT_YES)
		basket_add_file(file_spec);
	return (result != ALERT_CANCEL) ? TRUE : FALSE;
}	/* delete_file */

