/****  Alert.c  ****  simple alert box implementation  ****/


#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#ifdef X11R3
#include <X11/Command.h>
#include <X11/Form.h>
#include <X11/Label.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#endif

#include "AlertP.h"


static XtResource resources[] = {
	{XiNmessage, XiCMessage, XtRString, sizeof(char *),
		XtOffset(XiAlertWidget, alert.msg), XtRString, ""},
	{XiNbeepDuration, XiCBeepDuration, XtRInt, sizeof(int),
		XtOffset(XiAlertWidget, alert.std_beep_duration),
		XtRImmediate, (caddr_t) ALERT_STD_BEEP_DURATION},
	{XiNfont, XiCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffset(XiAlertWidget, alert.font),
		XtRString, "XtDefaultFont"},
};


/*
Class Methods:
*/

static void Initialize();
static void ConstraintInitialize();
static Boolean SetValues();
static void Destroy();


/*
Private support functions:
*/

static void internal_callback_proc();
static int get_query_result();
static void alert_beep();
static void alert_reset_proper_keyboard_focus();


/*
This is a pointer to a function to execute if the keyboard focus
needs resetting after pop-up operations.  See the functions
XiAlertSetResetProperKeyboardFocus() and
alert_reset_proper_keyboard_focus() in this module.
*/

static void (*reset_keyboard_focus_proc)();


/*
Define storage for the class here:
*/

XiAlertClassRec XialertClassRec = {
	{ /* core_class variables */
		(WidgetClass) &formClassRec,	/* ancestor */
		"Alert",						/* class name */
		sizeof(XiAlertRec),				/* widget size */
#ifdef X11R3
		NULL,							/* class initialize */
#else
		XawInitializeWidgetSet,			/* class initialize */
#endif
		NULL,							/* class part init. */
		FALSE,							/* class inited */
		Initialize,						/* initialize */
		NULL,							/* initialize hook */
		XtInheritRealize,				/* realize */
		NULL,							/* actions */
		0,								/* number of actions */
		resources,						/* resources */
		XtNumber(resources),			/* number of resources */
		NULLQUARK,						/* xrm class */
		TRUE,							/* compress motions */
		TRUE,							/* compress exposures */
		TRUE,							/* compress enter/leave */
		FALSE,							/* visibility interest */
		Destroy,						/* destroy */
		XtInheritResize,				/* resize */
		XtInheritExpose,				/* expose */
		SetValues,						/* set values */
		NULL,							/* set values hook */
		XtInheritSetValuesAlmost,		/* set values almost */
		NULL,							/* get values hook */
		NULL,							/* accept focus */
		XtVersion,						/* version */
		NULL,							/* callback private */
		NULL,							/* translation table */
		XtInheritQueryGeometry,			/* query geometry */
		XtInheritDisplayAccelerator,	/* display accelerator */
		NULL,							/* extension */
	},
	{ /* composite_class variables */
		XtInheritGeometryManager,		/* geometry manager */
		XtInheritChangeManaged,			/* change managed */
		XtInheritInsertChild,			/* insert child */
		XtInheritDeleteChild,			/* delete child */
		NULL,							/* extension */
	},
	{ /* constraint_class fields */
		NULL,							/* subresources */
		0,								/* number of subresources */
		sizeof(XiAlertConstraintsRec),	/* record size */
		ConstraintInitialize,			/* initialize */
		NULL,							/* destroy */
		NULL,							/* set values */
		NULL,							/* extension */
	},
	{ /* form_class fields */
#ifdef X11R3
		0,
#else
		XtInheritLayout,
#endif
	},
	{ /* alert_class variables */
		0,
	},
}; /* XialertClassRec */


WidgetClass xiAlertWidgetClass = (WidgetClass) &XialertClassRec;


/*
XiAlertWidget methods:
*/

/*
Initialize() creates the label widget for the alert box
message; it has a zero-width border.  Other variables are
initialized as well.
*/
/*ARGSUSED*/
static void Initialize(request, new)
XiAlertWidget request, new;
{
	Arg args[5];
	int i;

	if (new->core.width == 0)
		new->core.width = ALERT_DEFAULT_WIDTH;
	if (new->core.height == 0)
		new->core.height = ALERT_DEFAULT_HEIGHT;
	if (new->alert.std_beep_duration < 0 ||
			new->alert.std_beep_duration > ALERT_MAX_BEEP_DURATION)
		new->alert.std_beep_duration = ALERT_STD_BEEP_DURATION;
	i = 0;
	XtSetArg(args[i], XtNlabel,
		(XtArgVal) ALERT_DEFAULT_MSG); i++;
	XtSetArg(args[i], XtNborderWidth, (XtArgVal) 0); i++;
	XtSetArg(args[i], XtNresizable, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XtNfont, (XtArgVal) new->alert.font); i++;
	new->alert.msgW = XtCreateManagedWidget("message",
		labelWidgetClass, new, args, i);
	new->alert.num_buttons = 0;
}	/* Initialize */


/*
ConstraintInitialize() sets up the form widget that
houses the alert message and buttons.
*/
/*ARGSUSED*/
static void ConstraintInitialize(request, new)
Widget request, new;
{
	XiAlertWidget aw = (XiAlertWidget) new->core.parent;
	Widget *child, *children = aw->composite.children;
	XiAlertConstraints constraint =
		(XiAlertConstraints) new->core.constraints;

	if (!XtIsSubclass(new, commandWidgetClass))
		return;
	constraint->form.left = constraint->form.right = XtChainLeft;
	constraint->form.vert_base = aw->alert.msgW;
	if (aw->composite.num_children > 1) {
		for (child = children + aw->composite.num_children - 1;
				child > children; child--) {
			if (XtIsManaged(*child) &&
					XtIsSubclass(*child, commandWidgetClass)) {
				constraint->form.horiz_base = *child;
				break;
			}
		}
	}
}	/* ConstraintInitialize */


/*
SetValues() updates the stored message or the beep duration.
*/
/*ARGSUSED*/
static Boolean SetValues(current, request, new)
Widget current, request, new;
{
	XiAlertWidget aw = (XiAlertWidget) new;
	XiAlertWidget old_aw = (XiAlertWidget) current;
	Arg args[2];
	int i;

	if (aw->alert.msg != NULL) {
		i = 0;
		XtSetArg(args[i], XtNlabel, (XtArgVal) aw->alert.msg); i++;
		XtSetValues(aw->alert.msgW, args, i);
	}
	if (aw->alert.std_beep_duration < 0 ||
			aw->alert.std_beep_duration > ALERT_MAX_BEEP_DURATION)
		aw->alert.std_beep_duration = ALERT_STD_BEEP_DURATION;
	return FALSE;
}	/* SetValues */


/*
Destroy() frees encapsulated widgets.
*/

static void Destroy(aw)
XiAlertWidget aw;
{
	int i;

	XtDestroyWidget(aw->alert.msgW);
	for (i = 0; i < aw->alert.num_buttons; i++)
		XtDestroyWidget(aw->alert.buttonW[i].wid);
}	/* Destroy */


/*
Public functions:
*/

/*
XiAlertAddButton() adds buttons to an alert box, one at a time.
A value is associated with each button; this value is returned
by XiAlertPopup(), the function that invokes the alert box.  A
callback function may be registered with the button as well.
A hidden/internal callback function is used to pop down the
alert box after a button selection; see internal_callback_proc().
*/

void XiAlertAddButton(aw, button_name, button_value,
	callback_proc, client_data)
XiAlertWidget aw;
char *button_name;
int button_value;
void (*callback_proc)();	/* or, XtCallbackProc callback_proc; */
caddr_t client_data;
{
	Arg args[7];
	int i;

	if (aw->alert.num_buttons >= ALERT_MAX_BUTTONS) {
		fprintf(stderr,
			"alert box: too many buttons for the alert box\n");
		fprintf(stderr,
			"alert box: the maximum number of buttons is: %d.\n",
			ALERT_MAX_BUTTONS);
		return;
	}
	i = 0;
	XtSetArg(args[i], XtNlabel, (XtArgVal) button_name); i++;
	XtSetArg(args[i], XtNfont, (XtArgVal) aw->alert.font); i++;
	aw->alert.buttonW[aw->alert.num_buttons].wid =
		XtCreateManagedWidget(button_name, commandWidgetClass,
			aw, args, i);
	aw->alert.buttonW[aw->alert.num_buttons].value = button_value;
	XtAddCallback(aw->alert.buttonW[aw->alert.num_buttons].wid,
		XtNcallback, internal_callback_proc, button_value);
	if (callback_proc != NULL)
		XtAddCallback(aw->alert.buttonW[aw->alert.num_buttons].wid,
			XtNcallback, callback_proc, client_data);
	aw->alert.num_buttons++;
}	/* XiAlertAddButton */


/*
XiAlertPopup() pops up a message window.  It is the working
interface to the alert box that's used by the application.
*/

int XiAlertPopup(aw, msg, ring_bell)
XiAlertWidget aw;
char *msg;
Boolean ring_bell;
{
	Arg args[3];
	int i;

	query_set_pos(aw);
	if (ring_bell)
		XiAlertBeep(aw, aw->alert.std_beep_duration);
	i = 0;
	XtSetArg(args[i], XiNmessage, (XtArgVal) msg); i++;
	XtSetValues(aw, args, i);
	XtPopup(aw->core.parent, XtGrabExclusive);
	return get_query_result(aw);
}	/* XiAlertPopup */


/*
XiAlertMap() is like XiAlertPopup(), except it uses the message
previously set with XtSetValues().  You can turn off the (automatic)
beep by setting the beep-duration resource to 0.  
*/

int XiAlertMap(aw)
XiAlertWidget aw;
{
	query_set_pos(aw);
	XiAlertBeep(aw, aw->alert.std_beep_duration);
	XtPopup(aw->core.parent, XtGrabExclusive);
	return get_query_result(aw);
}	/* XiAlertMap */


/*
XiAlertBeep() is provided for consistency to allow independent
beeps by the application programmer.
*/

void XiAlertBeep(w, num_beeps)
Widget w;
int num_beeps;
{
	alert_beep(w, num_beeps);
}	/* XiAlertBeep */


/*
XiAlertResetProperKeyboardFocus() establishes a pointer to
the client's procedure that handles keyboard focus operations.
See the procedure alert_reset_proper_keyboard_focus().
*/

void XiAlertResetProperKeyboardFocus(rkfp)
void (*rkfp)();
{
	reset_keyboard_focus_proc = rkfp;
}	/* XiAlertResetProperKeyboardFocus */


/*
Support functions:
*/

/*
internal_callback_proc() is installed as the first callback
function for each button; see XiAlertAddButton().  Primarily,
it handles removal of the alert box.
*/
/*ARGSUSED*/
static void internal_callback_proc(w, button_value, call_data)
Widget w;	/* the button */
int button_value;
caddr_t call_data;
{
	XiAlertWidget aw = (XiAlertWidget) w->core.parent;

	aw->alert.query_result = button_value;
	aw->alert.query_wait = FALSE;
	XtPopdown(aw->core.parent);
	alert_reset_proper_keyboard_focus();
}	/* internal_callback_proc */


/*
get_query_result() blocks until a button is pressed.
*/

static int get_query_result(aw)
XiAlertWidget aw;
{
	XEvent event;

	aw->alert.query_wait = TRUE;
	while (aw->alert.query_wait) {
		XtNextEvent(&event);
		XtDispatchEvent(&event);
	}
	return aw->alert.query_result;
}	/* get_alert_result */


/*
alert_beep() rings the system bell n times.
*/

static void alert_beep(aw, num_beeps)
XiAlertWidget aw;
int num_beeps;
{
	while (num_beeps-- > 0)
		XBell(XtDisplay(aw->core.parent), 50);
}	/* alert_beep */


/*
alert_reset_proper_keyboard_focus() allows the keyboard focus to be
reestablished after after a transient shell is popped up and down.
Calling this routine is necessary with (old) twm, but not with uwm.
*/

static void alert_reset_proper_keyboard_focus()
{
	if (reset_keyboard_focus_proc == NULL)
		return;
	reset_keyboard_focus_proc();
}	/* alert_reset_proper_keyboard_focus */

