/**  Button.c  ** a simple button implementation  **/


#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#ifdef X11R3
#include <X11/Simple.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Simple.h>
#endif

#include "ButtonP.h"


/*
Class Methods:
*/

static void Initialize();
static void Destroy();
static Boolean SetValues();
static void Redisplay();


/*
Private support functions:
*/

static int create_button_pixmap();
static int allocate_and_center_button_text();
static void fill_pixmap_and_redisplay();
static void fill_pixmap();
static void redisplay_pixmap();
static void select_button();
static void deselect_button();


/*
Translations and actions:
*/

static void SelectButton();

static char button_translations[] =
	"<Btn1Down>:	select()";
	
static XtActionsRec button_actions[] = {
	{"select", (XtActionProc) SelectButton},
};


/*
Resource table:
*/

#define res_offset(field) XtOffset(XiButtonWidget, field)

static XtResource resources[] = {
	{XiNforeground, XiCForeground, XtRPixel, sizeof(Pixel),
		res_offset(button.foreground),
		XtRString, XtDefaultForeground},
	{XiNlabel, XiCLabel, XtRString, sizeof(char *),
		res_offset(button.label),
		XtRString, BTN_NOTHING},
	{XiNlength, XiCLength, XtRInt, sizeof(int),
		res_offset(button.length),
		XtRImmediate, (caddr_t) BTN_DEFAULT_LENGTH},
	{XiNvalue, XiCValue, XtRInt, sizeof(int),
		res_offset(button.value),
		XtRImmediate, (caddr_t) BTN_DEFAULT_VALUE},
	{XiNfont, XiCFont, XtRFontStruct, sizeof(XFontStruct *),
		res_offset(button.font),
		XtRString, "XtDefaultFont"},
	{XiNselectCallback, XiCSelectCallback, XtRCallback,
		sizeof(caddr_t), res_offset(button.cb_list),
		XtRCallback, (caddr_t) NULL},
};


/*
Define storage for the class here:
*/

XiButtonClassRec XibuttonClassRec = {
	{ /* core_class variables */
		(WidgetClass) &simpleClassRec,	/* ancestor */
		"Button",						/* class name */
		sizeof(XiButtonRec),			/* widget size */
#ifdef X11R3
		NULL,							/* class initialize */
#else
		XawInitializeWidgetSet,			/* class initialize */
#endif
		NULL,							/* class part init. */
		FALSE,							/* class inited */
		Initialize,						/* initialize */
		NULL,							/* initialize hook */
		XtInheritRealize,				/* realize */
		button_actions,					/* actions */
		XtNumber(button_actions),		/* number of actions */
		resources,						/* resources */
		XtNumber(resources),			/* number of resources */
		NULLQUARK,						/* xrm class */
		TRUE,							/* compress motions */
		TRUE,							/* compress exposures */
		TRUE,							/* compress enter/leave */
		FALSE,							/* visibility interest */
		Destroy,						/* destroy */
		NULL,							/* resize */
		Redisplay,						/* expose */
		SetValues,						/* set values */
		NULL,							/* set values hook */
		XtInheritSetValuesAlmost,		/* set values almost */
		NULL,							/* get values hook */
		NULL,							/* accept focus */
		XtVersion,						/* version */
		NULL,							/* callback private */
		button_translations,			/* translation table */
		XtInheritQueryGeometry,			/* query geometry */
		XtInheritDisplayAccelerator,	/* display accelerator */
		NULL,							/* extension */
	},
	{ /* simple_class variables */
		XtInheritChangeSensitive,
	},
	{ /* button_class variables */
		0,
	},
}; /* XibuttonClassRec */


WidgetClass xiButtonWidgetClass = (WidgetClass) &XibuttonClassRec;


/*
XiButtonWidget methods:
*/

/*
Initialize() initializes each instance of a button.
A button implements a pixmap in a Simple widget.
*/
/*ARGSUSED*/
static void Initialize(request, new)
XiButtonWidget request, new;
{
	if (!new->button.label)
		new->button.label = BTN_NOTHING;
	create_button_pixmap(new);
	new->button.selected = FALSE;
	fill_pixmap(new);
}	/* Initialize */


/*
Destroy() cleans up dynamic data structures.
*/

static void Destroy(w)
XiButtonWidget w;
{
	if (w->button.private_label != NULL)
		XtFree(w->button.private_label);
	if (w->button.gc != NULL)
		XFreeGC(XtDisplay(w), w->button.gc);
	if (w->button.cgc != NULL)
		XFreeGC(XtDisplay(w), w->button.cgc);
	if (w->button.pixmap != NULL)
		XFreePixmap(XtDisplay(w), w->button.pixmap);
}	/* Destroy */


/*
SetValues() handles dynamic modifications to resource-related
instance variables, where necessary.  Most resources are used
only during widget creation.  The `value' field can be set by
a resource either during creation or dynamically; however, its
value doesn't need to be checked here.  If a button's `label'
field is changed dynamically via a resource setting, it's
necessary to free and then reallocate space for the label.
This implementation does NOT dynamically resize the pixmap, if
the label text changes.
*/
/*ARGSUSED*/
static Boolean SetValues(current, request, new)
Widget current, request, new;
{
	XiButtonWidget bw = (XiButtonWidget) new;
	XiButtonWidget old_bw = (XiButtonWidget) current;

	if (!bw->button.label)
		bw->button.label = BTN_NOTHING;
	if (strcmp(bw->button.label, old_bw->button.label)) {
		if (old_bw->button.private_label != NULL)
			XtFree(old_bw->button.private_label);
		/*
		does NOT resize pixmap:
		*/
		allocate_and_center_button_text(bw, strlen(bw->button.label));
		fill_pixmap(bw);
	}
	return TRUE;
}	/* SetValues */


/*
Redisplay() calls redisplay_pixmap() to update the pixmap
window upon an exposure.
*/
/*ARGSUSED*/
static void Redisplay(w, event)
Widget w;
XEvent *event;
{
	redisplay_pixmap(w);
}	/* Redisplay */


/*
Actions functions:
*/
/*ARGSUSED*/
static void SelectButton(w, event)
XiButtonWidget w;
XEvent *event;
{
	if (w->button.selected)
		deselect_button(w);
	else
		select_button(w, BTN_WITH_CB);
}	/* SelectButton */


/*
Public functions:
*/

/*
XiButtonGetValue() returns the button's value.
This is an optional field.
*/

int XiButtonGetValue(w)
XiButtonWidget w;
{
	return w->button.value;
}	/* XiButtonGetValue */


/*
XiButtonSetValue() sets the button's value.
This is an optional field.
*/

void XiButtonSetValue(w, value)
XiButtonWidget w;
int value;
{
	w->button.value = value;
}	/* XiButtonSetValue */


/*
XiButtonIsSelected() determines whether or not
the button is currently selected.
*/

int XiButtonIsSelected(w)
XiButtonWidget w;
{
	return (int) w->button.selected;
}	/* XiButtonIsSelected */


/*
XiButtonSelect() programmatically selects a button.
*/

void XiButtonSelect(w)
XiButtonWidget w;
{
	if (!w->button.selected)
		select_button(w, BTN_WITH_CB);
}	/* XiButtonSelect */


/*
XiButtonSelectNoCB() programmatically selects a button,
but doesn't attempt to execute any associated callbacks.
*/

void XiButtonSelectNoCB(w)
XiButtonWidget w;
{
	if (!w->button.selected)
		select_button(w, BTN_WITHOUT_CB);
}	/* XiButtonSelect */


/*
XiButtonDeselect() programmatically deselects a button.
*/

void XiButtonDeselect(w)
XiButtonWidget w;
{
	if (w->button.selected)
		deselect_button(w);
}	/* XiButtonDeselect */


/*
Support functions:
*/

/*
create_button_pixmap() creates the GCs, allocates the button's
text, and builds the button's pixmap.
*/

static int create_button_pixmap(w)
XiButtonWidget w;
{
	XGCValues values;
	Display *display = XtDisplay(w);
	int num_btn_chars, screen = XDefaultScreen(display);
	int depth = XDisplayPlanes(display, screen);

	values.foreground = w->button.foreground;
	values.background = w->core.background_pixel;
	values.font = w->button.font->fid;
	values.fill_style = FillSolid;
	w->button.gc = XCreateGC(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		GCForeground | GCBackground | GCFont | GCFillStyle, &values);
	values.background = w->button.foreground;
	values.foreground = w->core.background_pixel;
	w->button.cgc = XCreateGC(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		GCForeground | GCBackground | GCFont | GCFillStyle, &values);

	num_btn_chars = (w->button.length == BTN_DEFAULT_LENGTH) ?
		strlen(w->button.label) : w->button.length;
	if (!allocate_and_center_button_text(w, num_btn_chars))
		fprintf(stderr,
		"button: unable to allocate space for the button's text!\n");

	if (w->core.width == 0) {
		if (*w->button.private_label)
			w->core.width = strlen(w->button.private_label) *
				w->button.font->max_bounds.width;
		else
			w->core.width = BTN_DEFAULT_WIDTH;
	}
	if (w->core.height == 0) {
		if (*w->button.private_label)
			w->core.height = w->button.font->max_bounds.ascent +
				w->button.font->max_bounds.descent;
		else
			w->core.height = BTN_DEFAULT_HEIGHT;
	}
	w->core.width = w->button.width =
		w->core.width + (2 * BTN_INTERNAL_HORIZ_PAD);
	w->core.height = w->button.height =
		w->core.height + (2 * BTN_INTERNAL_VERT_PAD);
	w->button.pixmap = XCreatePixmap(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		w->button.width, w->button.height, depth);
}	/* create_button_pixmap */


/*
allocate_and_center_button_text() creates a private (safe)
copy of the client's button text string.
*/

static int allocate_and_center_button_text(w, total_len)
XiButtonWidget w;
int total_len;
{
	int j, odd_num, pad_len;

	j = strlen(w->button.label);		/* center each item */
	if ((pad_len = (total_len - j) / 2) < 0)
		return FALSE;
	odd_num = ((((total_len - j) / 2) * 2) != (total_len - j)) ?
		TRUE : FALSE;
	w->button.private_label =
		(char *) XtMalloc((unsigned) (j + (2 * pad_len) + 2));
	if (w->button.private_label == NULL)
		return FALSE;
	for ( ; j > -1; j--)
		w->button.private_label[j + pad_len] = w->button.label[j];
	for (j = 0; j < pad_len; j++)
		w->button.private_label[j] = ' ';
	while (pad_len--)
		strcat(w->button.private_label, " ");
	if (odd_num)
		strcat(w->button.private_label, " ");
	return TRUE;
}	/* allocate_and_center_button_text */


/*
fill_pixmap_and_redisplay() builds the pixmap and refreshes
the window.
*/

static void fill_pixmap_and_redisplay(w)
XiButtonWidget w;
{
	fill_pixmap(w);
	redisplay_pixmap(w);
}	/* fill_pixmap_and_redisplay */


/*
fill_pixmap() tests for errors and then displays the label
on the pixmap.
*/

static void fill_pixmap(w)
XiButtonWidget w;
{
	if (!w->button.label || !*w->button.label)
		return;
	deselect_button(w);
	/*
	clear pixmap before it filling with a new label:
	*/
	XFillRectangle(XtDisplay(w), w->button.pixmap,
		w->button.cgc, 0, 0, w->button.width, w->button.height);
	XDrawImageString(XtDisplay(w), w->button.pixmap,
		w->button.gc, BTN_INTERNAL_HORIZ_PAD,
		w->button.font->max_bounds.ascent + BTN_INTERNAL_VERT_PAD,
		w->button.private_label, strlen(w->button.private_label));
}	/* fill_pixmap */


/*
redisplay_pixmap() copies the pixmap to the window.
*/

static void redisplay_pixmap(w)
XiButtonWidget w;
{
	if (!XtIsRealized(w))
		return;
	XCopyArea(XtDisplay(w), w->button.pixmap,
		XtWindow(w),
		w->button.gc, 0, 0, w->button.width,
		w->button.height, 0, 0);
}	/* redisplay_pixmap */


/*
select_button() draws a one pixel wide rectangle
on the inside of the pixmap, and then, if requested,
calls functions on the client-specified callback list.
*/

static void select_button(w, call_CB)
XiButtonWidget w;
int call_CB;
{
	w->button.selected = TRUE;
	XDrawRectangle(XtDisplay(w), w->button.pixmap, w->button.gc,
		1, 1, w->button.width - 3, w->button.height - 3);
	redisplay_pixmap(w);
	if (call_CB == BTN_WITH_CB)
		XtCallCallbacks(w, XiNselectCallback,
			(caddr_t) &w->button.value);
}	/* select_button */


/*
deselect_button() erases the one pixel wide rectangle
on the inside of the pixmap.
*/

static void deselect_button(w)
XiButtonWidget w;
{
	w->button.selected = FALSE;
	XDrawRectangle(XtDisplay(w), w->button.pixmap, w->button.cgc,
		1, 1, w->button.width - 3, w->button.height - 3);
	redisplay_pixmap(w);
}	/* deselect_button */

