/**  Choice.c  ** a simple choice box implementation  **/


#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#ifdef X11R3
#include <X11/Form.h>
#include <X11/Label.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#endif

#include <Ximisc.h>
#include "ChoiceP.h"


/*
Class Methods:
*/

static void Initialize();
static void ConstraintInitialize();
static void Destroy();


/*
Private support functions:
*/

static void internal_callback_proc();
static void remove_menu();


/*
Cursor place-holder:
*/

static Cursor default_item_cursor = None;


/*
Resource table:
*/

#define res_offset(field) XtOffset(XiChoiceWidget, field)

static XtResource resources[] = {
	{XiNlabel, XiCLabel, XtRString,
		sizeof(char *), res_offset(choice.label),
		XtRString, CHOICE_DEFAULT_LABEL},
	{XiNexclusive, XiCExclusive, XtRBoolean, sizeof(Boolean),
		res_offset(choice.exclusive),
		XtRImmediate, (caddr_t) FALSE},
	{XiNitemBorderWidth, XiCItemBorderWidth, XtRInt,
		sizeof(int), res_offset(choice.btn_border_width),
		XtRImmediate, (caddr_t) CHOICE_DEFAULT_BTN_BORDER_WIDTH},
	{XiNdefaultItem, XiCDefaultItem, XtRInt, sizeof(int),
		res_offset(choice.default_item),
		XtRImmediate, (caddr_t) CHOICE_DEFAULT_ITEM},
	{XiNlabelOnLeft, XiCLabelOnLeft, XtRBoolean, sizeof(Boolean),
		res_offset(choice.label_on_left),
		XtRImmediate, (caddr_t) FALSE},
	{XiNvertical, XiCVertical, XtRBoolean, sizeof(Boolean),
		res_offset(choice.vertical),
		XtRImmediate, (caddr_t) FALSE},
	{XiNmenuForm, XiCMenuForm, XtRBoolean, sizeof(Boolean),
		res_offset(choice.menu_form),
		XtRImmediate, (caddr_t) FALSE},
	{XiNitemCursor, XiCItemCursor, XtRCursor, sizeof(Cursor),
		res_offset(choice.item_cursor),
		XtRCursor, (caddr_t) &default_item_cursor},
	{XiNfont, XiCFont, XtRFontStruct, sizeof(XFontStruct *),
		res_offset(choice.font),
		XtRString, "XtDefaultFont"},
};


/*
Define storage for the class here:
*/

XiChoiceClassRec XichoiceClassRec = {
	{ /* core_class variables */
		(WidgetClass) &formClassRec,	/* ancestor */
		"Choice",						/* class name */
		sizeof(XiChoiceRec),			/* widget size */
#ifdef X11R3
		NULL,							/* class initialize */
#else
		XawInitializeWidgetSet,			/* class initialize */
#endif
		NULL,							/* class part init. */
		FALSE,							/* class inited */
		Initialize,						/* initialize */
		NULL,							/* initialize hook */
		XtInheritRealize,				/* realize */
		NULL,							/* actions */
		0,								/* number of actions */
		resources,						/* resources */
		XtNumber(resources),			/* number of resources */
		NULLQUARK,						/* xrm class */
		TRUE,							/* compress motions */
		TRUE,							/* compress exposures */
		TRUE,							/* compress enter/leave */
		FALSE,							/* visibility interest */
		Destroy,						/* destroy */
		XtInheritResize,				/* resize */
		XtInheritExpose,				/* expose */
		NULL,							/* set values */
		NULL,							/* set values hook */
		XtInheritSetValuesAlmost,		/* set values almost */
		NULL,							/* get values hook */
		NULL,							/* accept focus */
		XtVersion,						/* version */
		NULL,							/* callback private */
		NULL,							/* translation table */
		XtInheritQueryGeometry,			/* query geometry */
		XtInheritDisplayAccelerator,	/* display accelerator */
		NULL,							/* extension */
	},
	{ /* composite_class variables */
		XtInheritGeometryManager,		/* geometry manager */
		XtInheritChangeManaged,			/* change managed */
		XtInheritInsertChild,			/* insert child */
		XtInheritDeleteChild,			/* delete child */
		NULL,							/* extension */
	},
	{ /* constraint_class fields */
		NULL,							/* subresources */
		0,								/* number of subresources */
		sizeof(XiChoiceConstraintsRec),	/* record size */
		ConstraintInitialize,			/* initialize */
		NULL,							/* destroy */
		NULL,							/* set values */
		NULL,							/* extension */
	},
	{ /* form_class fields */
#ifdef X11R3
		0,
#else
		XtInheritLayout,
#endif
	},
	{ /* choice_class variables */
		0,
	},
}; /* XichoiceClassRec */


WidgetClass xiChoiceWidgetClass = (WidgetClass) &XichoiceClassRec;


/*
XiChoiceWidget methods:
*/

/*
Initialize() initializes each instance of a choice box.
*/
/*ARGSUSED*/
static void Initialize(request, new)
XiChoiceWidget request, new;
{
	Arg args[5];
	int i;

	if (new->core.width == 0)
		new->core.width = CHOICE_DEFAULT_WIDTH;
	if (new->core.height == 0)
		new->core.height = CHOICE_DEFAULT_HEIGHT;

	if (*new->choice.label) {
		i = 0;
		XtSetArg(args[i], XtNlabel,
			(XtArgVal) new->choice.label); i++;
		XtSetArg(args[i], XtNborderWidth, (XtArgVal) 0); i++;
		XtSetArg(args[i], XtNresizable, (XtArgVal) TRUE); i++;
		XtSetArg(args[i], XtNfont, (XtArgVal) new->choice.font); i++;
		new->choice.labelW = XtCreateManagedWidget("label",
			labelWidgetClass, new, args, i);
	}
	else
		new->choice.labelW = NULL;
	new->choice.num_items = 0;
	new->choice.items_selected[0] = 0;

	if (new->choice.menu_form) {			/* force these values */
		new->choice.vertical = TRUE;
		new->choice.exclusive = TRUE;
		new->choice.default_item = FALSE;
		new->choice.max_item_len = CHOICE_IGNORE_ITEM_LEN;
		XtAddEventHandler(new, ButtonPressMask, FALSE, remove_menu,
			NULL);
	}
	else if (new->choice.exclusive) {
		if (new->choice.default_item < 1)
			new->choice.default_item = CHOICE_DEFAULT_ITEM;
	}
}	/* Initialize */


/*
ConstraintInitialize() organizes the form widget that
houses the choice label and items.  Note that the label
is optional; its presence or absence must be considered
when traversing the list of children.
*/
/*ARGSUSED*/
static void ConstraintInitialize(request, new)
Widget request, new;
{
	XiChoiceWidget cw = (XiChoiceWidget) new->core.parent;
	Widget *child, *children = cw->composite.children;
	XiChoiceConstraints constraint =
		(XiChoiceConstraints) new->core.constraints;
	int label;

	if (cw->choice.label_on_left &&
			XtIsSubclass(new, labelWidgetClass) &&
			cw->choice.btn_border_width !=
				CHOICE_DEFAULT_BTN_BORDER_WIDTH)
		constraint->form.dy +=
			constraint->form.dy + cw->choice.btn_border_width - 1;
	if (!XtIsSubclass(new, xiButtonWidgetClass))
		return;
	if (cw->choice.vertical)
		constraint->form.top = constraint->form.bottom = XtChainTop;
	else
		constraint->form.left = constraint->form.right = XtChainLeft;
	if (!cw->choice.label_on_left &&
			(!cw->choice.vertical || cw->choice.menu_form))
		constraint->form.vert_base = cw->choice.labelW;
	if (cw->choice.label_on_left && cw->choice.vertical)
		constraint->form.horiz_base = cw->choice.labelW;
	if (cw->composite.num_children == 0)
		return;
	label = (cw->choice.labelW != NULL && !cw->choice.label_on_left) ?
		1 : 0;
	if (cw->composite.num_children == 1 && label == 1)
		return;
	for (child = children + cw->composite.num_children - 1;
			child >= (children + label); child--) {
		if (XtIsManaged(*child)) {
			if (cw->choice.menu_form)
				constraint->form.vert_base = *child;
			else if (cw->choice.vertical &&
					XtIsSubclass(*child, xiButtonWidgetClass))
				constraint->form.vert_base = *child;
			else
				constraint->form.horiz_base = *child;
			break;
		}
	}
}	/* ConstraintInitialize */


/*
Destroy() cleans up data structures, if any, and removes the
event handler.
*/

static void Destroy(w)
XiChoiceWidget w;
{
	int i;

	XtDestroyWidget(w->choice.labelW);
	for (i = 0; i < w->choice.num_items; i++)
		XtDestroyWidget(w->choice.itemW[i].wid);
	XtRemoveEventHandler(w, ButtonPressMask, FALSE, remove_menu,
		NULL);
}	/* Destroy */


/*
Public functions:
*/

/*
XiChoiceMenuPopup() uses query_set_pos() to pop up a
choice widget under the cursor.
*/

void XiChoiceMenuPopup(w)
XiChoiceWidget w;
{
	if (!w->choice.menu_form)
		return;
	query_set_pos(w);
	XtPopup(w->core.parent, XtGrabNone);
}	/* XiChoiceMenuPopup */


/*
XiChoiceIsExclusive() tests whether or not the choice box
is operating in exclusive mode.
*/

int XiChoiceIsExclusive(w)
XiChoiceWidget w;
{
	return (int) w->choice.exclusive;
}	/* XiChoiceIsExclusive */


/*
XiChoiceAddItems() is a convenience function for adding multiple items
to a choice box after the box has been created; see XiChoiceAddItem()
and Choice.h.  Note that the second argument varies in structure,
depending on the type of choice box.  For a menu, its type is
XiChoiceMenuItem, and for a standard choice box, its type is
XiChoiceItem.  Example #1 (standard choice box):

	static XiChoiceItem item_list[] = {
		{"Bin", 2, ChoiceCB, "Binary Data"},
		{"Dec", 10, ChoiceCB, "Decimal Data"},
		{NULL, NULL, NULL, NULL},
	};
	...
	i = 0;
	...
	example = XtCreateManagedWidget("example",
		xiChoiceWidgetClass, exampleParent, args, i);
	XiChoiceAddItems(example, item_list);
	...

Example #2 (menu):

static XiChoiceMenuItem item_list[] = {
	{"Menu Label 1", MenuCB, "Menu Data 1"},
	{"Menu Label 2", MenuCB, "Menu Data 2"},
	{NULL, NULL, NULL},
};
	...
	i = 0;
	...
	example = XtCreateManagedWidget("example",
		xiChoiceWidgetClass, exampleParent, args, i);
	XiChoiceAddItems(example, item_list);
	...

*/

void XiChoiceAddItems(cw, item_addr)
XiChoiceWidget cw;
caddr_t item_addr;
{
	int i, len, more_items = FALSE;

	if (item_addr == NULL) {
		fprintf(stderr, "choice box: item list is null!\n");
		return;
	}
	if (cw->choice.menu_form) {
		XiChoiceMenuItem *item = (XiChoiceMenuItem *) item_addr;
		cw->choice.max_item_len = 0;
		for (i = 0; item[i].label && i < CHOICE_MAX_ITEMS; i++)
			if ((len = strlen(item[i].label)) > cw->choice.max_item_len)
				cw->choice.max_item_len = len;
		for (i = 0; item[i].label && i < CHOICE_MAX_ITEMS; i++)
			XiChoiceAddItem(cw, item[i].label, FALSE,
				item[i].callback_proc, item[i].client_data);
		if (item[i].label != NULL)
			more_items = TRUE;
	}
	else {
		XiChoiceItem *item = (XiChoiceItem *) item_addr;
		for (i = 0; item[i].label && i < CHOICE_MAX_ITEMS; i++)
			XiChoiceAddItem(cw, item[i].label, item[i].value,
				item[i].callback_proc, item[i].client_data);
		if (item[i].label != NULL)
			more_items = TRUE;
	}
	if (i == CHOICE_MAX_ITEMS && more_items) {
		fprintf(stderr,
			"choice box: too many items for the choice box!\n");
		fprintf(stderr,
			"choice box: the maximum number of items is: %d.\n",
			CHOICE_MAX_ITEMS);
	}
}	/* XiChoiceAddItems */


/*
XiChoiceAddItem() is used to add items to a choice box after
the box has been created.  Buttons/items are added in left-
to-right order.  This form of item entry probably shouldn't be
publically accessible; use XiChoiceAddItems() below.
*/

void XiChoiceAddItem(cw, item_name, item_value,
	callback_proc, client_data)
XiChoiceWidget cw;
char *item_name;
int item_value;
void (*callback_proc)();	/* or, XtCallbackProc callback_proc; */
caddr_t client_data;
{
	Arg args[10];
	int i;

	if (cw->choice.num_items >= CHOICE_MAX_ITEMS) {
		fprintf(stderr,
			"choice box: too many items for the choice box!\n");
		fprintf(stderr,
			"choice box: the maximum number of items is: %d.\n",
			CHOICE_MAX_ITEMS);
		return;
	}
	i = 0;
	XtSetArg(args[i], XiNlabel, (XtArgVal) item_name); i++;
	/*
	item 1 ==> itemW[0]:
	*/	XtSetArg(args[i], XiNvalue,
		(XtArgVal) (cw->choice.num_items + 1)); i++;
	if (cw->choice.btn_border_width !=
			CHOICE_DEFAULT_BTN_BORDER_WIDTH) {
		XtSetArg(args[i], XiNborderWidth,
			(XtArgVal) cw->choice.btn_border_width); i++;
	}
	if (cw->choice.item_cursor != None) {
		XtSetArg(args[i], XtNcursor,
			(XtArgVal) cw->choice.item_cursor); i++;
	}
	XtSetArg(args[i], XiNfont, (XtArgVal) cw->choice.font); i++;
	if (cw->choice.menu_form &&
			cw->choice.max_item_len != CHOICE_IGNORE_ITEM_LEN) {
		XtSetArg(args[i], XiNlength,
			(XtArgVal) cw->choice.max_item_len); i++;
	}
	cw->choice.itemW[cw->choice.num_items].wid = (XiButtonWidget)
		XtCreateManagedWidget(item_name, xiButtonWidgetClass,
			cw, args, i);
	XtAddCallback(cw->choice.itemW[cw->choice.num_items].wid,
		XiNselectCallback, internal_callback_proc, NULL);
	if (callback_proc != NULL)
		XtAddCallback(cw->choice.itemW[cw->choice.num_items].wid,
			XiNselectCallback, callback_proc, client_data);
	cw->choice.itemW[cw->choice.num_items].value = item_value;
	cw->choice.num_items++;
	if (cw->choice.exclusive &&
			(cw->choice.num_items == cw->choice.default_item))
		XiButtonSelect(cw->choice.itemW[cw->choice.num_items - 1].wid);
}	/* XiChoiceAddItem */


/*
XiChoiceGetItems() returns a (pointer to a) zero-terminated list
of the currently selected items.  Item numbers are NOT zero-
based, that is: item 1 ==> itemW[0].  The list is an array of
integers.
*/

int *XiChoiceGetItems(w)
XiChoiceWidget w;
{
	int i, j;

	for (i = j = 0; i < w->choice.num_items; i++)
		if (XiButtonIsSelected(w->choice.itemW[i].wid))
			w->choice.items_selected[j++] = i + 1;
	w->choice.items_selected[j] = 0;
	return w->choice.items_selected;
}	/* XiChoiceGetItems */


/*
XiChoiceGetSelectedItem() returns the currently
selected item for an exclusive choice box.
Note:  item 1 ==> itemW[0].
*/

int XiChoiceGetSelectedItem(w)
XiChoiceWidget w;
{
	int i;

	if (!w->choice.exclusive)
		return FALSE;
	for (i = 0; i < w->choice.num_items; i++)
		if (XiButtonIsSelected(w->choice.itemW[i].wid))
			return i + 1;
	return FALSE;
}	/* XiChoiceGetSelectedItem */


/*
XiChoiceGetDefaultItem() returns the default
item number.  Note:  item 1 ==> itemW[0].
*/

int XiChoiceGetDefaultItem(w)
XiChoiceWidget w;
{
	if (!w->choice.exclusive)
		return FALSE;
	return w->choice.default_item;
}	/* XiChoiceGetDefaultItem */


/*
XiChoiceSelectedItemIsDefault() tests whether or not
the selected item is the default item.
*/

int XiChoiceSelectedItemIsDefault(w)
XiChoiceWidget w;
{
	if (!w->choice.exclusive)
		return FALSE;
	if (XiChoiceGetNumItems(w) != 1)
		return FALSE;
	return (XiChoiceGetSelectedItem(w) == w->choice.default_item);
}	/* XiChoiceSelectedItemIsDefault */


/*
XiChoiceGetNumItems() returns the number (quantity)
of currently selected items.
*/

int XiChoiceGetNumItems(w)
XiChoiceWidget w;
{
	int i, num_items_selected;

	for (i = num_items_selected = 0; i < w->choice.num_items; i++)
		if (XiButtonIsSelected(w->choice.itemW[i].wid))
			num_items_selected++;
	return num_items_selected;
}	/* XiChoiceGetNumItems */


/*
XiChoiceGetValue() retrieves the user-specified value of
the i'th item.  The returned value is the value
associated with the item using XiChoiceAddItem().
Note:  item 1 ==> itemW[0].
*/

int XiChoiceGetValue(w, item)
XiChoiceWidget w;
int item;
{
	return w->choice.itemW[item - 1].value;
}	/* XiChoiceGetValue */


/*
XiChoiceSelectItem() allows programmatic selection
of an item.
*/

void XiChoiceSelectItem(w, item)
XiChoiceWidget w;
int item;
{
	int i;

	if (item < 1 || item > w->choice.num_items)
		return;
	XiButtonSelectNoCB(w->choice.itemW[item - 1].wid);
	if (!w->choice.exclusive)
		return;
	for (i = 1; i <= w->choice.num_items; i++)
		if (i != item)
			XiButtonDeselect(w->choice.itemW[i - 1].wid);
}	/* XiChoiceSelectItem */


/*
XiChoiceDeselectItem() allows programmatic de-selection
of an item.
*/

void XiChoiceDeselectItem(w, item)
XiChoiceWidget w;
int item;
{
	if (item < 1 || item > w->choice.num_items)
		return;
	XiButtonDeselect(w->choice.itemW[item - 1].wid);
}	/* XiChoiceDeselectItem */


/*
XiChoiceItemIsSelected() allows programmatic testing of
of an item's status.
*/

int XiChoiceItemIsSelected(w, item)
XiChoiceWidget w;
int item;
{
	if (item < 1 || item > w->choice.num_items)
		return FALSE;
	return XiButtonIsSelected(w->choice.itemW[item - 1].wid);
}	/* XiChoiceItemIsSelected */


/*
Support functions:
*/

/*
internal_callback_proc() is installed as a private callback
function for each (internal) button; see XiChoiceAddItem().
The "exclusive" policy is enforced here -- if other items are
already selected, they must be de-selected.
*/
/*ARGSUSED*/
static void internal_callback_proc(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
	XiButtonWidget bw = (XiButtonWidget) w;
	XiChoiceWidget cw = (XiChoiceWidget) bw->core.parent;
	int i;

	if (!cw->choice.exclusive)
		return;
	for (i = 0; i < cw->choice.num_items; i++)
		if (cw->choice.itemW[i].wid != bw)
			XiButtonDeselect(cw->choice.itemW[i].wid);
	if (!cw->choice.menu_form)
		return;
	XtPopdown(cw->core.parent);
	XiButtonDeselect(bw);
}	/* internal_callback_proc */


/*
For choice boxes in menu form, remove_menu() pops down the menu
if the user clicks the left mouse button in anywhere inside the
widget other than on a button.
*/
/*ARGSUSED*/
static void remove_menu(w, client_data, event)
XiChoiceWidget w;
caddr_t client_data;
XEvent *event;
{
	if (event->xbutton.button == Button1)
		XtPopdown(w->core.parent);
}	/* remove_menu */


