/**  FileSelect.c  ** simple file selection box implementation  **/


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>	/* stat() is used to verify file existence */

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/cursorfont.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#ifdef X11R3
#include <X11/AsciiText.h>
#include <X11/Box.h>
#include <X11/Command.h>
#include <X11/Viewport.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#endif

#include "StrSelect.h"
#include "FileSelecP.h"


/*
Cursor place-holder:
*/

static Cursor default_pixmap_cursor = None;


/*
Resource table:
*/

#define res_offset(field) XtOffset(XiFileSelectWidget, field)

static XtResource resources[] = {
	{XiNfilter, XiCFilter, XtRString, sizeof(char *),
		res_offset(file_select.default_filter),
		XtRString, FILE_DEFAULT_FILTER},
	{XiNfilterWidth, XiCFilterWidth, XtRDimension, sizeof(Dimension),
		res_offset(file_select.filter_width),
		XtRImmediate, (caddr_t) FILE_DEFAULT_FILTER_WIDTH},
	{XiNrows, XiCRows, XtRInt, sizeof(int),
		res_offset(file_select.rows),
		XtRImmediate, (caddr_t) FILE_DEFAULT_ROWS},
	{XiNcolumns, XiCColumns, XtRInt, sizeof(int),
		res_offset(file_select.columns),
		XtRImmediate, (caddr_t) FILE_DEFAULT_COLUMNS},
	{XiNpixmapRows, XiCPixmapRows, XtRInt, sizeof(int),
		res_offset(file_select.pixmap_rows),
		XtRImmediate, (caddr_t) FILE_DEFAULT_PIXMAP_ROWS},
	{XiNpixmapColumns, XiCPixmapColumns, XtRInt, sizeof(int),
		res_offset(file_select.pixmap_columns),
		XtRImmediate, (caddr_t) FILE_DEFAULT_PIXMAP_COLUMNS},
	{XiNrowSpacing, XiCRowSpacing, XtRInt, sizeof(int),
		res_offset(file_select.row_spacing),
		XtRImmediate, (caddr_t) FILE_DEFAULT_ROW_SPACING},
	{XiNpixmapCursor, XiCPixmapCursor, XtRCursor, sizeof(Cursor),
		res_offset(file_select.pixmap_cursor),
		XtRCursor, (caddr_t) &default_pixmap_cursor},
	{XiNpixmapFont, XiCPixmapFont, XtRFontStruct,
		sizeof(XFontStruct *), res_offset(file_select.font),
		XtRString, "XtDefaultFont"},
};


/*
Class Methods:
*/

static void Initialize();
static void Destroy();
static Boolean SetValues();

/*
Action functions:
*/

static void Beep();

/*
Private support functions:
*/

static void FileSelectBoxDismiss();
static void FileSelectBoxFilter();
static void get_font_width_height();
static void set_filename_reference_buffer();
static char *expand_file_spec();
static int load_buffer();
static int file_size();
static int get_home_directory();


/*
Define storage for the class here:
*/

XiFileSelectClassRec XifileSelectClassRec = {
	{ /* core_class variables */
#ifdef X11R3
		(WidgetClass) &vPanedClassRec,	/* ancestor */
#else
		(WidgetClass) &panedClassRec,	/* ancestor */
#endif
		"FileSelect",					/* class name */
		sizeof(XiFileSelectRec),		/* widget size */
#ifdef X11R3
		NULL,							/* class initialize */
#else
		XawInitializeWidgetSet,			/* class initialize */
#endif
		NULL,							/* class part init. */
		FALSE,							/* class inited */
		Initialize,						/* initialize */
		NULL,							/* initialize hook */
		XtInheritRealize,				/* realize */
		NULL,							/* actions */
		0,								/* number of actions */
		resources,						/* resources */
		XtNumber(resources),			/* number of resources */
		NULLQUARK,						/* xrm class */
		TRUE,							/* compress motions */
		TRUE,							/* compress exposures */
		TRUE,							/* compress enter/leave */
		FALSE,							/* visibility interest */
		Destroy,						/* destroy */
		XtInheritResize,				/* resize */
		XtInheritExpose,				/* expose */
		SetValues,						/* set values */
		NULL,							/* set values hook */
		XtInheritSetValuesAlmost,		/* set values almost */
		NULL,							/* get values hook */
		NULL,							/* accept focus */
		XtVersion,						/* version */
		NULL,							/* callback private */
		NULL,							/* translation table */
		XtInheritQueryGeometry,			/* query geometry */
		XtInheritDisplayAccelerator,	/* display accelerator */
		NULL,							/* extension */
	},
	{ /* composite_class variables */
		XtInheritGeometryManager,		/* geometry manager */
		XtInheritChangeManaged,			/* change managed */
		XtInheritInsertChild,			/* insert child */
		XtInheritDeleteChild,			/* delete child */
		NULL,							/* extension */
	},
	{ /* constraint_class fields */
		NULL,							/* subresources */
		0,								/* number of subresources */
		sizeof(XiFileSelectConstraintsRec),		/* record size */
		NULL,							/* initialize */
		NULL,							/* destroy */
		NULL,							/* set values */
		NULL,							/* extension */
	},
	{ /* vpaned_class variables */
#ifdef X11R3
		XtInheritSetMinMax,
		XtInheritRefigureMode,
#else
		0,
#endif
	},
	{ /* file_select_class variables */
		0,
	},
}; /* XifileSelectClassRec */


WidgetClass xiFileSelectWidgetClass =
	(WidgetClass) &XifileSelectClassRec;


/*
XiFileSelectWidget methods:
*/

/*
Initialize() initializes each instance of a file selection box.
A file selection box is a compound widget containing several
subordinate widgets.  NOTE:  Viewport width/height are approx.
due to the presence of scrollbars; Initialize() could provide
more complex calculations, but it's probably not necessary.  Note
that the file selection box may be resized (automatically) in an
application.
   As currently designed, a number of features can't be modified by
the user, e.g., the border width of the file filter text entry box.
However, it is straightforward to provide resources for the
hard-coded features, as has been done with, for example, the file
filter box width.  For simplicity, the client program CANNOT add
additional buttons, etc.
*/
/*ARGSUSED*/
static void Initialize(request, new)
XiFileSelectWidget request, new;
{
	static char filter_translations[] =
		"#override\n\
		Ctrl<Key>J:		beep()\n\
		Ctrl<Key>M:		beep()\n\
		Ctrl<Key>O:		beep()\n\
		<Key>Linefeed:	beep()\n\
		<Key>Return:	beep()";
	static XtActionsRec filter_actions[] = {
		{"beep", (XtActionProc) Beep},
	};
	XtTranslations filter_trans_table;

	Arg args[15];
	int i, col_width, row_height;

	new->file_select.filenames = NULL;
	if (new->file_select.rows < FILE_MIN_ROWS)
		new->file_select.rows = FILE_DEFAULT_ROWS;
	if (new->file_select.columns < FILE_MIN_COLUMNS)
		new->file_select.columns = FILE_DEFAULT_COLUMNS;
	if (new->file_select.pixmap_rows < FILE_MIN_ROWS)
		new->file_select.pixmap_rows = FILE_DEFAULT_PIXMAP_ROWS;
	if (new->file_select.pixmap_columns < FILE_MIN_COLUMNS)
		new->file_select.pixmap_columns = FILE_DEFAULT_PIXMAP_COLUMNS;
	if (!strlen(new->file_select.default_filter))
		new->file_select.default_filter = FILE_DEFAULT_FILTER;

	get_font_width_height(new, &col_width, &row_height);
	row_height += new->file_select.row_spacing;

	/*
	A VPaned widget doesn't allow a (normal) width change.
	Here, we force the width of the pane to be that of the
	viewport (at least before resizing)--the area for
	displaying files is more important that the text area
	for displaying the file filter.
	*/
	new->core.width = (new->file_select.columns + 2) * col_width;
	new->core.height = (new->file_select.rows + 2) * row_height;

	/*
	add/create actions/translations:
	*/
	XtAddActions(filter_actions, XtNumber(filter_actions));
	filter_trans_table =
		XtParseTranslationTable(filter_translations);

	/*
	create a box to house the filter label and text entry area:
	*/
	new->file_select.filterBoxW = XtCreateManagedWidget("filterBox",
		boxWidgetClass, new, NULL, 0);

	/*
	create the filter label:
	*/
	i = 0;
	XtSetArg(args[i], XtNborderWidth, (XtArgVal) 0); i++;
	XtSetArg(args[i], XtNlabel, (XtArgVal) "Filter:"); i++;
	new->file_select.labelW = XtCreateManagedWidget("filterLabel",
		labelWidgetClass, new->file_select.filterBoxW, args, i);

	/*
	create the text entry area for the file filter:
	*/
	i = 0;
#ifdef X11R3
	new->file_select.filter = XtMalloc(FILE_MAX_FILTER + 1);
	strncpy(new->file_select.filter, new->file_select.default_filter,
		FILE_MAX_FILTER);
	if (strlen(new->file_select.default_filter) >= FILE_MAX_FILTER)
		new->file_select.filter[FILE_MAX_FILTER] = EOS;
	XtSetArg(args[i], XtNtextOptions,
		(XtArgVal) (resizeWidth | resizeHeight)); i++;
	XtSetArg(args[i], XtNeditType, (XtArgVal) XttextEdit); i++;
#else
	new->file_select.filter = new->file_select.default_filter;
	XtSetArg(args[i], XtNresize, (XtArgVal) XawtextResizeBoth); i++;
	XtSetArg(args[i], XtNeditType, (XtArgVal) XawtextEdit); i++;
#endif
	XtSetArg(args[i], XtNstring,
		(XtArgVal) new->file_select.filter); i++;
	XtSetArg(args[i], XtNborderWidth, (XtArgVal) 1); i++;
	XtSetArg(args[i], XtNlength, (XtArgVal) FILE_MAX_FILTER); i++;
	XtSetArg(args[i], XtNwidth,
		(XtArgVal) new->file_select.filter_width); i++;
	XtSetArg(args[i], XtNresizable, TRUE); i++;
	new->file_select.filterW = XtCreateManagedWidget("filterText",
#ifdef X11R3
		asciiStringWidgetClass,
#else
		asciiTextWidgetClass,
#endif
		new->file_select.filterBoxW, args, i);
	XtOverrideTranslations(new->file_select.filterW,
		filter_trans_table);

	/*
	create the button box:
	*/
	new->file_select.buttonBoxW =
		XtCreateManagedWidget("fileButtonBox",
			boxWidgetClass, new, NULL, 0);

	/*
	create the viewport to house the StrSelect widget:
	*/
	i = 0;
	XtSetArg(args[i], XtNallowHoriz, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XtNallowVert, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XtNforceBars, (XtArgVal) TRUE); i++;
	XtSetArg(args[i], XtNheight,
		(XtArgVal) new->file_select.rows * row_height); i++;
	XtSetArg(args[i], XtNwidth,
		(XtArgVal) new->file_select.columns * col_width); i++;
	new->file_select.viewportW = XtCreateManagedWidget("fileViewport",
		viewportWidgetClass, new, args, i);

	/*
	create the StrSelect widget:
	*/
	i = 0;
	XtSetArg(args[i], XiNdelimiters, (XtArgVal) FILE_DELIMITERS); i++;
	XtSetArg(args[i], XiNfont,
		(XtArgVal) new->file_select.font); i++;
	XtSetArg(args[i], XiNpixmapColumns,
		(XtArgVal) new->file_select.pixmap_columns); i++;
	XtSetArg(args[i], XiNpixmapRows,
		(XtArgVal) new->file_select.pixmap_rows); i++;
	if (new->file_select.pixmap_cursor != None) {
		XtSetArg(args[i], XtNcursor,
			(XtArgVal) new->file_select.pixmap_cursor); i++;
	}
	new->file_select.strSelectW = XtCreateManagedWidget("fileListBox",
		xiStrSelectWidgetClass, new->file_select.viewportW, args, i);
	set_filename_reference_buffer(new);

	/*
	create the individual buttons:
	*/
	i = 0;
	XtSetArg(args[i], XtNlabel, (XtArgVal) "Filter"); i++;
	new->file_select.filterButtonW =
		XtCreateManagedWidget("fileButtonFilter",
			commandWidgetClass, new->file_select.buttonBoxW, args, i);
	XtAddCallback(new->file_select.filterButtonW, XtNcallback,
		FileSelectBoxFilter, NULL);

	i = 0;
	XtSetArg(args[i], XtNlabel, (XtArgVal) "Apply"); i++;
	new->file_select.applyButtonW =
		XtCreateManagedWidget("fileButtonApply",
			commandWidgetClass, new->file_select.buttonBoxW, args, i);
	/*
	Note:  The client should use XiFileSelectAddApplyProc() to
	add the callback for the previous button.
	*/

	if (XtIsSubclass(XtParent(new), shellWidgetClass)) {
		i = 0;
		XtSetArg(args[i], XtNlabel, (XtArgVal) "Dismiss"); i++;
		new->file_select.dismissButtonW =
			XtCreateManagedWidget("fileButtonDismiss",
				commandWidgetClass, new->file_select.buttonBoxW,
				args, i);
		XtAddCallback(new->file_select.dismissButtonW, XtNcallback,
			FileSelectBoxDismiss, NULL);
	}
}	/* Initialize */


/*
Destroy() cleans up dynamic data structures.
*/

static void Destroy(w)
XiFileSelectWidget w;
{
	XtDestroyWidget(w->file_select.labelW);
	XtDestroyWidget(w->file_select.filterW);
	XtDestroyWidget(w->file_select.filterBoxW);
	XtDestroyWidget(w->file_select.viewportW);
	XtDestroyWidget(w->file_select.strSelectW);
	XtDestroyWidget(w->file_select.applyButtonW);
	XtDestroyWidget(w->file_select.dismissButtonW);
	XtDestroyWidget(w->file_select.filterButtonW);
	XtDestroyWidget(w->file_select.buttonBoxW);
	if (w->file_select.filenames != NULL)
		free(w->file_select.filenames);
}	/* Destroy */


/*
SetValues() handles modifications to resource-related instance
variables.  Typically, these don't need to be checked--most
resources are used during widget initialization only; others
simply assume the new values.
*/
/*ARGSUSED*/
static Boolean SetValues(current, request, new)
Widget current, request, new;
{
	/*
	at present, nothing to update...
	*/
	return FALSE;
}	/* SetValues */


/*
Actions functions:
*/

/*
Beep() is a general-purpose function; one use is to beep the
user if they try to enter a character that results in a line
feed operation; see Initialize().
*/
/*ARGSUSED*/
static void Beep(w, event)
Widget w;
XEvent *event;
{
	XBell(XtDisplay(w), 50);
}	/* Beep */


/*
Public functions:
*/

/*
XiFileSelectPopup() invokes (pops up) the file selection box.  This
convenience function is useful ONLY if the the file selection box's
parent is a popup shell.
*/

void XiFileSelectPopup(fsw)
XiFileSelectWidget fsw;
{
	/*
	shell -> file selection box -> ...
	*/
	if (XtIsSubclass(XtParent(fsw), shellWidgetClass))
		XtPopup(fsw->core.parent, XtGrabNone);
}	/* XiFileSelectPopup */


/*
XiFileSelectGetSelection() retrieves the value stored in the
private field `selection'; this is the currently selected
file.  Unlike XiStrSelectGetSelection(), this (higher level)
function doesn't allow the return value to be null.
*/

char *XiFileSelectGetSelection(fsw)
XiFileSelectWidget fsw;
{
	char *select;

	if ((select = XiStrSelectGetSelection(fsw->file_select.strSelectW))
			!= NULL)
		return select;
	else
		return "";
}	/* XiFileSelectGetSelection */


/*
XiFileSelectAddApplyProc() allows the user to register a function
(procedure) to be executed when the apply button is pressed.  Note
that no data is returned; an application should retrieve the
current selection with XiFileSelectGetSelection().
*/

void XiFileSelectAddApplyProc(fsw, callback_proc)
XiFileSelectWidget fsw;
void (*callback_proc)();
{
	if (callback_proc != NULL)
		XtAddCallback(fsw->file_select.applyButtonW, XtNcallback,
			callback_proc, NULL);
}	/* XiFileSelectAddApplyProc */


/*
XiFileSelectRefresh() allows the application programmer to update
the file selection box window manually.  This function does NOT
check for changes to the filter.
*/

void XiFileSelectRefresh(fsw)
XiFileSelectWidget fsw;
{
	set_filename_reference_buffer(fsw);
	XiStrSelectRedisplay(fsw->file_select.strSelectW);
}	/* XiFileSelectRefresh */


/*
Internal Callback functions:
*/

/*ARGSUSED*/
static void FileSelectBoxDismiss(w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	/*
	shell -> file selection box -> button box -> button
	*/
	XiFileSelectWidget fsw =
		(XiFileSelectWidget) XtParent(w->core.parent);
	if (XtIsSubclass(XtParent(fsw), shellWidgetClass)) 
		XtPopdown(XtParent(fsw));
}	/* FileSelectBoxDismiss */


/*ARGSUSED*/
static void FileSelectBoxFilter(w, client_data, call_data)
Widget w;
caddr_t client_data, call_data;
{
	/*
	shell -> file selection box -> button box -> button
	*/
	XiFileSelectWidget fsw =
		(XiFileSelectWidget) XtParent(w->core.parent);

#ifdef X11R3
#else
	Arg args[1];
	char *filter_text;

	XtSetArg(args[0], XtNstring, &filter_text);
	XtGetValues(fsw->file_select.filterW, args, 1);
	fsw->file_select.filter = filter_text;
#endif
	set_filename_reference_buffer(fsw);
	XiStrSelectRedisplay(fsw->file_select.strSelectW);
}	/* FileSelectBoxFilter */


/*
Support functions:
*/

/*
get_font_width_height() determines the font width and height so
that the row and column resources will be in sync with the pixmap
created by the StrSelect widget.
*/

static void get_font_width_height(w, font_width, font_height)
XiFileSelectWidget w;
int *font_width, *font_height;
{
	*font_width = w->file_select.font->max_bounds.width;
	*font_height = w->file_select.font->max_bounds.ascent +
		w->file_select.font->max_bounds.descent;
}	/* get_font_width_height */


/*
set_filename_reference_buffer() expands the file specification
given by the filter, and then updates the StrSelect widget's
reference string.
*/

static void set_filename_reference_buffer(w)
XiFileSelectWidget w;
{
	char *filename_buffer, *filter;
	int match;

	if (!*w->file_select.filter) {
		fprintf(stderr,						/** !!! remove? **/
			"file selection box: no value for filter!\n");
		XiStrSelectReset(w->file_select.strSelectW);
		return;
	}

	filename_buffer = expand_file_spec(w->file_select.filter);
	
	if (filename_buffer == NULL) {
		fprintf(stderr,
		"file selection box: error attempting to expand filter!\n");
		XiStrSelectReset(w->file_select.strSelectW);
		return;
	}

	if (w->file_select.filenames != NULL)
		free(w->file_select.filenames);
	w->file_select.filenames = filename_buffer;

	/*
	Check to see if the expansion is nothing more than the filter,
	and if so, make sure that there is no matching file.  Note
	that the expanded filename will have a linefeed appended.
	*/
	if (strlen(w->file_select.filter) + 1 ==
			strlen(filename_buffer)) {
		match = TRUE;
		filter = w->file_select.filter;
		while (*filename_buffer && *filter)
			if (*filename_buffer++ != *filter++)
				match = FALSE;
		if (match && file_size(w->file_select.filter) == -1) {
			fprintf(stderr,					/** !!! remove? **/
				"file selection box: no match for filter!\n");
			free(w->file_select.filenames);
			w->file_select.filenames = NULL;
			XiStrSelectReset(w->file_select.strSelectW);
		}
		else
			XiStrSelectSetString(w->file_select.strSelectW,
				w->file_select.filenames);
	}
	else
		XiStrSelectSetString(w->file_select.strSelectW,
			w->file_select.filenames);
}	/* set_filename_reference_buffer */


/*
expand_file_spec() takes a file spec. as an argument and
returns a pointer to a dynamically allocated list of filenames
that are produced by expanding the file spec.  The calling
function is responsible for freeing the dynamic storage when
it is no longer needed.
*/

static char *expand_file_spec(file_spec)
char *file_spec;
{
	static char *file_spec_buffer;
	char temp_file[40], alt_file_spec[FILE_MAX_FILTER + 1];
	char cmdstring[FILE_MAX_FILTER + 9 + 40];

	if (!*file_spec)
		return NULL;
	if (strlen(file_spec) >= FILE_MAX_FILTER) {
		fprintf(stderr,
		"file selection box: filter has too many characters!\n");
		return NULL;
	}
	while (*file_spec && *file_spec == ' ')
		file_spec++;
	if (*file_spec == '~') {
		if (get_home_directory(alt_file_spec)) {
			file_spec++;	/* skip the '~' */
			strncat(alt_file_spec, file_spec,
				FILE_MAX_FILTER - strlen(alt_file_spec));
			file_spec = alt_file_spec;
		}
	}
	sprintf(temp_file, "%s.%d", "Xi.temp.filenames", getpid());
	unlink(temp_file);		/* if it exists */
	sprintf(cmdstring, "echo %s > %s", file_spec, temp_file);
	system(cmdstring);
	if (file_size(temp_file) == -1) {
		fprintf(stderr,
		"file selection box: unknown error while expanding filter!\n");
		return NULL;
	}
	if ((file_spec_buffer =
			(char *) malloc((unsigned) (file_size(temp_file) + 1)))
			== NULL) {
		fprintf(stderr,
		"file selection box: memory allocation error\n\
 while expanding filter!\n");
		unlink(temp_file);
		return NULL;
	}
	if (!load_buffer(temp_file, file_spec_buffer)) {
		fprintf(stderr,
		"file selection box: unknown temp file error\n\
 while expanding filter!\n");
		free(file_spec_buffer);
		unlink(temp_file);
		return NULL;
	}
	unlink(temp_file);
	return file_spec_buffer;
}	/* expand_file_spec */


/*
load_buffer() loads a buffer with text from a file.
*/

static int load_buffer(file_spec, buffer)
char *file_spec, *buffer;
{
	FILE *file_ptr;

	if ((file_ptr = fopen(file_spec, "r")) == NULL)
		return FALSE;
	while ((*buffer = fgetc(file_ptr)) != EOF)
		buffer++;
	*buffer = EOS;
	fclose(file_ptr);
	return TRUE;
}	/* load_buffer */


/*
file_size() checks for the existence of a file using stat(),
returning -1 for a nonexistent file and the file size otherwise.
*/

static int file_size(file_spec)
char *file_spec;
{
	struct stat	statbuf;

	if (stat(file_spec, &statbuf) < 0)
		return -1;
	else
		return (int)statbuf.st_size;
}	/* file_size */


/*
get_home_directory() examines the environment for the value of HOME.
*/

static int get_home_directory(directory)
char *directory;
{
	char *home;

	if ((home = (char *) getenv("HOME")) == NULL) {
		fprintf(stderr,
	"file selection box: unable to determine your home directory!\n");
		return FALSE;
	}
	strcpy(directory, home);
	return TRUE;
}	/* get_home_directory */

