/**  StrSelect.c  ** simple string selection box implementation  **/


#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#ifdef X11R3
#include <X11/Simple.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Simple.h>
#endif

#include "StrSelectP.h"


/*
Class Methods:
*/

static void Initialize();
static void Destroy();
static Boolean SetValues();
static void Redisplay();


/*
Private support functions:
*/

static void create_offscreen_pixmap();
static void fill_pixmap_and_redisplay();
static void fill_pixmap();
static void redisplay_pixmap();
static void select_str();


/*
Translations and actions:
*/

static void SelectStr();

static char pixmap_translations[] =
	"<Btn1Down>:	select()";
	
static XtActionsRec pixmap_actions[] = {
	{"select", (XtActionProc) SelectStr},
};


/*
Resource table:
*/

#define res_offset(field) XtOffset(XiStrSelectWidget, field)

static XtResource resources[] = {
	{XiNreferenceString, XiCReferenceString, XtRString,
		sizeof(char *), res_offset(str_select.str),
		XtRString, STR_NOTHING},
	{XiNdelimiters, XiCDelimiters, XtRString, sizeof(char *),
		res_offset(str_select.delimiters),
		XtRString, STR_DELIMITERS},
	{XiNfont, XiCFont, XtRFontStruct, sizeof(XFontStruct *),
		res_offset(str_select.font),
		XtRString, "XtDefaultFont"},
	{XiNforeground, XiCForeground, XtRPixel, sizeof(Pixel),
		res_offset(str_select.foreground),
		XtRString, XtDefaultForeground},
	{XiNpixmapRows, XiCPixmapRows, XtRInt, sizeof(int),
		res_offset(str_select.pixmap_rows),
		XtRImmediate, (caddr_t) STR_DEFAULT_PIXMAP_ROWS},
	{XiNpixmapColumns, XiCPixmapColumns, XtRInt, sizeof(int),
		res_offset(str_select.pixmap_columns),
		XtRImmediate, (caddr_t) STR_DEFAULT_PIXMAP_COLUMNS},
	{XiNrowSpacing, XiCRowSpacing, XtRInt, sizeof(int),
		res_offset(str_select.row_spacing),
		XtRImmediate, (caddr_t) STR_DEFAULT_ROW_SPACING},
};


/*
Define storage for the class here:
*/

XiStrSelectClassRec XistrSelectClassRec = {
	{ /* core_class variables */
		(WidgetClass) &simpleClassRec,	/* ancestor */
		"StrSelect",					/* class name */
		sizeof(XiStrSelectRec),			/* widget size */
#ifdef X11R3
		NULL,							/* class initialize */
#else
		XawInitializeWidgetSet,			/* class initialize */
#endif
		NULL,							/* class part init. */
		FALSE,							/* class inited */
		Initialize,						/* initialize */
		NULL,							/* initialize hook */
		XtInheritRealize,				/* realize */
		pixmap_actions,					/* actions */
		XtNumber(pixmap_actions),		/* number of actions */
		resources,						/* resources */
		XtNumber(resources),			/* number of resources */
		NULLQUARK,						/* xrm class */
		TRUE,							/* compress motions */
		TRUE,							/* compress exposures */
		TRUE,							/* compress enter/leave */
		FALSE,							/* visibility interest */
		Destroy,						/* destroy */
		NULL,							/* resize */
		Redisplay,						/* expose */
		SetValues,						/* set values */
		NULL,							/* set values hook */
		XtInheritSetValuesAlmost,		/* set values almost */
		NULL,							/* get values hook */
		NULL,							/* accept focus */
		XtVersion,						/* version */
		NULL,							/* callback private */
		pixmap_translations,			/* translation table */
		XtInheritQueryGeometry,			/* query geometry */
		XtInheritDisplayAccelerator,	/* display accelerator */
		NULL,							/* extension */
	},
	{ /* simple_class variables */
		XtInheritChangeSensitive,
	},
	{ /* str_select_class variables */
		0,
	},
}; /* XistrSelectClassRec */


WidgetClass xiStrSelectWidgetClass =
	(WidgetClass) &XistrSelectClassRec;


/*
XiStrSelectWidget methods:
*/

/*
Initialize() initializes each instance of a string selection box.
A string selection box implements a pixmap in a SimpleWidget.
*/
/*ARGSUSED*/
static void Initialize(request, new)
XiStrSelectWidget request, new;
{
	if (new->str_select.pixmap_rows < STR_MIN_ROWS)
		new->str_select.pixmap_rows = STR_DEFAULT_PIXMAP_ROWS;
	if (new->str_select.pixmap_columns < STR_MIN_COLUMNS)
		new->str_select.pixmap_columns = STR_DEFAULT_PIXMAP_COLUMNS;
	if (!new->str_select.str)
		new->str_select.str = STR_NOTHING;
	if (!new->str_select.delimiters)
		new->str_select.delimiters = STR_DELIMITERS;
	if (!strlen(new->str_select.delimiters))
		new->str_select.delimiters = STR_DELIMITERS;

	create_offscreen_pixmap(new);

	new->str_select.selection = NULL;
	new->str_select.buffer = NULL;
	new->core.width = new->str_select.pixmap_width;
	new->core.height = new->str_select.pixmap_height;

	fill_pixmap(new, STR_NO_HIGHLIGHT);
}	/* Initialize */


/*
Destroy() cleans up dynamic data structures.
*/

static void Destroy(w)
XiStrSelectWidget w;
{
	if (w->str_select.buffer != NULL)
		XtFree(w->str_select.buffer);
	if (w->str_select.selection != NULL)
		XtFree(w->str_select.selection);
	if (w->str_select.gc != NULL)
		XFreeGC(XtDisplay(w), w->str_select.gc);
	if (w->str_select.cgc != NULL)
		XFreeGC(XtDisplay(w), w->str_select.cgc);
	if (w->str_select.hgc != NULL)
		XFreeGC(XtDisplay(w), w->str_select.hgc);
	if (w->str_select.pixmap != NULL)
		XFreePixmap(XtDisplay(w), w->str_select.pixmap);
}	/* Destroy */


/*
SetValues() handles modifications to resource-related
instance variables.  Since all pixmap-related processing is
based on `pixmap_width' and `pixmap_height', and these fields
are modified only during widget creation, it really isn't
necessary to check for modified row and column metrics; this
applies to several other resources as well.
*/
/*ARGSUSED*/
static Boolean SetValues(current, request, new)
Widget current, request, new;
{
	XiStrSelectWidget sw = (XiStrSelectWidget) new;
	XiStrSelectWidget old_sw = (XiStrSelectWidget) current;

	if (!sw->str_select.str)
		sw->str_select.str = STR_NOTHING;
	if (!sw->str_select.delimiters)
		sw->str_select.delimiters = STR_DELIMITERS;
	if (!strlen(sw->str_select.delimiters))
		sw->str_select.delimiters = STR_DELIMITERS;
	if (strcmp(sw->str_select.str, old_sw->str_select.str))
		fill_pixmap(sw, STR_NO_HIGHLIGHT);
	if (strcmp(sw->str_select.delimiters,
			old_sw->str_select.delimiters))
		fill_pixmap(sw, STR_NO_HIGHLIGHT);
	return FALSE;
}	/* SetValues */


/*
Redisplay() calls redisplay_pixmap() to update the pixmap
window upon an exposure.  Note that the pixmap is not updated--
just the pixmap window.
*/
/*ARGSUSED*/
static void Redisplay(w, event)
Widget w;
XEvent *event;
{
	redisplay_pixmap(w);
}	/* Redisplay */


/*
Actions functions:
*/

/*
SelectStr() simply calls the private function select_str().
*/

static void SelectStr(w, event)
Widget w;
XEvent *event;
{
	select_str(w, event);
}	/* SelectStr */


/*
Public functions:
*/

/*
XiStrSelectPopup() invokes (pops up) the string selection box.
This convenience is useful ONLY if the string selection box's
parent is a pop-up shell.
*/

void XiStrSelectPopup(ssw)
XiStrSelectWidget ssw;
{
/*	if (XtIsShell(ssw->core.parent)) **** won't work with X11R4 ****/
	if (XtIsSubclass(ssw->core.parent, (WidgetClass) shellWidgetClass))
		XtPopup(ssw->core.parent, XtGrabNone);
}	/* XiStrSelectPopup */


/*
XiStrSelectGetSelection() retrieves the value stored in the private
field `selection'; this is the currently selected string.
*/

char *XiStrSelectGetSelection(ssw)
XiStrSelectWidget ssw;
{
	return ssw->str_select.selection;
}	/* XiStrSelectGetSelection */


/*
XiStrSelectSetString() establishes a pointer to the string
(buffer) that the widget displays.  It assumes that a zero-
length string is OK.
*/

void XiStrSelectSetString(ssw, str)
XiStrSelectWidget ssw;
char *str;
{
	if (!str)			/* check for null pointer */
		fprintf(stderr,
		"string selection box: the reference string is null!\n");
	ssw->str_select.str = str;
	fill_pixmap(ssw, STR_NO_HIGHLIGHT);
}	/* XiStrSelectSetString */


/*
XiStrSelectRedisplay() allows an application to force a
redisplay of the pixmap.
*/

void XiStrSelectRedisplay(ssw)
XiStrSelectWidget ssw;
{
	redisplay_pixmap(ssw);
}	/* XiStrSelectRedisplay */


/*
XiStrSelectReset() allows an application to force the
clearing of the pixmap and related variables.
redisplay of the pixmap.
*/

void XiStrSelectReset(ssw)
XiStrSelectWidget ssw;
{
	ssw->str_select.current_str = STR_NO_STR;
	ssw->str_select.str = NULL;
	if (ssw->str_select.selection != NULL) {
		XtFree(ssw->str_select.selection);
		ssw->str_select.selection = NULL;
	}
	if (ssw->str_select.buffer != NULL)
		XtFree(ssw->str_select.buffer);
	XFillRectangle(XtDisplay(ssw), ssw->str_select.pixmap,
		ssw->str_select.cgc, 0, 0, ssw->core.width, ssw->core.height);
	if (XtIsRealized(ssw))
		XClearWindow(XtDisplay(ssw), XtWindow(ssw));
}	/* XiStrSelectReset */


/*
Support functions:
*/

/*
create_offscreen_pixmap() creates and sets the dimensions
of the string selection box pixmap.  It also creates three GCs:
	gc  -- for standard text display,
	hgc -- for highlighted/selected text, and 
	cgc -- for clearing the pixmap.
There are numerous data type inconsistencies in the various X
data structures for fonts, pixmaps, etc., so just use `int';
expressions will be converted to `unsigned int'.
*/

static void create_offscreen_pixmap(w)
XiStrSelectWidget w;
{
	XGCValues values;
	Display *display = XtDisplay(w);
	int screen = XDefaultScreen(display);
	int depth = XDisplayPlanes(display, screen);

	values.foreground = w->str_select.foreground;
	values.background = w->core.background_pixel;
	values.font = w->str_select.font->fid;
	values.fill_style = FillSolid;
	w->str_select.gc = XCreateGC(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		GCForeground | GCBackground | GCFont | GCFillStyle, &values);
	values.background = w->str_select.foreground;
	values.foreground = w->core.background_pixel;
	w->str_select.cgc = XCreateGC(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		GCForeground | GCBackground | GCFont | GCFillStyle, &values);
	w->str_select.hgc = XCreateGC(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)),
		GCForeground | GCBackground | GCFont, &values);
	w->str_select.row_increment =
		w->str_select.font->max_bounds.ascent +
		w->str_select.font->max_bounds.descent +
		w->str_select.row_spacing;
	w->str_select.pixmap_width = w->str_select.pixmap_columns *
		w->str_select.font->max_bounds.width;
	w->str_select.pixmap_height = w->str_select.pixmap_rows *
		w->str_select.row_increment;
	w->str_select.pixmap = XCreatePixmap(XtDisplay(w),
		RootWindowOfScreen(XtScreen(w)), w->str_select.pixmap_width,
		w->str_select.pixmap_height, depth);
}	/* create_offscreen_pixmap */


/*
fill_pixmap_and_redisplay() builds the pixmap and refreshes
the window.
*/

static void fill_pixmap_and_redisplay(w, highlight_position)
XiStrSelectWidget w;
int highlight_position;
{
	fill_pixmap(w, highlight_position);
	redisplay_pixmap(w);
}	/* fill_pixmap_and_redisplay */


/*
fill_pixmap() tests for errors and then displays the string
on the pixmap.  NOTE: This function sets the current selection.
*/

static void fill_pixmap(w, highlight_position)
XiStrSelectWidget w;
int highlight_position;
{
	int len, offset;
	char *next;

	if (!w->str_select.str)
		return;
	if (!strlen(w->str_select.str)) {
		fprintf(stderr,			/** !!! remove? **/
		"string selection box: reference string is empty!\n");
		return;
	}
	w->str_select.current_str = STR_NO_STR;
	if (w->str_select.selection != NULL) {
		XtFree(w->str_select.selection);
		w->str_select.selection = NULL;
	}
	if (w->str_select.buffer != NULL)
		XtFree(w->str_select.buffer);
	/*
	copy the user's string -- strtok() modifies its string argument
	*/
	if ((len = strlen(w->str_select.str)) > -1) {
		w->str_select.buffer = XtMalloc((unsigned) (len + 1));
		strcpy(w->str_select.buffer, w->str_select.str);
	}
	else
		w->str_select.buffer = "";
	/*
	clear pixmap before it filling with a new string:
	*/
	XFillRectangle(XtDisplay(w), w->str_select.pixmap,
		w->str_select.cgc, 0, 0, w->core.width, w->core.height);
	next = (char *)
		strtok(w->str_select.buffer, w->str_select.delimiters);
	for (offset = w->str_select.row_increment,
			w->str_select.current_num_strs = 0;
			next != NULL && offset < w->str_select.pixmap_height;
			offset += w->str_select.row_increment,
			w->str_select.current_num_strs++) {
		len = strlen(next);
		XDrawImageString(XtDisplay(w), w->str_select.pixmap,
			w->str_select.gc, 0, offset, next, len);
		if (highlight_position != STR_NO_HIGHLIGHT)
			if (highlight_position ==
					w->str_select.current_num_strs) {
				/* this is it */
				XDrawImageString(XtDisplay(w), w->str_select.pixmap,
					w->str_select.hgc, 0, offset, next, len);
				w->str_select.current_str =
					w->str_select.current_num_strs;
				w->str_select.selection =
					XtMalloc((unsigned) (len + 1));
				strcpy(w->str_select.selection, next);
			}
		next = (char *) strtok(NULL, w->str_select.delimiters);
	}
}	/* fill_pixmap */


/*
redisplay_pixmap() copies the pixmap to the window.
*/

static void redisplay_pixmap(w)
XiStrSelectWidget w;
{
	if (!XtIsRealized(w))
		return;
	XCopyArea(XtDisplay(w), w->str_select.pixmap,
		XtWindow(w),
		w->str_select.gc, 0, 0, w->str_select.pixmap_width,
		w->str_select.pixmap_height, 0, 0);
}	/* redisplay_pixmap */


/*
select_str() calculates the "row" offset for a string/item and then
calls fill_pixmap_and_redisplay() to either clear the current
highlighted string or highlight another string, depending on the
legitimacy of the position.  The vertical button position is
"adjusted" by 1/2 the height of a row.  NOTE: This function
indirectly sets/modifies the current selection.
*/

static void select_str(w, event)
XiStrSelectWidget w;
XEvent *event;
{
	int highlight_position;

	highlight_position =
		(event->xbutton.y - (w->str_select.row_increment / 2)) /
			w->str_select.row_increment;
	if (highlight_position > w->str_select.current_num_strs)
		fill_pixmap_and_redisplay(w, STR_NO_HIGHLIGHT);
	else
		fill_pixmap_and_redisplay(w, highlight_position);
}	/* select_str */

