// BMP_QUAN.H - MS-Windows Color Quantization Class

#ifndef _BMP_QUAN_H
#define _BMP_QUAN_H

#include "dib_dos.h"
#include "oct_quan.h"

class BMP_Quantize : public OctQuant
{
  private:
    char *pin_name;     // Input file name
    char *pout_name;    // Output file name

    int num_color;      // Number of palette colors

    size_t in_len;      // Input scan line length
    size_t out_len;     // Output scan line length

    BYTE *pin_scan;     // Input scan line buffer pointer
    BYTE *pout_scan;    // Output scan line buffer pointer

    DIB_FileHeader f_header;    // DIB file header
    DIB_InfoHeader i_header;    // DIB info header
    DIB_Palette dib_pal[256];   // DIB color palette

    DWORD in_offset;    // Input file bitmap offset

    FILE *pin_file;     // Input file pointer
    FILE *pout_file;    // Output file pointer

    BOOL GetPixel( int col, int row, OctColor &c )
    {
      BYTE *ppixel;     // Scan line buffer pixel pointer

      if (col == 0)
      {
        // Read in next scan line
        if (fread(pin_scan, sizeof(BYTE), in_len, pin_file)
            != in_len)
          return FALSE;
      }

      // Get pixel colors
      ppixel = pin_scan + col * 3;
      c.SetBlue(*ppixel++);
      c.SetGreen(*ppixel++);
      c.SetRed(*ppixel);

      return TRUE;
    }

    BOOL MapColors();

    BOOL SetPixel( int col, int row, BYTE color )
    {
      // Set pixel color
      *(pout_scan + col) = color;

      if (col == (i_header.biWidth - 1))
      {
        // Write scan line
        if (fwrite(pout_scan, sizeof(BYTE), out_len,
            pout_file) != out_len)
          return FALSE;
      }

      return TRUE;
    }

    // Reset input file pointer to beginning of bitmap
    BOOL Reset()
    {
      return (fseek(pin_file, in_offset, SEEK_SET) == 0) ?
          TRUE : FALSE;
    }

    // Scan line width in bytes must be multiple of four to
    // avoid memory segmentation problems
    size_t GetScanLineWidth( size_t num_pixel )
    { return (((num_pixel + 3) / 4) * 4); }

    void InitPalette();

    void ReportError( char *msg, char *var )
    {
      char temp_buff[128];      // Temporary buffer

      sprintf(temp_buff, msg, var);
      fprintf(stderr, "ERROR: %s\n", temp_buff);
    }

  public:
    BMP_Quantize() : OctQuant()
    {
      int i;    // Loop index

      pin_file = pout_file = NULL;
      pin_name = pout_name = NULL;
      pin_scan = pout_scan = NULL;
      num_color = 0;

      for (i = 0; i < 256; i++)
      {
        dib_pal[i].rgbBlue = (BYTE) 0;
        dib_pal[i].rgbGreen = (BYTE) 0;
        dib_pal[i].rgbRed = (BYTE) 0;
        dib_pal[i].rgbReserved = (BYTE) 0;
      }
    }

    BOOL Quantize( char *, char * );
};

#endif

