
/*
 *  TOKENRNG.C
 *
 *  Intercepts Media Access Control (MAC) token ring frames
 *  and displays data from the frames.
 *
 *  You must load the IBM "LAN Support Program" or equivalent
 *  before running this program. LSP consists of device drivers
 *  named DXMA0MOD, DXMC0MOD, etc. An equivalent would be the
 *  SMARTLSP program for Madge or Xircom adapters.
 *
 *  Compile with Borland C/C++ 3.0 or later.
 *  Use the "LARGE" (-ml command line option) memory model.
 *
 *  Copyright (c) 1994 Barry R. Nance
 *  All Rights Reserved
 *
 *                      Barry R. Nance
 *                      47 Cider Brook Drive
 *                      Wethersfield, CT 06109
 *
*/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <ctype.h>
#include <string.h>
#include <process.h>
#include <time.h>
#include <math.h>
#include <graphics.h>

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#define DIR_INTERRUPT           0x00
#define DIR_OPEN_ADAPTER        0x03
#define DIR_INITIALIZE          0x20
#define RECEIVE                         0x28
#define RECEIVE_CANCEL          0x29
#define BUFFER_FREE                     0x27
#define DLC_MAX_SAP         2
#define DLC_MAX_STATIONS    4
#define STANDBY_MONITOR     0x06
#define ACTIVE_MONITOR      0x05
#define BEACON              0x02
#define CLAIM_TOKEN         0x03
#define ERROR_REPORT        0x00
#define RING_PURGE          0x04
#define FALSE               0
#define TRUE                1

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

typedef unsigned char       BYTE;
typedef unsigned int        WORD;
typedef unsigned long       ULONG;

typedef struct  {
        BYTE    Name [21];
        BYTE    StringAddress [15];
        BYTE    Address [6];
        } USERLIST;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        BYTE    Value [46];
        } VECTOR;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        BYTE    Address [6];
        } NAUN;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        BYTE    PhysicalLocation [4];
        } PHYSICAL_LOCATION;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        BYTE    LineErrors;
        BYTE    InternalErrors;
        BYTE    BurstErrors;
        BYTE    ACErrors;
        BYTE    AbortDelimiterErrors;
        BYTE    Reserved;
        } ISOLATING_ERRORS;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        BYTE    LostFrameErrors;
        BYTE    ReceiverCongestionErrors;
        BYTE    FrameCopiedErrors;
        BYTE    FrequencyErrors;
        BYTE    TokenErrors;
        BYTE    Reserved;
        } NON_ISOLATING_ERRORS;

typedef struct  {
        BYTE    Length;
        BYTE    Command;
        WORD    BeaconType;
        } BEACON_TYPE;

typedef struct  {
        BYTE    AccessControl;
        BYTE    FrameControl;
        BYTE    Destination[6];
        BYTE    Source[6];
        WORD    MajorVectorLen;
        BYTE    Class;
        BYTE    Command;
        union   {
                VECTOR                  Vector;
                NAUN                    Vect02;
                PHYSICAL_LOCATION       Vect0B;
                ISOLATING_ERRORS        Vect2D;
                NON_ISOLATING_ERRORS    Vect2E;
                BEACON_TYPE             Vect01;
                } SubV;
        } MACFRAME;

typedef struct  {
        BYTE    EventCode;
        BYTE    EventTime [10];
        BYTE    EventVector [46];
        } EVENT;

typedef struct  {
        BYTE    adapter;
        BYTE    command;
        BYTE    retcode;
        BYTE    work;
        void    *pointer;
        void    *cmd_cplt;
        void    *parm_tab;
        } CCB;

typedef struct {
        WORD    bring_up_err;
        WORD    sram_addr;
        BYTE    reserved[4];
        void    *adptr_chk_exit;
        void    *ring_status_exit;
        void    *pc_error_exit;
        } DIR_INIT_PT;
 
typedef struct {
        void    *adapter_parms;
        void    *direct_parms;
        void    *dlc_parms;
        void    *msg_parms;
        } DIR_OPEN_PT;
 
typedef struct {
        WORD    open_error_code;
        WORD    open_options;
        BYTE    node_addr[6];
        BYTE    group_addr[4];
        BYTE    func_addr[4];
        WORD    num_rcv_buffers;
        WORD    rcv_buffer_len;
        WORD    dhb_length;
        BYTE    dhb_number;
        BYTE    reserved;
        WORD    lock_code;
        void    *prod_id_addr;
        } DIR_OPEN_AD_PT;

typedef struct {
        WORD    dir_buf_size;
        WORD    dir_pool_blocks;
        void    *dir_pool_address;
        void    *adpt_chk_exit;
        void    *ring_status_exit;
        void    *pc_error_exit;
        void    *work_addr;
        WORD    work_len_req;
        WORD    work_len_act;
        } DIR_OPEN_DIR_PT;
 
typedef struct {
        BYTE    max_sap;
        BYTE    max_sta;
        BYTE    max_gsap;
        BYTE    max_gmem;
        BYTE    t1_tick_one;
        BYTE    t2_tick_one;
        BYTE    ti_tick_one;
        BYTE    t1_tick_two;
        BYTE    t2_tick_two;
        BYTE    ti_tick_two;
        } DIR_OPEN_DLC_PT;

typedef struct {
        WORD    station_id;
        WORD    buffers_left;
        BYTE    numbuffs;
        BYTE    reserved[3];
        BYTE    *first_buffer;
        } BUFFER_PT;

struct Lanheader  {
        BYTE access_ctl;
        BYTE frame_ctl;
        BYTE destaddr[6];
        BYTE srcaddr[6];
        BYTE rinfo[18];
        };
 
struct Dlcheader  {
    BYTE dsap;
    BYTE ssap;
    WORD control;
        };

typedef struct {
        BYTE    *next_buf_ptr;
        BYTE    data[1];
        } BUFFER;

typedef struct {
        BUFFER *next_buf_ptr;
        WORD    rcv_len;
        WORD    len_in_buffer;
        WORD    adapter_offset;
        WORD    user_length;
        WORD    station_id;
        BYTE    options;
        BYTE    msg_type;
        WORD    buffers_left;
        BYTE    receive_fs;
        BYTE    adapter_num;
    BYTE    lan_hdr_len;
    BYTE    dlc_hdr_len;
        struct Lanheader lan_hdr;
        struct Dlcheader dlc_hdr;
        } RECV_BUFFER_ONE;

typedef struct {
        BUFFER  *next_buf_ptr;
        WORD    rcv_len;
        WORD    len_in_buffer;
        WORD    adapter_offset;
        WORD    user_length;
        WORD    station_id;
        BYTE    options;
        BYTE    msg_type;
        WORD    buffers_left;
        BYTE    receive_fs;
        BYTE    adapter_num;
    BYTE    recvd_data[1];
        } BUF1_CONTIG;

typedef struct {
        WORD      station_id;
        WORD      user_length;
        void      *recv_exit;
        RECV_BUFFER_ONE  *first_buffer;
        BYTE      recv_options;
        } RECV_PT;

typedef struct {
    BYTE        Address [6];
    BYTE        Name [15];
    BYTE        NeighborAddr [6];
    BYTE        NeighborName [15];
    BYTE        FirstSeenTime [10];
    WORD        X;
    WORD        Y;
    BYTE        HeardFromFlag;
    BYTE        Status;
    BYTE        ErrorTime [10];
    BYTE        ErrorFlag;
    long        SecondsSinceError;
    WORD        LineError;
    WORD        InternalError;
    WORD        BurstError;
    WORD        ACError;
    WORD        AbortDelimError;
    WORD        LostFrameError;
    WORD        ReceiverCongestionError;
    WORD        FrameCopiedError;
    WORD        FrequencyError;
    WORD        TokenError;
    WORD        BeaconError;
    } NODE;

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void    interrupt CCBCpltApp(void);
void    interrupt RecDataApp(void);
int     DoInt5C(void *cmdptr);
void    Terminate(BYTE *message, int code);
int     AdapterInterrupt(CCB *ccbptr);
int     AdapterInitialize(CCB *ccbptr);
int     AdapterOpen(CCB *ccbptr);
int     ReceiveMACFrames(CCB *ccbptr, WORD stationid);
int     ReceiveCancel(CCB *ccbptr, WORD stationid);
int     FreeBuffer(CCB *ccbptr, WORD stationid, void *BufPtr);
void    ReportError(char *Message);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

unsigned char   Buffers[4096];
unsigned char   FrameTable[64 * 256];
int             HeadCount, TailCount;
BYTE            *HeadPtr, *TailPtr;
int             StandbyCount;
int             BeaconFlag;
long            TotalFrames;
long            Now, Then;
int             FrameLen;
int             VectorLen;

FILE            *TextFile;
char            String [1024];
char            String2 [32];
USERLIST        UserList [255];
int             UserCount;

NODE            Node [255];
int             NodeCount;
int             ThisNode;

EVENT           Events [50];
int             EventCount;
int             EventFlag;

unsigned int    SaveSS;
unsigned int    SaveSP;
unsigned int    OurSS;
unsigned int    OurSP;
unsigned char   OurStack [512];

CCB             *CCBPtr;
BUF1_CONTIG     *DataPtr;
BUF1_CONTIG     *MACPtr;
BUF1_CONTIG     *RecvBuff;

union   {
        VECTOR                  Vector;
        NAUN                    Naun;
        PHYSICAL_LOCATION       PhysicalLocation;
        ISOLATING_ERRORS        IsolatingErrors;
        NON_ISOLATING_ERRORS    NonIsolatingErrors;
        BEACON_TYPE             BeaconType;
        } SubVector;

CCB             workccb;
CCB             recvccb;
BYTE            funcaddr[4] = {0x80, 0x00, 0x00, 0x08};
MACFRAME        *MACframeptr;
WORD            Stationid;
WORD            i, j;

int             rc;
int             intrrc;
int             initrc;
int             openrc;
int             FirstTimeFlag;
BYTE            ch;
BYTE            ch1;
BYTE            ch2;
BYTE            *ch_ptr;
WORD            RegES, RegBX;
double          Dummy;

DIR_INIT_PT     InitParms;
DIR_OPEN_PT     openpt;
DIR_OPEN_AD_PT  adpt;
DIR_OPEN_DIR_PT dirpt;
DIR_OPEN_DLC_PT dlcpt;
RECV_PT         RecvParms;
RECV_PT         RecvCancel;
BUFFER_PT       BufferPoolPT;

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    interrupt CCBCpltApp(void)
        {
        disable();
        RegES = _ES;
        RegBX = _BX;
        CCBPtr = MK_FP(RegES, RegBX);
        enable();
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    interrupt RecDataApp(void)
        {
        disable();

        RegES  = _ES;
        RegBX  = _BX;
        MACPtr = MK_FP(RegES, RegBX);

        SaveSS = _SS;
        SaveSP = _SP;
        _SS = OurSS;
        _SP = OurSP;
        enable();

        memcpy(HeadPtr, MACPtr, 64);
        HeadPtr += 64;
        HeadCount++;
        if (HeadCount > 255)
            {
            HeadCount = 0;
            HeadPtr   = FrameTable;
            }

        FreeBuffer(&workccb, Stationid, MACPtr);
        ReceiveMACFrames(&recvccb, Stationid);

        disable();
        _SS = SaveSS;
        _SP = SaveSP;
        enable();
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     DoInt5C(void *cmdptr)
        {
        _ES = FP_SEG(cmdptr);
        _BX = FP_OFF(cmdptr);
        geninterrupt(0x5C);
        _BX = _AX;
        _BH = 0;
        return _BX;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    Terminate(BYTE *message, int code)
        {
        clrscr();
        gotoxy(1, 2);
        cprintf("ERROR: %s. return code = %d", message, code);
        gotoxy(1, 5);
        exit(code);
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     AdapterInterrupt(CCB *ccbptr)
        {
        ccbptr->command  = DIR_INTERRUPT;
        ccbptr->adapter  = 0;
        ccbptr->parm_tab = ccbptr->pointer = ccbptr->cmd_cplt = NULL;
        DoInt5C(ccbptr);
        while( ccbptr->retcode == 0xFF )
            ;
        return (int) ccbptr->retcode;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     AdapterInitialize(CCB *ccbptr)
        {
        ccbptr->command  = DIR_INITIALIZE;
        ccbptr->adapter  = 0;
        ccbptr->retcode  = 0xFF;
        ccbptr->parm_tab = (void *) &InitParms;
        ccbptr->cmd_cplt = ccbptr->pointer = NULL;
        memset(&InitParms, 0, sizeof(DIR_INIT_PT));
        return (DoInt5C(ccbptr));
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     AdapterOpen(CCB *ccbptr)
        {
        ccbptr->command  = DIR_OPEN_ADAPTER;
        ccbptr->adapter  = 0;
        ccbptr->parm_tab = (void *) &openpt;
        ccbptr->pointer  = NULL;
        ccbptr->cmd_cplt = (void *) CCBCpltApp;
        openpt.adapter_parms = (void *) &adpt;
        openpt.direct_parms  = (void *) &dirpt;
        openpt.dlc_parms     = (void *) &dlcpt;
        openpt.msg_parms     = NULL;
        memset(&adpt,  0, sizeof(DIR_OPEN_AD_PT));
        memset(&dirpt, 0, sizeof(DIR_OPEN_DIR_PT));
        memset(&dlcpt, 0, sizeof(DIR_OPEN_DLC_PT));
        dlcpt.max_sap   = DLC_MAX_SAP;
        dlcpt.max_sta   = DLC_MAX_STATIONS;
        dirpt.dir_buf_size     = 160;
        dirpt.dir_pool_blocks  = 20;
        dirpt.dir_pool_address = Buffers;
        adpt.open_options = 0x7880;             /* for MAC frames */
        memcpy(adpt.func_addr, funcaddr, 4);
        CCBPtr = NULL;
        DoInt5C(ccbptr);
        while (CCBPtr == NULL)
            ;
        return CCBPtr->retcode;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     ReceiveMACFrames(CCB *ccbptr, WORD stationid)
        {
        ccbptr->command  = RECEIVE;
        ccbptr->adapter  = 0;
        ccbptr->pointer  = NULL;
        ccbptr->parm_tab = (void *) &RecvParms;
        ccbptr->cmd_cplt = (void *) CCBCpltApp;
        memset( &RecvParms, 0, sizeof(RECV_PT) );
        RecvParms.station_id   = stationid;
        RecvParms.recv_exit    = (void *) RecDataApp;
        RecvParms.recv_options = 0x80;                  /* MAC frames */
        return (DoInt5C(ccbptr));
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     ReceiveCancel(CCB *ccbptr, WORD stationid)
        {
        memset(&RecvCancel, 0, sizeof(RECV_PT));
        ccbptr->command  = RECEIVE_CANCEL;
        ccbptr->adapter  = 0;
        ccbptr->pointer  = NULL;
        ccbptr->cmd_cplt = NULL;
        ccbptr->parm_tab = MK_FP(0x0000, stationid);
        RecvCancel.station_id    = stationid;
        RecvCancel.recv_exit     = NULL;
        return (DoInt5C(ccbptr));
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     FreeBuffer(CCB *ccbptr, WORD stationid, void *BufPtr)
        {
        memset(&BufferPoolPT, 0, sizeof(BUFFER_PT));
        ccbptr->command  = BUFFER_FREE;
        ccbptr->adapter  = 0;
        ccbptr->pointer  = NULL;
        ccbptr->parm_tab = (void *) &BufferPoolPT;
        ccbptr->cmd_cplt = NULL;
        BufferPoolPT.station_id   = stationid;
        BufferPoolPT.first_buffer = BufPtr;
        return (DoInt5C(ccbptr));
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
int     GetSubVector(BYTE SubCommand)
        {
        int   ByteCount;
        int   VectorLen;
        BYTE  VectorID;

        swab((char *) &MACframeptr->MajorVectorLen, (char *) &ByteCount, 2);
        ByteCount -= 4;
        ch_ptr = (BYTE *) &MACframeptr->SubV;

        while (ByteCount > 0)
            {
            VectorLen = (int) *ch_ptr;
            VectorID  = *(ch_ptr+1);
            if (VectorID == SubCommand)
                {
                memcpy(&SubVector, ch_ptr, VectorLen);
                return TRUE;
                }
            ch_ptr    += VectorLen;
            ByteCount -= VectorLen;
            }

        return FALSE;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    InitErrorTotals(int Sub)
        {
        Node[Sub].ErrorFlag = FALSE;

        strcpy(Node[i].ErrorTime, " ");
        Node[Sub].SecondsSinceError = 0;

        Node[Sub].LineError = 0;
        Node[Sub].InternalError = 0;
        Node[Sub].BurstError = 0;
        Node[Sub].ACError = 0;
        Node[Sub].AbortDelimError = 0;
        Node[Sub].LostFrameError = 0;
        Node[Sub].ReceiverCongestionError = 0;
        Node[Sub].FrameCopiedError = 0;
        Node[Sub].FrequencyError = 0;
        Node[Sub].TokenError = 0;
        Node[Sub].BeaconError = 0;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    UpdateNodeList(void)
        {
        int i, j;

        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, MACframeptr->Source, 6) == 0)
                goto Update;
            i++;
            }

        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, SubVector.Naun.Address, 6) == 0)
                {
                i++;
                for (j=NodeCount; j>i; j--)
                    Node[j] = Node[j-1];
                goto Insert;
                }
            i++;
            }

AddToList:
        i = NodeCount;

Insert:
        memcpy(Node[i].Address,      MACframeptr->Source, 6);
        strcpy(Node[i].Name, " ");
        for (j=0; j<UserCount; j++)
            {
            if (memcmp(Node[i].Address, UserList[j].Address, 6) == 0)
                {
                strcpy(Node[i].Name, UserList[j].Name);
                break;
                }
            }

        if (FirstTimeFlag)
            _strtime(Node[i].FirstSeenTime);
        else
            _strtime(Node[i].FirstSeenTime);

        Node[i].X = 0;
        Node[i].Y = 0;
        Node[i].HeardFromFlag = 4;

        InitErrorTotals(i);
        NodeCount++;

Update:
        ThisNode = i;
        memcpy(Node[i].NeighborAddr, SubVector.Naun.Address, 6);
        strcpy(Node[i].NeighborName, " ");
        for (j=0; j<UserCount; j++)
            {
            if (memcmp(Node[i].NeighborAddr, UserList[j].Address, 6) == 0)
                {
                strcpy(Node[i].NeighborName, UserList[j].Name);
                break;
                }
            }

        Node[i].HeardFromFlag = 4;
        if (MACframeptr->FrameControl == STANDBY_MONITOR)
            Node[i].Status = 'S';
        else
            Node[i].Status = 'A';
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessActiveMonitor(void)
        {
        StandbyCount = 0;
        if (GetSubVector(0x02))
            UpdateNodeList();
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessStandbyMonitor(void)
        {
        StandbyCount++;
        if (GetSubVector(0x02))
            UpdateNodeList();
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ReportError(char *Message)
        {
        BYTE    CurrentTime [15];

        EventFlag = TRUE;
        window(1, 13, 80, 25);
        gotoxy(1, 1);
        cprintf(
"-- EVENTS ---------------------------------------------------------------------");

        i = ThisNode;
        _strtime(CurrentTime);

        gotoxy(1, 2);
        delline();

        if (Node[i].Name[0] == ' ')
            sprintf(String, "%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X",
                (int) Node[i].Address[0],
                (int) Node[i].Address[1],
                (int) Node[i].Address[2],
                (int) Node[i].Address[3],
                (int) Node[i].Address[4],
                (int) Node[i].Address[5]);
        else
            strcpy(String, Node[i].Name);

        if (Node[i].NeighborName[0] == ' ')
            sprintf(String2, "%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X",
                (int) Node[i].NeighborAddr[0],
                (int) Node[i].NeighborAddr[1],
                (int) Node[i].NeighborAddr[2],
                (int) Node[i].NeighborAddr[3],
                (int) Node[i].NeighborAddr[4],
                (int) Node[i].NeighborAddr[5]);
        else
            strcpy(String2, Node[i].NeighborName);

        gotoxy(1, 12);
        cprintf("%-9.9s  %-13.13s  %-30.30s   NAUN: %-13.13s",
                CurrentTime,
                String,
                Message,
                String2);

        time(&Node[i].SecondsSinceError);
        _strtime(Node[i].ErrorTime);

        TextFile = fopen("TOKENRNG.LOG", "a+");
        if (TextFile != NULL)
            {
            fprintf(TextFile, "%-9.9s  %-13.13s  %-30.30s   NAUN: %-13.13s\n",
                        CurrentTime,
                        String,
                        Message,
                        String2);
            fclose(TextFile);
            }

        window(1, 1, 80, 25);
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessRingPurge(void)
        {
        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, MACframeptr->Source, 6) == 0)
                goto RecordRingPurge;
            i++;
            }

        return;

RecordRingPurge:
        ThisNode = i;
        ReportError("Ring purge");
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessClaimToken(void)
        {
        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, MACframeptr->Source, 6) == 0)
                goto RecordClaimToken;
            i++;
            }

        return;

RecordClaimToken:
        ThisNode = i;
        ReportError("Claim token");
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessBeacon(void)
        {
        BeaconFlag = TRUE;

        sound(440);
        delay(75);
        nosound();
        sound(660);
        delay(50);
        nosound();

        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, MACframeptr->Source, 6) == 0)
                goto RecordBeacon;
            i++;
            }

        return;

RecordBeacon:
        ThisNode = i;
        Node[i].ErrorFlag = TRUE;
        Node[i].BeaconError++;

        ReportError("BEACON error!");
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    ProcessErrorReport(void)
        {
        i = 0;
        while (i < NodeCount)
            {
            if (memcmp(Node[i].Address, MACframeptr->Source, 6) == 0)
                goto RecordError;
            i++;
            }

        return;

RecordError:
        ThisNode = i;
        if (GetSubVector(0x2D))
            {
            Node[i].LineError       += SubVector.IsolatingErrors.LineErrors;
            Node[i].InternalError   += SubVector.IsolatingErrors.InternalErrors;
            Node[i].BurstError      += SubVector.IsolatingErrors.BurstErrors;
            Node[i].ACError         += SubVector.IsolatingErrors.ACErrors;
            Node[i].AbortDelimError += SubVector.IsolatingErrors.AbortDelimiterErrors;
            }

        if (GetSubVector(0x2E))
            {
            Node[i].LostFrameError  += SubVector.NonIsolatingErrors.LostFrameErrors;
            Node[i].ReceiverCongestionError += SubVector.NonIsolatingErrors.ReceiverCongestionErrors;
            Node[i].FrameCopiedError += SubVector.NonIsolatingErrors.FrameCopiedErrors;
            Node[i].FrequencyError   += SubVector.NonIsolatingErrors.FrequencyErrors;
            Node[i].TokenError       += SubVector.NonIsolatingErrors.TokenErrors;
            }

        Node[i].ErrorFlag = TRUE;

        if (GetSubVector(0x2D))
            {
            if (SubVector.IsolatingErrors.LineErrors != 0)
                ReportError("Line error");
            if (SubVector.IsolatingErrors.InternalErrors != 0)
                ReportError("Internal error");
            if (SubVector.IsolatingErrors.BurstErrors != 0)
                ReportError("Burst error");
            if (SubVector.IsolatingErrors.ACErrors != 0)
                ReportError("A-C error");
            if (SubVector.IsolatingErrors.AbortDelimiterErrors != 0)
                ReportError("Abort Delim. error");
            }
        if (GetSubVector(0x2E))
            {
            if (SubVector.NonIsolatingErrors.LostFrameErrors != 0)
                ReportError("Lost frame error");
            if (SubVector.NonIsolatingErrors.ReceiverCongestionErrors != 0)
                ReportError("Receiver congestion error");
            if (SubVector.NonIsolatingErrors.FrameCopiedErrors != 0)
                ReportError("Frame copied error");
            if (SubVector.NonIsolatingErrors.FrequencyErrors != 0)
                ReportError("Frequency error");
            if (SubVector.NonIsolatingErrors.TokenErrors != 0)
                ReportError("Token error");
            }
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    DrawRing(void)
        {
        int  row;

        if (!EventFlag)
            goto ShowRingData;

        TextFile = fopen("TOKENRNG.LOG", "a+");
        if (TextFile == NULL)
            goto ShowRingData;

        _strtime(String);
        _strdate(String2);
        fprintf(TextFile, "\n\nToken Ring log entries @ %s on %s\n\n",
                String, String2);

        fprintf(TextFile, 
    "       On at St Line Brst  A/C  Abt Lost Cong  Cpd Freq  Tok Beac\n");
        fprintf(TextFile, 
    "       ----- -- ---- ---- ---- ---- ---- ---- ---- ---- ---- ----\n");

        for (i=0; i<NodeCount; i++)
            {
            if (Node[i].NeighborName[0] == ' ')
                sprintf(String2, "%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X",
                    (int) Node[i].NeighborAddr[0],
                    (int) Node[i].NeighborAddr[1],
                    (int) Node[i].NeighborAddr[2],
                    (int) Node[i].NeighborAddr[3],
                    (int) Node[i].NeighborAddr[4],
                    (int) Node[i].NeighborAddr[5]);
            else
                strcpy(String2, Node[i].NeighborName);
            fprintf(TextFile, "%-10.10s [%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X]  ",
                    Node[i].Name,
                    (int) Node[i].Address[0],
                    (int) Node[i].Address[1],
                    (int) Node[i].Address[2],
                    (int) Node[i].Address[3],
                    (int) Node[i].Address[4],
                    (int) Node[i].Address[5]);
            fprintf(TextFile, " NAUN: %s\n", String2);
            fprintf(TextFile, "    %s (%c)",
                    Node[i].FirstSeenTime,
                    Node[i].Status);
            fprintf(TextFile, "%4d %4d %4d %4d %4d %4d %4d %4d %4d %4d\n",
                    Node[i].LineError,
                    Node[i].BurstError,
                    Node[i].ACError,
                    Node[i].AbortDelimError,
                    Node[i].LostFrameError,
                    Node[i].ReceiverCongestionError,
                    Node[i].FrameCopiedError,
                    Node[i].FrequencyError,
                    Node[i].TokenError,
                    Node[i].BeaconError);
            }

        fclose(TextFile);

ShowRingData:
        window(1, 1, 80, 12);
        clrscr();
        for (i=0; i<NodeCount; i++)
            if (Node[i].Status == 'A')
                {
                gotoxy(1, 1);
              cprintf("Active monitor is %-10.10s [%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X]",
                    Node[i].Name,
                    (int) Node[i].Address[0],
                    (int) Node[i].Address[1],
                    (int) Node[i].Address[2],
                    (int) Node[i].Address[3],
                    (int) Node[i].Address[4],
                    (int) Node[i].Address[5]);
                }

        gotoxy(50, 1);
        cprintf("   Total standby monitors: %d", StandbyCount);
        gotoxy(60, 2);
        cprintf("Total frames: %ld", TotalFrames);

        gotoxy(1, 3);
        cprintf(
"In Error      NAUN          Line Brst  A/C  Abt Lost Cong  Cpd Freq  Tok Beac");
        gotoxy(1, 4);
        cprintf(
"------------- ------------- ---- ---- ---- ---- ---- ---- ---- ---- ---- ----");

        row = 5;
        for (i=0; i<NodeCount; i++)
            {
            if (Node[i].ErrorFlag)
                {
                if (Node[i].Name[0] == ' ')
                    sprintf(String2, "%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X",
                        (int) Node[i].Address[0],
                        (int) Node[i].Address[1],
                        (int) Node[i].Address[2],
                        (int) Node[i].Address[3],
                        (int) Node[i].Address[4],
                        (int) Node[i].Address[5]);
                else
                    strcpy(String2, Node[i].Name);
                gotoxy(1, row);
                cprintf("%-13.13s", String2);
                if (Node[i].NeighborName[0] == ' ')
                    sprintf(String2, "%2.2X%2.2X%2.2X %2.2X%2.2X%2.2X",
                        (int) Node[i].NeighborAddr[0],
                        (int) Node[i].NeighborAddr[1],
                        (int) Node[i].NeighborAddr[2],
                        (int) Node[i].NeighborAddr[3],
                        (int) Node[i].NeighborAddr[4],
                        (int) Node[i].NeighborAddr[5]);
                else
                    strcpy(String2, Node[i].NeighborName);
                gotoxy(15, row);
                cprintf("%-13.13s", String2);
                gotoxy(29, row);
                cprintf("%4d %4d %4d %4d %4d %4d %4d %4d %4d %4d",
                    Node[i].LineError,
                    Node[i].BurstError,
                    Node[i].ACError,
                    Node[i].AbortDelimError,
                    Node[i].LostFrameError,
                    Node[i].ReceiverCongestionError,
                    Node[i].FrameCopiedError,
                    Node[i].FrequencyError,
                    Node[i].TokenError,
                    Node[i].BeaconError);
                row++;
                if (row > 12)
                    break;
                }
            }
        window(1, 1, 80, 25);
        EventFlag = FALSE;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
void    main(int argc, char *argv[])
        {
        _AH = 15;
        geninterrupt(0x10);
        _AH = 0;
        i = _AX;
        if (i == 7)
            textattr(0x07);
        else        
            textattr(0x02);

        clrscr();
        gotoxy(1, 1);
        cprintf("Starting Token Ring Monitor...");
        gotoxy(1, 5);

        TextFile = fopen("TOKENRNG.LOG", "a+");
        if (TextFile != NULL)
            {
            _strtime(String);
            _strdate(String2);
            fprintf(TextFile, "\nToken Ring monitor started @ %s on %s\n\n",
                String, String2);
            fclose(TextFile);
            }

        FirstTimeFlag = TRUE;
        EventFlag     = TRUE;

        TextFile = fopen("USER.LST", "r");
        UserCount = 0;
        if (TextFile != NULL)
            {
            while ( (fgets(String, 1000, TextFile)) != NULL)
                {
                if (strlen(String) < 5)
                    continue;
                if (strchr(String, '[') == NULL)
                    continue;
                ch_ptr = &String[12];
                i = 0;
                while (*ch_ptr != ' ' && i < 10)
                    {
                    UserList[UserCount].Name[i] = *ch_ptr;
                    ch_ptr++;
                    i++;
                    }
                strupr(String);
                ch_ptr = &String[40];
                i = 0;
                while (*ch_ptr != ']' && i < 12)
                    {
                    if (*ch_ptr == ' ')
                        *ch_ptr = '0';
                    UserList[UserCount].StringAddress[i] = *ch_ptr;
                    ch_ptr++;
                    i++;
                    }
                j = 40;
                for (i=0; i<6; i++)
                    {
                    ch1 = String[j];
                    if (ch1 >= '0' && ch1 <= '9')
                        ch1 -= '0';
                    else
                        {
                        ch1 -= 'A';
                        ch1 += 10;
                        }
                    j++;
                    ch2 = String[j];
                    if (ch2 >= '0' && ch2 <= '9')
                        ch2 -= '0';
                    else
                        {
                        ch2 -= 'A';
                        ch2 += 10;
                        }
                    ch = (BYTE) (ch1 << 4) | ch2;
                    UserList[UserCount].Address[i] = ch;
                    j++;
                    }
                UserCount++;
                }
            fclose(TextFile);
            }

        OurSP     = FP_OFF( (void *) &OurStack[510] );
        OurSS     = FP_SEG( (void *) &OurStack[510] );

        Stationid = 1;
        TotalFrames = 0;
        HeadCount = 0;
        HeadPtr   = FrameTable;
        TailCount = 0;
        TailPtr   = FrameTable;

        if (getvect(0x5C) == NULL)
            Terminate("Adapter handler not installed.", 16);

        intrrc = AdapterInterrupt(&workccb);
        if ( (intrrc != 0) && (intrrc != 0x09) )
            Terminate( "Could not communicate with network adapter", intrrc);

        initrc = AdapterInitialize(&workccb);
        if (initrc)
            Terminate( "Could not initialize network adapter", initrc);

        openrc = AdapterOpen(&workccb);
        if (openrc && openrc != 0x10)
            Terminate("Could not open network adapter", openrc);

        time(&Then);
        ReceiveMACFrames(&recvccb, Stationid);

CheckActivity:
        if (kbhit())
            {
            ch = getch();
            if (ch != 'q' && ch != 'Q')
                goto GetFrameFromTable;
            TextFile = fopen("TOKENRNG.LOG", "a+");
            if (TextFile != NULL)
                {
                _strtime(String);
                _strdate(String2);
                fprintf(TextFile, "\nToken Ring monitor ended @ %s on %s\n\n",
                    String, String2);
                fclose(TextFile);
                }
            ReceiveCancel(&recvccb, Stationid);
            gotoxy(1,24);
            clreol();
            cprintf("Ring Error Monitor completed.");
            while (recvccb.retcode == 0xFF)
                ;
            exit(0);
            }

GetFrameFromTable:
        if (HeadCount == TailCount)
            goto TimeToDisplay;

        RecvBuff = (void *) TailPtr;
        TailPtr += 64;
        TailCount++;
        if (TailCount > 255)
            {
            TailCount = 0;
            TailPtr   = FrameTable;
            }

        MACframeptr = (MACFRAME *) &RecvBuff->recvd_data[0];
        TotalFrames++;

        if (MACframeptr->FrameControl == STANDBY_MONITOR)
            {
            ProcessStandbyMonitor();
            goto TimeToDisplay;
            }

        if (MACframeptr->FrameControl == ACTIVE_MONITOR)
            {
            ProcessActiveMonitor();
            goto TimeToDisplay;
            }

        if (MACframeptr->FrameControl == BEACON)
            {
            ProcessBeacon();
            goto TimeToDisplay;
            }

        if (MACframeptr->FrameControl == CLAIM_TOKEN)
            {
            ProcessClaimToken();
            goto TimeToDisplay;
            }

        if (MACframeptr->FrameControl == ERROR_REPORT)
            {
            ProcessErrorReport();
            goto TimeToDisplay;
            }

        if (MACframeptr->FrameControl == RING_PURGE)
            {
            ProcessRingPurge();
            goto TimeToDisplay;
            }

        goto CheckActivity;


TimeToDisplay:
        time(&Now);
        if (Now < Then + 6 && !BeaconFlag)
            goto CheckActivity;
        Then = Now;

        for (ThisNode=0; ThisNode<NodeCount; ThisNode++)
            {
            Node[ThisNode].HeardFromFlag--;
            if (Node[ThisNode].HeardFromFlag == 0)
                Node[ThisNode].Status = 'O';
            if (Node[ThisNode].SecondsSinceError < Now - 600)
                InitErrorTotals(ThisNode);
            }

        DrawRing();

        BeaconFlag = FALSE;
        FirstTimeFlag = FALSE;
        goto CheckActivity;
        }

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

