
#include "BuiltinClasses.h"
#include "DynamicString.h"

extern SymbolTable classes;


NativeInstance* MakeNativeList(CL_List<Expr*>& args);
NativeInstance* MakeNativeStruct(CL_List<Expr*>& args);
NativeInstance* MakeNativeUnion(CL_List<Expr*>& args);

typedef Expr* (NativeInstance::* MethPtr)(CL_List<Expr*>&);

void AddBuiltinClasses() {
  NativeClassDef*  n;
  NativeMethod*    method;


  /* ---------------      NativeList      -------------------- */
  n=new NativeClassDef("List", MakeNativeList);

  /* Length */
  method=new NativeMethod("Length", (MethPtr)&NativeList::Length);
  n->AddMethod(*method);

  /* Add */
  method=new NativeMethod("Add", (MethPtr)&NativeList::Add);
  n->AddMethod(*method);

  /* At */
  method=new NativeMethod("At", (MethPtr)&NativeList::At);
  n->AddMethod(*method);

  /* AtPut */
  method=new NativeMethod("AtPut", (MethPtr)&NativeList::AtPut);
  n->AddMethod(*method);


  /* AddFirst */
  method=new NativeMethod("AddFirst", (MethPtr)&NativeList::AddFirst);
  n->AddMethod(*method);

  /* Delete   */
  method=new NativeMethod("Delete", (MethPtr)&NativeList::Delete);
  n->AddMethod(*method);

  ::classes.Lookup("List")->SetVal(*n);
  /* ---------------      End NativeList  -------------------- */


  /* ---------------      NativeStruct      -------------------- */
  n=new NativeClassDef("Struct", MakeNativeStruct);

  /* Length */
  method=new NativeMethod("Length", (MethPtr)&NativeStruct::Length);
  n->AddMethod(*method);

  /* Add */
  method=new NativeMethod("Add", (MethPtr)&NativeStruct::Add);
  n->AddMethod(*method);

  /* At */
  method=new NativeMethod("At", (MethPtr)&NativeStruct::At);
  n->AddMethod(*method);

  /* AtPut */
  method=new NativeMethod("AtPut", (MethPtr)&NativeStruct::AtPut);
  n->AddMethod(*method);


  /* AddFirst */
  method=new NativeMethod("AddFirst", (MethPtr)&NativeStruct::AddFirst);
  n->AddMethod(*method);

  /* Delete   */
  method=new NativeMethod("Delete", (MethPtr)&NativeStruct::Delete);
  n->AddMethod(*method);

  ::classes.Lookup("Struct")->SetVal(*n);
  /* ---------------      End NativeStruct  -------------------- */


  /* ---------------      NativeUnion      -------------------- */
  n=new NativeClassDef("Union", MakeNativeUnion);
  ::classes.Lookup("Union")->SetVal(*n);
  /* ---------------      End NativeStruct  -------------------- */

}





///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////   NativeList  /////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////


/* ---------------------------------------------------------------------------
   The contents of 'args' is always deleted by the caller. Callee *must* copy
   arguments !
   --------------------------------------------------------------------------- */

/* Constructor */
NativeInstance* MakeNativeList(CL_List<Expr*>& args) {
  return new NativeList(args);
}

NativeList::NativeList() : NativeInstance("List") {
  
}

NativeList::NativeList(CL_List<Expr*>& args) 
                      : NativeInstance("List") {
  DO(args, Expr*, item)
     if(item)
       _elements.Add(item->Copy());
  OD
}


NativeList::~NativeList() {
  DO(_elements, Expr*, item)
    delete item;
  OD
}


NativeList* NativeList::Copy() {
  return new NativeList(_elements);
}


void NativeList::Print(ostream& os) {
  short first=1;
  os << "#(";
  DO(_elements, Expr*, item)
    if(!first)
      os << ", ";
    else
      first=0;
    item->Print(os);
  OD
  os << ")";
}


void NativeList::Inspect(ostream& os) {
  Print(os);
  os << "\nMethods:\n--------:\n";
  PrintMethods(os);
  os << endl;
}


void NativeList::Dump(ostream& os) {
  Literal*  tmpval=0;
  os << NATIVE_INST_TYPE << " List ";
  os << START_DEL;
  
  DO(_elements, Expr*, item)
    tmpval=(Literal*)item;
    if(tmpval) {
      os << START_DEL;
      tmpval->Dump(os);
      os << END_DEL;
    }
  OD

  os << END_DEL;
}


void NativeList::Read(istream& is) {
  Literal*     lit=0;
  while(is) {
    DynamicString buf;
    ReadValue (is, buf);
    lit=::CreateLiteralFromString(buf);
    if(lit)
      _elements.Add(lit);
  }
}



Expr* NativeList::Length(CL_List<Expr*>&) {
  return new Int(_elements.Size());
}


Expr* NativeList::Add(CL_List<Expr*>& args) {
  DO(args, Expr*, item)
    _elements.Add(item->Copy());
  OD
  return 0;
}


Expr* NativeList::At(CL_List<Expr*>& args) {
  Int* index=(Int*)args[0];
  Expr* ret=_elements[(long)index->GetInt()];
  return ret? ret->Copy() : 0;
  return 0;
}


Expr* NativeList::AtPut(CL_List<Expr*>& args) {
  if(args.Size() != 2) {
    cerr << "List::AtPut() has 2 parameters !" << endl;
    return 0;
  }
  int index=(int)((Int*)args[0])->GetInt();
  Expr* new_element=(args[1])->Copy();
  _elements.Replace(index, new_element);
  return 0;
}


Expr* NativeList::AddFirst(CL_List<Expr*>& args) {
  DO(args, Expr*, item)
    _elements.Add(item->Copy(), ::head);
  OD
  return 0;
}

Expr* NativeList::Delete(CL_List<Expr*>& args) {
  int index=(int)((Int*)args[0])->GetInt();
  _elements.RemoveAt(index);
  return 0;
}



///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////   NativeStruct  ///////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////



/* Constructor */
NativeInstance* MakeNativeStruct(CL_List<Expr*>& args) {
  return new NativeStruct(args);
}

NativeStruct::NativeStruct(CL_List<Pair*>& args) : NativeInstance("Struct") {
  char* name;
  Expr* expr;
  DO(args, Pair*, item)
    _elements.Add(item->Copy());
  OD
}

NativeStruct::NativeStruct(CL_List<Expr*>& args) : NativeInstance("Struct") {
   Str*        name;
   Literal*    expr;
  /* elements must be pairs of <Str*, Expr*> */
  if(args.Size() % 2) {   // number of elements must be divisible by 2 !
    cerr << "Element list to ctor for NativeStruct must consists of pairs <char*, Expr*> !" << endl;
    return;
  }			
  for(int i=0; i < args.Size(); i+=2) {
    name=(Str*)args[i];
    expr=(Literal*)args[i+1];
    if(name && expr)
      _elements.Add(new Pair(name->GetStr(), expr));
  }
}

NativeStruct::NativeStruct() : NativeInstance("Struct") {}


NativeStruct::~NativeStruct() {
  DO(_elements, Pair*, item)
    delete item;
  OD
}


NativeStruct* NativeStruct::Copy() {
  return new NativeStruct(_elements);
}


void NativeStruct::Print(ostream& os) {
  short  first=1;
  char*  name;
  Expr*  expr;
  os << "{";

  DO(_elements, Pair*, item)
    if(!first)
      os << ", ";
    else
      first=0;
    item->Print(os);
  OD
  os << "}";
}


void NativeStruct::Inspect(ostream& os) {
  Print(os);
  os << "\nMethods:\n--------:\n";
  PrintMethods(os);
  os << endl;
}


Literal* NativeStruct::GetVal(char* name) {
  DO(_elements, Pair*, item)
    if(!strcmp(item->GetName(), name))
      return item->GetLiteral();
  OD
  return 0;
}


void NativeStruct::SetVal(char* name, Literal& v) {
  DO(_elements, Pair*, item)
    if(!strcmp(item->GetName(), name)) {
      item->SetLiteral(v);
      return;
    }
  OD
}


Symbol* NativeStruct::Find(char* name) {
  Literal* l=GetVal(name);
  if(!l)
    return 0;
  return new NativeSymbol(name, this, l->Copy());
}


void NativeStruct::Dump(ostream& os) {
  Literal*  tmpval=0;
  char*     tmpname=0;
  os << NATIVE_INST_TYPE << " Struct";
  os << START_DEL;
  
  DO(_elements, Pair*, item)
    tmpname=item->GetName();
    tmpval=item->GetLiteral();
    if(tmpname && tmpval) {
      os << tmpname << START_DEL;
      tmpval->Dump(os);
      os << END_DEL;
    }
  OD

  os << END_DEL;
}


void NativeStruct::Read(istream& is) {
  char         tmpname[0xff];
  Literal*     lit=0;
  Pair*        p;
  while(is) {
    DynamicString buf;
    is >> tmpname;
    if(!is)
      break;
    ReadValue(is, buf);
    lit=::CreateLiteralFromString(buf);
    p=new Pair(tmpname, lit);
    delete lit;  // was copied by Pair's ctor
    _elements.Add(p);
  }
}




Expr* NativeStruct::Length(CL_List<Expr*>&) {
  return new Int(_elements.Size());
}


Expr* NativeStruct::Add(CL_List<Expr*>& args) {
  int        s;
  Str*       name;
  Literal*   expr;
  if((s=args.Size()) % 2) {
    cerr << "Argument list to NativeStruct::Add() must be a sequence of pairs !" << endl;
    return 0;
  }
  for(int i=0; i < s; i+=2) {
    name=(Str*)args[i];
    expr=(Literal*)args[i+1];
    if(name && expr)
      _elements.Add(new Pair(strdup(name->GetStr()), expr), ::tail, ::replace);
  }
  return 0;
}


Expr* NativeStruct::At(CL_List<Expr*>& args) {
  Int* index=(Int*)args[0];
  Expr* ret=_elements[(long)index->GetInt()];
  return ret? ret->Copy() : 0;
  return 0;
}

Expr* NativeStruct::AtPut(CL_List<Expr*>& args) {
  if(args.Size() != 2) {
    cerr << "Struct::AtPut() has 2 parameters (attrname, new_value) !" << endl;
    return 0;
  }
  Str*      n=(Str*)args[0];
  Literal*  v=(Literal*)args[1];

  DO(_elements, Pair*, item)
    if(!strcmp(item->GetName(), n->GetStr()))
      item->SetLiteral(*v);
  OD
  return 0;
}


Expr* NativeStruct::AddFirst(CL_List<Expr*>& args) {
  Str*       name;
  Literal*   expr;
  if(args.Size() %2) {
    cerr << "Arguments to NativeStruct::AddFirst() must be a number of pairs !" << endl;
    return 0;
  }
  for(int i=0; i < args.Size(); i+=2) {
    name=(Str*)args[i];
    expr=(Literal*)args[i+1];
    if(name && expr)
      _elements.Add(new Pair(strdup(name->GetStr()), expr), ::head, ::replace);
  }
  return 0;
}


Expr* NativeStruct::Delete(CL_List<Expr*>& args) {
  int index=(int)((Int*)args[0])->GetInt();
  _elements.RemoveAt(index);
  return 0;
}



///////////////////////////////////////////////////////////////////////////////////////
///////////////////////////   NativeUnion  ////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////


NativeInstance* MakeNativeUnion(CL_List<Expr*>& args) {
  return new NativeUnion(args);
}



NativeUnion::NativeUnion(char* name, Literal* v) 
  : NativeInstance("Union"), _name(0), _val(0) {
    if(name)
      _name=strdup(name);
    if(v)
      _val=v->Copy();
  }

NativeUnion::NativeUnion(CL_List<Expr*>& args) 
  : NativeInstance("Union"), _name(0), _val(0) {
    if(args.Size() == 2) {
      Str*      name=(Str*)args[0];
      Literal*  val=(Literal*)args[1];
      _name=strdup(name->GetStr());
      _val=val->Copy();
    }  
  }


NativeUnion::~NativeUnion() {
  delete [] _name;
  delete _val;
}

NativeUnion* NativeUnion::Copy() {
  return new NativeUnion(_name, _val);
}

void NativeUnion::Print(ostream& os) {
  char*  name;
  Expr*  expr;
  os << "{";
  if(_name)
    os << _name << ": ";
  else
    os << "<unnamed>: ";
  if(_val)
    _val->Print(os);
  else
    os << "<novalue>";
  os << "}";
}

void NativeUnion::Inspect(ostream& os) {
  Print(os);
  os << "\nMethods:\n--------:\n";
  PrintMethods(os);
  os << endl;  
}

Literal* NativeUnion::GetVal(char* name) {
  if(!_name)
    return 0;
  if(!name)
    return _val;
  if(!strcmp(_name, name))
    return _val;
  return 0;
}

void NativeUnion::SetVal(char* name, Literal& v) {
  if(name) {
    delete [] _name;
    delete _val;
    _name=strdup(name);
    _val=&v;
  }
}

Symbol* NativeUnion::Find(char* name) {
  if(_name && name) {
    Literal*      retval=0;
    if(!strcmp(_name, name)) {
      retval=_val? _val->Copy() : 0;
      return new NativeSymbol(_name, this, retval);
    }
    else
      return new NativeSymbol(strdup(name), this, new UndefinedLiteral);
  }
  return 0;
}


void NativeUnion::Dump(ostream& os) {
  os << NATIVE_INST_TYPE << " Union ";
  os << START_DEL;
  
  if(_name && _val) {
    os << _name << " ";
    os << START_DEL;
    _val->Dump(os);
    os << END_DEL;
  }
  
  os << END_DEL;
}

void NativeUnion::Read(istream& is) {
  DynamicString buf;
  char         name[0xff];
  Literal*     lit=0;
  is >> name;
  if(!is)
    return;
  ReadValue(is, buf);
  lit=::CreateLiteralFromString(buf);
  if(lit)
    SetVal(name, *lit);
}
