#ifndef CL_LIST
#define CL_LIST
#include <iostream.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <new.h>

/*---------------------------------------------------------------------------*/
/*  Implementation of a doubly linked list by means of templates. The        */
/*  following member functions of class T have to be defined:                */
/*  - ostream& operator<<(ostream&, T&):                                     */
/*  - ostream& operator<<(ostream&, T*):                                     */
/*    ==================================                                     */
/*    If subclasses will be defined, it is best to define a virtual function */
/*    such as Print(ostream&) that should be called from the above function. */
/*    This has to be done since the '<<'-operator is *not* virtual !         */
/*  - copy constructor       --> T(T&);                                      */
/*  - assignment operator    --> operator=(T);                               */
/*  - operator== *should* be defined                                         */
/*  - IsEqual(T, T) if default comparison is not suitable                   */
/*---------------------------------------------------------------------------*/

template<class T> class CL_Node;
template<class T> class CL_List;

#ifdef _AIX  // compiler on AIX doesn't yet support type 'bool'
enum bool            {false=0, true};
#endif

enum insertion_order {tail, head};
enum insertion_mode  {normal,  replace};


/*----------------------------------------------------------------------------*/
/* Template functions.                                                        */
/* If default comparison (operator==) is not suited,                          */
/* specify this function !                                                    */
/*----------------------------------------------------------------------------*/

#ifdef __linux__
template<class T> static bool IsEqual(T a, T b) {
  return a==b? true : false;
}
#else
template<class T> bool IsEqual(T a, T b) {
  return a==b? true : false;
}

/* Default comparison for strings: */
bool IsEqual(char* a, char* b) {
  return !strcmp(a, b)? true : false;
}

template<class T> void Destroy(T element) {
  ;
}

void Destroy(char* element) {
  delete [] element;
}

#endif


/*----------------------------------------------------------------------------*/
/* CL_Node                                                                    */
/*----------------------------------------------------------------------------*/

template<class T> class CL_Node {
  CL_Node<T>*         next;
  CL_Node<T>*         prev;
  T                   element;
  public:
                      CL_Node(T&);
                      ~CL_Node()          {}
  CL_Node<T>*&        GetNext()           {return next;}
  CL_Node<T>*&        GetPrev()           {return prev;}
  T                   GetElement()        {return element;}

  #ifdef __linux__
  void                SetElement(T el)    {element=el;}
  #else
  void                SetElement(T el)    {Destroy(element); element=el;}
  #endif

  friend ostream&     operator<<(ostream&, CL_Node<T>&);
  friend ostream&     operator<<(ostream&, CL_Node<T>*);
};

/*----------------------------------------------------------------------------*/
/* CL_List                                                                    */
/*----------------------------------------------------------------------------*/

template<class T> class CL_List {
protected:
  long                size;
  CL_Node<T>*         first;
  CL_Node<T>*         last;
  CL_Node<T>*         GetNode(long index);
  public:
                      CL_List();
                      ~CL_List();
  bool                Add(T, insertion_order=::tail, insertion_mode m=normal);
  T                   Remove(T);
  void                Remove(CL_Node<T>&);
  virtual void        RemoveAll();
  void                RemoveAt(long index);
  long                Size()       {return size;}
  CL_Node<T>*&        GetFirst()   {return first;}
  CL_Node<T>*&        GetLast()    {return last;}
  CL_Node<T>*         Find(T);
  T                   operator[](long);
  bool                Replace(long index, T element);
  friend ostream&     operator<<(ostream&, CL_List<T>&);
  friend ostream&     operator<<(ostream&, CL_List<T>*);
  void                Apply(int (*foo)(T));
};

/*----------------------------------------------------------------------------*/
/* Macro definitions                                                          */
/*----------------------------------------------------------------------------*/
// Both macros are put in a separate block so that vars with the same name
// can be used all the time...

#define DO(list, type, name)\
   {for(CL_Node<type>* node=list.GetFirst(); node; node=node->GetNext())\
     {type name=node->GetElement();
#define OD }}

#define DOP(list, type, name)\
   {for(CL_Node<type>* node=list->GetFirst(); node; node=node->GetNext())\
     {type name=node->GetElement();
#define ODP }}


/*----------------------------------------------------------------------------*/
/* Implementation                                                  */
/*----------------------------------------------------------------------------*/

template<class T> CL_Node<T>::CL_Node(T& el) : next(0), prev(0), element(el) {}

template<class T> ostream& operator<<(ostream& os, CL_Node<T>& n) {
  return os << n.GetElement();
}

template<class T> ostream& operator<<(ostream& os, CL_Node<T>* n) {
  return os << n->GetElement();
}


template<class T> CL_Node<T>* CL_List<T>::GetNode(long index) {
  long count=0;
  if(index < 0 || index >= size) {
    cerr << "Index " << index << " GetNode() is out of bounds!\n";
    return 0;
  }
  CL_Node<T>* tmp=first;
  while(count++ < index)
    tmp=tmp->GetNext();
  return tmp;
}

template<class T> CL_List<T>::CL_List() : size(0), first(0), last(0) {}

template<class T> CL_List<T>::~CL_List() {
  RemoveAll();
}

template<class T> bool CL_List<T>::Add(T el, insertion_order o, insertion_mode m) {
  // 1. list is empty
  if(!first) {
    first=new CL_Node<T>(el);
    if(!first) return false;
    last=first;
    size=1;
    return true;
  }

  if(m == ::replace)            // don't add if already present
    if(Find(el))
      return false;

  // 2. insertion at tail
  if(o==::tail) {
    last->GetNext()=new CL_Node<T>(el);
    if(!last->GetNext()) return false;
    last->GetNext()->GetPrev()=last;
    last=last->GetNext();
    size++;
    return true;
  }
  // 3. insertion at head
  if(o==::head) {
    CL_Node<T>* tmp=new CL_Node<T>(el);
    if(!tmp) return false;
    tmp->GetNext()=first;
    first->GetPrev()=tmp;
    first=tmp;
    size++;
    return true;
  }
  return true;
}

template<class T> T CL_List<T>::Remove(T el) {
  CL_Node<T>*    tmp=Find(el);
  T*             t=0;              
  if(!tmp) return *t;
  T tt=tmp->GetElement();
  Remove(*tmp);
  return tt;
}

template<class T> void CL_List<T>::Remove(CL_Node<T>& n) {
  if(&n==first) {
    if(n.GetNext()) {
      first=n.GetNext();
      first->GetPrev()=0;
    }
    else
      first=last=0;
  }
  /* 2. at tail */
  else {
    if(&n==last) {
      if(last->GetPrev()) {
        last=last->GetPrev();
        last->GetNext()=0;
      }
      else
        last=first=0;
    }
    /* 3. normal deletion */
    else {
      n.GetPrev()->GetNext()=n.GetNext();
      n.GetNext()->GetPrev()=n.GetPrev();
    }
  }
  /* 4. delete node itself */
  delete &n;
  size--;
}

template<class T> void CL_List<T>::RemoveAll() {
  if(Size() < 1)
    return;
  /* Remove all nodes by deleting the first until there are no more nodes */
  while(size)
    Remove(*first);
}


template<class T> void CL_List<T>::RemoveAt(long index) {
  CL_Node<T>* rem=GetNode(index);
  if(rem)
    Remove(*rem);
}


template<class T> CL_Node<T>* CL_List<T>::Find(T el) {
  for(CL_Node<T>* tmp=first; tmp; tmp=tmp->GetNext())
    if(IsEqual(tmp->GetElement(), el))
       return tmp;
  return 0;
}


template<class T> T CL_List<T>::operator[](long index) {
  long count=0;
  if(index<0 || index >= size) {
    cerr << "Index " << index << " of operator \"[]\" is out of bounds!\n";
    T* dummy=0;
    return *dummy;
  }
  CL_Node<T>* tmp=first;
  while(count++ < index)
    tmp=tmp->GetNext();
  return tmp->GetElement();
}


template<class T> bool CL_List<T>::Replace(long index, T new_element) {
  CL_Node<T>* found=GetNode(index);
  if(found) {
    found->SetElement(new_element);
    return true;
  }
  return false;
}


template <class T> ostream& operator<<(ostream& os, CL_List<T>& l) {
  for(CL_Node<T>* tmp=l.GetFirst(); tmp; tmp=tmp->GetNext())
    os << *tmp << "\n";
  return os;
}

template <class T> ostream& operator<<(ostream& os, CL_List<T>* l) {
  for(CL_Node<T>* tmp=l->GetFirst(); tmp; tmp=tmp->GetNext())
    os << *tmp << "\n";
  return os;
}

template<class T> void CL_List<T>::Apply(int (*foo)(T)) {
  for(CL_Node<T>* tmp=first; tmp; tmp=tmp->GetNext())
    foo(tmp->GetElement());
}



#endif
