
#include <string.h>
#include <memory.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <fstream.h>
#include <strstream.h>
#include "Input.h"
#include "Debug.h"

#ifdef __linux__
  #include <dlfcn.h>
  #define LIB_PATH "LD_ELF_LIBRARY_PATH"
#else
  #include <load.h>
  #define LIB_PATH "LIBPATH"
#endif

extern int yyparse();

#include "Grammar.h"
#include "SymTab.h"
#include "vbx/defs.h"

#define DLL_EXT  ".dll"

extern SymbolTable   symtab;
extern SymbolTable   classes;

short                interactive=1;
short                prompt=1;

#ifdef __linux__
/* Important ! Forward template class declaration for GNU G++ compiler.
  Won't link if omitted ! */
template class CL_List<DLL_Info*>;
CL_List<DLL_Info*>   dll_handles;

#else
CL_List<DLL_Info*>   dll_handles;
#endif


int file_desc=0;

/*
   ""Bela""    ->   "Bela"
    "Bela"     ->   "Bela"
*/

#ifdef __linux__
void* LoadLinuxSharedLib(char* n) {
    void    (*EntryPoint)()=0;
    char*   err=0;
    void*   handle=dlopen(n, RTLD_LAZY);
    if(!handle) {
	cerr << dlerror() << endl;
	return 0;
    }
    EntryPoint=(void (*)())dlsym(handle, "EntryPoint__Fv");
    if((err=(char*)dlerror()) != 0) {
	cerr << err << endl;
	dlclose(handle);
	return 0;
    }
    return EntryPoint;
}
#endif


void ParseString(DynamicString& d) {
    char          c;
    short         esc=0;
    while((c=Input::GetInput()->GetChar()) != EOF) {
	if(esc) {
	    d << c;
	    esc=0;
	    continue;
	}
	switch(c) {
	case '\\':
	    esc=1;
	    continue;
	case '\"':
	    return;
	default:
	    d << c;		
	}
    }
}


void InterpretString(char* s) {
    Input* inp=Input::GetInput();

    inp->Push(s);
    inp->IncrNestingLevel();

    /* ---------------------------------------------------------------------------
       We HAVE to make a recursive call to yyparse(), otherwise the following will
       happen: the string will be pushed onto the Input's stack though, but it will 
       NOT be used until we return from our window (./vbx/Win.dll) !
       Therefore we have to make the interpreter interpret the string now !
       We must ensure that yyparse() is reentrant (./Docs/RecursiveParser.txt).
       The recursively called yyparse() can modify state by referring to a global
       instance of Interpreter. When yyparse() returns, we will continue as normal.
       If the recursive call to yyparse() is given stdin, then it will NOT return.
       Therefore we should use strings or files !
       --------------------------------------------------------------------------- */


    yyparse();
}

char* RemoveQuotes(char* s) {
  if(!s)
    return 0;
  char* buf=new char[strlen(s)+2];
  memset(buf, 0, strlen(s)+2);
  if(*s == '"') s++;
  strcpy(buf, s);
  if(buf[strlen(buf)-1] == '"') buf[strlen(buf)-1]=0;
  return buf;
}


void DumpSymbolTable(char* filename) {
    ofstream ofs(filename);
    if(ofs)
	::symtab.Dump(ofs);
}

void ReadSymbolTable(char* filename) {
    ifstream ifs(filename);
    if(ifs)
        ::symtab.Read(ifs);
}


void SetScript(char* filename) {
  ifstream    ifs(filename);
  ostrstream  ostr((char*)0, 0xff);
  while(ifs)
    ostr << (char)ifs.get();
    
  if(::agent_script && ostr.str() && strlen(ostr.str()))
    delete [] ::agent_script;

  if(ostr.str() && strlen(ostr.str()) > 0)
    ::agent_script=strdup(ostr.str());
}


void LoadFile(char* filename) {
  char           buf[0xff];
  if(!filename) 
      return;
  char*          fname=RemoveQuotes(filename);
  FILE*          fp=fopen(fname, "r");
  if(!fp)
    cerr << "File " << fname << " not found !" << endl;
  else
    Input::GetInput()->Push(fp);
  delete [] fname;
}


char* Op2Str(Operator op) {
  switch(op) {
  case op_plus:
    return "+";
  case op_minus:
    return "-";
  case op_mult:
    return "*";
  case op_div:
    return "/";
  case op_mod:
    return "mod";
  default:
    return "<no-op>";
  }
  return "<no-op>";
}

char* CompOp2Str(CompOperator op) {
  switch(op) {
    case lt:
    return "<";
  case lteq:
    return "<=";
  case gt:
    return ">";
  case gteq:
    return ">=";
  case eq:
    return "==";
  case not_eq:
    return "<>";
  default:
    return "<undefined>";
  }
}

char* BoolOp2Str(BoolOperator op) {
  switch(op) {
  case and_op:
    return "and";
  case or_op:
    return "or";
  case not_op:
    return "not";
  default:
    return "<undefined>";
  }
}


Literal* RetrieveFunctionFromSymbolTable(char* funcname, SymbolTable& context) {
  Literal* f;
  Symbol* s=context.Find(funcname);

  if(!s || ((f=(Literal*)s->GetVal()) == 0))
    return 0;
  if(f->type != FUNC_TYPE && f->type != NATIVE_FUNC_TYPE) {
    cerr << "Symbol '" << funcname << " is not a function !" << endl;
    return 0;
  }  
  return f->Copy();
}



short LoadVBXs(char* directory) {
  dirent*     entry;
  char*       dllname;
  short       ret=1;

  DIR*        d=opendir(directory);
  if(!d) {
    cerr << "Directory " << directory << " could not be opened !" << endl;
    return 0;
  }  
  while(entry=readdir(d)) {
    dllname=entry->d_name;
    if(strstr(dllname, DLL_EXT)) {
      cout << "--> Loading " << dllname << "..."; cout.flush();
      if(LoadDLL(dllname, ::symtab))
	cout << "Done." << endl;
      else {
	ret=0;
	cerr << "Failed." << endl;
      }
    }      
  }
  return ret;
}



short UnloadVBXs() {
  short t=::dll_handles.Size();
  DO(::dll_handles, DLL_Info*, item)
    if(item->unload_func)
      (*item->unload_func)();
    #ifdef _AIX
    ::terminateAndUnload((int (*)())item->dll_handle);
    #else
    dlclose(item->dll_handle);
    #endif
    delete item;
  OD
  if(t)
    cout << "VBxs were unloaded." << endl;
  ::dll_handles.RemoveAll();
  return 1;
}




short LoadDLL(char* dllname, SymbolTable&) {
  FuncDescr*          funcs;
  ClassDescr*         classes;
  NativeFunction*     nf;
  NativeClassDef*     ncd;
  Symbol*             s;
  static char         buf[1024];
  void*               dll_handle=0;
  char*               tmpdir;
  memset(buf, '\0', sizeof(buf));

#ifdef _AIX
  char*               old_libpath=getenv(LIB_PATH);
  char*               new_libpath=0;
  char*               vbx_path=getenv("VBXDIR") ? strdup(getenv("VBXDIR")) : 0;
  if(!old_libpath)
    old_libpath=".:/lib:/usr/lib";
  if(!vbx_path) {
    tmpdir=getcwd(buf, sizeof(buf));
    vbx_path=new char[strlen(tmpdir)*2 + 10];
    sprintf(vbx_path, "%s:%s/%s", tmpdir, tmpdir, "vbx");
  }

  new_libpath=new char[strlen(old_libpath) + strlen(vbx_path) + 10];
  sprintf(new_libpath, ".:%s:%s", old_libpath, vbx_path);
  dll_handle=::loadAndInit(dllname, 1, new_libpath);
  delete [] new_libpath;
  delete [] vbx_path;

#else

  tmpdir=getenv("VBXDIR")? getenv("VBXDIR") : 0;
  char* full_name=0;

  if(!tmpdir) {
      tmpdir=getcwd(buf, sizeof(buf));
      full_name=new char[strlen(tmpdir) + strlen(dllname) + 10];
      sprintf(full_name, "%s/vbx/%s", tmpdir, dllname);
  }
  else {
      full_name=new char[strlen(tmpdir)+strlen(dllname)+10];
      sprintf(full_name, "%s/%s", tmpdir, dllname);
  }
  dll_handle=LoadLinuxSharedLib(full_name);
  delete [] full_name;

#endif

  if(dll_handle == 0) {
    #ifdef _AIX
    perror("");
    #endif
    return 0;
  }

  Contents*           (*Fn)()=(Contents* (*)())dll_handle;
  Contents*           contents=0;
  Ident*              superclass;
  CL_List<Method*>*   method_list;
  Method*             meth;
  CL_List<BType>*     type_list;
  
  if(!Fn)
    return 0;
  DLL_Info* dll_info=new DLL_Info;
  dll_info->unload_func=0;
  dll_info->dll_handle=dll_handle;
  ::dll_handles.Add(dll_info);

  if(!(contents=Fn())) {
    cerr << "Contents of " << dllname << " could not be retrieved ! " << endl;
    return 0;
  }

  if(contents->load_func)
    (*contents->load_func)();
  if(contents->unload_func)
    dll_info->unload_func=contents->unload_func;

  /* Load functions if available */
  if((funcs=contents->functions)) {
    for(FuncDescr* tmp=funcs; tmp->name; tmp++) {
      if(!tmp->parm_types)
	nf=new NativeFunction(tmp->name, tmp->addr, tmp->number_of_args, 0);
      else {
	CL_List<BType> types;
	BType*         t=tmp->parm_types;
	for(int i=0; i < tmp->number_of_args; i++)
	  types.Add(*t++);
	nf=new NativeFunction(tmp->name, tmp->addr, tmp->number_of_args, &types);
      }
      s=::symtab.Lookup(tmp->name);
      s->SetVal((Literal&)*nf);
    }
  }

  /* Load classes if available */
  if((classes=contents->classes)) {
    for(ClassDescr* cd=classes; cd->name; cd++) {
      superclass=cd->superclass? new Ident(cd->superclass) : 0;
      method_list=cd->methods ? new CL_List<Method*> : 0;
      if(method_list) {
	for(MethodDescr* md=cd->methods; md->name; md++) {
	  type_list=md->parm_types? new CL_List<BType> : 0;
	  if(type_list) {
	    for(BType* t=md->parm_types; t; t++)
	      type_list->Add(*t);
	  }
	  meth=new NativeMethod(md->name, md->mptr, 0, md->number_of_formal_parms, type_list);
	  method_list->Add(meth);
	}
      }
      ncd=new NativeClassDef(cd->name, cd->ctor, superclass, (CL_List<Function*>*)method_list);
      ::classes.Lookup(cd->name)->SetVal((Literal&)*ncd);
    }
  }
  return 1;
}



short ReloadVBXs(char* directory) {
  UnloadVBXs();
  short ret=LoadVBXs(directory);  
  ::symtab.Print(cout);
  return ret;
}


void InspectInstance(Expr& ex, SymbolTable& context) {
  Expr* result=ex.Interpret(context);
  Instance* inst;
  if(!result)
    return;
  switch(result->type) {
  case INST_TYPE:
  case NATIVE_INST_TYPE:
    inst=(Instance*)result;
    inst->Inspect();
    break;
  default:
    cerr << "Value to be inspected is not an instance !" << endl;
    break;
  }
  delete result;


}



ClassDef* RetrieveMetaclass(char* classname) {
  Symbol*    s=::classes.Find(classname);
  ClassDef*  cl_def;
  if(!s) {
    cerr << "Class " << classname << " not found !" << endl;
    return 0;
  }
  if(!s->GetVal() || (s->GetVal()->type != CLASS_TYPE && s->GetVal()->type != NATIVE_CLASS_TYPE)) {
    cerr << classname << " not bound or not a class !" << endl;
    return 0;
  }
  cl_def=(ClassDef*)s->GetVal();
  return cl_def;
}


/* ---------------------------------------------------------------------------
                                  Statements
   --------------------------------------------------------------------------- */

RetVal::~RetVal() {
  delete ex;
}

Stmt::Stmt(Ident* id, Expr* ex) {
  ident=id;
  expr=ex;
  stmt_type=NORMAL;
}

Stmt::~Stmt() {
//  delete ident;  // +++ temporary ! +++
  delete expr;
}

Stmt& Stmt::operator=(Stmt& s) {
  if(this == &s) return *this;
  if(ident) delete ident;
  if(expr)  delete expr;
  ident=s.ident;
  expr=s.expr;
  return *this;
}

void Stmt::Print(ostream& os) {
  if(ident)
    ident->Print(os); 
  if(expr) {
    os << " = ";
    expr->Print(os);
    os << ";";
  }
  else
    os << ";";
  os << endl;
}

void Stmt::Trace(ostream& os) {
  if(ident)
    ident->Print(os); os << " = ";
  if(expr)
    expr->Print(os);
  os << endl;
}

/*
   Normal statements don't return an expression. The ret val should always be 0.
   Return stataments return an expression or 0.
*/
RetVal* Stmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug deb(*this);
#endif
  Literal* tmp=(Literal*)expr->Interpret(context, this_ptr); // expr must always eval to a Literal
  if(!tmp)
    return 0;
  if(!ident) {
    if(interactive)
      cout << "> "; 
    tmp->Print(cout); // +++
    if(interactive) {
      cout << endl << "> "; 
      prompt=0;
    }
    cout.flush();
    delete tmp;  // therefore Interpret always has to return a *copy* of the expr !!
    return 0;
  }
  /* Assignment expression */
  Symbol* ret=ident->GetSymbol(context);    
  if(ret) {
    ret->SetVal(*tmp);
    if(ret->_sym_type == native_sym)
      delete ret;
  }
  else {
    cerr << "Symbol ";
    ident->Print(cerr);
    cerr << " not found !" << endl;
  }
  cout.flush();
  return 0;
}

Stmt* Stmt::Copy() {
  if(ident)
    return new Stmt(ident->Copy(), expr->Copy());
  else
    return new Stmt(0, expr->Copy());
}


VarsStmt::VarsStmt(CL_List<Ident*>& vars) {
  DO(vars, Ident*, item)
    _vars.Add(item);
  OD
}

VarsStmt::~VarsStmt() {
  DO(_vars, Ident*, item)
    delete item;
  OD
}

RetVal* VarsStmt::Interpret(SymbolTable& context, Instance* this_ptr) { /* Create local variables */
#ifdef TRACE
  Debug  deb(*this);
#endif
  DO(_vars, Ident*, item)
    context.Lookup(item->GetName());
  OD
  return 0;
}


void VarsStmt::Trace(ostream& os) {
  os << "vars: ";
  short first=1;
  DO(_vars, Ident*, item)
    item->Print(os);
    if(first)
      first=0;
    else
      os << ", ";
  OD
  os << ";" << endl;
}


RetVal* PrintStmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug  deb(*this);
#endif
  Expr*  tmp;
  if(expr) {
    tmp=expr->Interpret(context, this_ptr);
    if(tmp) {
      tmp->Print(cout);
      delete tmp;
    }
  }
  if(_cgi)
    printf("%c%c", 10, 10);
  if(_println)
    cout << endl;  
  return 0;
}


void PrintStmt::Print(ostream& os) {
  if(expr)
    expr->Print(os);
}


void PrintStmt::Trace(ostream& os) {
  if(_println)
    os << "println ";
  else
    os << "print ";
  if(expr)
    expr->Print(os);
  os << ";" << endl;
}


PrintStmt* PrintStmt::Copy() {
  Expr* e=expr? expr->Copy() : 0;
  PrintStmt* ret=new PrintStmt(e, _println, _cgi);
  return ret;
}


RetVal* EvalStmt::Interpret(SymbolTable& context, Instance* i) {
#ifdef TRACE
    Debug  deb(*this);
#endif
    char*  eval_str=0;
    Expr*  tmp=0;
    if(expr) {
	switch(expr->type) {
	case STRING_TYPE:
	    eval_str=((Str*)expr)->GetStr();
	    break;
	case IDENT_TYPE:
	    tmp=expr->Interpret(context, i);
	    if(tmp->type == STRING_TYPE) {
		eval_str=((Str*)tmp)->GetStr();
		break;
	    }
	default:
	    cerr << "Identifier does not evaluate to a string !" << endl;	    
	}
	if(eval_str)
	    InterpretString(eval_str);
	if(tmp)
	    delete tmp;
    }

    return 0;
}

void EvalStmt::Print(ostream& os) {
  if(expr)
    expr->Print(os);
}



void EvalStmt::Trace(ostream& os) {
  os << "eval ";
  if(expr) {
    os << "\"";
    expr->Print(os);
    os << "\"";
  }
  os << ";" << endl;
}



EvalStmt* EvalStmt::Copy() {
    Expr* e=expr? expr->Copy() : 0;
    return new EvalStmt(e);
}


ReturnStmt* ReturnStmt::Copy() {
  if(expr)
    return new ReturnStmt(expr->Copy());
  else
    return new ReturnStmt;
}



void ReturnStmt::Trace(ostream& os) {
  os << "return ";
  if(expr)
    expr->Print(os);
  os << ";" << endl;
}


RetVal* ReturnStmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug  deb(*this);
#endif
  return expr? new RetVal(RET_STMT, expr->Interpret(context, this_ptr)) : new RetVal(RET_STMT);
}



WhileStmt::WhileStmt(BoolExpr* cond, CL_List<Stmt*>* stmt_list) {
  stmt_type=WHILE_STMT;
  _cond=cond;
  if(stmt_list) {
    DOP(stmt_list, Stmt*, item)
      _stmt_list.Add(item->Copy(), ::tail);
    ODP
  }
}


WhileStmt::~WhileStmt() {
  delete _cond;
  DO(_stmt_list, Stmt*, item)
    delete item;
  OD
}


RetVal* WhileStmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug       deb(*this);
#endif
  if(!_cond)  return 0;
  gomBool*    condition;
  RetVal*     ex;
  short       is_done=0;

  _local_vars.SetPrevScope(context);
  while(!is_done) {
    condition=(gomBool*)_cond->Interpret(context, this_ptr);
    if(!condition->IsTrue())
      break;
    DO(_stmt_list, Stmt*, item)
      if(is_done)
	break;
      ex=item->Interpret(_local_vars, this_ptr);
      if(ex) {
	switch(ex->type) {
	case RET_STMT:
	  delete condition;
	  return ex;
	case BREAK_STMT:
	  delete ex;
	  is_done=1;
	  break;
	default:
	  delete ex;
	  break;
	}
      }
    OD
  }
  delete condition;
  return 0;
}




void WhileStmt::Print(ostream& os) {
  os << "WHILE-stmt" << endl;
}


void WhileStmt::Trace(ostream& os) {
  os << "while(...) {" << endl;
}


WhileStmt* WhileStmt::Copy() {
  WhileStmt* ret=new WhileStmt(_cond->Copy(), &_stmt_list);  // latter is copied
  return ret;
}


IfStmt::IfStmt(BoolExpr* cond, CL_List<Stmt*>* if_part, CL_List<Stmt*>* else_part) {
  stmt_type=IF_STMT;
  _cond=cond;
  if(if_part) {
    DOP(if_part, Stmt*, item)
      _if_part.Add(item, ::tail);
    ODP
  }
  if(else_part) {
    DOP(else_part, Stmt*, item)
      _else_part.Add(item, ::tail);
    ODP
  }
}


IfStmt::~IfStmt() {
  delete _cond;
  DO(_if_part, Stmt*, item)
    delete item;
  OD
  DO(_else_part, Stmt*, item)
    delete item;
  OD
}


RetVal* IfStmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug     deb(*this);
#endif
  gomBool*  condition=(gomBool*)_cond->Interpret(context, this_ptr);
  RetVal*   ex;
  _local_vars.SetPrevScope(context);
  if(condition->IsTrue()) {  /*  if-part    */
    DO(_if_part, Stmt*, item)
      ex=item->Interpret(_local_vars, this_ptr);
      if(ex) {
	switch(ex->type) {
	case RET_STMT:
	case BREAK_STMT:
	  delete condition;
	  return ex;
	default:
	  delete ex;
	  break;
	}
      }
    OD
  }
  else {                     /*  else-part  */
    DO(_else_part, Stmt*, item)
      ex=item->Interpret(_local_vars, this_ptr);
      if(ex) {
	switch(ex->type) {
	case RET_STMT:
	case BREAK_STMT:
	  delete condition;
	  return ex;
	default:
	  delete ex;
	  break;
	}
      }
    OD
  }
  return 0;
}


void IfStmt::Print(ostream& os) {
  os << "IF-stmt" << endl;
}


void IfStmt::Trace(ostream& os) {
  os << "if(";
  if(_cond)
    _cond->Print(os);
  os << ") { ..." << endl;
}



IfStmt* IfStmt::Copy() {
  CL_List<Stmt*> if_part_copy, else_part_copy;
  DO(_if_part, Stmt*, item)
    if_part_copy.Add(item->Copy(), ::tail);
  OD
  DO(_else_part, Stmt*, item)
    else_part_copy.Add(item->Copy(), ::tail);
  OD
  IfStmt* ret=new IfStmt(_cond->Copy(), &if_part_copy, &else_part_copy);
  return ret;
}


ForStmt::ForStmt(Ident& id, Expr& start, int up_or_down, Expr& end, CL_List<Stmt*>* stmt_list) {
  stmt_type=FOR_STMT;
  ident=&id;
  _a=_b=0;  // will be set when evaluated
  _start=&start;
  _end=&end;
  _up_or_down=up_or_down;
  if(stmt_list) {
    DOP(stmt_list, Stmt*, item)
      _stmt_list.Add(item->Copy());
    ODP
  }
}


ForStmt::~ForStmt() {
  delete _start;
  delete _end;
  DO(_stmt_list, Stmt*, item)
    delete item;
  OD
}


RetVal* ForStmt::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug    deb(*this);
#endif
  RetVal*  ret=0, *ex=0;
  short    is_done=0;
  Symbol*  s;
  Expr*    start_expr=_start->Interpret(context, this_ptr), 
           *end_expr=_end->Interpret(context, this_ptr);
  if(start_expr->type != INT_TYPE || end_expr->type != INT_TYPE) {
    cerr << "start- or end-expr in for-stmt is not integer !" << endl;
    delete start_expr;
    delete end_expr;
    return 0;
  }
  _a=(int)((Int*)start_expr)->GetInt();
  _b=(int)((Int*)end_expr)->GetInt();  
  delete start_expr;
  delete end_expr;
  _local_vars.Lookup(ident->GetName());  // e.g. 'i' in for(i = 1 to 10) {...}
  _local_vars.SetPrevScope(context);  
  while(!is_done) {    
    if(!_up_or_down) {  // up
      if(_a > _b) {
	is_done=1;
	break;
      }	
    }
    else {              // down
      if(_a < _b) {
	is_done=1;
	break;
      }
    }    
    s=_local_vars.Find(ident->GetName());
    if(!s)
      return ret;
    s->SetVal(*new Int(_a));
    // iterate through _stmt_list
    DO(_stmt_list, Stmt*, item)
      if(is_done)
	break;
      ex=item->Interpret(_local_vars, this_ptr);
      if(ex) {
	switch(ex->type) {
	case RET_STMT:
	  return ex;
	case BREAK_STMT:
	  is_done=1;
	  delete ex;
	  break;
	default:
	  delete ex;
	  break;
	}
      }
    OD   
    _up_or_down? _a-- : _a++;
  }
  return ret;
}


void ForStmt::Print(ostream& os) {
  os << "FOR-stmt" << endl;
}


void ForStmt::Trace(ostream& os) {
  os << "for(...) {..." << endl;
}



ForStmt* ForStmt::Copy() {
  ForStmt* ret=new ForStmt(*ident->Copy(), *_start->Copy(), _up_or_down, *_end->Copy(), 
			   &_stmt_list);  
  return ret;
}

void BreakStmt::Trace(ostream& os) {
  os << "break;" << endl;
}


/* ---------------------------------------------------------------------------
                                  Expressions
   --------------------------------------------------------------------------- */

void UnaryExpr::Print(ostream& os) {
  os << Op2Str(op) << " ";
  expr->Print(os);
  os.flush();
}


Expr* UnaryExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  Literal*  tmp=(Literal*)expr->Interpret(context, this_ptr);
  Expr*     ret=0;
  if(tmp->type == INT_TYPE)
    switch(op) {
    case op_minus:
      ret=new Int(-((Int*)tmp)->GetInt());
      break;
    default:
      ret=new Int(((Int*)tmp)->GetInt());
      break;      
    }
  else
    cerr << "<operand for uminus is not integer !";
  delete tmp;
  return ret;
}


BinExpr::BinExpr(Expr* a, Operator b, Expr* c) {
  type=BIN_TYPE;
  left=a;
  op=b;
  right=c;
}

BinExpr::~BinExpr() {
  delete left; delete right;
}


/* 
   - Call Interpret on the left and right expr
   - Combine the result to a new expr and return it
   Assumption: left and right are present.
             : Interpret on left and right *always* has to return a Literal !
*/
Expr* BinExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  Int*  ret=0;
  int a, b;
  Literal* l, *r;
  l=(Literal*)left->Interpret(context, this_ptr); 
  r=(Literal*)right->Interpret(context, this_ptr);
  if(l->type == INT_TYPE) {
    if(TypeCheck(l, r, INT_TYPE)) {  // can r be assigned to l and are both of type integer ?
      switch(op) {
      case op_plus:
	ret=new Int(((Int*)l)->GetInt() + ((Int*)r)->GetInt());
	break;
      case op_minus:
	ret=new Int(((Int*)l)->GetInt() - ((Int*)r)->GetInt());
	break;
      case op_mult:
	ret=new Int(((Int*)l)->GetInt() * ((Int*)r)->GetInt());
	break;
      case op_div:
	ret=new Int(((Int*)l)->GetInt() / ((Int*)r)->GetInt());
	break;
      case op_mod:
	a=(int)((Int*)l)->GetInt();
	b=(int)((Int*)r)->GetInt();
	ret=new Int(a % b);
	break;
      default:
	break;
      }
      delete l; 
      delete r;   
    }
    return ret;
  }
  if(l->type == STRING_TYPE && r->type == INT_TYPE) {
    char* a=((Str*)l)->GetStr();
    int   b=(int)((Int*)r)->GetInt();
    char* c=new char[strlen(a)+20];
    sprintf(c, "%s%li", a, b);
    Str* retval=new Str(c);
    delete [] c;
    delete l;
    delete r;
    return retval;
  }
  if(TypeCheck(l, r, STRING_TYPE) && op == op_plus) {
    char* a=((Str*)l)->GetStr(), *b=((Str*)r)->GetStr();
    Str* retval=0;
    if(a && b) {
      char* c=new char[strlen(a) + strlen(b)+2];
      sprintf(c, "%s%s", a, b);
      retval=new Str(c);
      delete [] c;
      delete l;
      delete r;    
    }
    return retval;
  }
  
  delete l; 
  delete r;
  return 0;
}


void BinExpr::Print(ostream& os) {
  left->Print(os);
  os << " " << Op2Str(op) << " ";
  right->Print(os);
  os << endl;
}


Ident::Ident(char* id, Ident* n) : name(0), next(n) {
  type=IDENT_TYPE;
  if(id)
    name=strdup(id);
}

Ident::~Ident() {
  delete [] name;
  delete next;
}


void Ident::SetName(char* n) {
  if(n) {
    if(name)
      delete [] name;
    name=strdup(n);
  }
}

void Ident::SetNext(Ident& n) {
  if(!next)
    next=&n;
  else {
    Ident* tmp=next;
    while(tmp->next)
      tmp=tmp->next;
    tmp->next=&n;
  }
}

void Ident::Print(ostream& os) {
  if(name) os << name;
  if(next) {
    os << ".";
    next->Print(os);
  }
  os.flush();
}


Expr* Ident::Interpret(SymbolTable& context, Instance* this_ptr) {    
  Symbol*   s=GetSymbol(context);
  if(!s) {
    cerr << "Symbol ";
    Print(cerr);
    cerr << " not found !" << endl;
    return 0;
  }
  Literal*  l=s->CopyVal();
  if(s->_sym_type == native_sym)
    delete s;
  return l;
}


Symbol* Ident::GetSymbol(SymbolTable& context) {
  Symbol* ret;
  if(next)
    return GetInstanceSymbol(context);
  ret=context.Find(name);
  if(!ret)
    ret=context.Lookup(name);
  if(!ret)
    return ret;
  if(ret->GetValType() == IDENT_TYPE)  // resolve Idents as far as possible !
    ret=((Ident*)ret->GetVal())->GetSymbol(context);
  return ret;
}


Symbol* Ident::GetInstanceSymbol(SymbolTable& context) {
  Symbol*       ret=0;
  Instance*     inst;
  SymbolTable*  new_context=0;
  Ident*        next_ident;
  ret=context.Find(name);
  if(!ret) {
    cerr << name << " not found !";
    return 0;
  }
  if(ret->GetVal()->type != INST_TYPE && ret->GetVal()->type != NATIVE_INST_TYPE) {
    cerr << name << " is not an instance !" << endl;
    return 0;
  }
  inst=(Instance*)ret->GetVal();
  new_context=inst->GetContext();      
  if(next)
    ret=next->GetSymbol(*new_context);
  
  return ret;
}



Ident* Ident::Copy() {
  Ident* copy=next? next->Copy() : 0;
  return new Ident(name, copy);
}


Int::Int(double i) {
  _i=i; type=INT_TYPE;
}


Int& Int::operator=(Int& i) {
  if(this == &i)
    return *this;
  _i=i._i;
  return *this;
}

int Int::operator==(Int& i) {
  if(_i == i._i) return 0;
  if(_i < i._i)  return -1;
  return 1;
}

Expr* Int::Interpret(SymbolTable& context, Instance* this_ptr) {
  return Copy();
}


void Int::Dump(ostream& os) {
    os << INT_TYPE << " ";
    os << _i;
}


/* Remove double quotes in strings such as ""Bela""  --> "Bela" */
Str::Str(char* s) {
  type=STRING_TYPE;
  if(s)
    _s=::RemoveQuotes(s);  
  else 
    _s=0;
}

Str::Str(Str& s) {
  if(s._s)
    _s=strdup(s._s);
  else
    _s=0;
  type=STRING_TYPE;
}

Str::~Str() {
  delete [] _s;
}

Str& Str::operator=(Str& s) {
  if(this == &s)
    return *this;
  delete [] _s;
  if(s._s)
    _s=strdup(s._s);
  return *this;  
}

int Str::operator==(Str& s) {
  return strcmp(_s, s._s);
}



void Str::Dump(ostream& os) {
    os << STRING_TYPE << " ";
    if(!_s)
	return;
    os << "\"";
    char* start=_s, *end=&_s[strlen(_s)-1];
    if(*start == '"')
	start++;
    if(*end == '"')
	end--;
    while(start <= end)
	os << *start++;
    os << "\"";
}


int IsEqual(Pair* a, Pair* b) {
  return (*a == *b)? 1 : 0;
}

Pair::Pair(char* name, Literal* l) : _name(0), _literal(0) {
  type=PAIR_TYPE;
  if(name)
    _name=strdup(name);
  if(l)
    _literal=l->Copy();
}

Pair::Pair(Pair& other) {
  type=PAIR_TYPE;
  if(&other == this)
    return;
  Pair(other._name, other._literal);
}

Pair::~Pair() {
  delete [] _name;
  delete _literal;
}

Pair& Pair::operator=(Pair& other) {
  if(&other == this)
    return *this;
  if(other._name) {
    delete [] _name;
    _name=strdup(other._name);
  }
  if(other._literal) {
    delete _literal;
    _literal=other._literal->Copy();
  }
  return *this;
}

int Pair::operator==(Pair& other) {
  if((!other._name && !_name) || (other._name && _name))
    return !strcmp(other._name, _name)? 1 : 0;
  return 0;
}

Expr* Pair::Interpret(SymbolTable& context, Instance* i) {
  return Copy();
}

void Pair::Print(ostream& os) {
  if(_name)
    os << _name << ": ";
  else
    os << "<unnamed>: ";
  if(_literal)
    _literal->Print(os);
  else
    os << "<novalue>";
  os .flush();
}



Expr* Str::Interpret(SymbolTable& context, Instance* this_ptr) {
  return Copy();
}



void Str::Print(ostream& os) {
  if(!_s)
    return;
  char* start=_s, *end=&_s[strlen(_s)-1];
  if(*start == '\"')
    start++;
  if(*end == '\"')
    end--;
  while(start <= end)
    os << *start++;
  os.flush();
}


NilVal::NilVal() {
  type=NILVAL_TYPE;
}


gomBool::gomBool(short b) {
  type=BOOL_TYPE;
  if(!b)
    _bool=0;
  else
    _bool=1;
}

gomBool::~gomBool() {
  
}

Expr* gomBool::Interpret(SymbolTable& context, Instance* this_ptr) {
  return Copy();
}

void gomBool::Print(ostream& os) {
  if(_bool)
    os << "true";
  else
    os << "false";
  os.flush();
}


void gomBool::Dump(ostream& os) {
    os << BOOL_TYPE << " ";
    if(_bool)
	os << "true";
    else
	os << "false";
}


int TypeCheck(Literal* a, Literal* b, BType t, short show_error) {
  if(b->type == NILVAL_TYPE)  // NULL value can be assigned to any other value
    return 1;
  if(a->type == b->type && a->type == t && b->type == t)
    return 1;
  else
    if(show_error)
      cerr << "type mismatch between vars or var(s) undefined !" << endl;
  return 0;
}


Function::Function(char* name, CL_List<Ident*>* formal_parms, CL_List<Stmt*>* stmt_list) {
  if(name)
    _name=strdup(name);
  else
    _name=0;
  type=FUNC_TYPE;
  /* Add parameters to parameter list */  
  if(formal_parms) {
    DOP(formal_parms, Ident*, item)
      _formal_parms.Lookup(item->GetName());
    ODP
  }
  if(stmt_list) {
    DOP(stmt_list, Stmt*, item)
      _statements.Add(item, ::tail);
    ODP
  }
}


Function::~Function() {
  delete [] _name;
  DO(_statements, Stmt*, item)
    delete item;
  OD
}



/* After call, remove all entries from the symbol table ! */
 // context refers to global symbol table...
Expr* Function::Interpret(SymbolTable& context, Instance* this_ptr) { 
#ifdef TRACE
  Debug   deb(*this, _name);
#endif
  RetVal* ret=0;
  Expr*   ret_ex;
  if(type != METHOD_TYPE && type != NATIVE_METHOD_TYPE)
    _local_vars.SetPrevScope(::symtab);  // functions don't know nested scopes (unlike Pascal) !
  DO(_statements, Stmt*, item)
    ret=item->Interpret(_local_vars, this_ptr);
    if(ret) {
      switch(ret->type) {
      case RET_STMT:
	_local_vars.DeleteAll();
	ret_ex=ret->ex ? ret->ex->Copy() : 0;
	delete ret;
	return ret_ex;
	break;
      default:
	delete ret;
	return 0;
      }
    }
  OD
  _local_vars.DeleteAll();
  return 0;
}


Function* Function::Copy() {
  Function* ret=new Function(_name);
  DO(_statements, Stmt*, item)
    ret->AddStmt(*(item->Copy()));
  OD
  DO(_formal_parms.GetSymbols(), Symbol*, item)
    ret->AddParm(item->GetName());
  OD
  return ret;
}


/* 1. Bind the formal parameters to the args and add them to the symbol table
   2. Function::Interpret will later remove all bound values from the symtab again */

int Function::BindArgs(CL_List<Expr*>& args) {
  int args_size=args.Size();
  int sym_size=_formal_parms.Size();
  if(args_size != sym_size) {
    cerr << "Arglist has " << args_size << ", but formal parameter list has " << sym_size;
    cerr << " !" << endl;
    return 0;
  }
  // add the formal parameters to the local vars:
  Expr*    arg;
  Symbol*  s, *to_add;
  /* The order of the arguments must be the same in both arglist and _local_vars ! */
  for(int i=0; i < sym_size; i++) {
    s=_formal_parms[i];
    arg=args[i];
    to_add=_local_vars.Lookup(s->GetName());
    if(to_add)
      to_add->SetVal((Literal&)*arg->Copy()); // args will be deleted by caller, so copy !
  }
  return 1;
}


NativeFunction::NativeFunction(char* name, void* native_addr, int number_of_parms,
			       CL_List<BType>* parm_types) {
  type=NATIVE_FUNC_TYPE;
  if(name)
    _name=strdup(name);
  _native_addr=native_addr;
  _number_of_formal_parms=number_of_parms;
  if(parm_types) {
    _parm_types=new CL_List<BType>;
    DOP(parm_types, BType, item)
      _parm_types->Add(item);
    ODP
  }
  else
    _parm_types=0;
}

NativeFunction::~NativeFunction() {
  delete [] _name;
  delete _parm_types;
}

Expr* NativeFunction::Interpret(CL_List<Expr*>* arglist) {
#ifdef TRACE
  Debug deb(*this);
#endif
  Expr* ret=0;
  Expr* (*func) (CL_List<Expr*>*);
  if(_native_addr) {
    func=(Expr* (*)(CL_List<Expr*>*))_native_addr;
    if(arglist && TypeCheckArgs(*arglist))
      ret=func(arglist);
  }
  return ret;
}


NativeFunction* NativeFunction::Copy() {
  return new NativeFunction(_name, _native_addr, _number_of_formal_parms, _parm_types);
}


short NativeFunction::TypeCheckArgs(CL_List<Expr*>& args) {
  BType a, b;
  if(!_parm_types)
    return 1;
  if(args.Size() != _number_of_formal_parms) {
    cerr << "Function " << _name << " has " << _number_of_formal_parms << " parameters";
    cerr << ", but number of args is " << args.Size() << " !" << endl;    
    return 0;
  }
  for(int i=0; i < _number_of_formal_parms; i++) {
    a=(*_parm_types)[i];
    b=args[i]->type;
    if(a != b) {
      cerr << "Type mismatch of formal parm(s) with arg(s), function " << _name << endl;
      return 0;
    }
  }
  return 1;
}


FunctionCall::FunctionCall(Ident& func_name, CL_List<Expr*>* arguments) {
  type=FUNC_CALL_TYPE;
  _func_name=&func_name;
  if(arguments) {
    DOP(arguments, Expr*, item)
      _args.Add(item, ::tail);  // don't copy item
    ODP
  }
}


FunctionCall::~FunctionCall() {
  delete _func_name;
  DO(_args, Expr*, item)
    delete item;
  OD
}

void FunctionCall::AddArg(Expr* arg) {
  _args.Add(arg, ::tail);
}


void FunctionCall::Print(ostream& os) {
  os << "Function call" << endl;
}

/*
   1. Find the Function with _func_name in the symbol table (RetrieveFunctionFromSymbolTable)
   2. Evaluate the args and bind them to the function's parameters
      Bindings are done on the order of the parameters with the args. 
      Order is therefore important !
   3. Interpret the function and return the value.
*/
Expr* FunctionCall::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug            deb(*this);
#endif
  Literal*         l;
  Expr*            ret=0;
  CL_List<Expr*>   evaluated_args;
  Function*        f;
  NativeFunction*  nf;
  
  l=RetrieveFunctionFromSymbolTable(_func_name->GetName(), context);  // should be a copy as well !
  if(!l)
    return 0;
  DO(_args, Expr*, item)
    evaluated_args.Add(item->Interpret(context, this_ptr));
  OD
  switch(l->type) {
  case FUNC_TYPE:
    f=(Function*)l;
    if(!f->BindArgs(evaluated_args)) {
      delete f;
      return 0;
    }
    ret=f->Interpret(context, this_ptr);
    break;
  case NATIVE_FUNC_TYPE:
    nf=(NativeFunction*)l;
    ret=nf->Interpret(&evaluated_args);
    break;
  default:
    break;
  }
  delete l;
  DO(evaluated_args, Expr*, item);
  delete item;
  OD
  return ret;
}



FunctionCall* FunctionCall::Copy() {
  FunctionCall* f=new FunctionCall(*_func_name->Copy());
  DO(_args, Expr*, item)
    f->AddArg(item->Copy());
  OD
  return f;
}


/* ---------------------------------------------------------------------------
                               Boolean expressions
   --------------------------------------------------------------------------- */

Expr* BoolExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  if(_b) {
    return _b->Interpret(context, this_ptr);
  }
  return 0;
}


void BoolExpr::Print(ostream& os) {
  if(_b) {
    if(_b->IsTrue())
      os << "true";
    else
      os << "false";
    os.flush();
  }
}


CompBoolExpr::CompBoolExpr(Expr* left, CompOperator op, Expr* right) {
  _left=left;
  _right=right;
  _comp_op=op;
}

CompBoolExpr::~CompBoolExpr() {
  delete _left;
  delete _right;
}

Expr* CompBoolExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  gomBool*  ret=0;
  int       comp, a, b;
  Literal*  l=(Literal*)_left->Interpret(context, this_ptr), 
           *r=(Literal*)_right->Interpret(context, this_ptr);
  if(TypeCheck(l, r, INT_TYPE, 0) || TypeCheck(l, r, STRING_TYPE, 0)) {     
    switch(l->type) {
    case STRING_TYPE:
      comp=strcmp(((Str*)l)->GetStr(), ((Str*)r)->GetStr());
      switch(_comp_op) {
      case lt:
	if(comp < 0) ret=new gomBool(1);
	break;
      case gt:
	if(comp > 0) ret=new gomBool(1);
	break;
      case lteq:
	if(comp < 0 || comp == 0) ret=new gomBool(1);
	break;
      case gteq:
	if(comp > 0 || comp == 0) ret=new gomBool(1);
	break;
      case eq:
	if(comp == 0) ret=new gomBool(1);
	break;
      case not_eq:
	if(comp =! 0) ret=new gomBool(1);
	break;
      default:
	ret=new gomBool(0);
      }
      break;

    case INT_TYPE:
      a=(int)((Int*)l)->GetInt();
      b=(int)((Int*)r)->GetInt();
      switch(_comp_op) {
      case lt:
	if(a < b) ret=new gomBool(1);
	break;
      case gt:
	if(a > b) ret=new gomBool(1);
	break;
      case lteq:
	if(a <= b) ret=new gomBool(1);
	break;
      case gteq:
	if(a >= b) ret=new gomBool(1);
	break;
      case eq:
	if(a == b) ret=new gomBool(1);
	break;
      case not_eq:
	if(a != b) ret=new gomBool(1);
	break;
      default:      
	ret=new gomBool(0);  // false
      }
      break;

    case NILVAL_TYPE:
      if(r->type == NILVAL_TYPE && _comp_op == eq)
	ret=new gomBool(1);
      break;    
    }
  }

  if(r->type == NILVAL_TYPE) {
    switch(_comp_op) {
    case eq:
      if(l->type == NILVAL_TYPE)
	ret=new gomBool(1);
      break;
    case not_eq:
      if(l->type != NILVAL_TYPE)
	ret=new gomBool(1);
      break;
    }
  }

  delete l; 
  delete r;
  return ret? ret : new gomBool(0);
}


void CompBoolExpr::Print(ostream& os) {
  _left->Print(os);
  os << " " << CompOp2Str(_comp_op) << " ";
  _right->Print(os);
  os << endl;
}


BinBoolExpr::BinBoolExpr(BoolExpr* left, BoolOperator op, BoolExpr* right) {
  _left=left;
  _right=right;
  _op=op;
}

BinBoolExpr::~BinBoolExpr() {
  delete _left;
  delete _right;
}


Expr* BinBoolExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  gomBool* a=(gomBool*)_left->Interpret(context, this_ptr), 
          *b=(gomBool*)_right->Interpret(context, this_ptr);
  gomBool* ret=0;
  switch(_op) {
  case and_op:
    if(a->IsTrue() && b->IsTrue()) ret=new gomBool(1);
    break;
  case or_op:
    if(a->IsTrue() || b->IsTrue()) ret=new gomBool(1);
    break;
  default:
    ret=new gomBool(0);
    break;
  }  
  delete a;
  delete b;
  return ret? ret : new gomBool(0);
}


void BinBoolExpr::Print(ostream& os) {
  _left->Print(os);
  os << " " << BoolOp2Str(_op) << " ";
  _right->Print(os);
  os << endl;
}



/* Currently contains no bool_op, only 'not' is used */
Expr* UnaryBoolExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  gomBool* ret=0, *a=(gomBool*)_expr->Interpret(context, this_ptr);
  if(a->IsTrue())
    ret=new gomBool(0);
  else
    ret=new gomBool(1);
  delete a;
  return ret;
}


void UnaryBoolExpr::Print(ostream& os) {
  os << " not ";
  _expr->Print(os);
  os << endl;
}


CreationExpr::CreationExpr(char* classname, CL_List<Expr*>* args) : _classname(0), _args(args) {
  type=CREATION_TYPE;
  if(classname)
    _classname=strdup(classname);
}


CreationExpr::~CreationExpr() {
  delete [] _classname;
  if(_args) {
    DOP(_args, Expr*, item)
      delete item;
    ODP
    delete _args;
  }
}

void CreationExpr::Print(ostream& os) {
  os << _classname; os.flush();
}

Expr* CreationExpr::Interpret(SymbolTable& context, Instance* this_ptr) {
  Symbol* s=::classes.Find(_classname);
  if(!s) {
    cerr << "Class " << _classname << " was not found !" << endl;
    return 0;
  }
  if(s->GetVal() && 
     (s->GetVal()->type == CLASS_TYPE || s->GetVal()->type == NATIVE_CLASS_TYPE))
    return ((ClassDef*)s->GetVal())->CreateInstance(_args, &context);

  cerr << _classname << " is unbound or not a class !" << endl;
  return 0;
}


CreationExpr* CreationExpr::Copy() {
  CL_List<Expr*>* new_args=0;
  if(_args) {
    new_args=new CL_List<Expr*>;
    DOP(_args, Expr*, item)
      new_args->Add(item->Copy());
    ODP
  }
  return new CreationExpr(_classname, new_args);
}


/* ---------------------------------------------------------------------------
                          Object-Oriented Extensions
   --------------------------------------------------------------------------- */

void ClassDef::AddSymbolsFromSuperclasses(char* superclass, SymbolTable& target,
					  CL_List<char*>& visited_classes) {
  ClassDef* cl;
  if(!superclass || visited_classes.Find(superclass))
    return;  
  visited_classes.Add(superclass); // no need for copy, only temporary
  
  if(!(cl=::RetrieveMetaclass(superclass)))
    return;
  if(cl->_vars) {
    DOP(cl->_vars, Ident*, item)
      target.Lookup(item->GetName());  // only adds new vars if not yet existing;
    ODP                                // old vars are not overwritten by new ones
  }
  if(cl->_superclass)
    AddSymbolsFromSuperclasses(cl->_superclass->GetName(), target, visited_classes);
}

ClassDef::ClassDef(char* name, Ident* superclass, CL_List<Ident*>* vars, 
		   CL_List<Function*>* methods) : _name(0) {
  type=CLASS_TYPE;		     
  if(name)
    _name=strdup(name);
  _superclass=superclass;
  _vars=vars;
  _methods=(CL_List<Method*>*)methods;
}


ClassDef::~ClassDef() {
  delete [] _name;
  delete _superclass;
  if(_vars) {
    DOP(_vars, Ident*, item)
      delete item;
    ODP
    delete _vars;
  }
  if(_methods) {
    DOP(_methods, Method*, item)
      delete item;
    ODP
    delete _methods;
  }
}


void ClassDef::GetAllMethods(CL_List<Method*>& methods_list) {
  if(_methods) {
    DOP(_methods, Method*, item)
      methods_list.Add(item->Copy());
    ODP
  }
  // check for superclasses, doesn't yet check for cycles
  if(_superclass) {
    ClassDef* cl=::RetrieveMetaclass(_superclass->GetName());
    if(cl)
      cl->GetAllMethods(methods_list);
  }
}


ClassDef* ClassDef::Copy() {
  CL_List<Ident*>*     new_vars=0;
  CL_List<Method*>*    new_methods=0;
  Ident*               new_superclass=_superclass? _superclass->Copy() : 0;  
  if(_vars) {
    new_vars=new CL_List<Ident*>;
    DOP(_vars, Ident*, item)
      new_vars->Add(item->Copy());
    ODP
  }
  if(_methods) {
    new_methods=new CL_List<Method*>;
    DOP(_methods, Method*, item)
      new_methods->Add(item->Copy());
    ODP
  }
  return new ClassDef(_name, new_superclass, new_vars, (CL_List<Function*>*)new_methods);
}


Instance* ClassDef::CreateInstance(CL_List<Expr*>* args, SymbolTable* context) {
  Instance* ret=new Instance(_name);
  CL_List<Expr*> evaluated_args;
  if(_vars) {  /* Create instance variables */
    DOP(_vars, Ident*, item)
      ret->_instance_vars->Lookup(item->GetName());
    ODP
  }
  if(_superclass) {
    CL_List<char*> visited_classes;
    AddSymbolsFromSuperclasses(_superclass->GetName(), *ret->_instance_vars, visited_classes);
  }
  ret->_instance_vars->SetPrevScope(::symtab);

  /* Call constructor if args is not null:
     1.) Look for method that has the same name as class
     2.) Call that method using args as arguments
  */
  if(args) {
    Method* ctor=FindMethod(_name);
    if(ctor) {
      SymbolTable* con=context? context : &::symtab;
      DOP(args, Expr*, item)
	evaluated_args.Add(item->Interpret(*con));
      ODP
      if(ctor->BindArgs(evaluated_args)) {
	ctor->SetThisPtr(*ret);
	ctor->GetLocalVars().SetPrevScope(*ret->_instance_vars);
	Expr* r=ctor->Interpret(*ret->_instance_vars, ret);
	delete r;
      }
      DO(evaluated_args, Expr*, item)
	delete item;
      OD
      delete ctor;
    }
    else
      cerr << "Warning: no constructor for class " << _name << " was found !" << endl;
  }
  return ret;
}


Method* ClassDef::FindMethod(char* methodname, short super) {
  if(!_methods || !methodname)
    return 0;
  if(!super) {
    DOP(_methods, Method*, item)
      if(!strcmp(item->GetName(), methodname))
	return item->Copy();
    ODP
  }
  if(_superclass) {
    ClassDef* cl=::RetrieveMetaclass(_superclass->GetName());
    if(cl)
      return cl->FindMethod(methodname);
  }
  // cerr << "Method '" << methodname << "' was not found !" << endl;
  return 0;
}


Instance::Instance(char* metaclass) {
  type=INST_TYPE;
  _instance_vars=new SymbolTable;
  if(metaclass)
    _metaclass=strdup(metaclass);
  else
    _metaclass=0;
}


Instance::~Instance() {
  delete _instance_vars;
  delete [] _metaclass;
}


Method* Instance::FindMethod(char* methodname, short super) {
  Method* ret=0;
  if(!methodname) return 0;
  ClassDef* cl=::RetrieveMetaclass(_metaclass);
  if(!cl)
    return 0;
  ret=cl->FindMethod(methodname, super);
  if(ret) {
    
    /* 1.) scope is method -> instance -> global; set instance & global scope !  */
    /* 2.) set this-pointer                                                      */
    if(ret) {
      ((Method*)ret)->GetLocalVars().SetPrevScope(*this->GetContext());
      ret->SetThisPtr(*this);
    }    
    return ret;
  }
  return ret;
}


Instance* Instance::Copy() {
  Instance* ret=new Instance(_metaclass);
  _instance_vars->CopyTo(*ret->_instance_vars);
  return ret;
}


void Instance::PrintMethods(ostream& os) {
  if(!_metaclass) return;
  ClassDef* cl=::RetrieveMetaclass(_metaclass);
  if(!cl) return;
  CL_List<Method*> methods;
  cl->GetAllMethods(methods);
  DO(methods, Method*, item)
    if(item->GetName())
      os << item->GetName() << endl;
    delete item;
  OD
}


void Instance::Inspect(ostream& os) {
  os << "\nClass: " << _metaclass << endl;
  os << "Instance variables:\n------------------:\n";
  _instance_vars->Print(os);
  os << "\nMethods:\n--------:\n";
  PrintMethods(os);
  os << endl;
}


Method::Method(char* name, CL_List<Ident*>* formal_parms, 
	       CL_List<Stmt*>* stmt_list, Instance* this_ptr) 
               : Function(name, formal_parms, stmt_list), _this_ptr(this_ptr) {
  type=METHOD_TYPE;
}


Method::~Method() {
  
}


Expr* Method::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  char* classname=this_ptr? this_ptr->GetMetaclass() : 0;
  DynamicString d;
  d << classname << "::" << _name;
  Debug deb(*this, d);
#endif
  return Function::Interpret(_local_vars, _this_ptr);
}


Method* Method::Copy() {
  Method* ret=new Method(_name);
  DO(_statements, Stmt*, item)
    ret->AddStmt(*(item->Copy()));
  OD
  DO(_formal_parms.GetSymbols(), Symbol*, item)
    ret->AddParm(item->GetName());
  OD
  ret->_this_ptr=_this_ptr;
  return ret;  
}


MethodCall::MethodCall(Ident& method_name, CL_List<Expr*>* arguments) 
                      : FunctionCall(method_name, arguments) {
  type=METHOD_CALL_TYPE;
}


MethodCall::~MethodCall() {
  
}


void MethodCall::Print(ostream& os) {
  os << "Method call" << endl;
}

/*
   if scope is not null: start resolving the instance using scope's local variables.
   else use the global symbol table.
   return the instance in target and the name of the method in name (dyn. alloc.)
*/
short MethodCall::ResolveInstance(Ident& id, 
				  Instance*& found, 
				  char*& name, 
				  SymbolTable* context) {
  Symbol*    s;
  char*      tmp_name=id.GetName();
  Instance*  new_scope;
  if(context)                                               /* is member variable ?      */
    s=context->Find(tmp_name);
  else                                                      /* ... or global variable ?  */
    s=::symtab.Find(tmp_name);
  
  // If the variable is not found, create it in the enclosing scope:*/
  // !s) {
  // =context->Lookup(tmp_name);
  // 

  if(s) {
    if(id.GetNext() && id.GetNext()->GetNext()) {           /* more to resolve...        */
      if(s->GetVal() && (s->GetVal()->type == INST_TYPE || 
			 s->GetVal()->type == NATIVE_INST_TYPE)) {
	new_scope=(Instance*)s->GetVal();
	return ResolveInstance(*id.GetNext(), found, name, new_scope->GetContext());
      }
      else {
	cerr << tmp_name << " is not bound or not an instance !" << endl;
	return 0;	
      }

    }
    else {                                                  /* next *must* be method name */
      if(s->GetVal() && (s->GetVal()->type == INST_TYPE || s->GetVal()->type == NATIVE_INST_TYPE)) {
	found=(Instance*)s->GetVal();
	name=strdup(id.GetNext()->GetName());
	return 1;
      }
      else {
	cerr << tmp_name << " is not bound or not an instance !" << endl;
	return 0;	
      }
    }
  }

  cerr << "MethodCall::ResolveInstance: " << tmp_name << " was not found !" << endl;
  return 0;
}



Expr* MethodCall::Interpret(SymbolTable& context, Instance* this_ptr) {
#ifdef TRACE
  Debug            deb(*this);
#endif
  Literal*         l;
  Instance*        target=0;
  char*            method_name=0;
  short            rc;
  Expr*            ret=0;
  CL_List<Expr*>   evaluated_args;
  Function*        f;
  Method*          m;
  NativeMethod*    nm;
  NativeFunction*  nf;

  if(_func_name->GetNext()) {                     /* compound ident, e.g. 'a.Foo()'  */
    if(this_ptr && !strcmp(_func_name->GetName(), "super") && !_func_name->GetNext()->GetNext())
      l=this_ptr->FindMethod(_func_name->GetNext()->GetName(), 1);
    else {
      if(this_ptr) {

	rc=ResolveInstance(*_func_name, target, method_name, &context);

	// rc=ResolveInstance(*_func_name, target, method_name, this_ptr->GetContext());

      }
      else
	rc=ResolveInstance(*_func_name, target, method_name, &context);
      if(!rc) {
	_func_name->Print(cerr); cerr << " could not be resolved !" << endl;
	return 0;
      }
      else {                                      
	l=target->FindMethod(method_name);
	this_ptr=target;      // new scope !
	delete [] method_name;
      }
    }
  }
  else {
    if(this_ptr) {
      l=this_ptr->FindMethod(_func_name->GetName());
      if(!l) {
	SymbolTable* new_context=this_ptr->GetContext();
	if(new_context)
	  l=::RetrieveFunctionFromSymbolTable(_func_name->GetName(), *new_context);
	else
	  l=::RetrieveFunctionFromSymbolTable(_func_name->GetName(), context);
      }
    }
    else
      l=::RetrieveFunctionFromSymbolTable(_func_name->GetName(), context);
  }

  if(!l) {
    cerr << "Symbol \""; 
    _func_name->Print(cerr);
    cerr << "\" was not found !" << endl;
    return 0;
  }

  DO(_args, Expr*, item)
    evaluated_args.Add(item->Interpret(context, this_ptr));
  OD
  switch(l->type) {
  case METHOD_TYPE:
    m=(Method*)l;
    if(!m->BindArgs(evaluated_args)) {
      delete m;
      break;
    }
    ret=m->Interpret(context, this_ptr);
    break;
  case NATIVE_METHOD_TYPE:
    nm=(NativeMethod*)l;    
    nm->SetThisPtr(*this_ptr);
    ret=evaluated_args.Size() ? nm->Interpret(&evaluated_args) : nm->Interpret();
    break;
  case FUNC_TYPE:
    f=(Function*)l;
    if(!f->BindArgs(evaluated_args)) {
      delete f;
      break;
    }
    ret=f->Interpret(context, this_ptr);
    break;
  case NATIVE_FUNC_TYPE:
    nf=(NativeFunction*)l;
    ret=nf->Interpret(&evaluated_args);
    break;
  default:
    break;
  }
  DO(evaluated_args, Expr*, item)
    delete item;
  OD
  delete l;
  return ret;
}


MethodCall* MethodCall::Copy() {
  MethodCall* f=new MethodCall(*_func_name->Copy());
  DO(_args, Expr*, item)
    f->AddArg(item->Copy());
  OD
  return f;
}


/* ---------------------------------------------------------------------------
                   Object-Oriented Extensions - Native
   --------------------------------------------------------------------------- */

NativeClassDef::NativeClassDef(char* name, NativeInstance* (*ctor)(CL_List<Expr*>&), 
			       Ident* superclass, CL_List<Function*>* methods) 
                               : ClassDef(name, superclass, 0, methods), _constructor(ctor) {
  type=NATIVE_CLASS_TYPE;  
}


void NativeClassDef::AddMethod(NativeMethod& method) {
  if(!_methods)
    _methods=new CL_List<Method*>;
  _methods->Add(&method);
}


NativeClassDef* NativeClassDef::Copy() {
  CL_List<Function*>*  new_methods=0;
  Ident*               new_superclass=_superclass? _superclass->Copy() : 0;  
  if(_methods) {
    new_methods=new CL_List<Function*>;
    DOP(_methods, Method*, item)
      new_methods->Add(item->Copy());
    ODP
  }
  return new NativeClassDef(_name, _constructor, new_superclass, 
			    (CL_List<Function*>*)new_methods);
}


Instance* NativeClassDef::CreateInstance(CL_List<Expr*>* args, SymbolTable* context) {
  NativeInstance*  ret=0;
  CL_List<Expr*>   evaluated_args;
  if(_constructor) {
    if(args) {
      SymbolTable* con=context? context : &::symtab;
      DOP(args, Expr*, item)
	evaluated_args.Add(item->Interpret(*con));
      ODP	
    }   
    ret=(*_constructor)(evaluated_args);
    DO(evaluated_args, Expr*, item)
      delete item;
    OD
  }
  else
    cerr << "Constructor for class " << _name << " is null !" << endl;
  if(ret)
      ret->GetContext()->SetPrevScope(::symtab);
  return ret;
}


NativeMethod::NativeMethod(char* name, MethodPtr mptr, NativeInstance* inst,
			   int number_of_parms, CL_List<BType>* parm_types)
                          : Method(name, 0, 0, inst), 
                            _method_ptr(mptr), _number_of_formal_parms(number_of_parms) {
  type=NATIVE_METHOD_TYPE;
  _parm_types=parm_types;
}



NativeMethod::~NativeMethod() {
  if(_parm_types)
    delete _parm_types;
}


Expr* NativeMethod::Interpret(CL_List<Expr*>* arglist) {
#ifdef TRACE
  Debug  deb(*this);
#endif
  Expr*  ret=0;
  if(_method_ptr && _this_ptr)
    if(arglist) {
      if(TypeCheckArgs(*arglist))
	ret=(((NativeInstance*)_this_ptr)->*_method_ptr)(*arglist);
    }
    else {
      CL_List<Expr*> tmp_args;
      ret=(((NativeInstance*)_this_ptr)->*_method_ptr)(tmp_args);
    }
  return ret;
}


NativeMethod* NativeMethod::Copy() {
  return new NativeMethod(_name, _method_ptr, (NativeInstance*)_this_ptr,
			  _number_of_formal_parms, _parm_types);
}

short NativeMethod::TypeCheckArgs(CL_List<Expr*>& args) {
  BType a, b;
  if(!_parm_types)
    return 1;
  if(args.Size() != _number_of_formal_parms) {
    cerr << "NativeMethod " << _name << " has " << _number_of_formal_parms << " parameters";
    cerr << ", but number of args is " << args.Size() << " !" << endl;    
    return 0;
  }
  for(int i=0; i < _number_of_formal_parms; i++) {
    a=(*_parm_types)[i];
    b=args[i]->type;
    if(a != b) {
      cerr << "Type mismatch of formal parm(s) with arg(s), function " << _name << endl;
      return 0;
    }
  }
  return 1;
}



NativeInstance::NativeInstance(char* metaclass) : Instance(metaclass) {
  type=NATIVE_INST_TYPE;
  if(_instance_vars)
    delete _instance_vars;
  _instance_vars=new NativeSymbolTable(this); // will be deleted by SymbolTable's dtor !
}


NativeInstance::~NativeInstance() {}


NativeInstance* NativeInstance::Copy() {
  cerr << "NativeInstance::Copy() should be overridden !" << endl;
  return 0;
}


void NativeInstance::Inspect(ostream& os) {
  os << "NativeInstance: Sorry, 'Inspect' not yet implemented !" << endl;
}


Literal* NativeInstance::GetVal(char* name) {
  cerr << "NativeInstance::GetVal() was not overwritten !" << endl;
  return 0;
}


void NativeInstance::SetVal(char* name, Literal& v) {
  cerr << "NativeInstance::SetVal(Literal&) was not overwritten !" << endl;
}

void NativeInstance::PrintMethods(ostream& os) {  
  Instance::PrintMethods(os);
}
