#ifndef GRAMMAR_H
#define GRAMMAR_H

#include <iostream.h>
#include "global.h"
#include "SymTab.h"




class Stmt;
class Expr;
class BinExpr;
class Ident;
class Literal;
class UndefinedLiteral;
class NilVal;
class Int;
class Str;
class Pair;
class SymbolTable;
class Symbol;
class Function;
class NativeFunction;
class BoolExpr;
class CompBoolExpr;
class BinBoolExpr;
class UnaryBoolExpr;
class ClassDef;
class Instance;
class Method;
class MethodCall;
class NativeClassDef;
class NativeMethod;
class NativeInstance;

void       ParseString(DynamicString&);
void       InterpretString(char*);
void       LoadFile(char* fname);
void       SetScript(char* fname);
void       DumpSymbolTable(char* filename="symtab.db");
void       ReadSymbolTable(char* filename="symtab.db");
Literal*   RetrieveFunctionFromSymbolTable(char* funcname, SymbolTable&);
           /* Can b be assigned to a ? */
int        TypeCheck(Literal* a, Literal* b, BType t=INT_TYPE, short show_error=1);
short      LoadVBXs(char* directory="./vbx");
short      UnloadVBXs();
short      LoadDLL(char* dllname, SymbolTable&);
short      ReloadVBXs(char* directory="./vbx");
void       InspectInstance(Expr&, SymbolTable&);
ClassDef*  RetrieveMetaclass(char* classname);


/* ---------------------------------------------------------------------------
                                  Statements
   --------------------------------------------------------------------------- */

struct RetVal {
  StmtType         type;
  Expr*            ex;  
                   RetVal(StmtType st, Expr* exp=0) : type(st), ex(exp) {}
                   ~RetVal();
};


class Stmt {
protected:
  Ident*           ident;
  Expr*            expr;
  StmtType         stmt_type;
public:
                   Stmt(Ident* id=0, Expr* ex=0);
                   ~Stmt();
  StmtType         GetStmtType() {return stmt_type;}
  void             SetStmtType(StmtType s) {stmt_type=s;}
  Stmt&            operator=(Stmt&);
  virtual RetVal*  Interpret(SymbolTable& context, Instance* i=0);  // for future subclasses
  virtual void     Print(ostream& os=cout);                         // for future subclasses
  virtual void     Trace(ostream& os=cout);                         // for future subclasses
  virtual Stmt*    Copy();
};


class VarsStmt : public Stmt {
  CL_List<Ident*>  _vars;
public:
                   VarsStmt(CL_List<Ident*>& vars);
		   ~VarsStmt();
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout)  {os << "VARS-stmt"; os.flush();}
  void             Trace(ostream& os=cout);
  VarsStmt*        Copy()                   {return new VarsStmt(_vars);}
};



class PrintStmt : public Stmt {  
  short            _println;
  short            _cgi;
public:
                   PrintStmt(Expr* ex, int println=0, short cgi=0) 
		     : Stmt(0, ex), _println(println), _cgi(cgi) {}
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  void             Trace(ostream& os=cout);
  PrintStmt*       Copy();
};


class EvalStmt : public Stmt {  
public:
                   EvalStmt(Expr* ex=0) : Stmt(0, ex) {}
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  void             Trace(ostream& os=cout);
  EvalStmt*        Copy();
};




/*
   Return statements return an expression or 0;
*/
class ReturnStmt : public Stmt {
public:
                   ReturnStmt(Expr* ret_expr=0) : Stmt(0, ret_expr) {stmt_type=RET_STMT;}
  RetVal*          Interpret(SymbolTable& context, Instance* i=0); 
  void             Print(ostream& os=cout) {os << "RETURN-stmt"; os.flush();}
  void             Trace(ostream& os=cout);
  ReturnStmt*      Copy();
};



class WhileStmt : public Stmt {
  BoolExpr*        _cond;
  SymbolTable      _local_vars;     // only local variables
  CL_List<Stmt*>   _stmt_list;
public:  
                   WhileStmt(BoolExpr* cond=0, CL_List<Stmt*>* stmt_list=0);
		   ~WhileStmt();
  void             SetCond(BoolExpr& cond) {_cond=&cond;}
  void             AddStmt(Stmt& s) {_stmt_list.Add(&s, ::tail);}  // s already a copy !
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  void             Trace(ostream& os=cout);
  WhileStmt*       Copy();
};


class IfStmt : public Stmt {
  BoolExpr*        _cond;
  SymbolTable      _local_vars;     // only local variables
  CL_List<Stmt*>   _if_part;
  CL_List<Stmt*>   _else_part;
public:
                   IfStmt(BoolExpr* cond, CL_List<Stmt*>* if_part=0, CL_List<Stmt*>* else_part=0);
                   ~IfStmt();
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  void             Trace(ostream& os=cout);
  IfStmt*          Copy();
};


class ForStmt : public Stmt {
  SymbolTable      _local_vars;
  CL_List<Stmt*>   _stmt_list;
  int              _a, _b;
  Expr*            _start, *_end;
  int              _up_or_down; // 0: up, 1: down
public:
                   ForStmt(Ident& id, Expr& start, int up_or_down, Expr& end, 
			   CL_List<Stmt*>* stmt_list=0);
                   ~ForStmt();
  RetVal*          Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  void             Trace(ostream& os=cout);
  ForStmt*         Copy();
};



class BreakStmt : public Stmt {
public:  
                   BreakStmt()  {stmt_type=BREAK_STMT;}
		   ~BreakStmt() {}
  RetVal*          Interpret(SymbolTable& context, Instance* i=0) {return new RetVal(BREAK_STMT);}
  void             Print(ostream& os=cout) {os << "BREAK-stmt" << endl;}
  void             Trace(ostream& os=cout);
  BreakStmt*       Copy() {return new BreakStmt;}
};



/* ---------------------------------------------------------------------------
                                  Expressions
   --------------------------------------------------------------------------- */

class Expr {       // abstract base class, defines common interface
public:
  BType            type;
                   Expr() : type(UNDEFINED) {}
  virtual          ~Expr() {}	   
  virtual void     Print(ostream& os=cout)=0;
  virtual Expr*    Interpret(SymbolTable& context, Instance* i=0)=0;
  virtual Expr*    Copy()=0;
};


class UnaryExpr : public Expr {
  Expr*            expr;
  Operator         op;
public:
                   UnaryExpr(Operator o, Expr* e) {op=o; expr=e; type=UNARY_TYPE;}
  void             Print(ostream& os=cout);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  UnaryExpr*       Copy() {return new UnaryExpr(op, expr);}
};


class BinExpr : public Expr {  // +, -, *, /
  Expr*            left, *right;
  Operator         op;
public:
                   BinExpr(Expr*, Operator, Expr*);
		   ~BinExpr();
  Expr*		   Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  BinExpr*         Copy() {return new BinExpr(left->Copy(), op, right->Copy());}
};


/*
   Ident is used for identification of entries in the symbol table. When an ident is 
   encountered, it's associated symbol will be retrieved from the symbol table. If not
   yet in the symbol table, a new entry will be created and inserted.
*/

class Ident : public Expr {
  char*            name;
  Ident*           next;
public:
                   Ident(char*, Ident* next=0);
                   ~Ident();  
  char*            GetName()                         {return name;}
  void             SetName(char*);
  Ident*           GetNext()                         {return next;}
  void             SetNext(Ident& n);
  void             Print(ostream& os=cout);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  Symbol*          GetSymbol(SymbolTable& context);
  Symbol*          GetInstanceSymbol(SymbolTable& context);
  Ident*           Copy();
};


class Literal : public Expr {    // abstract class
public:
                   Literal() {type=LITERAL_TYPE;}
  Expr*            Interpret(SymbolTable& context, Instance* i=0)  {cerr << "<literal>"; return 0;}
  void             Print(ostream& os=cout) {os << "<literal>" << endl;}
  Literal*         Copy() {cerr << "Literal::Copy(): Literal is abstract !" << endl; return 0;}
  virtual void     Dump(ostream& os) {os << UNDEFINED;}
  virtual void     Read(istream&)    {}
};


class UndefinedLiteral : public Literal {
public:
                   UndefinedLiteral()               {type=UNDEFINED;}
  Expr*            Interpret(SymbolTable& context, Instance* i=0){cerr << "<undefined literal>"; return 0;}  
  void             Print(ostream& os=cout)          {os << "<undefined literal>"; os.flush();}
  UndefinedLiteral* Copy()                          {return new UndefinedLiteral;}
};


/* Strange name chosen because NULL is often used in other systems */
class NilVal : public Literal {
public:
                   NilVal();
  Expr*            Interpret(SymbolTable& context, Instance* i=0) {return Copy();}  
  void             Print(ostream& os=cout)          {os << "NULL"; os.flush();}
  NilVal*          Copy()                           {return new NilVal;}
  void             Dump(ostream& os)                {os << NILVAL_TYPE;}
};



class gomBool : public Literal {  /* 1 == true, 0 == false */
  short            _bool;
public:
                   gomBool(short b);
		   ~gomBool();
  short            IsTrue()                         {return _bool;}
  void             SetTrue()                        {_bool=1;}
  void             SetFalse()                       {_bool=0;}
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  gomBool*         Copy()                           {return new gomBool(_bool);}
  void             Dump(ostream& os);
};


class Int : public Literal {
  double           _i;
public:
                   Int(double i=0);
  double           GetInt() {return _i;}
  void             SetInt(double i) {_i=i;}
  Int&             operator=(Int&);
  int              operator==(Int&);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout) {os <<  _i; os.flush();}
  Int*             Copy() {return new Int(_i);}
  void             Dump(ostream& os);
};


class Str : public Literal {
  char*            _s;
public:
                   Str(char* s=0);
                   Str(Str&);
                   ~Str();
  char*            GetStr() {return _s;}
  Str&             operator=(Str&);
  int              operator==(Str&);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  Str*             Copy() {return new Str(_s);}
  void             Dump(ostream& os);
};



class Pair : public Literal {
  char*            _name;
  Literal*         _literal;
public:
                   Pair(char* name=0, Literal* l=0);
                   Pair(Pair&);
                   ~Pair();
  char*            GetName()       {return _name;}
  Literal*         GetLiteral()    {return _literal;}
  void             SetLiteral(Literal& l) {if(_literal) delete _literal; _literal=l.Copy();}
  Pair&            operator=(Pair&);
  int              operator==(Pair&);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  Pair*            Copy() {return new Pair(_name, _literal);}
};


class CreationExpr : public Expr {
  char*            _classname;
  CL_List<Expr*>*  _args;
public:
                   CreationExpr(char* classname, CL_List<Expr*>* args=0);
                   ~CreationExpr();
  void             Print(ostream& os=cout);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  CreationExpr*    Copy();

};



/* ----------------------------------------------------------------------
   Function is a class that captures the definition of a class.
   Function is created by the parser when it encounters a function_def-
   statement.
   When interpreted, it interpretes its statement list.
   Since it is generated, functions cannot be defined in other functions;
   they may only be defined in the toplevel loop.

   FunctionCall is the invocation of a Function. When interpreted, the
   function name is looked up and the arguments are bound to the looked-up
   Function. The the function is executed.   
   ---------------------------------------------------------------------- */

class Function : public Literal {
protected:
  char*            _name;
  SymbolTable      _formal_parms;   // parameters are put in here rather than in _symbols !
  SymbolTable      _local_vars;     // only local variables
  CL_List<Stmt*>   _statements;
public:
                   Function(char* name, CL_List<Ident*>* formal_parms=0, 
			    CL_List<Stmt*>* stmt_list=0);
                   ~Function();
  char*            GetName()                {return _name;}
  SymbolTable&     GetLocalVars()           {return _local_vars;}
  Symbol*          Lookup(char* name)       {return _local_vars.Lookup(name);}
  Symbol*          Find(char* name)         {return _local_vars.Find(name);}
  void             AddStmt(Stmt& stmt)      {_statements.Add(&stmt, ::tail);}
  void             Print(ostream& os=cout)  {os << "Function"; os.flush();}
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             AddParm(char* n)         {_formal_parms.Lookup(n);}
  Function*        Copy();
  int              BindArgs(CL_List<Expr*>& args);
  void             SetLocalVars(SymbolTable& symtab) {_local_vars=symtab;}
};



class NativeFunction : public Literal {
protected:
  void*            _native_addr;
  char*            _name;
  int              _number_of_formal_parms;
  CL_List<BType>*  _parm_types;
public:
                   NativeFunction(char* name, void* native_addr, int number_of_parms=0,
				  CL_List<BType>* parm_types=0);
                   ~NativeFunction();
  int              GetNumberOfFormalParms() {return _number_of_formal_parms;}
  void             SetNativeAddr(void* ptr) {_native_addr=ptr;}
  char*            GetName()                {return _name;}
  void             Print(ostream& os=cout)  {os << "NativeFunction"; os.flush();}
  Expr*            Interpret(SymbolTable& context, Instance* i=0) { /* not used */  return 0;};
  virtual Expr*    Interpret(CL_List<Expr*>* arglist=0);
  NativeFunction*  Copy();
  short            TypeCheckArgs(CL_List<Expr*>& args);
};


class FunctionCall : public Expr {
protected:
  CL_List<Expr*>   _args;
  Ident*           _func_name;
public:
                   FunctionCall(Ident& func_name, CL_List<Expr*>* arguments=0);
                   ~FunctionCall();
  void             AddArg(Expr* arg);
  void             Print(ostream& os=cout);
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  FunctionCall*    Copy();
};


/* ---------------------------------------------------------------------------
                             Boolean expressions
   --------------------------------------------------------------------------- */

class BoolExpr : public Expr { 
  gomBool*         _b;
public:
                   BoolExpr(gomBool* b=0) {_b=b;}
                   ~BoolExpr() {delete _b;}
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  BoolExpr*        Copy() {return new BoolExpr(_b);}
};


class CompBoolExpr : public BoolExpr {
  Expr*            _left;
  Expr*            _right;
  CompOperator     _comp_op;
public:
                   CompBoolExpr(Expr* left, CompOperator op, Expr* right);
                   ~CompBoolExpr();
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  CompBoolExpr*    Copy()  {return new CompBoolExpr(_left->Copy(), _comp_op, _right->Copy());}
};


class BinBoolExpr : public BoolExpr {
  BoolExpr*        _left;
  BoolExpr*        _right;
  BoolOperator     _op;
public:
                   BinBoolExpr(BoolExpr* left, BoolOperator op, BoolExpr* right);
                   ~BinBoolExpr();
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  BinBoolExpr*     Copy() {return new BinBoolExpr(_left->Copy(), _op, _right->Copy());}
};


class UnaryBoolExpr : public BoolExpr {
  BoolExpr*        _expr;
public:
                   UnaryBoolExpr(BoolExpr* expr) : _expr(expr) {}
                   ~UnaryBoolExpr() {delete _expr;}
  Expr*            Interpret(SymbolTable& context, Instance* i=0);
  void             Print(ostream& os=cout);
  UnaryBoolExpr*   Copy() {return new UnaryBoolExpr(_expr->Copy());}
};


/* ---------------------------------------------------------------------------
                          Object-Oriented Extensions
   --------------------------------------------------------------------------- */

class ClassDef : public Literal {  // VERY awkward, but necessary for SymbolTable !!
protected:
  char*              _name;
  Ident*             _superclass;
  CL_List<Ident*>*   _vars;
  CL_List<Method*>*  _methods;
  void               AddSymbolsFromSuperclasses(char* superclass, SymbolTable& target,
					       CL_List<char*>& visited_classes);
public:
                     ClassDef(char* name, Ident* superclass=0, 
			      CL_List<Ident*>* vars=0, CL_List<Function*>* methods=0);
                     ~ClassDef();
  char*              GetName()               {return _name;}
  CL_List<Method*>*  GetMethods()            {return _methods;}
  void               GetAllMethods(CL_List<Method*>&);  // local plus parent classes' methods
  Expr*              Interpret(SymbolTable& context, Instance* i=0) {return 0;}
  void               Print(ostream& os=cout) {os << "ClassDef"; os.flush();}
  virtual Method*    FindMethod(char* methodname, short super=0);
  ClassDef*          Copy();
  virtual Instance*  CreateInstance(CL_List<Expr*>* args=0, SymbolTable* context=0);
};


class Instance : public Literal {
  friend class ClassDef;
protected:
  SymbolTable*       _instance_vars;
  char*              _metaclass;
public:
                     Instance(char* metaclass);
                     ~Instance();
  char*              GetMetaclass()          {return _metaclass;}
  SymbolTable*       GetContext()            {return _instance_vars;}
  virtual Method*    FindMethod(char* methodname, short super=0);
  Symbol*            FindVar(char* varname) {return _instance_vars->Find(varname);}
  Expr*              Interpret(SymbolTable& context, Instance* i=0) {return Copy();}
  void               Print(ostream& os=cout) {os << "Instance of " << _metaclass; os.flush();}
  virtual void       PrintMethods(ostream& os=cout);
  Instance*          Copy();
  virtual void       Inspect(ostream& os=cout);
};



class Method : public Function {
protected:
  Instance*          _this_ptr;
public:
                     Method(char* name, CL_List<Ident*>* formal_parms=0, 
			    CL_List<Stmt*>* stmt_list=0, Instance* this_ptr=0);
                     ~Method();
  void               SetThisPtr(Instance& this_ptr) {_this_ptr=&this_ptr;}
  void               Print(ostream& os=cout)  {os << "Method"; os.flush();}
  Expr*              Interpret(SymbolTable& context, Instance* this_ptr=0);
  Method*            Copy();
};


class MethodCall : public FunctionCall {
  short              ResolveInstance(Ident& id, Instance*& found, char*& name, SymbolTable* contxt=0);
public:
                     MethodCall(Ident& method_name, CL_List<Expr*>* arguments=0);
                     ~MethodCall();
  void               Print(ostream& os=cout);
  Expr*              Interpret(SymbolTable& context, Instance* i=0);
  MethodCall*        Copy();
};


/* ---------------------------------------------------------------------------
                   Object-Oriented Extensions - Native
   --------------------------------------------------------------------------- */

class NativeClassDef : public ClassDef {
  NativeInstance*    (*_constructor)(CL_List<Expr*>&);
public:
                     NativeClassDef(char* name, NativeInstance* (*ctor)(CL_List<Expr*>&)=0,
				    Ident* superclass=0, CL_List<Function*>* methods=0);
  void               AddMethod(NativeMethod& method);
  Expr*              Interpret(SymbolTable& context, Instance* i=0) {return 0;}
  void               Print(ostream& os=cout) {os << "NativeClassDef"; os.flush();}
  NativeClassDef*    Copy();
 Instance*           CreateInstance(CL_List<Expr*>* args=0, SymbolTable* context=0);
};


class NativeInstance : public Instance {  
public:
                     NativeInstance(char* metaclass);
		     ~NativeInstance();
  Expr*              Interpret(SymbolTable& context, Instance* i=0) {return Copy();}
  void               Print(ostream& os=cout) {os << "NativeInstance of " << _metaclass; os.flush();}
  virtual NativeInstance* Copy();
  void               Inspect(ostream& os=cout);
  virtual Literal*   GetVal(char*);
  virtual void       SetVal(char*, Literal& v);
  virtual Symbol*    Lookup(char* name)=0;
  virtual Symbol*    Find(char* name)=0;     
  void               PrintMethods(ostream& os=cout);
};


typedef Expr* (NativeInstance::* MethodPtr)(CL_List<Expr*>& args);

class NativeMethod : public Method {
  MethodPtr          _method_ptr;
  int                _number_of_formal_parms;
  CL_List<BType>*    _parm_types;
public:
                     NativeMethod(char* name, MethodPtr mptr=0, NativeInstance* inst=0,
				  int number_of_parms=0, CL_List<BType>* parm_types=0);
                     ~NativeMethod();
  void               Print(ostream& os=cout)  {os << "NativeMethod"; os.flush();}
  Expr*              Interpret(SymbolTable& context, Instance* this_ptr=0) {return 0;}
  virtual Expr*      Interpret(CL_List<Expr*>* arglist=0);
  NativeMethod*      Copy();
  short              TypeCheckArgs(CL_List<Expr*>& args);
};







#endif


