
#include <string.h>
#include <iomanip.h>
#include <strstream.h>
#include "SymTab.h"
#include "Grammar.h"

SymbolTable symtab;
SymbolTable classes;

char* agent_script=0;
char* agent_start_functions=0;
char* agent_symtab=0;



Literal* ParseString(istream& is) {
    DynamicString buf;
    char        c;
    Literal*    ret=0;
    is >> ws;
    is.get(c);
    if(c != '"') {
      cerr << "ParseString(): string must start with a '\"' !" << endl;
      return 0;
    }
    while(!is.eof()) {
      is.get(c);      
      if(is) {
	switch(c) {
	case '"':
	  is >> ws;
	  break;
	default:
	  buf << c;
	}
      }
    }
    ret=new Str(buf);
    return ret;
}

Literal* ParseNativeInst(istream& is) {
    Literal*    ret=0;
    char        classname[0xff];
    ClassDef*   classdef;
    DynamicString buf;

    is >> ws >> classname;
    classdef=RetrieveMetaclass(classname);
    if(!classdef)
	return 0;
    ret=classdef->CreateInstance();
    ReadValue(is, buf);
    istrstream input((char*)buf);
    ret->Read(input);

    return ret;
}


Literal* CreateLiteralFromString(char* s) {
    istrstream  is(s);
    char        buf[0xff];
    BType       type=UNDEFINED;
    Literal*    ret=0;
    double      i=0;
    int         ii=1;
    if(!is)
	return 0;
    is >> ii;
    if(!is)
	return 0;
    type=(BType)ii;
    switch(type) {
    case UNDEFINED:
	break;
    case NILVAL_TYPE:
	ret=new NilVal;
	break;
    case BOOL_TYPE:
	is >> buf;
	if(!strcmp(buf, "false"))
	    ret=new gomBool(0);
	else
	    ret=new gomBool(1);
	break;
    case INT_TYPE:
	is >> i;
	ret=new Int(i);
	break;
    case STRING_TYPE:
	ret=ParseString(is);
	break;
    case NATIVE_INST_TYPE:
	ret=ParseNativeInst(is);
	break;
    }
    return ret;
}


short MatchDelimiter(char* tok, int& num_starts, int& num_ends) {
    if(!strcmp(tok, "<START>")) {
	num_starts++;
	return 1;
    }
    if(!strcmp(tok, "<END>")) {
	num_ends++;
	return 1;
    }
    return 0;
}


/* Copies the contents from is between <START> and matching <END>
   to buf */
void ReadValue(istream& is, DynamicString& buf) {
    int num_of_start_dels=0, num_of_end_dels=0;
    DynamicString tok;  
    tok.ReadToken(is);
    if(!is)
	return;
    MatchDelimiter(tok, num_of_start_dels, num_of_end_dels);
    while(is && num_of_start_dels > num_of_end_dels) {
	DynamicString tok2;
	tok2.ReadToken(is);
	if(MatchDelimiter(tok2, num_of_start_dels, num_of_end_dels)) {
	    // the delimiting <START> and <END> tags are not copied
	    if(num_of_start_dels > num_of_end_dels)
		buf << (char*)tok2 << " ";
	}
	else
	    buf << (char*)tok2 << " ";
    }
}



Symbol::Symbol(char* name, Literal* val) : _name(0) {
  _sym_type=normal_sym;
  if(name)
    _name=strdup(name);
  if(val) {
    _val=val;
  }
  else
    _val=new NilVal;
}

Symbol::~Symbol() {
  if(_name) {
    delete [] _name;
    _name=0;
  }
  if(_val) {
    delete _val;
    _val=0;
  }
}

BType Symbol::GetValType() {
  return _val->type;
}

void Symbol::SetVal(Literal& v) {  // v need not be copied !
  if(!_val) {
    _val=&v;
    return;
  }
  if(_val == &v) return;
  delete _val;
  _val=&v;
}

void Symbol::Print(ostream& os) {
  os << _name << setw(20) << setiosflags(ios::left) << ": ";
  if(_val)
    _val->Print(os);
  else
    os << "Symbol " << _name << " has no name !" << endl;
}


Symbol* Symbol::Copy() {
  return new Symbol(_name, _val->Copy());
}

Literal* Symbol::CopyVal() {
  if(_val)
    return _val->Copy();
  return 0;
}

void Symbol::Dump(ostream& os) {
    os << _name << " ";
    os << START_DEL;
    if(!_val)
	os << UNDEFINED << END_DEL;
    else {
	_val->Dump(os);
	os << END_DEL;
    }
}



void SymbolTable::ReadTokens(istream& is, CL_List<char*>& tokens) {
    static char  n[0xff];
    char*        name=0, *val=0;
    while(!is.eof()) {
	DynamicString buf;
	is >> ws >> n;
	if(!is)
	    break;
	name=strdup(n);
	ReadValue(is, buf);
	val=strdup(buf);
	if(name && val) {
	    tokens.Add(name);
	    tokens.Add(val);
	}	    
    }
}

void SymbolTable::InsertValues(CL_List<char*>& tokens) {
    Symbol*    sym;
    char*      name=0, *val=0;
    Literal*   lit=0;
    long       syms_added=0;

    for(int i=0; i < tokens.Size(); i+=2) {
	name=tokens[i];
	val=tokens[i+1];
	if(!name && !val)
	    break;
	lit=CreateLiteralFromString(val);
	if(lit) {
	    sym=::symtab.Find(name);
	    if(sym)
		sym->SetVal(*lit);
	    else {
		sym=new Symbol(name, lit);
		_symbols.Add(sym, tail, replace);
	    }
	    syms_added++;
	}
    }
    cout << syms_added << " symbols were added to symbol table." << endl;
}



SymbolTable::~SymbolTable() {
  DO(_symbols, Symbol*, item)
    if(item)
      delete item;
  OD
}

Symbol* SymbolTable::Lookup(char* name) {
  Symbol* found=0;
  DO(_symbols, Symbol*, item)
    if(!strcmp(item->GetName(), name))
      return item;
  OD
  found=new Symbol(name);
  _symbols.Add(found, ::tail);
  return found;
}

/* Recursively looks up enclosing scopes */
Symbol* SymbolTable::Find(char* name) {
  DO(_symbols, Symbol*, item)
    if(!strcmp(item->GetName(), name))
      return item;
  OD
  if(_prev_scope)
    return _prev_scope->Find(name);
  return 0;
}

void SymbolTable::Print(ostream& os) {
  os << "\nSymbol table contents:\n";
  os << "----------------------" << endl;
  DO(_symbols, Symbol*, item)
    item->Print(os);
    os << endl;
  OD
  os << endl;
}


SymbolTable& SymbolTable::operator=(SymbolTable& other) {
  if(this == &other)
    return *this;
  DO(_symbols, Symbol*, item)
    delete item;
  OD
  _symbols.RemoveAll();
  DO(other._symbols, Symbol*, item)
    Lookup(item->GetName());  // create variable *without* any binding yet !
  OD
  return *this;
}


void SymbolTable::DeleteAll() {
  DO(_symbols, Symbol*, item)
    delete item;
  OD
  _symbols.RemoveAll();
}


SymbolTable& SymbolTable::Copy() {
  SymbolTable* ret=new SymbolTable;
  ret->_prev_scope=_prev_scope;
  DO(_symbols, Symbol*, item)
    ret->_symbols.Add(item->Copy());
  OD
  return *ret;
}

void SymbolTable::CopyTo(SymbolTable& other) {
  other._prev_scope=_prev_scope;
  DO(_symbols, Symbol*, item)
    other._symbols.Add(item->Copy());
  OD
}


void SymbolTable::Dump(ostream& os) {
  BType t;  
  DO(_symbols, Symbol*, item)
    t=item->GetValType();
    if(t == UNDEFINED || t == INT_TYPE || t == STRING_TYPE || t == BOOL_TYPE ||
       t == NATIVE_INST_TYPE || t == NILVAL_TYPE)
      item->Dump(os);
  OD
}

/* tokens is a list of strings where the first is the name of the 
   value and the second is the value as string. CL_Dict<char*,char*>
   was tried first, but caused too many problems with inline methods */

void SymbolTable::Read(istream& is) {
  CL_List<char*> tokens;
  ReadTokens(is, tokens);
  InsertValues(tokens);

  // is this needed ? Aren's char* s automatically deleted ?
  DO(tokens, char*, item)
      delete [] item;
  OD
}



Symbol* NativeSymbolTable::Lookup(char* name) {
  return _this_ptr->Lookup(name);
}


Symbol* NativeSymbolTable::Find(char* name) {
  return _this_ptr->Find(name);
}


NativeSymbol::NativeSymbol(char* n, NativeInstance* i, Literal* v)
             : Symbol(n, v), this_ptr(i) {
  _sym_type=native_sym;
}


Literal* NativeSymbol::GetVal() {
  if(this_ptr)
    return this_ptr->GetVal(_name);
  return 0;
}

void NativeSymbol::SetVal(Literal& v) {
  if(this_ptr)
    this_ptr->SetVal(_name, v);
}


NativeSymbol* NativeSymbol::Copy() {
  Literal* v=_val? _val->Copy() : 0;
  return new NativeSymbol(_name, this_ptr, v);
}


