#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <signal.h>
#include <unistd.h>
#include <memory.h>
#include <strings.h>
#include <strstream.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "Grammar.h"
#include "BuiltinClasses.h"
#include "Contact.h"
#include "Input.h"
#include "Debug.h"

/* globals */

int                 server_socket=-1, client_socket=-1;
extern FILE*        yyin;
extern SymbolTable  symtab;
extern short        interactive;

extern "C" {

  void HandleCtrlC(int) {
    cout << "+++ Interpreter daemon interrupted by CTRL-C or KILL +++" << endl;
    UnloadVBXs();
    if(server_socket != -1)
      close(server_socket);
    if(client_socket != -1)
      close(client_socket);
    exit(-1);
  }

  void HandleSigPipe(int) {
    cerr << "\n+++ Interpreter daemon received SIGPIPE signal +++\n";
    if(server_socket != -1)
      close(server_socket);
    if(client_socket != -1)
      close(client_socket);
    UnloadVBXs();
    exit(-1);
  }

}


extern int yyparse();
extern int yydebug;


void Usage(char* n) {
  cout << "\n\nUsage: " << n << " [-d] [-h] [-f <filename>]\n";
  cout << "\n-v: show debugging info (verbose)\n";
  cout << "\n-V: show version\n";
  cout << "-d: VBX directory\n";
  cout << "-h: this help\n";
  cout << "-n: don't load VBXs\n";
  cout << "-p <port>: listen on specified port (i.e. start as daemon)\n";
  cout << "-i: start in intelligent agent mode (only together with -p !)\n";
  cout << "-t: trace mode\n";
  cout << "-f: input file\n\n" << endl;
  exit(1);
}

void Credentials() {
  cout << "\n          -----------------------------------------------------\n";
  cout << "                      Basic-like interpreter written by \n";
  cout << "                                 Bela Ban\n";
  cout << "                       IBM Zurich Research Laboratory\n";
  cout << "                              bba@zurich.ibm.com\n";
  cout << "                                  Version 0.91\n";
  cout << "                       Built on " << __DATE__ << " at " <<  __TIME__ << "\n";
  cout << "          -----------------------------------------------------\n\n" << endl;  
}

char* CopyTextBetween(char* buf, char* start, char* end) {
  char* first, *last;

  first=strstr(buf, start);
  if(!first) {
    cerr << "Delimiter \"" << start << "\" not found !" << endl;
    return 0;
  }
  first+=strlen(start);
  last=strstr(buf, end);
  if(!last) {
    cerr << "Delimiter \"" << last << "\" not found !" << endl;
    return 0;
  }
  char* ret=new char[last-first+2];
  memset(ret, 0, last-first+2);
  strncpy(ret, first, last-first);
  return ret;
}




void ReceiveAgent(int sock) {
  DynamicString d;
  char buf[0xff];
  int numbytes=0;
  
  // Read entire stuff (script, intial functions and symbol table
  memset(buf, 0, sizeof(buf));


  /* Leave 1 byte of the buffer for the terminating zero !!
     This is already in the buffer through memset ! */

  while((numbytes=read(sock, buf, sizeof(buf)-1)) > 0) {
    //    write(2, buf, numbytes);
    d << buf;
    memset(buf, 0, sizeof(buf));
  }

  if(::agent_script)
    delete [] ::agent_script;
  ::agent_script=CopyTextBetween(d, "<SCRIPT>", "</SCRIPT>");

  if(::agent_start_functions)
    delete [] ::agent_start_functions;
  ::agent_start_functions=CopyTextBetween(d, "<INITIAL_FUNCTIONS>", "</INITIAL_FUNCTIONS>");

  if(::agent_symtab)
    delete [] ::agent_symtab;
  ::agent_symtab=CopyTextBetween(d, "<SYMBOL_TABLE>", "</SYMBOL_TABLE>");
}




int main(int argc, char* argv[]) {
  char*         filename=0;
  char*         vbx_path=getenv("VBXDIR")? getenv("VBXDIR") : "vbx";
  FILE*         fp=0;
  short         load_vbxs=1;
  int           child_id, tmpfd, rc, server_port=0;
  size_t        client_addr_len;
  sockaddr_in   server_addr, client_addr;
  short         agent_mode=0;
  char*         script=0;
  Input*        input=0;

  SendUsageToServer("Interpreter Daemon");

  for(int i=1; i < argc; i++) {
    if(!strncmp(argv[i], "-d", 2)) {
      if(i+1 >= argc)
	Usage(argv[0]);
      else {
	vbx_path=argv[i+1];
	i++;
	continue;
      }
    }
    if(!strncmp(argv[i], "-v", 2)) {
      yydebug=1;
      continue;
    }
    if(!strncmp(argv[i], "-t", 2)) {
      Debug::debug_mode=1;
      continue;
    }
    if(!strncmp(argv[i], "-V", 2)) {
      Credentials();
      exit(0);
      continue;
    }
    if(!strncmp(argv[i], "-n", 2)) {
      load_vbxs=0;
      continue;
    }
    if(!strncmp(argv[i], "-h", 2)) {
      Usage(argv[0]);
      continue;
    }
    if(!strncmp(argv[i], "-i", 2)) {
      agent_mode=1;
      continue;
    }
    if(!strncmp(argv[i], "-f", 2)) {
      if(i+1 >= argc)
	Usage(argv[0]);
      else {
	filename=argv[i+1];
	i++;
	continue;
      }
    }
    if(!strncmp(argv[i], "-p", 2)) {
      if(i+1 >= argc)
	Usage(argv[0]);
      else {
	server_port=atoi(argv[i+1]);
	i++;
	continue;
      }
    }
    /* Read the rest as GOMscript instructions: */
    script=argv[i];
    filename=script;
  }



  if(server_port) {       /* start as daemon */
    server_socket=socket(AF_INET, SOCK_STREAM, 0);
    if(server_socket == -1) {
      perror("Interpreter daemon");
      return -1;
    }    
    bzero((char*)&server_addr, sizeof(server_addr));
    server_addr.sin_family=AF_INET;
    server_addr.sin_addr.s_addr=htonl(INADDR_ANY);
    server_addr.sin_port=htons(server_port);    
    rc=bind(server_socket, (sockaddr*)&server_addr, sizeof(server_addr));
    if(rc) {
      perror("Interpreter daemon");
      close(server_socket);
      return -1;
    }        
    rc=listen(server_socket, 5);  // 5 is max. queue length
    if(rc < 0) {
      perror("Interpreter daemon");
      close(server_socket);
      return -1;
    }    
  }
  

  if(load_vbxs) {
    cout << "Loading VBXs..." << endl;
    short rc=LoadVBXs(vbx_path);
    if(rc)
      cout << "\nDone." << endl;
    else {
      cerr << "\nFailed !" << endl;
      cerr << "No VBXs will be available !" << endl;
    }
  }

  if(filename) {
    ::SetScript(filename);
    fp=fopen(filename, "r");
    if(!fp) {
      cerr << "Filename " << filename << " could not be opened !" << endl;
      return -1;
    }
    input=Input::GetInput();        
    input->Push(fp);
    interactive=0;
  }

  AddBuiltinClasses();  // adds NativeClassDefs to ::classes

  signal(SIGINT, HandleCtrlC);
  signal(SIGTERM, HandleCtrlC);
  signal(SIGPIPE, HandleSigPipe);
  signal(SIGCLD, SIG_IGN);          // dont' allow children processes to become zombies


  if(server_port) {                 /* start as daemon */
    cout << "Basic interpreter service is listening on port ";
    cout << server_port << " for incoming requests..." << endl;
    if(agent_mode)
      cout << "(Enabled as GOMscript execution environment)" << endl;
    for(;;) {
      client_socket=accept(server_socket, (sockaddr*)&client_addr, &client_addr_len);
      if(client_socket == -1) {
	perror("Interpreter daemon");
	continue;
      }

      sockaddr_in peer;
      size_t      peerlen=sizeof(peer);
      char*       peername_tmp, *peername="<Unknown host>";
      rc=getpeername(client_socket, (sockaddr*)&peer, &peerlen);
      if(!rc) {
	hostent* hent2=gethostbyaddr((char*)&peer.sin_addr, sizeof(peer.sin_addr), AF_INET);
	if(hent2 && hent2->h_name[0])
	  peername=(char*)hent2->h_name;
	else
	  peername=inet_ntoa(peer.sin_addr);      	
      }
      cout << "A client connected from host " << peername << "." << endl;
      SendUsageToServer("Interpreter", peername);
      
      child_id=fork();      
      if(child_id == 0) {     // child
	close(server_socket);
	
        if(agent_mode) {
	  
	  ReceiveAgent(client_socket);

	  // Symbol table has to be resurrected first !
	  if(::agent_symtab && strlen(::agent_symtab) > 0) {
	    // Add symbols to symbol table:
	    istrstream input(::agent_symtab);
	    if(input)
	      ::symtab.Read(input);
	  }
	  
	  char tmpbuf[L_tmpnam+32];
	  char *tmp_file=tmpnam(tmpbuf);
	  if(tmp_file) {
	    FILE* fp=fopen(tmp_file, "w");
	    if(fp) {
	      fprintf(fp, "/* Temporary script for GOMscript */\n");
	      if(::agent_script)
		fprintf(fp, "%s\n", ::agent_script);
	      if(::agent_start_functions)
		fprintf(fp, "%s\n", ::agent_start_functions);
	      fclose(fp);
	    }

	    fp=fopen(tmp_file, "r");
	    if(fp) {
		Input::GetInput()->Push(fp);
		yyparse();
		fclose(fp);
		unlink(tmp_file);
	    }


#ifdef bela
	    fp=freopen(tmp_file, "r", stdin);
	    if(fp) {
	      try {
		yyparse();
	      }
	      catch(char* msg) {
		cerr << "\n\nAn Exception was caught: " << msg << endl;
	      }
	      fclose(fp);
	      unlink(tmp_file);
	    }
#endif



	  }
	  	  
	  delete [] ::agent_script;
	  delete [] ::agent_start_functions;
	  delete [] ::agent_symtab;
	  close(client_socket);
	  exit(0);
	}
	else {
	  yyin=fdopen(client_socket, "r");
	  close(1);
	  close(2);
	  tmpfd=dup(client_socket);  // 1    stdout
	  tmpfd=dup(client_socket);  // 2    stderr        
	}
		
	try {
	  if(interactive) {
	    cout << "> "; 
	    cout.flush();
	  }
	  yyparse();    
	}
	catch(char* msg) {
	  cerr << "\n\nAn Exception was caught: " << msg << endl;
	}	
	close(client_socket);
	exit(0);      
      }
      // parent
      close(client_socket);      
    }    
  }
  else {                            /* start normal */
    try {
      if(interactive) {
	cout << "> "; 
	cout.flush();
      }
      yyparse();    
    }
    catch(char* msg) {
      cerr << "\n\nAn Exception was caught: " << msg << endl;
    }
  }  
  return 0;
}
