
#include "defs.h"
#include "./GOM/MID.h"
#include "./GOM/genObj.h"
#include "./GOM/Attr.h"
#include "../BuiltinClasses.h"


// #include <GenObj.xh>
// #include <DSOM_Adapter.xh>

class GomObjMethod;
class GomObj;


/* ---------------------------------------------------------------------------
   Function prototypes
   --------------------------------------------------------------------------- */
Expr* Find(Elist& arglist); BType Find_types[]={STRING_TYPE};
Expr* DiscoverRoots(Elist& arglist); BType DiscoverRoots_types[]={INT_TYPE};
Expr* SetSyncPolicy(Elist& arglist); BType SetSyncPolicy_types[]={INT_TYPE};
Expr* PrintSyncPolicy(Elist& arglist);
Expr* GetRDN(Elist& arglist); BType GetRDN_types[]={STRING_TYPE};
Expr* ShowCorbaClasses(Elist& arglist);
Expr* GetException(Elist& arglist);
Expr* ResetEx(Elist& arglist);
Expr* FindServer(Elist& arglist); BType FindServer_types[]={STRING_TYPE};
Expr* FindServerByClass(Elist& arglist); BType FindServerByClass_types[]={STRING_TYPE};

#ifdef NEW
Expr* CreateFunc(Elist& arglist); BType CreateFunc_types[]={STRING_TYPE, 
							    STRING_TYPE, 
							    STRING_TYPE};

SOMObject*   CreateDSOMObject(char* classname, char* location);
#endif

Environment* Env() {return somGetGlobalEnvironment();}

/* ---------------------------------------------------------------------------
   Function list and entry point
   --------------------------------------------------------------------------- */

static FuncDescr functions[]={
  /* name               address      num of parms    parm types                */
  /* --------           ---------    ------------    ----------                */

  {"Find",              Find,              1,        Find_types},
  {"DiscoverRoots",     DiscoverRoots,     1,        DiscoverRoots_types},
  {"SetSyncPolicy",     SetSyncPolicy,     1,        SetSyncPolicy_types},
  {"PrintSyncPolicy",   PrintSyncPolicy,   0,        0},
  {"GetRDN",            GetRDN,            1,        GetRDN_types},
  {"ShowCorbaClasses",  ShowCorbaClasses,  0,        0},
  {"GetEx",             GetException,      0,        0},
  {"ResetEx",           ResetEx,           0,        0},
  {"FindServer",        FindServer,        1,        FindServer_types},
  {"FindServerByClass", FindServerByClass, 1,        FindServerByClass_types},
  //  {"Create",            CreateFunc,        3,        CreateFunc_types},
  0
};



boolean ErrorCheck(Environment* env) {
    if(!env)
	return FALSE;
    boolean retval=FALSE;
    switch(env->_major) {
    case SYSTEM_EXCEPTION:
    case USER_EXCEPTION:
	cerr << somExceptionId(env) << endl;
	retval=TRUE;
	break;
    default:
	retval=FALSE;
    }    
    somExceptionFree(env);
    return retval;
}


void ResetExceptionInternal(Environment* env) {
    somExceptionFree(env);
}


typedef Expr* (NativeInstance::* MethPtr)(Elist&);


class GomObj : public NativeInstance {  
protected:
  Literal*   Gom2Basic(Val&);
  Val*       Basic2Gom(Literal&);
private:
  genObj*    _real_obj;
public:
             GomObj(genObj* real_obj=0) : _real_obj(real_obj), NativeInstance("Obj") {}
             GomObj(Elist& args);  // classname args
             ~GomObj();
  GomObj*    Copy();
  void       Print(ostream& os=cout);
  void       Inspect(ostream& os=cout);
  Literal*   GetVal(char*);
  void       SetVal(char*, Literal& v);
  Literal*   Execute(char* name, Elist& args);
  Symbol*    Lookup(char* name);
  Symbol*    Find(char* name);  
  Method*    FindMethod(char* methodname, short super=0);
  void       Dump(ostream&);
  void       Read(istream&);


  Expr*      Ctor(Elist&              args);
  Expr*      GetName(Elist&           args);
  Expr*      GetClass(Elist&          args);
  Expr*      Synchronize(Elist&       args);
  Expr*      Delete(Elist&            args);
  Expr*      DiscoverChildren(Elist&  args);
  Expr*      GetChildren(Elist&       args);
  Expr*      Exec(Elist&              args);
  Expr*      AttrsAsString(Elist&     args);
};



class GomObjMethod : public NativeMethod {
public:
                     GomObjMethod(char* name, Instance* inst);
                     ~GomObjMethod();
  void               Print(ostream& os=cout)  {os << "GomObjMethod"; os.flush();}
  Expr*              Interpret(SymbolTable& context, Instance* this_ptr=0) {return 0;}
  Expr*              Interpret(CL_List<Expr*>* arglist=0);
  GomObjMethod*      Copy();
};





NativeInstance* MakeGomObj(Elist& args) {
  return new GomObj(args);
}


MethodDescr GomObj_methods[]={
  {"Synchronize",      (MethPtr)&GomObj::Synchronize,      0, 0},
  {"DiscoverChildren", (MethPtr)&GomObj::DiscoverChildren, 0, 0},
  {"GetChildren",      (MethPtr)&GomObj::GetChildren,      0, 0},
  {"GetName",          (MethPtr)&GomObj::GetName,          0, 0},
  {"GetClass",         (MethPtr)&GomObj::GetClass,         0, 0},
  {"Delete",           (MethPtr)&GomObj::Delete,           0, 0},
  {"Exec",             (MethPtr)&GomObj::Exec,             0, 0},
  {"AttrsAsString",    (MethPtr)&GomObj::AttrsAsString,    0, 0},
  0
};

static ClassDescr classes[]={
  {"Obj", 0, MakeGomObj, 
   GomObj_methods
  }, 
  0
  };


void LoadFunction() {
  ::InitializeGOM();
  ::sync_policy.SetSyncMode(SyncPolicy::sync_all_values);
  cout << "(Sync policy was set to 1)" << endl;
}

void UnLoadFunction() {
  ::UninitializeGOM();
}


static Contents contents={functions, classes, LoadFunction, UnLoadFunction};

Contents* EntryPoint() {return &contents;}



/* ---------------------------------------------------------------------------
                                 Implementation
   --------------------------------------------------------------------------- */


Literal* GomObj::Gom2Basic(Val& v) {
  long                               l=0;
  char*                              s=0, *assoc_key=0;
  CL_List<Expr*>                     items, tmp_items;
  CL_List<Val*>*                     vals;
  CL_Dict<char*, Val*>*              dict;
  CL_Node<CL_Assoc<char*, Val*>*>*   node;
  CL_Assoc<char*, Val*>*             assoc;
  Literal*                           val;
  Val*                               assoc_val;
  NativeList*                        nl, *tmp_list;
  NativeStruct*                      ns;
  NativeUnion*                       nu;
  CL_List<Pair*>*                    native_struct;
  switch(v.GetTCode()) {
  case NILType:
    return new NilVal;
  case BoolType:
    return new gomBool((Bool&)v);  // conversion op. short()
  case ShortType:
    return new Int((double)(short)(Short&)v);
  case IntegerType:
    return new Int((double)(int)(Integer&)v);
  case LongType:
    return new Int((double)(long)(Long&)v);
  case EnumType:
  case StringType:
  case ObjectIdType:
    return new Str((char*)(String&)v);
  case DistinguishedNameType:
    return new Str((char*)(String&)v);

  case ObjRefType:
    return new GomObj((genObj*)&v);

  case SequenceType:
    vals=((Sequence&)v).GetNativeVal();
    if(!vals) break;
    DOP(vals, Val*, item)
      val=Gom2Basic(*item);
      if(val)
	items.Add(val);
    ODP
    nl=new NativeList(items);
    return nl;
  case StructType:
    dict=((Struct&)v).GetNativeVal();
    if(!dict) break;
    ns=new NativeStruct;
    native_struct=ns->GetNativeStruct();

    for(node=dict->GetFirst(); node; node=node->GetNext()) {
      assoc=node->GetElement();
      if(!assoc) continue;
      assoc_key=assoc->GetKey();
      assoc_val=assoc->GetVal();
      if(!assoc_key || !assoc_val) continue;

      // convert value:
      val=Gom2Basic(*assoc_val);
      if(!val) continue;
      native_struct->Add(new Pair(assoc_key, val));
    }
    return ns;

  case UnionType:
    assoc_key=((Union&)v).GetName();
    assoc_val=((Union&)v).GetVal();
    if(assoc_key && assoc_val) {      
      if(val=Gom2Basic(*assoc_val)) {
	nu=new NativeUnion(assoc_key, val);
	return nu;
      }
    }
    break;
  }   
  
  if(items.Size()) {
    DO(items, Expr*, item)
      delete item;
    OD
  }
  return 0;
}



Val* GomObj::Basic2Gom(Literal& v) {
  Val*                              ret=0;
  CL_List<Expr*>*                   tmp_list;
  Literal*                          l;
  Val*                              val;
  Sequence*                         seq=0;
  Struct*                           ret_struct=0;
  Union*                            ret_union=0;
  char*                             metaclass, *name;
  NativeList*                       nl;
  NativeUnion*                      nu;
  CL_List<Pair*>*                   pair_list;
  CL_Node<CL_Assoc<char*, Expr*>*>* item;

  switch(v.type) {      
  case NILVAL_TYPE:
    ret=new NIL;
    break;

  case BOOL_TYPE:
    ret=new Bool(((gomBool&)v).IsTrue());
    break;

  case INT_TYPE:
    ret=new Integer(((Int&)v).GetInt());
    break;

  case STRING_TYPE:
    ret=new String(((Str&)v).GetStr());
    break;

  case NATIVE_INST_TYPE:
    metaclass=((NativeInstance&)v).GetMetaclass();

    if(!strcmp(metaclass, "List")) {      
      tmp_list=((NativeList&)v).GetNativeList();
      if(tmp_list) {
	seq=new Sequence;
	DOP(tmp_list, Expr*, item)
	  l=(Literal*)item;
          val=Basic2Gom(*l);
          if(val)
	   seq->Add(val);
	ODP
      }
      return seq;
    }    

    if(!strcmp(metaclass, "Struct")) {
      pair_list=((NativeStruct&)v).GetNativeStruct();
      if(pair_list) {
	ret_struct=new Struct;
	DOP(pair_list, Pair*, item)
	  name=item->GetName();
          if(l=item->GetLiteral()) {
	    val=Basic2Gom(*l);
	    if(name && val)
	      ret_struct->Add(strdup(name), val);
	  }
	ODP
      }
      return ret_struct;
    } 
    
    if(!strcmp(metaclass, "Union")) {
      nu=(NativeUnion*)&v;
      name=nu->GetName();
      l=nu->GetVal();
      if(name && l) {	
	if(val=Basic2Gom(*l)) {
	  ret_union=new Union(name, val);
	  return ret_union;
	}
      }
      return ret_union;
    }

    if(!strcmp(metaclass, "Obj")) {
	ret=((GomObj&)v)._real_obj;
	return ret;
    }
  }   
  return ret;
}




GomObj::GomObj(Elist& args) : NativeInstance("Obj"), _real_obj(0) {
  // 1. OM type: "Corba", "X700" ("" defaults to Corba)
  // 2. char* classname
  // 3. char* location (or Distinguished Name)
  // 4. X.700: Struct, Corba: --

  int                    numargs=args.Size();
  char*                  classname, *location, *om_type, *dn;
  NativeStruct*          arglist;
  CL_Dict<char*, char*>* arguments;

  if(!numargs)   // just create a proxy
    return;

  om_type=((Str*)args[0])->GetStr();
  if(!om_type) return;
  if(!strcmp(om_type, "Corba") || !strcmp(om_type, "")) {
    if(numargs != 3) {
      cerr << "Wrong number of args: GomObj(om_type, classname, location)" << endl;
      return;
    }
    classname=((Str*)args[1])->GetStr();
    location=((Str*)args[2])->GetStr();    
    if(!strlen(location))       // create object in local address space
      _real_obj=::mid->Create(classname, "", 0, 0);
    else
      _real_obj=::mid->Create(classname, location, 0, 0);        
  }
  else {
    if(!strcmp(om_type, "X700") || !strcmp(om_type, "X.700")) {
      if(numargs != 4) {
	cerr << "Wrong number of args: GomObj(om_type, classname, DN, arglist)" << endl;
	return;
      }
      classname=((Str*)args[1])->GetStr();
      dn=((Str*)args[2])->GetStr();    
      arglist=(NativeStruct*)args[3];
      arguments=new CL_Dict<char*, char*>;
      DOP(arglist->GetNativeStruct(), Pair*, item)  // no copy !
	arguments->Add(item->GetName(), ((Str*)item->GetLiteral())->GetStr());
      ODP
      
      _real_obj=::mid->Create(classname, dn, arguments);
      delete arguments;
    }
    else {
      cerr << om_type << " is not known !" << endl;
      return;
    }
  }
}


GomObj::~GomObj() {
  
}


GomObj* GomObj::Copy() {
  return new GomObj(_real_obj);
}


void GomObj::Print(ostream& os) {
  if(_real_obj) {
    char* tmp=_real_obj->GetClassName();
    os << "<Instance of " << tmp << ">";
    // _real_obj->Synchronize();
    // _real_obj->Print(os);
  }
}


void GomObj::Inspect(ostream& os) {
  Print(os);
  os << "\nMethods:\n--------:\n";
  if(_real_obj)
    _real_obj->PrintMethods(os);
  PrintMethods(os);
  os << endl;
}


Literal* GomObj::GetVal(char* name) {
  if(_real_obj) {
    Val* ret=_real_obj->Get(name);
    if(ret)
      return Gom2Basic(*ret);
  }
  return 0;
}



void GomObj::SetVal(char* name, Literal& v) {
  genErr    rc;
  Attr*     attr;
  Val*      val;
  
  if(_real_obj) {
    if(!(attr=_real_obj->FindAttr(name))) {
      cerr << "Attribute " << name << " not found !" << endl;
      return;
    }
    val=Basic2Gom(v);
    if(val) {
      rc=_real_obj->Set(name, val);
      if(rc)
	cout << rc;
      else
	cerr << "Failure to set " << name << " !" << endl;
    }
  }
}


Literal* GomObj::Execute(char* name, Elist& args) {
  Val*      tmpval;
  Literal*  retval=0;
  Arglist*  arglist;
  if(!_real_obj)
    return 0;
  arglist=new Arglist;
  DO(args, Expr*, item)
    tmpval=Basic2Gom(*(Literal*)item);
    if(tmpval)
      arglist->Add(tmpval);
  OD
  tmpval=_real_obj->Execute(name, arglist);
  retval=tmpval? Gom2Basic(*tmpval) : 0;
  delete arglist;
  return retval;
}



Symbol* GomObj::Lookup(char* name) {
  Literal* l=GetVal(name);
  if(l)
    return new NativeSymbol(name, this, l);
  return 0;
}

Symbol* GomObj::Find(char* name) {
  return Lookup(name);
}


void GomObj::Dump(ostream& os) {
  os << NATIVE_INST_TYPE << " Obj ";
  os << START_DEL;
  if(_real_obj)
    _real_obj->DumpPersistent(os);
  else
    os << "NULL";
  os << END_DEL;
}


// object_model class_name

void GomObj::Read(istream& is) {
  char  buf[0xff], *clname=0, *om="Corba";

  is >> ws >> buf >> ws;
  om=strdup(buf);
  
  is >> buf >> ws;
  clname=strdup(buf);  

  genObj* tmp=::mid->CreateProxy(om, clname);
  if(tmp) {
    tmp->ReadPersistent(is);
    _real_obj=tmp;
  }

  delete [] om;
  delete [] clname;
}


Method* GomObj::FindMethod(char* methodname, short super) {
  Method* ret=0;  // will be a GomObjMethod if found
  if(_real_obj && _real_obj->SupportsOperation(methodname))
    ret=new GomObjMethod(methodname, this);
  return ret? ret : Instance::FindMethod(methodname, super);
}


Expr* GomObj::Synchronize(Elist& args) {
  if(_real_obj)
    _real_obj->Synchronize();
  return 0;
}



Expr* Find(Elist& arglist) {
  Str*     dn=(Str*)arglist[0];
  genObj*  go=::mid->Find(dn->GetStr());
  if(!go)
    return new NilVal;
  GomObj*  retval=new GomObj(go);
  return retval;
}


Expr* DiscoverRoots(Elist& arglist) {
  int  flag=((Int*)arglist[0])->GetInt();
  bool rc, discover_all=flag? true : false;
  CL_List<genObj*> roots;
  if(!::DiscoveryFunction) {
    cerr << "Discovery function was not found !\nThe X700.dll component may not have been ";
    cerr << "loaded !" << endl;
    return new NilVal;
  }
  rc=(*::DiscoveryFunction)(&roots, discover_all);
  if(rc == false && !roots.Size())
    return new NilVal;
  NativeList* retval=new NativeList;
  DO(roots, genObj*, item)
    retval->GetNativeList()->Add(new GomObj(item));
  OD
  cout << endl;
  return retval;
}


Expr* GomObj::DiscoverChildren(Elist& arglist) {
  int  flag=((Int*)arglist[0])->GetInt();
  bool rc, discover_all=flag? true : false;
  if(!_real_obj)
    return new NilVal;
  rc=_real_obj->DiscoverChildren(discover_all);
  return 0;
}


Expr* GomObj::GetName(Elist& args) {
  if(_real_obj) {
    char* n=_real_obj->GetName();
    Str* s=new Str(n);
    return s;
  }
  return new NilVal;
}


Expr* GomObj::Delete(Elist& args) {
  int  flag=args.Size()? ((Int*)args[0])->GetInt() : 1;
  bool rc;
  DelMode m=flag? delete_children : fail_if_contains_children;
  if(!_real_obj)
    return new NilVal;
  rc=_real_obj->Delete(m);
  if(rc) {
    delete _real_obj;
    _real_obj=0;
  }    
  return 0;
}



Expr* GomObj::GetClass(Elist& args) {
  if(_real_obj) {
    char* n=_real_obj->GetClassName();
    Str* s=new Str(n);
    return s;
  }
  return new NilVal;
}


Expr* GomObj::GetChildren(Elist& args) {
  if(_real_obj) {    
    CL_List<genObj*>* kids;
    _real_obj->DiscoverChildren(true);   // discover only 1 level (lazy discovery)
    kids=_real_obj->GetKids();
    if(kids) {
      NativeList* retval=new NativeList;
      GomObj*     obj;
      DOP(kids, genObj*, item)
	obj=new GomObj(item);
        retval->GetNativeList()->Add((Expr*)obj);
      ODP
      return retval;
    }
  }
  return new NilVal;
}


Expr* GomObj::Exec(Elist& args) {
  Literal* retval=0;
  if(args.Size() < 1)
    return 0;
  char* name=((Str*)args[0])->GetStr();
  Elist newargs;
  for(int i=1; i < args.Size(); i++)
    newargs.Add(args[i]);  // no copy !
  retval=Execute(name, newargs);
  return retval;
}

Expr* GomObj::AttrsAsString(Elist& args) {
  Literal*   retval=0;
  char*      attrs;
  if(!_real_obj)
    return 0;
  attrs=_real_obj->AttrsAsString();
  if(attrs) {
      retval=new Str(attrs);
      delete [] attrs;
      return retval;
  }
  return retval;
}


Expr* SetSyncPolicy(Elist& arglist) {
  int policy=((Int*)arglist[0])->GetInt();
  if(policy)
    ::sync_policy.SetSyncMode(SyncPolicy::sync_all_values);
  else
    ::sync_policy.SetSyncMode(SyncPolicy::sync_missing_values_only);
  return 0;
}



Expr* PrintSyncPolicy(Elist& arglist) {
  if(::sync_policy.GetSyncMode() == SyncPolicy::sync_all_values)
    cout << "Sync all values" << endl;
  else
    cout << "Sync missing values only" << endl;
  return 0;
}

Expr* GetRDN(Elist& arglist) {
  char* dn=((Str*)arglist[0])->GetStr();
  if(!dn)
    return new NilVal;
  char* rdn=strrchr(dn, ';');
  if(!rdn)
    rdn=dn;
  else
    rdn++;
  return new Str(rdn);
}

Expr* ShowCorbaClasses(Elist&) {
  ::mid->ShowClasses("Corba");
  return 0;
}


Expr* GetException(Elist&) {
  char*     tmp=somExceptionId(somGetGlobalEnvironment());
  Literal*  ret=0;
  if(tmp)
    ret=new Str(tmp);

  // delete [] tmp;
  return ret;
}


Expr* ResetEx(Elist&) {
  somExceptionFree(somGetGlobalEnvironment());

  return 0;
}



Expr* FindServer(Elist& args) {
  char*         name=((Str*)args[0])->GetStr();
  char*         classname="SOMDServer";
  SOMDServer*   server=0;
  genObj*       ret=0;
  GomObj*       retVal=0;
  if(name) {
    ResetExceptionInternal(somGetGlobalEnvironment());
    server=SOMD_ObjectMgr->somdFindServerByName(somGetGlobalEnvironment(), name);
    ErrorCheck(somGetGlobalEnvironment());
    if(!server)
      return 0;
    classname=server->somGetClassName();
    ret=::mid->CreateProxy("Corba", classname);
    if(ret) {      
      ret->SetRef(server);
      retVal=new GomObj(ret);
      return retVal;
    }
  }
  return 0;
}


Expr* FindServerByClass(Elist& args) {
  char*        name=((Str*)args[0])->GetStr();
  char*        classname;
  SOMDServer*  server=0;
  genObj*      tmp;
  GomObj*      ret=0;
  if(name) {
    ResetExceptionInternal(somGetGlobalEnvironment());
    server=SOMD_ObjectMgr->somdFindAnyServerByClass(somGetGlobalEnvironment(), name);
    ErrorCheck(somGetGlobalEnvironment());
    if(!server)
      return 0;
    classname=server->somGetClassName();
    tmp=::mid->CreateProxy("Corba", classname);
    if(tmp) {      
      tmp->SetRef(server);
      return new GomObj(tmp);
    }
  }
  return ret;
}



// Instance* _this_ptr  points to GomObj instance
// char*     _name      points to name of method

GomObjMethod::GomObjMethod(char* name, Instance* inst) : NativeMethod(name, 0, (NativeInstance*)inst) {
  
}


GomObjMethod::~GomObjMethod() {
  
}


Expr* GomObjMethod::Interpret(CL_List<Expr*>* arglist) {
  Expr* ret=0;
  if(!_this_ptr)
    return 0;

  if(arglist)
    ret=((GomObj*)_this_ptr)->Execute(_name, *arglist);
  else {
    CL_List<Expr*> tmp_arglist;
    ret=((GomObj*)_this_ptr)->Execute(_name, tmp_arglist);
  }
  return ret;
}


GomObjMethod* GomObjMethod::Copy() {
  return new GomObjMethod(_name, _this_ptr);
}


#ifdef NEW
SOMObject* CreateDSOMObject(char* classname, char* location) {
    SOMClass*      cls=0;
    SOMObject*     ret=0;
    SOMDServer*    server=0;

    if(!location) {         // Create instance in the client's address space
	cls=SOMClassMgrObject->somFindClass(somIdFromString(classname), 0, 0);
	if(cls)
	    return cls->somNew();
	return 0;
    }

    if(!strlen(location)) // Create instance in any server that implements it
	server=SOMD_ObjectMgr->somdFindAnyServerByClass(Env(), classname);
    else                       // Create instance in server 'location'
	server=SOMD_ObjectMgr->somdFindServerByName(Env(), location);
    
    if(!server) {
	cerr << "Server for class " << classname << " was not found !" << endl;
	return 0;
    }

    ret=server->somdCreateObj(Env(), classname, 0);
    return ret;    
}


Expr* CreateFunc(Elist& args) {
    char*       object_model=((Str*)args[0])->GetStr();
    char*       classname=((Str*)args[1])->GetStr();
    char*       location=((Str*)args[2])->GetStr();
    SOMObject*  tmp, *real_obj=0;
    genObj*     proxy=0;
    GenObj*     obj=0;
    Adapter*    ad=0;
    GomObj*     ret=0;
    char        buf[0xff];


    /****************************************************************************
      VERY IMPORTANT to instantiate class objects (here, they are not loaded
      from a DLL) !
      ***************************************************************************/
    GenObjNewClass(0,0);
    // AdapterNewClass(0,0);
    DSOM_AdapterNewClass(0,0);
    /****************************************************************************/


    // Create target object first:
    real_obj=CreateDSOMObject(classname, location);
    if(!real_obj) {
	cerr << "Instance of " << classname << " could not be created !" << endl;
	return 0;
    }


    // Create an instance of genObj and its target object (GenObj):
    proxy=::mid->Create("GenObj", "", 0, 0);
    if(!proxy) {
	cerr << "Proxy instance for DSOM object GenObj could not be created !" << endl;
	delete real_obj;
	return 0;
    }


    // Create instance of DSOM_Adapter in local address space:
    ad=(Adapter*)CreateDSOMObject("DSOM_Adapter", 0);  // '0' means local addr. space
    if(!ad) {
	cerr << "Instance of DSOM_Adapter could not be created !" << endl;
	delete proxy;
	delete real_obj;
	return 0;
    }



    // Set GenObj::ad to DSOM_Adapter
    // Set GenObj::specific_data to target object
    obj=(GenObj*)proxy->GetRef();
    obj->_set_ad(Env(), ad);
    obj->_set_classname(Env(), classname);
    obj->_set_specific_data(Env(), (long)real_obj);
    sprintf(buf, "Instance of %s at location %s\n", classname, location);
    obj->_set_instance_name(Env(), buf);


    ret=new GomObj(proxy);
    return ret;
}

#endif



