/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.mock;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Expression;
import org.apache.camel.Predicate;
import org.apache.camel.support.ExpressionAdapter;
import org.apache.camel.support.ExpressionToPredicateAdapter;
import org.apache.camel.support.builder.ExpressionBuilder;
import org.apache.camel.support.builder.PredicateBuilder;
import org.apache.camel.util.json.JsonArray;
import org.apache.camel.util.json.JsonObject;
import org.apache.camel.util.json.Jsoner;

/**
 * A builder of expressions or predicates based on values.
 * <p/>
 * This implementation is a derived copy of the <tt>org.apache.camel.builder.ValueBuilder</tt> from camel-core, that are
 * specialized for being used with the mock component and separated from camel-core.
 */
public class MockValueBuilder implements Expression, Predicate {
    private final Expression expression;
    private boolean not;

    public MockValueBuilder(Expression expression) {
        this.expression = expression;
    }

    @Override
    public void init(CamelContext context) {
        expression.init(context);
    }

    @Override
    public <T> T evaluate(Exchange exchange, Class<T> type) {
        return expression.evaluate(exchange, type);
    }

    @Override
    public boolean matches(Exchange exchange) {
        return PredicateBuilder.toPredicate(getExpression()).matches(exchange);
    }

    public Expression getExpression() {
        return expression;
    }

    @Override
    public String toString() {
        return expression.toString();
    }

    // Predicate builders
    // -------------------------------------------------------------------------

    public Predicate matches(Expression expression) {
        return onNewPredicate(ExpressionToPredicateAdapter.toPredicate(expression));
    }

    public MockExpressionClause<Predicate> matches() {
        // chicken-and-egg situation as we need to return an ExpressionClause
        // which needs a right-hand side that is being built via the fluent
        // builder that is returned, and therefore we need to use a ref
        // to the expression (right hand side) that will be used below
        // in the onNewPredicate where the actual matching is executed
        final AtomicReference<Expression> ref = new AtomicReference<>();

        final MockExpressionClause<Predicate> answer = new MockExpressionClause<>(
                onNewPredicate(new Predicate() {
                    @Override
                    public boolean matches(Exchange exchange) {
                        Expression left = expression;
                        Expression right = ref.get();
                        return PredicateBuilder.isEqualTo(left, right).matches(exchange);
                    }

                    @Override
                    public String toString() {
                        return expression + " == " + ref.get();
                    }
                }));

        final Expression right = new ExpressionAdapter() {
            @Override
            public Object evaluate(Exchange exchange) {
                return answer.evaluate(exchange, Object.class);
            }
        };
        // okay, now we can set the reference to the right-hand-side
        ref.set(right);

        return answer;
    }

    public Predicate isTrue() {
        return onNewPredicate(PredicateBuilder.isTrue(expression));
    }

    public Predicate isFalse() {
        return onNewPredicate(PredicateBuilder.isFalse(expression));
    }

    public Predicate isNotEqualTo(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isNotEqualTo(expression, right));
    }

    public Predicate isEqualTo(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isEqualTo(expression, right));
    }

    public Predicate isEqualToIgnoreCase(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isEqualToIgnoreCase(expression, right));
    }

    public Predicate isLessThan(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isLessThan(expression, right));
    }

    public Predicate isLessThanOrEqualTo(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isLessThanOrEqualTo(expression, right));
    }

    public Predicate isGreaterThan(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isGreaterThan(expression, right));
    }

    public Predicate isGreaterThanOrEqualTo(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.isGreaterThanOrEqualTo(expression, right));
    }

    public Predicate isInstanceOf(Class<?> type) {
        return onNewPredicate(PredicateBuilder.isInstanceOf(expression, type));
    }

    public Predicate isNull() {
        return onNewPredicate(PredicateBuilder.isNull(expression));
    }

    public Predicate isNotNull() {
        return onNewPredicate(PredicateBuilder.isNotNull(expression));
    }

    public Predicate not(Predicate predicate) {
        return onNewPredicate(PredicateBuilder.not(predicate));
    }

    public Predicate in(Object... values) {
        List<Predicate> predicates = new ArrayList<>();
        for (Object value : values) {
            Expression right = asExpression(value);
            right = ExpressionBuilder.convertToExpression(right, expression);
            Predicate predicate = PredicateBuilder.isEqualTo(expression, right);
            predicates.add(predicate);
        }
        return in(predicates.toArray(new Predicate[0]));
    }

    public Predicate in(Predicate... predicates) {
        return onNewPredicate(PredicateBuilder.in(predicates));
    }

    public Predicate startsWith(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.startsWith(expression, right));
    }

    public Predicate endsWith(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.endsWith(expression, right));
    }

    /**
     * Create a predicate that the left-hand expression contains the value of the right-hand expression
     *
     * @param  value the element which is compared to be contained within this expression
     * @return       a predicate which evaluates to true if the given value expression is contained within this
     *               expression value
     */
    public Predicate contains(Object value) {
        Expression right = asExpression(value);
        return onNewPredicate(PredicateBuilder.contains(expression, right));
    }

    /**
     * Creates a predicate which is true if this expression matches the given regular expression
     *
     * @param  regex the regular expression to match
     * @return       a predicate which evaluates to true if the expression matches the regex
     */
    public Predicate regex(String regex) {
        return onNewPredicate(PredicateBuilder.regex(expression, regex));
    }

    /**
     * Creates a predicate that compares JSON content semantically, ignoring element order in arrays.
     * <p>
     * This method is useful when testing Camel routes with JSON payloads where the order of elements in JSON arrays or
     * objects can vary, making exact string comparison unreliable.
     * </p>
     * <p>
     * Example usage:
     * </p>
     *
     * <pre>
     * MockEndpoint mock = getMockEndpoint("mock:result");
     * mock.message(0).body().jsonEquals("{\"items\":[{\"id\":1},{\"id\":2}]}");
     * </pre>
     * <p>
     * The above will match even if the actual JSON has the items in a different order:
     * {@code {"items":[{"id":2},{"id":1}]}}
     * </p>
     *
     * @param  expected the expected JSON value (can be String, byte[], InputStream, JsonObject, or JsonArray)
     * @return          a predicate which evaluates to true if the JSON content matches semantically, ignoring array
     *                  element order
     */
    public Predicate jsonEquals(Object expected) {
        return jsonEquals(expected, true);
    }

    /**
     * Creates a predicate that compares JSON content semantically.
     * <p>
     * This method is useful when testing Camel routes with JSON payloads where the order of elements in JSON arrays or
     * objects can vary, making exact string comparison unreliable.
     * </p>
     * <p>
     * Example usage:
     * </p>
     *
     * <pre>
     * MockEndpoint mock = getMockEndpoint("mock:result");
     * // Ignore array element order
     * mock.message(0).body().jsonEquals("{\"items\":[{\"id\":1},{\"id\":2}]}", true);
     *
     * // Strict array order comparison
     * mock.message(1).body().jsonEquals("{\"items\":[{\"id\":1},{\"id\":2}]}", false);
     * </pre>
     *
     * @param  expected    the expected JSON value (can be String, byte[], InputStream, JsonObject, or JsonArray)
     * @param  ignoreOrder if true, ignores the order of elements in JSON arrays; if false, array order must match
     * @return             a predicate which evaluates to true if the JSON content matches semantically
     */
    public Predicate jsonEquals(Object expected, boolean ignoreOrder) {
        return onNewPredicate(new Predicate() {
            @Override
            public boolean matches(Exchange exchange) {
                Object actual = expression.evaluate(exchange, Object.class);
                try {
                    Object expectedJson = parseJson(expected);
                    Object actualJson = parseJson(actual);
                    return jsonDeepEquals(expectedJson, actualJson, ignoreOrder);
                } catch (Exception e) {
                    return false;
                }
            }

            @Override
            public String toString() {
                return expression + " jsonEquals(" + expected + ", ignoreOrder=" + ignoreOrder + ")";
            }
        });
    }

    // Expression builders
    // -------------------------------------------------------------------------

    /**
     * Creates an expression using the custom expression
     *
     * @param  function the custom function
     * @return          a builder with the expression
     */
    public MockValueBuilder expression(Function<Object, Object> function) {
        Expression newExp = ExpressionBuilder.customExpression(this.expression, function);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the given language
     *
     * @param  language the language
     * @param  value    the expression value
     * @return          a builder with the expression
     */
    public MockValueBuilder language(String language, String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, language, value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the simple language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder simple(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "simple", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the datasonnet language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder datasonnet(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "datasonnet", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the groovy language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder groovy(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "groovy", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the javascript language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder js(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "js", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the jq language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder jq(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "jq", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the jsonpath language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder jsonpath(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "jsonpath", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the mvel language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder mvel(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "mvel", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the ognl language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder ognl(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "ognl", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the python language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder python(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "python", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the spel language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder spel(String value) {
        Expression newExp = ExpressionBuilder.languageExpression(expression, "spel", value, Object.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the xpath language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder xpath(String value) {
        // work with string as result as xpath otherwise will use DOM types
        Expression newExp = ExpressionBuilder.languageExpression(expression, "xpath", value, String.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the xquery language
     *
     * @param  value the expression value
     * @return       a builder with the expression
     */
    public MockValueBuilder xquery(String value) {
        // work with string as result as xquery otherwise will use DOM types
        Expression newExp = ExpressionBuilder.languageExpression(expression, "xquery", value, String.class);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the tokenize language using new-line as tokenizer
     *
     * @return a builder with the expression
     */
    public MockValueBuilder tokenize() {
        return tokenize("\n");
    }

    /**
     * Creates an expression using the tokenize language
     *
     * @param  token the token to use
     * @return       a builder with the expression
     */
    public MockValueBuilder tokenize(String token) {
        Expression newExp = ExpressionBuilder.tokenizeExpression(expression, token);
        return onNewValueBuilder(newExp);
    }

    /**
     * Creates an expression using the tokenize language
     *
     * @param  token     the token to use
     * @param  group     number of elements to group
     * @param  skipFirst whether to skip first element
     * @return           a builder with the expression
     */
    public MockValueBuilder tokenize(String token, int group, boolean skipFirst) {
        return tokenize(token, Integer.toString(group), skipFirst);
    }

    /**
     * Creates an expression using the tokenize language
     *
     * @param  token     the token to use
     * @param  group     number of elements to group
     * @param  skipFirst whether to skip first element
     * @return           a builder with the expression
     */
    public MockValueBuilder tokenize(String token, String group, boolean skipFirst) {
        Expression newExp = ExpressionBuilder.tokenizeExpression(expression, token);
        if (group == null && skipFirst) {
            // wrap in skip first (if group then it has its own skip-first logic)
            newExp = ExpressionBuilder.skipFirstExpression(newExp);
        }
        newExp = ExpressionBuilder.groupIteratorExpression(newExp, token, group, skipFirst);
        return onNewValueBuilder(newExp);
    }

    /**
     * Tokenizes the string conversion of this expression using the given regular expression
     */
    public MockValueBuilder regexTokenize(String regex) {
        Expression newExp = ExpressionBuilder.regexTokenizeExpression(expression, regex);
        return onNewValueBuilder(newExp);
    }

    /**
     * Replaces all occurrences of the regular expression with the given replacement
     */
    public MockValueBuilder regexReplaceAll(String regex, String replacement) {
        Expression newExp = ExpressionBuilder.regexReplaceAll(expression, regex, replacement);
        return onNewValueBuilder(newExp);
    }

    /**
     * Replaces all occurrences of the regular expression with the given replacement
     */
    public MockValueBuilder regexReplaceAll(String regex, Expression replacement) {
        Expression newExp = ExpressionBuilder.regexReplaceAll(expression, regex, replacement);
        return onNewValueBuilder(newExp);
    }

    /**
     * Converts the current value to the given type using the registered type converters
     *
     * @param  type the type to convert the value to
     * @return      the current builder
     */
    public MockValueBuilder convertTo(Class<?> type) {
        Expression newExp = ExpressionBuilder.convertToExpression(expression, type);
        return onNewValueBuilder(newExp);
    }

    /**
     * Converts the current value to a String using the registered type converters
     *
     * @return the current builder
     */
    public MockValueBuilder convertToString() {
        return convertTo(String.class);
    }

    /**
     * Appends the string evaluation of this expression with the given value
     *
     * @param  value the value or expression to append
     * @return       the current builder
     */
    public MockValueBuilder append(Object value) {
        Expression newExp = ExpressionBuilder.append(expression, asExpression(value));
        return onNewValueBuilder(newExp);
    }

    /**
     * Prepends the string evaluation of this expression with the given value
     *
     * @param  value the value or expression to prepend
     * @return       the current builder
     */
    public MockValueBuilder prepend(Object value) {
        Expression newExp = ExpressionBuilder.prepend(expression, asExpression(value));
        return onNewValueBuilder(newExp);
    }

    /**
     * Sorts the current value using the given comparator. The current value must be convertable to a {@link List} to
     * allow sorting using the comparator.
     *
     * @param  comparator the comparator used by sorting
     * @return            the current builder
     */
    public MockValueBuilder sort(Comparator<?> comparator) {
        Expression newExp = ExpressionBuilder.sortExpression(expression, comparator);
        return onNewValueBuilder(newExp);
    }

    /**
     * Negates the built expression.
     *
     * @return the current builder
     */
    public MockValueBuilder not() {
        not = true;
        return this;
    }

    // Implementation methods
    // -------------------------------------------------------------------------

    /**
     * A strategy method to allow derived classes to deal with the newly created predicate in different ways
     */
    protected Predicate onNewPredicate(Predicate predicate) {
        if (not) {
            return PredicateBuilder.not(predicate);
        } else {
            return predicate;
        }
    }

    protected Expression asExpression(Object value) {
        if (value instanceof Expression exp) {
            return exp;
        } else {
            return ExpressionBuilder.constantExpression(value);
        }
    }

    protected MockValueBuilder onNewValueBuilder(Expression exp) {
        return new MockValueBuilder(exp);
    }

    /**
     * Parses a JSON value from various input types.
     *
     * @param  value     the value to parse (String, byte[], InputStream, JsonObject, JsonArray, or already parsed
     *                   object)
     * @return           the parsed JSON object (JsonObject, JsonArray, String, Number, Boolean, or null)
     * @throws Exception if parsing fails
     */
    private Object parseJson(Object value) throws Exception {
        if (value == null) {
            return null;
        }

        // Already parsed JSON objects
        if (value instanceof JsonObject || value instanceof JsonArray) {
            return value;
        }

        // Primitives that are valid JSON values
        if (value instanceof String || value instanceof Number || value instanceof Boolean) {
            // For strings, we need to check if it's a JSON string or a plain string
            if (value instanceof String str) {
                str = str.trim();
                // Check if it looks like JSON
                if (str.startsWith("{") || str.startsWith("[") || str.equals("null")
                        || str.equals("true") || str.equals("false")
                        || (str.startsWith("\"") && str.endsWith("\""))
                        || str.matches("-?\\d+(\\.\\d+)?([eE][+-]?\\d+)?")) {
                    try (Reader reader = new StringReader(str)) {
                        return Jsoner.deserialize(reader);
                    }
                }
                // Plain string - return as is for comparison
                return str;
            }
            return value;
        }

        // byte[] input
        if (value instanceof byte[] byteArray) {
            try (InputStream is = new ByteArrayInputStream(byteArray);
                 Reader reader = new InputStreamReader(is, StandardCharsets.UTF_8)) {
                return Jsoner.deserialize(reader);
            }
        }

        // InputStream input
        if (value instanceof InputStream inputstream) {
            try (Reader reader = new InputStreamReader(inputstream, StandardCharsets.UTF_8)) {
                return Jsoner.deserialize(reader);
            }
        }

        // Try to convert to string and parse
        String str = value.toString();
        try (Reader reader = new StringReader(str)) {
            return Jsoner.deserialize(reader);
        }
    }

    /**
     * Performs deep equality comparison of JSON values.
     *
     * @param  expected    the expected JSON value
     * @param  actual      the actual JSON value
     * @param  ignoreOrder if true, ignores the order of elements in JSON arrays
     * @return             true if the JSON values are equal, false otherwise
     */
    private boolean jsonDeepEquals(Object expected, Object actual, boolean ignoreOrder) {
        // Both null
        if (expected == null && actual == null) {
            return true;
        }

        // One is null
        if (expected == null || actual == null) {
            return false;
        }

        // Same reference
        if (expected == actual) {
            return true;
        }

        // JsonObject comparison
        if (expected instanceof JsonObject jsonobject && actual instanceof JsonObject actualJsonObject) {
            return jsonObjectEquals(jsonobject, actualJsonObject, ignoreOrder);
        }

        // Map comparison (in case of deserialized maps)
        if (expected instanceof Map && actual instanceof Map) {
            return jsonMapEquals((Map<?, ?>) expected, (Map<?, ?>) actual, ignoreOrder);
        }

        // JsonArray comparison
        if (expected instanceof JsonArray jsonarray && actual instanceof JsonArray actualJsonArray) {
            return jsonArrayEquals(jsonarray, actualJsonArray, ignoreOrder);
        }

        // List comparison (in case of deserialized lists)
        if (expected instanceof List && actual instanceof List) {
            return jsonListEquals((List<?>) expected, (List<?>) actual, ignoreOrder);
        }

        // Primitive comparison (String, Number, Boolean)
        return expected.equals(actual);
    }

    /**
     * Compares two JsonObjects for equality.
     */
    private boolean jsonObjectEquals(JsonObject expected, JsonObject actual, boolean ignoreOrder) {
        if (expected.size() != actual.size()) {
            return false;
        }

        for (Map.Entry<String, Object> entry : expected.entrySet()) {
            String key = entry.getKey();
            if (!actual.containsKey(key)) {
                return false;
            }
            if (!jsonDeepEquals(entry.getValue(), actual.get(key), ignoreOrder)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Compares two Maps for equality (used for deserialized JSON objects).
     */
    private boolean jsonMapEquals(Map<?, ?> expected, Map<?, ?> actual, boolean ignoreOrder) {
        if (expected.size() != actual.size()) {
            return false;
        }

        for (Map.Entry<?, ?> entry : expected.entrySet()) {
            Object key = entry.getKey();
            if (!actual.containsKey(key)) {
                return false;
            }
            if (!jsonDeepEquals(entry.getValue(), actual.get(key), ignoreOrder)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Compares two JsonArrays for equality.
     */
    private boolean jsonArrayEquals(JsonArray expected, JsonArray actual, boolean ignoreOrder) {
        if (expected.size() != actual.size()) {
            return false;
        }

        if (ignoreOrder) {
            // For unordered comparison, we need to find a matching element for each expected element
            List<Object> actualCopy = new ArrayList<>(actual);
            for (Object expectedElement : expected) {
                boolean found = false;
                for (int i = 0; i < actualCopy.size(); i++) {
                    if (jsonDeepEquals(expectedElement, actualCopy.get(i), ignoreOrder)) {
                        actualCopy.remove(i);
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    return false;
                }
            }
            return true;
        } else {
            // For ordered comparison, compare elements at the same index
            for (int i = 0; i < expected.size(); i++) {
                if (!jsonDeepEquals(expected.get(i), actual.get(i), ignoreOrder)) {
                    return false;
                }
            }
            return true;
        }
    }

    /**
     * Compares two Lists for equality (used for deserialized JSON arrays).
     */
    private boolean jsonListEquals(List<?> expected, List<?> actual, boolean ignoreOrder) {
        if (expected.size() != actual.size()) {
            return false;
        }

        if (ignoreOrder) {
            // For unordered comparison, we need to find a matching element for each expected element
            List<Object> actualCopy = new ArrayList<>(actual);
            for (Object expectedElement : expected) {
                boolean found = false;
                for (int i = 0; i < actualCopy.size(); i++) {
                    if (jsonDeepEquals(expectedElement, actualCopy.get(i), ignoreOrder)) {
                        actualCopy.remove(i);
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    return false;
                }
            }
            return true;
        } else {
            // For ordered comparison, compare elements at the same index
            for (int i = 0; i < expected.size(); i++) {
                if (!jsonDeepEquals(expected.get(i), actual.get(i), ignoreOrder)) {
                    return false;
                }
            }
            return true;
        }
    }
}
