/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.torque.dsfactory.converters;

import java.time.Duration;

import org.apache.commons.beanutils.converters.AbstractConverter;

/**
 * {@link org.apache.commons.beanutils.Converter} implementation that handles conversion
 * to and from <b>java.time.Duration</b> objects.
 * <p>
 * Can be configured to either return a <i>default value</i> or throw a
 * <code>ConversionException</code> if a conversion error occurs.
 *
 * @version $Id$
 * @since 1.3
 */
public final class DurationConverter extends AbstractConverter {

    boolean useTextFormat = false;
    /**
     * Construct a <b>java.time.Duration</b> <i>Converter</i> that throws
     * a <code>ConversionException</code> if an error occurs.
     */
    public DurationConverter() {
        super();
    }
    /**
     * Construct a <b>java.time.DurationL</b> <i>Converter</i> that returns
     * a default value if an error occurs.
     *
     * @param defaultValue The default value to be returned
     * if the value to be converted is missing or an error
     * occurs converting the value.
     * @param useTextFormat if values are in ISO8901 formats.
     */
    public DurationConverter(final Object defaultValue, boolean useTextFormat) {
        super(defaultValue);
        this.useTextFormat = useTextFormat;
    }

    /**
     * Construct a <b>java.time.DurationL</b> <i>Converter</i> that returns
     * a default value if an error occurs.
     *
     * @param defaultValue The default value to be returned
     * if the value to be converted is missing or an error
     * occurs converting the value.
     */
    public DurationConverter(final Object defaultValue) {
        super(defaultValue);
    }
    /**
     * Construct a <b>java.time.DurationL</b> <i>Converter</i> that throws
     * a <code>ConversionException</code> if an error occurs.
     *
     * @param useTextFormat if values are in ISO8901 formats.
     *
     * @see {@link Duration#parse(CharSequence)}
     */
    public DurationConverter(boolean useTextFormat) {
        super();
        this.useTextFormat = useTextFormat;
    }

    /**
     * Return the default type this <code>Converter</code> handles.
     *
     * @return The default type this <code>Converter</code> handles.
     * @since 1.8.0
     */
    @Override
    protected Class<?> getDefaultType() {
        return Duration.class;
    }

    /**
     * <p>Convert a String to a java.time.Duration.</p>
     *
     * @param <T> Target type of the conversion.
     * @param type Data type to which this value should be converted.
     * @param value The input value to be converted.
     * @return The converted value.
     * @throws Throwable if an error occurs converting to the specified type
     */
    @Override
    protected <T> T convertToType(final Class<T> type, final Object value) throws Throwable {
        if (Duration.class.equals(type)) {
            if (useTextFormat) {
                return type.cast( Duration.parse( value.toString() ));
            } else {
                Integer amount = Integer.valueOf( value.toString() );
                return type.cast(Duration.ofMillis(amount));
            }
        }

        throw conversionException(type, value);
    }

}
