/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode.snapshot;

import java.io.IOException;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.apache.hadoop.hdfs.server.namenode.snapshot.Snapshot;
import org.apache.hadoop.hdfs.server.namenode.snapshot.SnapshotTestHelper;
import org.apache.hadoop.hdfs.server.protocol.NamenodeProtocols;
import org.apache.hadoop.util.Time;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public class TestOpenFilesWithSnapshot {
    private static final Log LOG = LogFactory.getLog((String)TestOpenFilesWithSnapshot.class.getName());
    private final Configuration conf = new Configuration();
    MiniDFSCluster cluster = null;
    DistributedFileSystem fs = null;
    private static final long SEED = 0L;
    private static final short REPLICATION = 3;
    private static final long BLOCKSIZE = 1024L;
    private static final long BUFFERLEN = 512L;
    private static final long FILELEN = 2048L;

    @Before
    public void setup() throws IOException {
        this.conf.setBoolean("dfs.namenode.snapshot.capture.openfiles", true);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(3).build();
        this.conf.set("dfs.blocksize", "1048576");
        this.fs = this.cluster.getFileSystem();
    }

    @After
    public void teardown() throws IOException {
        if (this.fs != null) {
            this.fs.close();
            this.fs = null;
        }
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
    }

    @Test
    public void testUCFileDeleteWithSnapShot() throws Exception {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        this.fs.delete(new Path("/test/test/test2"), true);
        this.fs.delete(new Path("/test/test/test3"), true);
        this.restartNameNode();
    }

    @Test
    public void testParentDirWithUCFileDeleteWithSnapShot() throws Exception {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        this.fs.delete(new Path("/test/test"), true);
        this.restartNameNode();
    }

    @Test
    public void testWithCheckpoint() throws Exception {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        this.fs.delete(new Path("/test/test"), true);
        this.restartNameNode();
        String test2snapshotPath = Snapshot.getSnapshotPath((String)path.toString(), (String)"s1/test/test2");
        DFSTestUtil.readFile((FileSystem)this.fs, new Path(test2snapshotPath));
        String test3snapshotPath = Snapshot.getSnapshotPath((String)path.toString(), (String)"s1/test/test3");
        DFSTestUtil.readFile((FileSystem)this.fs, new Path(test3snapshotPath));
    }

    @Test
    public void testFilesDeletionWithCheckpoint() throws Exception {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        this.fs.delete(new Path("/test/test/test2"), true);
        this.fs.delete(new Path("/test/test/test3"), true);
        this.restartNameNode();
        String test2snapshotPath = Snapshot.getSnapshotPath((String)path.toString(), (String)"s1/test/test2");
        DFSTestUtil.readFile((FileSystem)this.fs, new Path(test2snapshotPath));
        String test3snapshotPath = Snapshot.getSnapshotPath((String)path.toString(), (String)"s1/test/test3");
        DFSTestUtil.readFile((FileSystem)this.fs, new Path(test3snapshotPath));
    }

    private void doWriteAndAbort(DistributedFileSystem fs, Path path) throws IOException {
        fs.mkdirs(path);
        fs.allowSnapshot(path);
        DFSTestUtil.createFile((FileSystem)fs, new Path("/test/test1"), 100L, (short)2, 100024L);
        DFSTestUtil.createFile((FileSystem)fs, new Path("/test/test2"), 100L, (short)2, 100024L);
        Path file = new Path("/test/test/test2");
        FSDataOutputStream out = fs.create(file);
        for (int i = 0; i < 2; ++i) {
            for (long count = 0L; count < 0x100000L; count += 4L) {
                out.writeBytes("hell");
            }
        }
        ((DFSOutputStream)out.getWrappedStream()).hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
        DFSTestUtil.abortStream((DFSOutputStream)out.getWrappedStream());
        Path file2 = new Path("/test/test/test3");
        FSDataOutputStream out2 = fs.create(file2);
        for (int i = 0; i < 2; ++i) {
            for (long count = 0L; count < 0x100000L; count += 4L) {
                out2.writeBytes("hell");
            }
        }
        ((DFSOutputStream)out2.getWrappedStream()).hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
        DFSTestUtil.abortStream((DFSOutputStream)out2.getWrappedStream());
        fs.createSnapshot(path, "s1");
    }

    @Test
    public void testOpenFilesWithMultipleSnapshots() throws Exception {
        this.doTestMultipleSnapshots(true);
    }

    @Test
    public void testOpenFilesWithMultipleSnapshotsWithoutCheckpoint() throws Exception {
        this.doTestMultipleSnapshots(false);
    }

    private void doTestMultipleSnapshots(boolean saveNamespace) throws IOException {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        this.fs.createSnapshot(path, "s2");
        this.fs.delete(new Path("/test/test"), true);
        this.fs.deleteSnapshot(path, "s2");
        this.cluster.triggerBlockReports();
        if (saveNamespace) {
            NameNode nameNode = this.cluster.getNameNode();
            NameNodeAdapter.enterSafeMode(nameNode, false);
            NameNodeAdapter.saveNamespace(nameNode);
            NameNodeAdapter.leaveSafeMode(nameNode);
        }
        this.cluster.restartNameNode(true);
    }

    @Test
    public void testOpenFilesWithRename() throws Exception {
        Path path = new Path("/test");
        this.doWriteAndAbort(this.fs, path);
        Path fileWithEmptyBlock = new Path("/test/test/test4");
        this.fs.create(fileWithEmptyBlock);
        NamenodeProtocols nameNodeRpc = this.cluster.getNameNodeRpc();
        String clientName = this.fs.getClient().getClientName();
        nameNodeRpc.addBlock(fileWithEmptyBlock.toString(), clientName, null, null, 0L, null, null);
        this.fs.createSnapshot(path, "s2");
        this.fs.rename(new Path("/test/test"), new Path("/test/test-renamed"));
        this.fs.delete(new Path("/test/test-renamed"), true);
        this.restartNameNode();
    }

    private void createFile(Path filePath) throws IOException {
        DFSTestUtil.createFile((FileSystem)this.fs, filePath, 512, 2048L, 1024L, (short)3, 0L);
    }

    private int writeToStream(FSDataOutputStream outputStream, byte[] buf) throws IOException {
        outputStream.write(buf);
        ((HdfsDataOutputStream)outputStream).hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));
        return buf.length;
    }

    @Test(timeout=120000L)
    public void testPointInTimeSnapshotCopiesForOpenFiles() throws Exception {
        this.conf.setBoolean("dfs.namenode.snapshot.capture.openfiles", true);
        Path level0A = new Path("/level_0_A");
        Path level0B = new Path("/level_0_B");
        Path level1C = new Path(level0A, "level_1_C");
        Path level1D = new Path(level0B, "level_1_D");
        Path level2E = new Path(level1C, "level_2_E");
        Path level3G = new Path(level2E, "level_3_G");
        HashSet<Path> dirPaths = new HashSet<Path>(Arrays.asList(level0A, level0B, level1C, level1D, level2E, level3G));
        for (Path dirPath : dirPaths) {
            this.fs.mkdirs(dirPath);
        }
        Path flumeSnapRootDir = level2E;
        Path hbaseSnapRootDir = level1D;
        String flumeFileName = "flume.log";
        String hbaseFileName = "hbase.log";
        String appAFileName = "appA.log";
        String appBFileName = "appB.log";
        String flumeSnap1Name = "flume_snap_s1";
        String flumeSnap2Name = "flume_snap_s2";
        String flumeSnap3Name = "flume_snap_s3";
        String hbaseSnap1Name = "hbase_snap_s1";
        String hbaseSnap2Name = "hbase_snap_s2";
        String hbaseSnap3Name = "hbase_snap_s3";
        String flumeRelPathFromSnapDir = "level_3_G/flume.log";
        Path flumeFile = new Path(level3G, "flume.log");
        this.createFile(flumeFile);
        FSDataOutputStream flumeOutputStream = this.fs.append(flumeFile);
        Path hbaseFile = new Path(level1D, "hbase.log");
        this.createFile(hbaseFile);
        FSDataOutputStream hbaseOutputStream = this.fs.append(hbaseFile);
        Path appAFile = new Path(level1C, "appA.log");
        this.createFile(appAFile);
        FSDataOutputStream appAOutputStream = this.fs.append(appAFile);
        Path appBFile = new Path(level0B, "appB.log");
        this.createFile(appBFile);
        FSDataOutputStream appBOutputStream = this.fs.append(appBFile);
        long appAFileInitialLength = this.fs.getFileStatus(appAFile).getLen();
        long appBFileInitialLength = this.fs.getFileStatus(appBFile).getLen();
        Path flumeS1Dir = SnapshotTestHelper.createSnapshot(this.fs, flumeSnapRootDir, "flume_snap_s1");
        Path flumeS1Path = new Path(flumeS1Dir, "level_3_G/flume.log");
        Path hbaseS1Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s1");
        Path hbaseS1Path = new Path(hbaseS1Dir, "hbase.log");
        long flumeFileLengthAfterS1 = this.fs.getFileStatus(flumeFile).getLen();
        long hbaseFileLengthAfterS1 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS1, (long)this.fs.getFileStatus(hbaseS1Path).getLen());
        Assert.assertEquals((long)appAFileInitialLength, (long)this.fs.getFileStatus(appAFile).getLen());
        Assert.assertEquals((long)appBFileInitialLength, (long)this.fs.getFileStatus(appBFile).getLen());
        long flumeFileWrittenDataLength = flumeFileLengthAfterS1;
        long hbaseFileWrittenDataLength = hbaseFileLengthAfterS1;
        long appAFileWrittenDataLength = appAFileInitialLength;
        int newWriteLength = 1536;
        byte[] buf = new byte[newWriteLength];
        Random random = new Random();
        random.nextBytes(buf);
        flumeFileWrittenDataLength += (long)this.writeToStream(flumeOutputStream, buf);
        hbaseFileWrittenDataLength += (long)this.writeToStream(hbaseOutputStream, buf);
        Path flumeS2Dir = SnapshotTestHelper.createSnapshot(this.fs, flumeSnapRootDir, "flume_snap_s2");
        Path flumeS2Path = new Path(flumeS2Dir, "level_3_G/flume.log");
        Path hbaseS2Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s2");
        Path hbaseS2Path = new Path(hbaseS2Dir, "hbase.log");
        long flumeFileLengthAfterS2 = this.fs.getFileStatus(flumeFile).getLen();
        long hbaseFileLengthAfterS2 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)flumeFileWrittenDataLength, (long)flumeFileLengthAfterS2);
        Assert.assertEquals((long)hbaseFileWrittenDataLength, (long)hbaseFileLengthAfterS2);
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS2, (long)this.fs.getFileStatus(hbaseS2Path).getLen());
        Assert.assertEquals((long)appAFileInitialLength, (long)this.fs.getFileStatus(appAFile).getLen());
        Assert.assertEquals((long)appBFileInitialLength, (long)this.fs.getFileStatus(appBFile).getLen());
        newWriteLength = 2560;
        buf = new byte[newWriteLength];
        random.nextBytes(buf);
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)(appAFileWrittenDataLength += (long)this.writeToStream(appAOutputStream, buf)), (long)this.fs.getFileStatus(appAFile).getLen());
        newWriteLength = 2560;
        buf = new byte[newWriteLength];
        random.nextBytes(buf);
        flumeFileWrittenDataLength += (long)this.writeToStream(flumeOutputStream, buf);
        Path flumeS3Dir = SnapshotTestHelper.createSnapshot(this.fs, flumeSnapRootDir, "flume_snap_s3");
        Path flumeS3Path = new Path(flumeS3Dir, "level_3_G/flume.log");
        Path hbaseS3Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s3");
        Path hbaseS3Path = new Path(hbaseS3Dir, "hbase.log");
        long flumeFileLengthAfterS3 = this.fs.getFileStatus(flumeFile).getLen();
        long hbaseFileLengthAfterS3 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)flumeFileWrittenDataLength, (long)flumeFileLengthAfterS3);
        Assert.assertEquals((long)hbaseFileWrittenDataLength, (long)hbaseFileLengthAfterS3);
        Assert.assertEquals((long)flumeFileLengthAfterS3, (long)this.fs.getFileStatus(flumeS3Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS3, (long)this.fs.getFileStatus(hbaseS3Path).getLen());
        Assert.assertEquals((long)appAFileWrittenDataLength, (long)this.fs.getFileStatus(appAFile).getLen());
        Assert.assertEquals((long)appBFileInitialLength, (long)this.fs.getFileStatus(appBFile).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS3, (long)this.fs.getFileStatus(flumeS3Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS1, (long)this.fs.getFileStatus(hbaseS1Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS2, (long)this.fs.getFileStatus(hbaseS2Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS3, (long)this.fs.getFileStatus(hbaseS3Path).getLen());
        flumeOutputStream.close();
        hbaseOutputStream.close();
        appAOutputStream.close();
        appBOutputStream.close();
    }

    @Test(timeout=120000L)
    public void testSnapshotsForOpenFilesWithNNRestart() throws Exception {
        Path level0A;
        Path flumeSnapRootDir = level0A = new Path("/level_0_A");
        String flumeFileName = "flume.log";
        String flumeSnap1Name = "flume_snap_1";
        String flumeSnap2Name = "flume_snap_2";
        Path flumeFile = new Path(level0A, "flume.log");
        this.createFile(flumeFile);
        FSDataOutputStream flumeOutputStream = this.fs.append(flumeFile);
        Path flumeS1Dir = SnapshotTestHelper.createSnapshot(this.fs, flumeSnapRootDir, "flume_snap_1");
        Path flumeS1Path = new Path(flumeS1Dir, "flume.log");
        long flumeFileLengthAfterS1 = this.fs.getFileStatus(flumeFile).getLen();
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        long flumeFileWrittenDataLength = flumeFileLengthAfterS1;
        int newWriteLength = 1536;
        byte[] buf = new byte[newWriteLength];
        Random random = new Random();
        random.nextBytes(buf);
        Path flumeS2Dir = SnapshotTestHelper.createSnapshot(this.fs, flumeSnapRootDir, "flume_snap_2");
        Path flumeS2Path = new Path(flumeS2Dir, "flume.log");
        long flumeFileLengthAfterS2 = this.fs.getFileStatus(flumeFile).getLen();
        Assert.assertEquals((long)(flumeFileWrittenDataLength += (long)this.writeToStream(flumeOutputStream, buf)), (long)flumeFileLengthAfterS2);
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        this.restartNameNode();
        this.cluster.waitActive();
        Assert.assertEquals((long)(flumeFileWrittenDataLength += (long)this.writeToStream(flumeOutputStream, buf)), (long)this.fs.getFileStatus(flumeFile).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        flumeOutputStream.close();
    }

    @Test(timeout=120000L)
    public void testSnapshotsForOpenFilesAndDeletion() throws Exception {
        Path snapRootDir = new Path("/level_0_A");
        String flumeFileName = "flume.log";
        String hbaseFileName = "hbase.log";
        String snap1Name = "snap_1";
        String snap2Name = "snap_2";
        String snap3Name = "snap_3";
        Path flumeFile = new Path(snapRootDir, "flume.log");
        this.createFile(flumeFile);
        Path hbaseFile = new Path(snapRootDir, "hbase.log");
        this.createFile(hbaseFile);
        FSDataOutputStream flumeOutputStream = this.fs.append(flumeFile);
        FSDataOutputStream hbaseOutputStream = this.fs.append(hbaseFile);
        Path snap1Dir = SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_1");
        Path flumeS1Path = new Path(snap1Dir, "flume.log");
        long flumeFileLengthAfterS1 = this.fs.getFileStatus(flumeFile).getLen();
        Path hbaseS1Path = new Path(snap1Dir, "hbase.log");
        long hbaseFileLengthAfterS1 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS1, (long)this.fs.getFileStatus(hbaseS1Path).getLen());
        long flumeFileWrittenDataLength = flumeFileLengthAfterS1;
        long hbaseFileWrittenDataLength = hbaseFileLengthAfterS1;
        int newWriteLength = 1536;
        byte[] buf = new byte[newWriteLength];
        Random random = new Random();
        random.nextBytes(buf);
        flumeFileWrittenDataLength += (long)this.writeToStream(flumeOutputStream, buf);
        hbaseFileWrittenDataLength += (long)this.writeToStream(hbaseOutputStream, buf);
        Path snap2Dir = SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_2");
        Path flumeS2Path = new Path(snap2Dir, "flume.log");
        Path hbaseS2Path = new Path(snap2Dir, "hbase.log");
        long flumeFileLengthAfterS2 = this.fs.getFileStatus(flumeFile).getLen();
        Assert.assertEquals((long)flumeFileWrittenDataLength, (long)flumeFileLengthAfterS2);
        long hbaseFileLengthAfterS2 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)hbaseFileWrittenDataLength, (long)hbaseFileLengthAfterS2);
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS2, (long)this.fs.getFileStatus(hbaseS2Path).getLen());
        this.writeToStream(flumeOutputStream, buf);
        hbaseFileWrittenDataLength += (long)this.writeToStream(hbaseOutputStream, buf);
        Assert.assertEquals((long)flumeFileLengthAfterS1, (long)this.fs.getFileStatus(flumeS1Path).getLen());
        Assert.assertEquals((long)flumeFileLengthAfterS2, (long)this.fs.getFileStatus(flumeS2Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS1, (long)this.fs.getFileStatus(hbaseS1Path).getLen());
        Assert.assertEquals((long)hbaseFileLengthAfterS2, (long)this.fs.getFileStatus(hbaseS2Path).getLen());
        boolean flumeFileDeleted = this.fs.delete(flumeFile, true);
        Assert.assertTrue((boolean)flumeFileDeleted);
        Assert.assertFalse((boolean)this.fs.exists(flumeFile));
        Assert.assertTrue((boolean)this.fs.exists(flumeS1Path));
        Assert.assertTrue((boolean)this.fs.exists(flumeS2Path));
        SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "tmp_snap");
        this.fs.deleteSnapshot(snapRootDir, "tmp_snap");
        this.fs.deleteSnapshot(snapRootDir, "snap_2");
        Assert.assertFalse((boolean)this.fs.exists(flumeS2Path));
        Assert.assertTrue((boolean)this.fs.exists(flumeS1Path));
        this.fs.deleteSnapshot(snapRootDir, "snap_1");
        Assert.assertFalse((boolean)this.fs.exists(flumeS2Path));
        Assert.assertFalse((boolean)this.fs.exists(flumeS1Path));
        Path snap3Dir = SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_3");
        Path hbaseS3Path = new Path(snap3Dir, "hbase.log");
        long hbaseFileLengthAfterS3 = this.fs.getFileStatus(hbaseFile).getLen();
        Assert.assertEquals((long)hbaseFileWrittenDataLength, (long)hbaseFileLengthAfterS3);
        Assert.assertEquals((long)hbaseFileLengthAfterS3, (long)this.fs.getFileStatus(hbaseS3Path).getLen());
        Assert.assertEquals((long)(hbaseFileWrittenDataLength += (long)this.writeToStream(hbaseOutputStream, buf)), (long)this.fs.getFileStatus(hbaseFile).getLen());
        hbaseOutputStream.close();
    }

    @Test(timeout=600000L)
    public void testOpenFileDeletionAndNNRestart() throws Exception {
        Path snapRootDir = new Path("/level_0_A/test");
        String hbaseFileName = "hbase.log";
        String snap1Name = "snap_1";
        Path hbaseFile = new Path(snapRootDir, "hbase.log");
        this.createFile(hbaseFile);
        FSDataOutputStream hbaseOutputStream = this.fs.append(hbaseFile);
        int newWriteLength = 1536;
        byte[] buf = new byte[newWriteLength];
        Random random = new Random();
        random.nextBytes(buf);
        this.writeToStream(hbaseOutputStream, buf);
        Path snap1Dir = SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_1");
        LOG.info((Object)("Open file status in snap: " + this.fs.getFileStatus(new Path(snap1Dir, "hbase.log"))));
        this.fs.delete(hbaseFile, true);
        this.fs.deleteSnapshot(snapRootDir, "snap_1");
        Assert.assertFalse((boolean)this.fs.exists(hbaseFile));
        this.cluster.restartNameNode(new String[0]);
        this.cluster.waitActive();
        Assert.assertFalse((boolean)this.fs.exists(hbaseFile));
    }

    @Test(timeout=240000L)
    public void testOpenFileWritingAcrossSnapDeletion() throws Exception {
        Path snapRootDir = new Path("/level_0_A");
        String flumeFileName = "flume.log";
        String hbaseFileName = "hbase.log";
        String snap1Name = "snap_1";
        String snap2Name = "snap_2";
        String snap3Name = "snap_3";
        final Path flumeFile = new Path(snapRootDir, "flume.log");
        FSDataOutputStream flumeOut = this.fs.create(flumeFile, false, 8000, (short)3, 0x100000L);
        flumeOut.close();
        final Path hbaseFile = new Path(snapRootDir, "hbase.log");
        FSDataOutputStream hbaseOut = this.fs.create(hbaseFile, false, 8000, (short)3, 0x100000L);
        hbaseOut.close();
        final AtomicBoolean writerError = new AtomicBoolean(false);
        final CountDownLatch startLatch = new CountDownLatch(1);
        final CountDownLatch deleteLatch = new CountDownLatch(1);
        Thread t = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    FSDataOutputStream flumeOutputStream = TestOpenFilesWithSnapshot.this.fs.append(flumeFile, 8000);
                    FSDataOutputStream hbaseOutputStream = TestOpenFilesWithSnapshot.this.fs.append(hbaseFile, 8000);
                    byte[] bytes = new byte[204];
                    Random r = new Random(Time.now());
                    for (int i = 0; i < 200000; ++i) {
                        r.nextBytes(bytes);
                        flumeOutputStream.write(bytes);
                        if (hbaseOutputStream != null) {
                            hbaseOutputStream.write(bytes);
                        }
                        if (i == 50000) {
                            startLatch.countDown();
                            continue;
                        }
                        if (i == 100000) {
                            deleteLatch.countDown();
                            continue;
                        }
                        if (i == 150000) {
                            hbaseOutputStream.hsync();
                            TestOpenFilesWithSnapshot.this.fs.delete(hbaseFile, true);
                            try {
                                hbaseOutputStream.close();
                            }
                            catch (Exception exception) {
                                // empty catch block
                            }
                            hbaseOutputStream = null;
                            continue;
                        }
                        if (i % 5000 != 0) continue;
                        LOG.info((Object)("Write pos: " + flumeOutputStream.getPos() + ", size: " + TestOpenFilesWithSnapshot.this.fs.getFileStatus(flumeFile).getLen() + ", loop: " + (i + 1)));
                    }
                }
                catch (Exception e) {
                    LOG.warn((Object)("Writer error: " + e));
                    writerError.set(true);
                }
            }
        });
        t.start();
        startLatch.await();
        Path snap1Dir = SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_1");
        Path flumeS1Path = new Path(snap1Dir, "flume.log");
        LOG.info((Object)("Snap1 file status: " + this.fs.getFileStatus(flumeS1Path)));
        LOG.info((Object)("Current file status: " + this.fs.getFileStatus(flumeFile)));
        deleteLatch.await();
        LOG.info((Object)("Snap1 file status: " + this.fs.getFileStatus(flumeS1Path)));
        LOG.info((Object)("Current file status: " + this.fs.getFileStatus(flumeFile)));
        LOG.info((Object)"Deleting snap_1");
        this.fs.deleteSnapshot(snapRootDir, "snap_1");
        SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_2");
        SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "snap_3");
        this.fs.deleteSnapshot(snapRootDir, "snap_3");
        this.fs.deleteSnapshot(snapRootDir, "snap_2");
        SnapshotTestHelper.createSnapshot(this.fs, snapRootDir, "test");
        t.join();
        Assert.assertFalse((String)"Client encountered writing error!", (boolean)writerError.get());
        this.restartNameNode();
        this.cluster.waitActive();
    }

    @Test(timeout=120000L)
    public void testOpenFilesSnapChecksumWithTrunkAndAppend() throws Exception {
        this.conf.setBoolean("dfs.namenode.snapshot.capture.openfiles", true);
        Path dir = new Path("/A/B/C");
        this.fs.mkdirs(dir);
        Path hbaseSnapRootDir = dir;
        String hbaseFileName = "hbase.wal";
        String hbaseSnap1Name = "hbase_snap_s1";
        String hbaseSnap2Name = "hbase_snap_s2";
        String hbaseSnap3Name = "hbase_snap_s3";
        String hbaseSnap4Name = "hbase_snap_s4";
        Path hbaseFile = new Path(dir, "hbase.wal");
        this.createFile(hbaseFile);
        FileChecksum hbaseWALFileCksum0 = this.fs.getFileChecksum(hbaseFile);
        FSDataOutputStream hbaseOutputStream = this.fs.append(hbaseFile);
        Path hbaseS1Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s1");
        Path hbaseS1Path = new Path(hbaseS1Dir, "hbase.wal");
        FileChecksum hbaseFileCksumS1 = this.fs.getFileChecksum(hbaseS1Path);
        Assert.assertEquals((String)"Live and snap1 file checksum doesn't match!", (Object)hbaseWALFileCksum0, (Object)this.fs.getFileChecksum(hbaseS1Path));
        int newWriteLength = 1536;
        byte[] buf = new byte[newWriteLength];
        Random random = new Random();
        random.nextBytes(buf);
        this.writeToStream(hbaseOutputStream, buf);
        Path hbaseS2Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s2");
        Path hbaseS2Path = new Path(hbaseS2Dir, "hbase.wal");
        FileChecksum hbaseFileCksumS2 = this.fs.getFileChecksum(hbaseS2Path);
        Assert.assertEquals((String)"Snap file checksum has changed!", (Object)hbaseFileCksumS1, (Object)this.fs.getFileChecksum(hbaseS1Path));
        Assert.assertNotEquals((String)"Snap1 and snap2 file checksum should differ!", (Object)hbaseFileCksumS1, (Object)hbaseFileCksumS2);
        newWriteLength = 2560;
        buf = new byte[newWriteLength];
        random.nextBytes(buf);
        this.writeToStream(hbaseOutputStream, buf);
        Path hbaseS3Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s3");
        Path hbaseS3Path = new Path(hbaseS3Dir, "hbase.wal");
        FileChecksum hbaseFileCksumS3 = this.fs.getFileChecksum(hbaseS3Path);
        hbaseOutputStream.close();
        FileChecksum hbaseFileCksumBeforeTruncate = this.fs.getFileChecksum(hbaseFile);
        Assert.assertEquals((String)"Snap3 and before truncate file checksum should match!", (Object)hbaseFileCksumBeforeTruncate, (Object)hbaseFileCksumS3);
        long currentFileLen = this.fs.getFileStatus(hbaseFile).getLen();
        boolean fileTruncated = this.fs.truncate(hbaseFile, currentFileLen / 2L);
        Assert.assertTrue((String)"File truncation failed!", (boolean)fileTruncated);
        FileChecksum hbaseFileCksumAfterTruncate = this.fs.getFileChecksum(hbaseFile);
        Assert.assertNotEquals((String)"Snap3 and after truncate checksum shouldn't match!", (Object)hbaseFileCksumS3, (Object)hbaseFileCksumAfterTruncate);
        hbaseOutputStream = this.fs.append(hbaseFile);
        newWriteLength = 5632;
        buf = new byte[newWriteLength];
        random.nextBytes(buf);
        this.writeToStream(hbaseOutputStream, buf);
        Path hbaseS4Dir = SnapshotTestHelper.createSnapshot(this.fs, hbaseSnapRootDir, "hbase_snap_s4");
        Path hbaseS4Path = new Path(hbaseS4Dir, "hbase.wal");
        FileChecksum hbaseFileCksumS4 = this.fs.getFileChecksum(hbaseS4Path);
        hbaseOutputStream.close();
        FileChecksum hbaseFileCksumAfterAppend = this.fs.getFileChecksum(hbaseFile);
        Assert.assertEquals((String)"Snap4 and after append file checksum should match!", (Object)hbaseFileCksumAfterAppend, (Object)hbaseFileCksumS4);
        hbaseFileCksumS3 = this.fs.getFileChecksum(hbaseS3Path);
        Assert.assertEquals((String)"Snap3 and before truncate file checksum should match!", (Object)hbaseFileCksumBeforeTruncate, (Object)hbaseFileCksumS3);
    }

    private void restartNameNode() throws Exception {
        this.cluster.triggerBlockReports();
        NameNode nameNode = this.cluster.getNameNode();
        NameNodeAdapter.enterSafeMode(nameNode, false);
        NameNodeAdapter.saveNamespace(nameNode);
        NameNodeAdapter.leaveSafeMode(nameNode);
        this.cluster.restartNameNode(true);
    }
}

