/* 
   - The contents of this file are subject to the Mozilla Public
   - License Version .1 (the "License"); you may not use this file
   - except in compliance with the License. You may obtain a copy of
   - the License at http://www.mozilla.org/MPL/
   -
   - Software distributed under the License is distributed on an "AS
   - IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
   - implied. See the License for the specific language governing
   - rights and limitations under the License.
   -
   - The Original Code is this file, written on 20 September 2000,
   - released on multizilla.mozdev.org on Aprils fools day in 2001.
   -
   - The Initial Developer of the Original Code is HJ van Rantwijk.
   - Portions created by HJ van Rantwijk are Copyright (C) 2000-2008
   - HJ van Rantwijk.  All Rights Reserved.
   - 
   - Contributor(s):
   -    David Perry <d.perry@utoronto.ca>
   -	Leland M. George <Leland.M.George@alum.mit.edu> (bug 9547,10119)
   -	Konstantin Svist <fry.kun@gmail.com> (bug 10119)
   -    Rajnish Bhaskar <raj@lordofthemoon.com>
   -	Philip Chee <philip.chee@gmail.com> (bug 16330/16332/16468)
   -	Michael Vincent van Rantwijk <mv_van_rantwijk@yahoo.com>
   - 
*/

function gMultiZilla() {
  
}

gMultiZilla.prototype =
{
  _closeConfirmationID    : -1,
  _showPopupIconOnTab     : true,
  _delayedPopupIconRemoval: 0,
  _firstBrowserWindow     : null,
  _multiZillaToolbar      : null,
  _nsIObserverService     : null,
  _nsIStringBundleService : null,

  /***
    * MultiZilla observers
    */
  _observer_Managers: /* XXX: Formerly known as: gManagerObserver */
  {
    /***
      * subject: 'tabManager-window', 'sessionManager-window', 
      *          'userAgentManager-window' or 'permissionManager-window'
      * topic  : 'MZManagers'
      * data   : 'start'/'stop' are triggered when a manager is started/closed
      */
    observe: function(subject, topic, data) {
      // prevent startup errors and filter out all unrelated notifications
      if (!gBrowser || topic != 'MZManagers')
        return;
  
      var bitValue = 0;
      subject = subject.id.replace(/-window/, ''); // remove '-window' part from subject 
  
      switch(subject)
      {
        /***
          * used bits: 00008421
          */
        case "tabManager":
          bitValue = 1;
          break;
        case "sessionManager":
          bitValue = 2;
          break;
        case "userAgentManager":
          bitValue = 4;
          break;
        case "permissionManager":
          bitValue = 8;
          break;
      }
      if (bitValue) {
        // 'start' and 'stop' are triggered when any of the MZ manager are started/closed
        gBrowser.mManagers = (data == 'start') ? gBrowser.mManagers |= bitValue
                                               : gBrowser.mManagers &= ~bitValue;
      }
    }
  },

  /* XXX: Formerly known as: mzFeedRefreshObserver */
  _observer_FeedRefresh:
  {
    observe: function(subject, topic, state) {
      var feedButton = document.getElementById("newFeedContent-button");
  
      if (feedButton)
        feedButton.setAttribute("state", (state ? String(state) : "0"));
    }
  },

  /* XXX: Formerly known as: gXPInstallObserver */
  _observer_XPInstall:
  {
    _multizillaBundle: null,
    _XPInstallEnabledPrefString : "xpinstall.enabled",
    _XPInstallWhiteListPrefString : "xpinstall.whitelist.required",
    _XPInstallSoundEnabledPrefString : "multizilla.blocked-xpinstall.sound_enabled",
    _XPInstallSoundPrefString : "multizilla.blocked-xpinstall.sound_url",
    _XPInstallImageURL : "chrome://multiviews/content/images/managers/xpinstall-small.png",
    _nsIPermissionManager : Components.classes["@mozilla.org/permissionmanager;1"]
                                      .getService(Components.interfaces.nsIPermissionManager),
    _nsIIOService : Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService),
    _nsIURI : Components.classes["@mozilla.org/network/standard-url;1"]
                        .createInstance(Components.interfaces.nsIURI),
    notificationIcon: "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAANbY1E9YMgAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAICSURBVHjaYvz//z8DJQAggJhwiDPvnmlzc2aR0O+JGezt+AwACCCsBhxfYhn59N41FWtXIxZOLu70niRGXVwGAAQQNgNYHj96O8HaWYdJW5ubwd4/mI2Ng7sblwEAAYRhwMm1URk/vn4SUNWVYGD8+YZBXZOZm5OLzRjoCmNsBgAEEKoBN82Y7l851GLrqMjM8Oc7A8O/3wwMP54wuAQFCXNycUzGZgBAAKEYcOaKZO2/f//5FbUVgBrfMoRVcgHpNwyKGjKMXDwCan0prFboBgAEELIBzDcvXyy2cVZhYPj9GWj7H4jo/38MDJ9OMDj7O/KzsjH3oxsAEEBwA/bNNipiZf7FI6cqwcDw8x2qqp8fGORUpVn4BEXlgGHhhCwFEEAwA9gfP3hdZ+Oizcjw+wvCdjgAuuLrFQbXIH9hTm7uqcgyAAEENuD4ctcebm5mbikFYRTbV7V/Q6j88Z5BSuY7q4CQgAjQFR4wYYAAAhtw89L5ZFsnRaDtn4CW/YXrAQcisit+PGVwDgrnZ2NnnwATBQggpsNLvGYLCAmxi8tLARWg+h3FBVBXSEj/ZZWQkRcCuiIQJAQQQCyvnj5KMDTkZ2JgYmRg4FchnHv+vmEwttLmeXT3VjKQtx4ggFgk5TXebV63UfT3ijOMxOZAVlZWdiB1EMQGCCBGSrMzQIABAFR3kRM3KggZAAAAAElFTkSuQmCC",
  
    observe: function(aSubject, aTopic, aData)
    {
      switch (aTopic)
      {
        case "xpinstall-install-blocked":
          var shell = aSubject.QueryInterface(Components.interfaces.nsIDocShell);
          var browser = this._getBrowser(shell);
  
          if (!this._multizillaBundle)
            this._multizillaBundle = gMultiZilla.utility.getStringBundle();

          if (browser) {
            var host = this.getHostFromBrowser(browser);
            var haveHost = (host != null);
  
            if (!haveHost)
              host = this._multizillaBundle.GetStringFromName("xpinstallHostUnavailable");
  
            var brandShortName = "MultiZilla";
            var notificationName, messageString, buttons;
  
            if (!this.getXPInstallPref()) {
              notificationName = "xpinstall-disabled";
  
              if (pref.prefIsLocked("xpinstall.enabled")) {
                messageString = this._multizillaBundle.GetStringFromName("xpinstallDisabledMessageLocked");
                buttons = [];
              }
              else {
                messageString = this._multizillaBundle.formatStringFromName("xpinstallDisabledMessage", [brandShortName, host], 2);
  
                buttons = [{
                  label: this._multizillaBundle.GetStringFromName("xpinstallDisabledButton"),
                  accessKey: this._multizillaBundle.GetStringFromName("xpinstallDisabledButton.accesskey"),
                  popup: null,
                  callback: function editPrefs() {
                    pref.setBoolPref("xpinstall.enabled", true);
                    return false;
                  }
                }];
              }
            }
            else {
              notificationName = "xpinstall-blocked";
              messageString = this._multizillaBundle.formatStringFromName("xpinstallBlockedMessage", [host], 1);
  
              buttons = [{
                label: this._multizillaBundle.GetStringFromName("xpinstallPromptWarningButton"),
                accessKey: this._multizillaBundle.GetStringFromName("xpinstallPromptWarningButton.accesskey"),
                popup: "XPInstallPopupOptions",
                callback: null // function() { return xpinstallEditPermissions(browser.docShell); }
              }];
            }
            var notificationBox = gBrowser.getNotificationBox(browser);
  
            if (!notificationBox.getNotificationWithValue(notificationName)) {
              const priority = notificationBox.PRIORITY_INFO_MEDIUM;
              const iconURL = this._XPInstallImageURL;
              notificationBox.appendNotification(messageString, notificationName, iconURL, priority, buttons);
            }
          }
          if (gMultiZilla.prefs.readBoolean(this._XPInstallSoundEnabledPrefString, true))
            mzPlaySound(gMultiZilla.prefs.readString(this._XPInstallSoundPrefString, ""));
          break;
      }
    },
  
    initPopup: function(aEvent, aSpecifiedHost)
    {
      var browser = gBrowser.mCurrentBrowser;
  
      if (browser) {
        var host = null;
        var haveHost = false;
        var permission = -1;
        var isXPIEnabled = this.getXPInstallPref();
        var requireWhiteList = this.getXPInstallWhiteListPref();
  
        if (!this._multizillaBundle)
          this._multizillaBundle = gMultiZilla.utility.getStringBundle();

        if (aSpecifiedHost == undefined) { // Called from Info Message or QuickPrefs Menu
          host = this.getHostFromBrowser(browser);
          haveHost = (host != null);
          // dump("\nhost: " + host);
        }
        else { // Called from XPInstall Confirmation Window
          host = aSpecifiedHost;
          haveHost = true;
        }
  
        if (!haveHost)
          host = this._multizillaBundle.GetStringFromName("xpinstallHostUnavailable");
        else
          permission = this.getXPInstallPermission(host);
  
        aEvent.target.childNodes[0].hidden = isXPIEnabled;
        aEvent.target.childNodes[1].hidden = !isXPIEnabled;
        aEvent.target.childNodes[2].setAttribute("checked", requireWhiteList);
        // aEvent.target.childNodes[3] is a menuseparator
        aEvent.target.childNodes[4].setAttribute("disabled", !haveHost);
        aEvent.target.childNodes[4].setAttribute("checked", (permission == this._nsIPermissionManager.ALLOW_ACTION));
        aEvent.target.childNodes[4].label = this._multizillaBundle.formatStringFromName("xpinstallAllow", [host], 1);
        aEvent.target.childNodes[5].setAttribute("disabled", !haveHost);
        aEvent.target.childNodes[5].setAttribute("checked", (permission == this._nsIPermissionManager.DENY_ACTION));
        aEvent.target.childNodes[5].label = this._multizillaBundle.formatStringFromName("xpinstallBlock", [host], 1);
        aEvent.target.childNodes[6].setAttribute("disabled", !haveHost);
        aEvent.target.childNodes[6].setAttribute("checked", (permission == this._nsIPermissionManager.UNKNOWN_ACTION));
        aEvent.target.childNodes[6].label = this._multizillaBundle.formatStringFromName("xpinstallDefault", [host], 1);
        // aEvent.target.childNodes[7] is a menuseparator
      }
    },
  
    getHostFromBrowser: function(aBrowser)
    {
      var lastBlockedXPInstallURL = aBrowser.lastBlockedXPInstallURL;
      var currentURI = aBrowser.docShell.QueryInterface(Components.interfaces.nsIWebNavigation).currentURI;
      var url = (lastBlockedXPInstallURL) ? lastBlockedXPInstallURL : currentURI.spec;
      var uri = this._nsIIOService.newURI(url, null, null);
  
      try {
        return (uri.scheme == "file") ? "file:" : uri.hostPort;
      } catch(ex) {
        // dump("\nin catch{}");
        // We must been called from: History, Bookmarks or the Location bar
        var host = url.replace(/^\s*([-\w]*:\/+)?/, "");
        host = host.substring(0, host.indexOf("/"));
  
        if (host != "")
          return host;
      }
      return null;
    },
  
    findChildShell: function(aDocShell, aSoughtShell)
    {
      if (aDocShell == aSoughtShell)
        return aDocShell;
  
      var node = aDocShell.QueryInterface(Components.interfaces.nsIDocShellTreeNode);
  
      for (var i = 0; i < node.childCount; ++i) {
        var docShell = node.getChildAt(i);
  
        docShell = this.findChildShell(docShell, aSoughtShell);
  
        if (docShell == aSoughtShell)
          return docShell;
      }
      return null;
    },
  
    _getBrowser: function(aDocShell)
    {
      var tabBrowser = getBrowser();
      const browsers = tabBrowser.browsers;
  
      for (var i = 0; i < browsers.length; ++i) {
        var browser = browsers[i];
  
        if (this.findChildShell(browser.docShell, aDocShell))
          return browser;
      }
      return null;
    },
  
    getXPInstallPref: function()
    {
       return gMultiZilla.prefs.readBoolean(this._XPInstallEnabledPrefString, true);
    },
  
    getXPInstallWhiteListPref: function(aEvent)
    {
       return gMultiZilla.prefs.readBoolean(this._XPInstallWhiteListPrefString, true);
    },
  
    getHostURI: function(aHost)
    {
      var browser = gBrowser.mCurrentBrowser;
      var url = aHost;
  
      if (url == undefined) // Called from QPrefs 'Extension Management'
        url = (browser.lastBlockedXPInstallURL) ? browser.lastBlockedXPInstallURL : browser.currentURI.spec;
  
      var isFileScheme = url.match(/(scheme:file|file:)/i);
      url = url.replace(/^\s*([-\w]*:\/+)?/, ""); // trim any leading space and scheme
      var host = isFileScheme ? 'file:' : 'http://' + url;
      return this._nsIIOService.newURI(host, null, null);
    },
  
    getXPInstallPermission: function(aHost)
    {
      var host = aHost;
  
      if (typeof aHost == "string")
        host = this.getHostURI(aHost);
      return this._nsIPermissionManager.testPermission(host, "install");
    },
  
    setXPInstallPref: function(aEvent)
    {
       gMultiZilla.prefs.writeBoolean( this._XPInstallEnabledPrefString, (aEvent.target.id == "enableXPInstall") );
    },
  
    setXPInstallWhiteListPref: function(aEvent)
    {
       gMultiZilla.prefs.writeBoolean( this._XPInstallWhiteListPrefString, (aEvent.target.getAttribute("checked") == "true") );
    },
  
    setXPInstallPermission: function(aHost, aPermission)
    {
      var host = aHost;
  
      if (aHost == undefined || typeof aHost == "string")
        host = this.getHostURI(aHost);
      this._nsIPermissionManager.add(host, "install", aPermission);
    }
  },

  /* XXX: Formerly known as: gDomainChecker */
  gDomainChecker:
  {
    _multizillaBundle: null,
    _warningImageURL : "chrome://multiviews/content/images/icons/notfound.png",
    _idnService : Components.classes["@mozilla.org/network/idn-service;1"].getService(Components.interfaces.nsIIDNService),
  
    onSSLChecked: function(aBrowser, aHostArray, aIsNewHostFlag, aIsIDNFlag, aCertificate, aNumberOfKeys)
    {
      try {
      var host = aHostArray[0];
      var textID = (aIsIDNFlag) ? "newSSLHostDetected" : "newSSLHostDetected"; // XXXMichael: what is this ???

      if (!this._multizillaBundle)
        this._multizillaBundle = gMultiZilla.utility.getStringBundle();

      var messageString = gMultiZilla._observer_XPInstall._multizillaBundle.formatStringFromName(textID, [host], 1);
      // var securityFlag = true;
      var dataObj = new Object();
      dataObj.host = host;
      dataObj.keys = Number(aNumberOfKeys);
      dataObj.sheetURL = this._dropDownSheetURL;
      var notificationName = "unknown-secure-host";
      var notificationBox = gBrowser.getNotificationBox(aBrowser);
      dataObj.notificationBox = notificationBox;
      var notification = notificationBox.getNotificationWithValue(notificationName);
  
      if (notification) {
        notification.label = messageString;
      }
      else {
        var buttons = [{
              label: this._multizillaBundle.GetStringFromName("newSecureSiteButton"),
              accessKey: this._multizillaBundle.GetStringFromName("newSecureSiteButton.accesskey"),
              popup: null,
              callback: function showSheet() {
                window.openDialog("chrome://multiviews/content/multizillaBrowserInfoMessages.xul", 
                                  "", "chrome, dialog, modal, dependent", dataObj);
                return false;
              }
            }];
        const priority = notificationBox.PRIORITY_WARNING_MEDIUM;
        notificationBox.appendNotification(messageString, notificationName, this._warningImageURL, priority, buttons);
      }
      } catch(ex) {
        dump("\nex(onSSLChecked): " + ex);    
      }
    }
  },

  /* XXX: Formerly known as:  */

  /* XXX: Formerly known as: mzToolboxVisibilityPrefListener */
  _prefListener_ToolboxVisibility:
  {
    domains: [ "multizilla.integrated.toolbar",
               "multizilla.integrated.toolbar.hide-all-other-buttons" ],
  
    observe: function(subject, topic, aPrefName) {
      // gMultiZilla._multiZillaToolbar is the GoogleBox placeholder!
      if (!gMultiZilla._multiZillaToolbar || topic != "nsPref:changed")
        return;
  
      var prefValue = pref.getBoolPref(aPrefName);
  
      if (aPrefName == this.domains[0]) {
        gMultiZilla._multiZillaToolbar.mzToolboxSetVisibility(prefValue);
        gMultiZilla.googlebox.refreshState(prefValue, false);
  
        if (prefValue == false)
           pref.setBoolPref(this.domains[1], false);
      }
      else if (aPrefName == this.domains[1])
        gMultiZilla._multiZillaToolbar.refreshMultiZillaButtons();
    }
  },

  /* XXX: Formerly known as: mzTabPrefListener */
  _prefListener_Tab:
  {
    domains: ["multizilla.tabs.mintabwidth",
              "multizilla.tabs.tabwidth",
              "multizilla.tabs.status-leds",
              "multizilla.tabs.title",
              "multizilla.tabs.cropping",
              "multizilla.tabs.ontop",
              "multizilla.tabs.iconclose",
              "multizilla.tabs.open-order",
              "multizilla.tabs.focus-order",
              "multizilla.tabs.meta-refresh",
              "multizilla.tabs.automatic-selection",
              "multizilla.tabs.label.color",
              "multizilla.tabs.label.backgroundcolor",
              "multizilla.tabs.label.selected.color",
              "multizilla.tabs.label.selected.backgroundcolor",
              "multizilla.tabs.label.selected.usebold"],
  
    observe: function(subject, topic, aPrefName) {
      if (!gBrowser || topic != "nsPref:changed")
        return;
  
      if (aPrefName == this.domains[0])
        gBrowser.setTabWidthTo("minwidth", pref.getIntPref(aPrefName));
      else if (aPrefName == this.domains[1])
        gBrowser.setTabWidthTo("maxwidth", pref.getIntPref(aPrefName));
      else if (aPrefName == this.domains[2])
        gBrowser.setTabLEDstateTo(pref.getBoolPref(aPrefName));
      else if (aPrefName == this.domains[3])
        gBrowser.setTabLabelsTo(pref.getBoolPref(aPrefName));
      else if (aPrefName == this.domains[4])
        gBrowser.setLabelCropping();
      else if (aPrefName == this.domains[5]) {
        if (pref.getBoolPref(aPrefName))
          gBrowser.activateTopTabs();
        else
          gBrowser.activateBottomTabs();
      }
      else if (aPrefName == this.domains[6])
        gBrowser.setTabCanClose(pref.getBoolPref(aPrefName));
      else if (aPrefName == this.domains[7])
        gBrowser.mTabOrdinance = pref.getIntPref(aPrefName);
      else if (aPrefName == this.domains[8])
        gBrowser.mTabFocus = pref.getIntPref(aPrefName);
      else if (aPrefName == this.domains[9])
        gBrowser.mMetaRefresh = pref.getIntPref(aPrefName);
      else if (aPrefName == this.domains[10])
        gBrowser.mAutomaticTabSelection =pref.getIntPref(aPrefName);
      else if (aPrefName == this.domains[11] || aPrefName == this.domains[12] ||
               aPrefName == this.domains[13] || aPrefName == this.domains[14] ||
               aPrefName == this.domains[15]) {
        gBrowser.setTabLabelColors(aPrefName);
      }
    }
  },

  /* XXX: Formerly known as: mzTabIconPrefListener */
  _prefListener_TabIcon:
  {
    domains: ["browser.chrome.site_icons",
              "browser.chrome.favicons" ],
  
    observe: function(subject, topic, prefName) {
      if (!gBrowser || topic != "nsPref:changed")
        return;
  
      if (prefName == this.domains[0])
        gBrowser.updateDisplayTabIcons();
      else if (prefName == this.domains[1])
        gBrowser.updateDisplayTabIcons();
    }
  },

  /* XXX: Formerly known as: mzTabPopupIconPrefListener */
  _prefListener_TabPopupIcon:
  {
    domains: ["multizilla.tabs.display-popup-icon",
              "multizilla.tabs.remove-popup-icon-delay"],
  
    observe: function(subject, topic, aPrefName) {
      if (topic != "nsPref:changed")
        return;
  
      if (aPrefName == this.domains[0])
        gMultiZilla._showPopupIconOnTab = pref.getBoolPref(aPrefName);
      else if (aPrefName == this.domains[1])
        gMultiZilla._delayedPopupIconRemoval = pref.getIntPref(aPrefName);
    }
  },

  /* XXX: Formerly known as: mzTabAdvancedOpenTabPrefListener */
  _prefListener_TabAdvancedOpenTab:
  {
    domains: ["multizilla.newtabfor.native-applications",
              "advanced.system.supportDDEExec"],
  
    observe: function(subject, topic, aPrefName) {
      if (!pref || topic != "nsPref:changed")
        return;
  
      if (aPrefName == "multizilla.newtabfor.native-applications")
        pref.setBoolPref(this.domains[1], !pref.getBoolPref(aPrefName));
      else if (aPrefName == "advanced.system.supportDDEExec")
        pref.setBoolPref(this.domains[0], !pref.getBoolPref(aPrefName));
    }
  },

  /* XXX: Formerly known as: mzTabbarButtonPrefListener */
  _prefListener_TabbarButton:
  {
    domains: ["multizilla.tabbar.newbutton",
              "multizilla.tabbar.closebutton"],
  
    observe: function(subject, topic, prefName) {
      if (!gBrowser || topic != "nsPref:changed")
        return;
  
      var button = pref.getBoolPref(prefName);
  
      if (prefName == this.domains[0])
        gBrowser.mTabContainer.setAttribute("displaynew", button ? "true" : "false");
      else if (prefName == this.domains[1])
        gBrowser.mTabContainer.setAttribute("displayclose", button ? "true" : "false");
    }
  },

  /* XXX: Formerly known as: mzTBButtonPrefListener */
  _prefListener_ToolbarButton:
  {
    domain: "multizilla.showbutton",
  
    observe: function(subject, topic, prefName) {
      if (topic != "nsPref:changed")
        return;
      var buttonName = prefName.substr(this.domain.length+1);
      var buttonId = buttonName + "-tbbutton";
      var button = document.getElementById(buttonId);
  
      if (!button)
        return;
  
      var show = pref.getBoolPref(prefName);
  
      if (show)
        button.removeAttribute("hidden");
      else
        button.setAttribute("hidden", "true");
    }
  },

  /* XXX: Formerly known as: mzTabSessionStartupPrefListener */
  _prefListener_TabSessionStartup:
  {
    domains: ["browser.startup.page",
              "multizilla.browsersession.skip.navigator.startup",
              "multizilla.browsersession.restore.behavior"],
  
    observe: function(subject, topic, aPrefName) {
      if (!pref || topic != "nsPref:changed")
        return;
  
      var prefValue;
  
      if (aPrefName == "multizilla.browsersession.skip.navigator.startup") {
        prefValue = pref.getBoolPref(aPrefName);
        if (prefValue)
          pref.setIntPref(this.domains[0], 0);
      }
      else if (aPrefName == "browser.startup.page") {
        prefValue = pref.getIntPref(aPrefName);
        if (prefValue > 0)
          pref.setBoolPref(this.domains[1], false);
      }
      else if (aPrefName == "multizilla.browsersession.restore.behavior") {
        prefValue = pref.getIntPref(aPrefName);
        if (prefValue > 0)
          pref.setBoolPref(this.domains[1], true);
      }
    }
  },

  /* XXX: Formerly known as: mzUserAgentOverridePrefListener */
  _prefListener_UserAgentOverride:
  {
    domains: ["general.useragent.override",
              "multizilla.general.useragent.override"],
  
    observe: function(subject, topic, prefName) {
      if (!gBrowser || topic != "nsPref:changed")
        return;
  
      var mozillaPref = "";
      var multiZillaPref = pref.getCharPref(this.domains[1]);
  
      if (pref.prefHasUserValue(this.domains[0]))
        mozillaPref = pref.getCharPref(this.domains[0]);
  
      if (prefName == this.domains[0] && mozillaPref != multiZillaPref) {
        if (multiZillaPref != "") {
          pref.setCharPref(prefName, multiZillaPref);       
        }
        else {
          pref.setCharPref(prefName, "");
          pref.clearUserPref(prefName);
          prefName = this.domains[1];
          pref.setCharPref(prefName, "");
          pref.clearUserPref(prefName);
        }
      }
      else if (prefName == this.domains[1]) {
        pref.setCharPref(this.domains[0], multiZillaPref);
  
        if (multiZillaPref == "")
          pref.clearUserPref(this.domains[0]);
      }
    }
  },

  /* XXX: Formerly known as: mzSidebarViewButtonPrefListener */
  _prefListener_SidebarViewButton:
  {
    domain: ["multizilla.sidebar-view.hideheaders"],
  
    observe: function(subject, topic, prefName) {
  
      if (topic != "nsPref:changed")
        return;
  
      var hideHeaders = pref.getBoolPref(prefName);
      var sidebarPanelsNode = document.getElementById("sidebar-panels");
  
      if (hideHeaders)
        sidebarPanelsNode.setAttribute("hide-headers", "true");
      else
        sidebarPanelsNode.removeAttribute("hide-headers");
    }
  },

  /* XXX: Formerly known as: mzStatusbarButtonPrefListener */
  _prefListener_StatusbarButton:
  {
    domains: ["multizilla.statusbar.display-newFeedContent-button",
              "multizilla.statusbar.display-blocked-windows-button",
              "multizilla.statusbar.display-allowPlugins-button",
              "multizilla.statusbar.display-allowJavascript-button",
              "multizilla.statusbar.display-allowMetaRedirects-button",
              "multizilla.statusbar.display-allowSubframes-button",
              "multizilla.statusbar.display-allowImages-button"],
  
    observe: function(subject, topic, aPrefName) {
      if (topic != "nsPref:changed")
        return;
  
      var prefValue = pref.getBoolPref(aPrefName)
      var id = aPrefName.replace(/multizilla.statusbar.display-/, '');
      var button = document.getElementById(id);
  
      if (button)
        button.setAttribute('show', prefValue ? 'true' : 'false');
    }
  },

  /* XXX: Formerly known as: mzJavaEnabledPrefListener */
  _prefListener_JavaEnabled:
  {
    domains: ["security.enable_java",
              "multizilla.content-blocker.java-enabled"],
  
    observe: function(subject, topic, prefName) {
  
      if (topic != "nsPref:changed")
        return;
      if (prefName == this.domains[0])
        pref.setBoolPref(this.domains[1], pref.getBoolPref(this.domains[0]));
      else if (prefName == this.domains[1])
        pref.setBoolPref(this.domains[0], pref.getBoolPref(this.domains[1]));
    }
  },

  /* XXX: Formerly known as: mzNewWindowRedirectorPrefListener */
  _prefListener_NewWindowRedirector:
  {
    domain: "multizilla.window-open.redirector",
  
    observe: function(subject, topic, prefName) {
  
      if (!gBrowser || topic != "nsPref:changed")
        return;
  
      var prefValue = pref.getBoolPref(prefName);
      gBrowser.newWindowRedirector(prefValue);
    }
  },

  /* XXX: Formerly known as: mzPopupPrefListener */
  _prefListener_Popups: // Replacement for 'gPopupPrefListener' in navigator.js
  {
    domain: "dom.disable_open_during_load",
  
    pref: Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefBranch),
  
    observe: function(subject, topic, prefName) {
      if (topic != "nsPref:changed")
        return;
  
      var whitelist = this.pref.getBoolPref(prefName);
  
      if (whitelist && this.pref.getBoolPref("privacy.popups.first_popup"))
        this.pref.setBoolPref("privacy.popups.first_popup", false);
  
      var hosts = [];
      var permissionManager = Components.classes["@mozilla.org/permissionmanager;1"].getService(Components.interfaces.nsIPermissionManager);
      var enumerator = permissionManager.enumerator;
      var capability = whitelist ? nsIPermissionManager.DENY_ACTION : nsIPermissionManager.ALLOW_ACTION;
  
      while (enumerator.hasMoreElements()) {
        var nextPermission = enumerator.getNext();
        nextPermission = nextPermission.QueryInterface(Components.interfaces.nsIPermission);
  
        if (nextPermission.type == "popup") {
          if (nextPermission.capability == capability)
            hosts[nextPermission.host] = nextPermission.host;
        }
      }
      var popupIcon = document.getElementById("popupIcon");
      var browsers = gBrowser.mPanelContainer.childNodes.length;
  
      for (var i = 0; i < browsers; i++) {
        var browser = gBrowser.browsers[i];
  
        if (!whitelist && "popupDomain" in browser && browser.popupDomain in hosts)
          break;
  
        browser.popupDomain = null;
        popupIcon.hidden = true;
      }
    }
  },

  /* XXX: Formerly known as: mzPopupBlocked() */
  _eventHandler_PopupBlocked: function(aEvent)
  {
    var browser = getBrowserForDocument(aEvent.target);
  
    if (browser) {
      var uri = null;
      var newPopupWindow = true;
      var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                                .getService(Components.interfaces.nsIIOService);
  
      if (!browser.mBlockedWindows)
        browser.mBlockedWindows = new Array();
  
      browser.mNumberOfBlockedWindows++;
  
      for (i in browser.mBlockedWindows) {
        if (browser.mBlockedWindows[i].popupWindowURI.equals(aEvent.popupWindowURI)) {
          browser.mBlockedWindows[i].blockedCounter++;
          newPopupWindow = false;
        }
      }
      if (newPopupWindow) {
        var obj = { popupWindowURI: aEvent.popupWindowURI,
                    popupWindowFeatures: aEvent.popupWindowFeatures,
                    blockedCounter: 1
                  };
        var uuidNumber = Components.interfaces.nsIDOMPopupBlockedEvent.number;
  
        if (uuidNumber == "{9e201104-78e9-4cb3-aff5-7f0a9cf446c0}") {
          // We're not allowed to call the nsIDOMWindow version from script,  
          // but we need a pointer to requestingWindow for SeaMonkey v1.1.x  
          var nwrEnabled = gMultiZilla.prefs.readBoolean("multizilla.window-open.redirector", false);
          obj.requestingWindow = (nwrEnabled) ? aEvent.requestingWindowURI
                                              : aEvent.requestingWindow;
          uri = aEvent.requestingWindowURI;
        }
        else if (uuidNumber == "{bb57e9bb-0c89-459c-977d-5cbc1831bdf5}") {
          obj.requestingWindow = aEvent.requestingWindowURI;
          uri = aEvent.requestingWindowURI;
        }
        else if (uuidNumber == "{05be571f-c3ea-4959-a340-c57b1591ae4b}") {
          obj.requestingWindow = aEvent.requestingWindow;
          obj.requestingDocument = aEvent.requestingWindow.document;
          obj.popupWindowName = aEvent.popupWindowName;
          uri = ioService.newURI(obj.requestingDocument.location, null, null);
        }
        browser.mBlockedWindows.push(obj);
      }
      gPopupBlockerObserver.onUpdatePageReport(browser);
  
      // Do we need to display the popup icon on the tab?
      if (gMultiZilla._showPopupIconOnTab && gMultiZilla._delayedPopupIconRemoval > 0) {
        var tabbrowser = getBrowser();
        var targetTab = tabbrowser.getTabForBrowser(browser);
  
        targetTab.setAttribute("popupwindow", "true");
        tabbrowser.mzPopupIconCheck(browser, targetTab, uri);
      }
    }
  },

  /* XXX: Formerly known as: mzOnScriptBlocked() */
  _eventHandler_ScriptBlocked: function(aEvent)
  {
    gMultiZilla.utility.playSoundForBlocked("script", aEvent);
  },

  /* XXX: Formerly known as: mzOnImageBlocked() */
  _eventHandler_ImageBlocked: function(aEvent)
  {
    gMultiZilla.utility.playSoundForBlocked("image", aEvent);
  },

  /* XXX: Formerly known as: mzOnObjectBlocked() */
  _eventHandler_ObjectBlocked: function(aEvent)
  {
    gMultiZilla.utility.playSoundForBlocked("object", aEvent);
  },

  /* XXX: Formerly known as: mzOnFrameBlocked() */
  _eventHandler_FrameBlocked: function(aEvent)
  {
    gMultiZilla.utility.playSoundForBlocked("frame", aEvent);
  },

  /* XXX: Formerly known as: mzOnDocumentBlocked() */
  _eventHandler_DocumentBlocked: function(aEvent)
  {
    // dump("\nDocument blocked URL: " + aEvent.blockedURI.spec);
    var browser = aEvent.browserElement;
  
    if (browser) {
      // dump("\nbrowser: " + browser);
      var currentURI = aEvent.blockedURI;
      // dump("\ncurrentURI: " +currentURI.spec);
      var host = (currentURI.scheme == "file") ? "file:" : currentURI.hostPort;
      // dump("\nhost: " + host);
      /* if (blockedPopups == 1) {
        messageString = this._multizillaBundle.formatStringFromName("popupWarning", [host], 1);
        tooltipString = this._multizillaBundle.GetStringFromName("blockedPopupTooltip");
        this._blockedWindowButton.setAttribute("tooltiptext", tooltipString);
      }
      else {
        messageString = this._multizillaBundle.formatStringFromName("popupWarningMultiple", [host, blockedPopups], 2);
        tooltipString = this._multizillaBundle.formatStringFromName("blockedPopupsTooltip", [blockedPopups], 1);
        this._blockedWindowButton.setAttribute("tooltiptext", tooltipString);
      } */
      var notificationBox = gBrowser.getNotificationBox(browser);
      var notification = notificationBox.getNotificationWithValue("document-blocked");
      var messageString = "Document Blocked";
      if (notification)
        notification.label = messageString;
      else {
        var buttons = [{
          label: "Unblock...", // this._multizillaBundle.GetStringFromName("popupWarningButton"),
          accessKey: "U", // this._multizillaBundle.GetStringFromName("popupWarningButton.accesskey"),
          popup: "blockedWindowsPopupmenu",
          callback: null
        }];
  
      const priority = notificationBox.PRIORITY_INFO_HIGH;
      notificationBox.appendNotification(messageString, 
                                         "document-blocked", 
                                         aEvent.blockedURI, 
                                         priority, 
                                         buttons);
      }
    }
    gMultiZilla.utility.playSoundForBlocked("document", aEvent);
  },

  /* XXX: Formerly known as: mzOnMetaRedirectBlocked() */
  _eventHandler_MetaRedirectBlocked: function(aEvent)
  {
    gMultiZilla.utility.playSoundForBlocked("meta", aEvent);
  },

  /* XXX: Formerly known as: onWebProgressSTART() */
  _webProgressSTART: function(aWindow, aFollowFrameFlag)
  {
    /***
      * Some handy/important debug information;
      * this    = [object ChromeWindow]
      * aWindow = [object Window]
      * Window  = [Window]
      */
    var frames = new XPCNativeWrapper(aWindow, 'frames').frames;
  
    if (frames && !aFollowFrameFlag) {
      for (var i = 0; i < frames.length; i++)
        this._webProgressSTART(frames[i]);
    }
    var methodOpen = Components.lookupMethod(aWindow, 'open');
  
    if ('mz_window_init' in aWindow || !methodOpen)
      return;
  
    var wrapper = new XPCNativeWrapper(aWindow, 'Window').Window;
    eval('wrapper.prototype.open =' + window.mzWindowOpen.toSource());
    aWindow = wrapper;
    aWindow.timeStamp = (new Date()).getTime();
    aWindow.mz_window_init = true;
  },
  
  /* XXX: Formerly known as: onWebProgressSTOP() */
  _webProgressSTOP: function(aWindow)
  {
    if (!('mz_window_init' in aWindow))
      this._webProgressSTART(aWindow, true);
  
    var frames = new XPCNativeWrapper(aWindow, 'frames').frames;
  
    if (frames) {
      for (var i = 0; i < frames.length; i++)
        this._webProgressSTOP(frames[i]);
    }
  },

  /* XXX: Formerly known as: mzAddPrefListener() */
  _addPrefListener: function(observer)
  {
    var domains = ("domains" in observer) ? observer.domains
                                          : [observer.domain];
    try {
      var pbi = pref.QueryInterface(Components.interfaces.nsIPrefBranchInternal);

      for (var i = 0; i < domains.length; i++)
        pbi.addObserver(domains[i], observer, false);
    } catch(ex) {
      dump("Failed to add observer: " + ex + "\n");
    }
  },
  
  /* XXX: Formerly known as: mzRemovePrefListener() */
  _removePrefListener: function(observer)
  {
    var domains = ("domains" in observer) ? observer.domains
                                          : [observer.domain];
    try {
      var pbi = pref.QueryInterface(Components.interfaces.nsIPrefBranchInternal);

      for (var i = 0; i < domains.length; i++)
        pbi.removeObserver(domains[i], observer, false);
    } catch(ex) {
      dump("Failed to remove observer: " + ex + "\n");
    }
  },

  init: function()
  {
    this.chrome.initGoMenu();
    this.chrome.initInstalledToolbox();
    this.chrome.initOrdinals();
    this.initEnhancedPopupBlocker();
    this.bookmarks.initTemplates();
    this.prefs.checkVendorPrefs()
  },

  /* XXX: Formerly known as: multiZillaInit() */
  startup: function()
  {
    var shouldInit = true;
  
    if ('arguments' in window &&
        window.arguments.length == 2 && 
        window.arguments[1].indexOf("turbo=yes") != -1) {
      shouldInit = false;
    }
    if (shouldInit) {
      this._addPrefListener(this._prefListener_ToolboxVisibility);
      this._addPrefListener(this._prefListener_Tab);
      this._addPrefListener(this._prefListener_TabIcon);
      this._addPrefListener(this._prefListener_TabPopupIcon);
      this._addPrefListener(this._prefListener_TabAdvancedOpenTab);
      this._addPrefListener(this._prefListener_TabbarButton);
      this._addPrefListener(this._prefListener_ToolbarButton);
      this._addPrefListener(this._prefListener_TabSessionStartup);
      this._addPrefListener(this._prefListener_UserAgentOverride);
      this._addPrefListener(this._prefListener_SidebarViewButton);
      this._addPrefListener(this._prefListener_StatusbarButton);
      this._addPrefListener(this._prefListener_JavaEnabled);
      this._addPrefListener(this._prefListener_NewWindowRedirector);
      // this._prefListener_Popups will be added by: initEnhancedPopupBlocker()
      this._nsIObserverService = Components.classes["@mozilla.org/observer-service;1"]
                                           .getService(Components.interfaces.nsIObserverService);
      this._nsIObserverService.addObserver(this._observer_XPInstall, "xpinstall-install-blocked", false);
      this._nsIObserverService.addObserver(this._observer_Managers, "MZManagers", false);
      this._nsIObserverService.addObserver(this._observer_FeedRefresh, "WebFeedService", false);
    }
    this.prefs.initPreferences();

    mzInitializeRDFStorageFile();
    mzCheckAvailabilityOfFeedPanel();
    mzFeedHandler.init();
    // gMultiZillaBundle = this.utility.getStringBundle();
  },

  /* XXX: Formerly known as: mzShutdown() */
  shutdown: function()
  {
    this._removePrefListener(this._prefListener_ToolboxVisibility);
    this._removePrefListener(this._prefListener_Tab);
    this._removePrefListener(this._prefListener_TabIcon);
    this._removePrefListener(this._prefListener_TabPopupIcon);
    this._removePrefListener(this._prefListener_TabAdvancedOpenTab);
    this._removePrefListener(this._prefListener_TabbarButton);
    this._removePrefListener(this._prefListener_ToolbarButton);
    this._removePrefListener(this._prefListener_TabSessionStartup);
    this._removePrefListener(this._prefListener_UserAgentOverride);
    this._removePrefListener(this._prefListener_SidebarViewButton);
    this._removePrefListener(this._prefListener_StatusbarButton);
    this._removePrefListener(this._prefListener_JavaEnabled);
    this._removePrefListener(this._prefListener_NewWindowRedirector);
    this._removePrefListener(this._prefListener_Popups);
  
    removeEventListener("DOMPopupBlocked", this._eventHandler_PopupBlocked, false);
    window.document.removeEventListener("MZScriptBlocked", this._eventHandler_ScriptBlocked, false);
    window.document.removeEventListener("MZImageBlocked", this._eventHandler_ImageBlocked, false);
    window.document.removeEventListener("MZObjectBlocked", this._eventHandler_ObjectBlocked, false);
    window.document.removeEventListener("MZFrameBlocked", this._eventHandler_FrameBlocked, false);
    // window.document.removeEventListener("MZMetaRedirectBlocked", this._eventHandler_MetaRedirectBlocked, false);
    window.document.removeEventListener("MZDocumentBlocked", this._eventHandler_DocumentBlocked, false);
  
    this._nsIObserverService.removeObserver(this._observer_XPInstall, "xpinstall-install-blocked");
    this._nsIObserverService.removeObserver(this._observer_Managers, "MZManagers");
    this._nsIObserverService.removeObserver(this._observer_FeedRefresh, "WebFeedService");
  
    var contentArea = document.getElementById("appcontent");
    contentArea.removeEventListener("pageshow", mzFeedHandler.checkForFeeds, true);
  
    Shutdown(); // navigator.xul
  },

  /* XXX: Formerly known as: mzWindowInit() */
  windowInit: function()
  {
    var url = ('arguments' in window) ? window.arguments[0] : null;
  
    // if (this.utility.isFirstWindow()) {
    if (this.firstBrowserWindow) {
      // Sanity check (mozilla crashed before writing our pref setting?).
      this.resetConfirmationPref();
      // Do we need to restore a previously saved browser session?
      if (this.prefs.readInteger("multizilla.browsersession.restore.behavior", 3) > 0) {
        var startup = pref.getIntPref("browser.startup.page");
        // Browser sessions will only be restored in limited cases
        if (startup == 0 && url == "about:blank")
          getBrowser().mRestoreBrowserSession = true;
        else if (startup == 1 && url == getHomePage())
          getBrowser().mRestoreBrowserSession = true;
        else if (startup == 2) {
          var globalHistory;
          // Builds 20040210 and up (see also mozilla bug 224829)
          if ("@mozilla.org/browser/global-history;2" in Components.classes)
            globalHistory = Components.classes["@mozilla.org/browser/global-history;2"];
          else
            globalHistory = Components.classes["@mozilla.org/browser/global-history;1"];
  
          globalHistory = globalHistory.getService(Components.interfaces.nsIBrowserHistory);
  
          if (URL == globalHistory.lastPageVisited)
            getBrowser().mRestoreBrowserSession = true;
        }
      }
    }
    Startup(); // Calling navigator.js
  
    if (this.prefs.readBoolean("multizilla.feed-viewer.enabled", true)) {
      var contentArea = document.getElementById("appcontent");
      contentArea.addEventListener("pageshow", mzFeedHandler.checkForFeeds, true);
    }
   return true;
  },

  /* Formerly known as: mzResetConfirmationPref() */
  resetConfirmationPref: function()
  {
    /***
      * This pref setting will be set to 'true' in function confirmCloseWS()
      * after the user confirms to close the this session. Other open windows
      * will check this pref setting, to enable a more relaxed/quiet browser
      * exit, with only a single confirmation asked for close session!
      */
    this.prefs.writeBoolean("multizilla.confirm.quit-session", false);
  },

  // XXX: Check me!  Do we still need this?  
  // Added for compatibility with pre 1.5 mozilla builds (20032406 and later)
  checkConfirmCloseWS: function(aConfirm)
  {
    var retValue;
    // Give 'tryToClose' (WindowIsClosing()) a chance to veto if it is defined
    if (typeof(window.tryToClose) == "function") {
      this._closeConfirmationID = aConfirm;

      if (aConfirm == 4) { // onclose="gMultiZilla.checkConfirmCloseWS(4)" in multizillaOverlay.xul
        retValue = WindowIsClosing();
  
        if (retValue)
          gBrowser.mObserverService.notifyObservers(window, "tabUpdate", "closeWindow");
        return retValue;
      }
    }
    else {
      retValue = gMultiZilla.confirmCloseWS(aConfirm);
  
      if (retValue)
        gBrowser.mObserverService.notifyObservers(window, "tabUpdate", "closeWindow");
      return retValue;
    }
    return true;
  },

  confirmCloseWS: function(aConfirm)
  {
    /* ::::
       1  = File Menu 'Close Window' & Ctrl-Shift-W
       2  = File Menu 'Exit' & Ctrl-Q
       4  = Click on [X] on Window
       8  = JavaScript close()
       16 = no confirmation for single tab
    :::: */
    var confirmQuitSessionPrefID = "multizilla.confirm.quit-session";
  
    if (aConfirm == 2) {
      var lastChoice = this.prefs.readBoolean(confirmQuitSessionPrefID, false);
  
      if (lastChoice)
        return mzSaveTabSession(false, true, null, true);
    }
    var confirmPref = this.prefs.readInteger("multizilla.confirm.quit", 11);
    var aHeader = this.utility.getTextFromBundle("closeSession");
    var aQuestion = this.utility.getTextFromBundle("closeSessionQuestion");
    var numberOfTabs = gBrowser.mTabContainer.childNodes.length;
    var saveSessionPref = this.prefs.readInteger("multizilla.browsersession.save.behavior",3);
    var storeSessions = (saveSessionPref > 0);
    var confirmSaveSession = (saveSessionPref == 3 && this.tabs.areAllTabsBlank(getBrowser()) == false);

    if (numberOfTabs == 1 && confirmPref & 16)
      return mzSaveTabSession(false, true, null, false);
    else if (aConfirm == 8 && document.documentElement.hasAttribute("chromehidden"))
      return mzSaveTabSession(false, true, null, false);
    else if (!confirmSaveSession && confirmPref && (confirmPref & aConfirm) == aConfirm) {
      var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                    .getService(Components.interfaces.nsIPromptService);
  
      if (aConfirm & 1 || aConfirm & 4) {
        aHeader = this.utility.getTextFromBundle("closeWindow");
        aQuestion = this.utility.getTextFromBundle("closeWindowQuestion");
      }
      if (promptService.confirm(window, aHeader, aQuestion)) {
        if (aConfirm == 2) { // Close session confirmed, remember this, write pref!
          this.prefs.writeBoolean(confirmQuitSessionPrefID, true);
  
          if (storeSessions)
            setSessionIDs();
        }
        return mzSaveTabSession(false, true, null, true);
      }
      else {
        if (aConfirm == 2) // Close session, clear pref
          this.prefs.writeBoolean(confirmQuitSessionPrefID, false);

        return false;
      }
    }
    if (aConfirm == 2 && storeSessions) {
      this.prefs.writeBoolean(confirmQuitSessionPrefID, true);
      setSessionIDs();
      return mzSaveTabSession(false, true, null, true);
    }
    return mzSaveTabSession(false, true, null, false);
  },

  /* Formerly known as: mzConfirmWindowOpen() */
  confirmWindowOpen: function(aHostPort)
  {
    var title = this.utility.getTextFromBundle('newWindowRedirectorTitle'); // 'mzWizard');
    var text =  this.utility.getTextFromBundle('confirmWindowOpenText', [aHostPort], null);
    var selectionPrefValue =  this.prefs.readInteger("multizilla.window-open.selection.behavior", 0);
    var checkbox = {value:(selectionPrefValue == 2)};
    var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                  .getService(Components.interfaces.nsIPromptService);
    // var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
    //                                 .getService(nsIWindowDataSource);
    // var win = windowManagerDS.getWindowForResource(mzGetWindowResourceID(window));
    // returns 0 for [Tab] 1 for [Block] or (ESC) and 2 [Window] button
    var result = promptService.confirmEx(window, title, text,
                                        (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_IS_STRING) +
                                        (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_IS_STRING) +
                                        (promptService.BUTTON_POS_2 * promptService.BUTTON_TITLE_IS_STRING),
                                        gMultiZilla.utility.getTextFromBundle("openInTabButton"),
                                        gMultiZilla.utility.getTextFromBundle("blockWindowButton"),
                                        gMultiZilla.utility.getTextFromBundle("openInWindowButton"),
                                        gMultiZilla.utility.getTextFromBundle("updatePermissionsCheckbox"), checkbox);
    // Tab Button = 0, Window Button = 2 and Block Button/ESC/Window X = 1
    // Tab Button returns 3, Window Button returns 1 and Block Button/ESC/Window X returns 2
    result = ((result == 0) ? 3 : (result == 1) ? 2 : 1);
    result += (checkbox.value ? 4 : 0);
    return result;
  },

  /* Formerly known as: mzCleanupOnExit() */
  cleanupOnExit: function()
  {
    this.resetConfirmationPref();

    if (this.prefs.readBoolean("multizilla.uam.set-default-useragent-onbrowserexit", true)) {
      var uaPrefIdentifier = "multizilla.general.useragent.override";
      pref.setCharPref(uaPrefIdentifier, "");
    }
    var xpinstallConfirmPref = "xpinstall.dialog.confirm";

    if (pref.prefHasUserValue(xpinstallConfirmPref)) { // "chrome://multiviews/content/xpinstall/xpinstallConfirm.xul"
      // dump("\nUser pref found for: 'xpinstall.dialog.confirm'");
      pref.setCharPref(xpinstallConfirmPref, "");
      pref.clearUserPref(xpinstallConfirmPref);
    }
    var vendorPref = "general.useragent.vendor"; // "MultiZilla"

    if (pref.prefHasUserValue(vendorPref)) {
      // dump("\nUser pref found for: 'general.useragent.vendor'");
      pref.setCharPref(vendorPref, "");
      pref.clearUserPref(vendorPref);
    }
    var vendorSubPref = "general.useragent.vendorSub"; // "1.8.0.1n"

    if (pref.prefHasUserValue(vendorSubPref)) {
      // dump("\nUser pref found for: 'general.useragent.vendorSub'");
      pref.setCharPref(vendorSubPref, "");
      pref.clearUserPref(vendorSubPref);
    }
  
    var list = "";
    var prefIdentifier = ["diskcache", "locationbar", "history", "cookies",
                          "downloads", "formdata", "passwords", "idn-history" ];
    // Initialize list based on pref values
    for (i in prefIdentifier) {
      if (this.prefs.readBoolean("multizilla.privacy.clear-on-exit." + prefIdentifier[i], false))
        list = list ? list + ", " + prefIdentifier[i] : prefIdentifier[i];
    }
    if (list.length > 0) { // Do we have anything to remove?
      var result = this.prefs.readBoolean("multizilla.privacy.clear-on-exit.confirm", true);
  
      if (result) {
        var checkbox = {value:0};
        var checkboxMsg = this.utility.getTextFromBundle("clearOnExitDontAskAgain");
        var title = this.utility.getTextFromBundle("clearOnExitTitle");
        var text = this.utility.getTextFromBundle("clearOnExitText");
        text = text + list; // Add list
        var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                      .getService(Components.interfaces.nsIPromptService);
        result = promptService.confirmEx(window, title, text,
                                         (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) +
                                         (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_NO),
                                         null, null, null, null, checkbox);
      }
      if (result == 0) {
        var newPrefIdentifier = list.split(", ");
  
        for (i in newPrefIdentifier) {
          if (newPrefIdentifier[i] == prefIdentifier[0])
            gMultiZilla.privacy.clearCache("ALL");
          if (newPrefIdentifier[i] == prefIdentifier[1])
            gMultiZilla.privacy.clearUrlBarHistory()
          if (newPrefIdentifier[i] == prefIdentifier[2])
            gMultiZilla.privacy.clearGlobalHistory();
          if (newPrefIdentifier[i] == prefIdentifier[3])
            gMultiZilla.privacy.clearAllCookies();
          if (newPrefIdentifier[i] == prefIdentifier[4])
            gMultiZilla.privacy.clearAllDownloads();
          if (newPrefIdentifier[i] == prefIdentifier[5])
            gMultiZilla.privacy.clearAllFormData();
          if (newPrefIdentifier[i] == prefIdentifier[6])
            gMultiZilla.privacy.clearAllPasswords();
          if (newPrefIdentifier[i] == prefIdentifier[7])
            gMultiZilla.dnp.purgeIDNHistory();
        }
      }
    }
  },

  initEnhancedPopupBlocker: function()
  {
    // Remove default popup pref listener, because it fails to work properly!
    if ('gPopupPrefListener' in window)
      removePrefListener(gPopupPrefListener);
    // Use a 'heavily' modified copy of it
    addPrefListener(this._prefListener_Popups);
    /****
    * We need to get notified of blocked popup windows so we replace the
    * original eventlistener for 'DOMPopupBlocked' with our (this) event
    * listener and call 'onPopupBlocked' (original function) when we're done.
    */
    addEventListener("DOMPopupBlocked", this._eventHandler_PopupBlocked, false);
    gMultiZilla._showPopupIconOnTab = this.prefs.readBoolean("multizilla.tabs.display-popup-icon", true);
    gMultiZilla._delayedPopupIconRemoval = this.prefs.readInteger("multizilla.tabs.remove-popup-icon-delay", 4000);
  
    window.document.addEventListener("MZScriptBlocked", this._eventHandler_ScriptBlocked, false);
    window.document.addEventListener("MZImageBlocked", this._eventHandler_ImageBlocked, false);
    window.document.addEventListener("MZObjectBlocked", this._eventHandler_ObjectBlocked, false);
    window.document.addEventListener("MZFrameBlocked", this._eventHandler_FrameBlocked, false);
    // window.document.addEventListener("MZMetaRedirectBlocked", this._eventHandler_MetaRedirectBlocked, false);
    window.document.addEventListener("MZDocumentBlocked", this._eventHandler_DocumentBlocked, false);
  }
};


gMultiZilla.utility =
{
  _prefs: gMultiZilla.prefs,

  /* Formerly known as: mzGetMostRecentWindow() */
  _getMostRecentWindow: function(aType, aChromeCheckFlag)
  {
    var defaultType = "navigator:browser";
  
    if (!aType)
      aType = defaultType;
  
    var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1'].getService();
    var windowManagerInterface = windowManager.QueryInterface(Components.interfaces.nsIWindowMediator);
    var navigatorWindow = windowManagerInterface.getMostRecentWindow(aType);
    // Prevent opening of new tabs in a chromeless window!
    if (aChromeCheckFlag && aType == "navigator:browser") {
      var enumerator = windowManager.getEnumerator(aType);
      var windowArray = new Array();
  
      while (enumerator.hasMoreElements()) {
        var win = enumerator.getNext();
  
        if (!win.document.documentElement.hasAttribute("chromehidden"))
          windowArray.push(win);
      }
      return windowArray[windowArray.length-1];
    }
    return navigatorWindow;
  },

  /* XXX: Formerly known as: mzGetStrBundle() */
  getStringBundle: function(aBundleName)
  {
    var bundleURL = aBundleName;
    
    if (gMultiZilla._nsIStringBundleService == null) {
      gMultiZilla._nsIStringBundleService = Components.classes["@mozilla.org/intl/stringbundle;1"]
                                                      .getService(Components.interfaces.nsIStringBundleService);
    }
    if (aBundleName == undefined ||
        (aBundleName && aBundleName.indexOf("chrome://") == -1)) {
      bundleURL = aBundleName ? "chrome://multiviews/locale/" + aBundleName :
                                "chrome://multiviews/locale/multizilla.properties";
    }
    return gMultiZilla._nsIStringBundleService.createBundle(bundleURL); // XXX: remove this when you're done!!!
    // return this._nsIStringBundleService.createBundle(bundleURL);
  },

  /* XXX: Formerly known as: mzGetTextFromBundle() */
  getTextFromBundle: function(aStr, aParam, aBundleName)
  {
    // aStr = string, aParam = array of params used to replace the %S arguments
    // (you need one param for each %S) and aBundleName = filename.properties
    if (aBundleName == undefined)
      aBundleName = "multizilla.properties";

    var aBundle = this.getStringBundle(aBundleName);
  
    if (!aBundle) {
      alert("aBundle error!");
      return "error";
    }
    try {
      var arguments = (aParam) ? aParam.length : 1;
      var text = aParam ? aBundle.formatStringFromName(aStr, aParam, arguments) :
                          aBundle.GetStringFromName(aStr);
    }
    catch(ex) {
      alert("Error: '"+aStr+"' not found in "+(aBundleName ? aBundleName
                                                           : "multizilla.properties") +
            (aParam ? "\n\nNote: 'aParam' is specified, make sure to use an [array]" : ""));
    }
    return text;
  },

  /* XXX: Formerly known as: mzGetPlatformExtras() */
  getPlatformExtras: function() 
  {
    var platformExtras = new Object();
    platformExtras.isClassicStyle = gMultiZilla.prefs.readBoolean("multizilla.windows.classic-style", false);
    platformExtras.isMac = /Mac/.test(navigator.platform) || 
                           gMultiZilla.prefs.readBoolean("multizilla.windows.mac-style", false);
    platformExtras.isLinux = /Linux/.test(navigator.platform) ||
                             gMultiZilla.prefs.readBoolean("multizilla.windows.linux-style", false);
    return platformExtras;
  },

  getMostRecentTabbrowser: function() // We don't seem to use this anymore?!?
  {
    var win = this._getMostRecentWindow();
    return win ? this._getMostRecentWindow().getBrowser() : null;
  },

  getDocumentReferrer: function(doc)
  {
    if (!doc)
      return null;
  
    if (doc == document) // compatibility
      doc = getContentFrameDocument(document.commandDispatcher.focusedWindow);
  
    // return makeURI(doc.location.href, doc.characterSet);
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService);
    
    try {
      return ioService.newURI(doc.location.href, doc.characterSet, null);
    } catch (ex) {
      return null;
    }
  },

  getLocationFromClipboard: function()
  {
    var URL = null;
    var clipboard = Components.classes["@mozilla.org/widget/clipboard;1"]
                              .getService(Components.interfaces.nsIClipboard);
    var trans = Components.classes["@mozilla.org/widget/transferable;1"]
                          .createInstance(Components.interfaces.nsITransferable);
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService);
  
    try {
      trans.addDataFlavor("text/unicode");
      clipboard.getData(trans, clipboard.kGlobalClipboard);
  
      var data = {};
      var dataLen = {};
      trans.getTransferData("text/unicode", data, dataLen);
  
      if (data) {
        data = data.value.QueryInterface(Components.interfaces.nsISupportsString);
        data = data.data.substring(0, dataLen.value / 2);
        // Convert clipboard data to URL
        try {
          var URI = ioService.newURI(data, null, null);
          URL = URI.spec;
        } catch(ex) {
          alert(gMultiZilla.utility.getTextFromBundle("noValidClipboardURL"));
        }
      }
      else
        alert(gMultiZilla.utility.getTextFromBundle("noClipboardData"));
    } catch (ex) {
      alert(gMultiZilla.utility.getTextFromBundle("clipboardReadError"));
    }
    return URL;
  },

  /* Formerly known as: mzIsFirstWindow() */
  isFirstWindow: function(aType)
  {
    var defaultType = "navigator:browser";
  
    if (aType == undefined) // default check for browser window
      aType = defaultType;
  
    var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1'].getService();
    var windowManagerInterface = windowManager.QueryInterface(Components.interfaces.nsIWindowMediator);
    var topWindow = windowManagerInterface.getMostRecentWindow(aType);
    var enumerator = windowManagerInterface.getEnumerator(aType);
    var firstWindow = enumerator.getNext().QueryInterface(Components.interfaces.nsIDOMWindowInternal);
    var windowType = topWindow.document.documentElement.getAttribute("windowtype");
    // Initialize 'firstBrowserWindow' the first time only!
    if (this._firstBrowserWindow == null && windowType == defaultType)
      this._firstBrowserWindow = firstWindow; // used in windowInit()
  
    if (firstWindow != topWindow)
      return false;
    return true;
  },

  isNavigatorWindow: function(aWindow) {
    var win = window.top;

    if (aWindow)
      win = aWindow;
    return (win.document.documentElement.getAttribute("windowtype") == "navigator:browser");
  },

  isLastWindow: function(aType)
  {
    return (mzGetNumberOfWindowsByType(aType) == 1);
  },

  isHostBlocked: function(aBrowser, aHost)
  {
    if (aBrowser) {
      for (i in aBrowser.mBlockedWindows) {
        const blockedWindowURI = aBrowser.mBlockedWindows[i].popupWindowURI;
        const scheme = blockedWindowURI.scheme;
  
        if ((/^http[s]?/.test(scheme) && blockedWindowURI.hostPort == aHost) || (scheme + ":" == aHost))
          return true;
      }
    }
    return false;
  },

  isImageCached: function(imageURL)
  {
    dump("\nisImageCached: " + imageURL);
    var isCached = false;
    var imageURI = makeURI(imageURL, getCharsetforSave(null));
    var imageCache = Components.classes["@mozilla.org/image/cache;1"]
                               .getService(Components.interfaces.imgICache);
  
    try {
      // this only works in Mozilla builds after 2005-03-23 13:05 PDT
      // see also https://bugzilla.mozilla.org/show_bug.cgi?id=287286
      if (imageCache && 'findEntryProperties' in imageCache)
        isCached = (imageCache.findEntryProperties(imageURI) != null);
      else
        throw "Compatibility error";
    } catch(ex) {
      var cacheService = Components.classes["@mozilla.org/network/cache-service;1"]
                                   .getService(Components.interfaces.nsICacheService);
      var cacheSession = cacheService.createSession("HTTP", 0, true);
      var cacheEntryDescriptor = null;
  
      try {
        cacheSession.openCacheEntry(imageURL, Components.interfaces.nsICache.ACCESS_READ, false);
      } catch(ex) {
        return false;
      }
      isCached = (cacheEntryDescriptor != null);
    }
    return isCached;
  },

  isAlphaNum: function(mixedCase)
  {
    var allCaps = mixedCase.toUpperCase();
    for (var i = allCaps.length - 1; i >= 0; i--) {
      var c = allCaps.charAt(i);
      if ((c < 'A' || c > 'Z') && (c < '0' || c > '9') && (c!='_'))
        return false;
    }
    return true;
  },

  /* Formerly known as: mzIsBlockedXPInstall() */
  isBlockedXPInstall: function(aURL, aBrowser)
  {
    var browser = (aBrowser == undefined) ? gBrowser.mCurrentBrowser : aBrowser;
    browser.lastBlockedXPInstallURL = null;
    /***
      * Prevent callers from loadURI to throw a JS error when they pass in a
      * typeof object instead of a typeof string (just like the GoogleBox did)
      * see also: http://bugzilla.mozdev.org/show_bug.cgi?id=10xxx
      */
    if (typeof(aURL) != 'string')
      aURL = String(aURL);
    var checkURL = aURL.replace(/\?(.*)/, ""); // remove crufty parameters!
  
    if (checkURL.indexOf(".xpi")) { //  || aURL.indexOf(".jar")) {
      // var isJarFile = (checkURL.substring(checkURL.length-4, checkURL.length) == ".jar");
      var isXPIFile = (checkURL.substring(checkURL.length-4, checkURL.length) == ".xpi");
  
      if (isXPIFile) { // || isJarFile) {
        if (!gMultiZilla.utility.checkXPInstallPermission(checkURL, browser)) {
          var observerService = Components.classes["@mozilla.org/observer-service;1"]
                                          .getService(Components.interfaces.nsIObserverService);
          observerService.notifyObservers(browser.docShell, "xpinstall-install-blocked", null);
          gBrowser.updateCurrentBrowser();
          return true;
        }
      }
    }
    return false;
  },

  addBlockedContentEntry: function(aEvent)
  {
    var blockedContent, contentType;
    var browser = aEvent.browserElement;
  
    if (!browser) { // This should never happen but I like to get notify when it does!
      /* alert("Please notify bugs4hj@aim.com when you see this!\n\nBrowser failed for:\n" + 
            "Content type: " + aEvent.type + "\nHostport: " + aEvent.requestingWindowURI.hostPort); */
      return false;
    }
    /***
      * This makes logging for XUL Elements (like favicons) possible, and it also prevents 
      * JS errors, because a chrome window doesn't have a 'mBlockedContent' property
      *
      * Note: 'browser instanceof ChromeWindow' is used to maintain backwards compatibility
      */
    if (aEvent.context instanceof XULElement || browser instanceof ChromeWindow) {
      // Do we need to add a log entry for this item?
      if (this._prefs.readBoolean("multizilla.content-blocker.log_enabled.chrome", false)) {
        // Yes, and chrome windows don't have a 'mBlockedContent' so we set that here
        if (browser.mBlockedContent == undefined)
          browser.mBlockedContent = new Object();
      }
      else {
        return -1; // No logging
      }
    }
    // only browser elements have a mBlockedContent property defined
    if (browser.mBlockedContent == undefined)
      browser = gBrowser.mCurrentBrowser;
  
    switch(aEvent.type)
    {
      case "MZImageBlocked": 
           blockedContent = (browser.mBlockedContent.images == undefined) ? browser.mBlockedContent.images = new Array()
                                                                          : browser.mBlockedContent.images; 
           contentType = "image";
           break;
      case "MZObjectBlocked":
           blockedContent = (browser.mBlockedContent.objects == undefined) ? browser.mBlockedContent.objects = new Array()
                                                                           : browser.mBlockedContent.objects;
           contentType = "object";
           break;
      case "MZScriptBlocked":
           blockedContent = (browser.mBlockedContent.scripts == undefined) ? browser.mBlockedContent.scripts = new Array()
                                                                           : browser.mBlockedContent.scripts;
           contentType = "script";
           break;
      case "MZFrameBlocked":
           blockedContent = (browser.mBlockedContent.frames == undefined) ? browser.mBlockedContent.frames = new Array()
                                                                          : browser.mBlockedContent.frames;
           contentType = "frame";
           break;
      case "MZDocumentBlocked":
           blockedContent = (browser.mBlockedContent.documents == undefined) ? browser.mBlockedContent.documents = new Array()
                                                                             : browser.mBlockedContent.documents;
           contentType = "document";
           break;
      case "MZMetaRedirectBlocked":
           blockedContent = (browser.mBlockedContent.meta == undefined) ? browser.mBlockedContent.meta = new Array()
                                                                        : browser.mBlockedContent.meta;
           contentType = "meta";
           break;
      default: dump("\nUNSUPPORTED EVENT TYPE USED for 'addBlockedContentEntry()'");
    }
    var logType = gMultiZilla.prefs.readInteger("multizilla.content-blocker.log_enabled." + contentType, 0);
    // 0 = logging disabled / 1 = add basic log entry (domain only) / 2 = add extended log entry (complete URL) 
  
    if (Boolean(logType)) {
      for (i in blockedContent) {
        if (blockedContent && blockedContent[i].filter && blockedContent[i].filter.match(aEvent.filter)) {
          // Yes, bail out early
          // dump("\n" + contentType + " blocked by filter: " + aEvent.filter + " (previously matched)");
          blockedContent[i].counter++;
          return false;
        }
      }
      // var entryData = (logType == 1) ? aEvent.blockedURI.hostPort : aEvent.blockedURI.spec;
      var dataObject = new Object();
      dataObject.counter = blockedContent.length;
      dataObject.blockedURI = (logType == 1) ? aEvent.blockedURI.hostPort : aEvent.blockedURI.spec;
      dataObject.filter = aEvent.filter;
      blockedContent.push(dataObject);
      // var index = blockedContent.length;
      // dump("\nindex: " + index);
      /* dump("\nblockedContent[index].counter: " + blockedContent[index].counter);
      dump("\nblockedContent[index].blockedURI: " + blockedContent[index].blockedURI);
      dump("\nblockedContent[index].filter: " + blockedContent[index].filter); */
  
      if (blockedContent.length > 100)
        blockedContent.shift();
  
      // dump("\n" + blockedContent.length + " " + contentType + "(s) blocked - last match: " + aEvent.filter);
      /* dump("\n" + contentType + " blocked by filter: " + aEvent.filter + " (new match)" +
           "\nAdding filter to " + contentType + " log (" + blockedContent.length + " filter(s) logged)"); */
    }
    return true;
  },

  /* Formerly known as: doCheck4Updates() */
  checkForUpdates: function()
  {
    // http://multizilla.mozdev.org/update.html?version=1.6.0.1&revision=e&gb-version=1.4&gb-revision=c
    var url = "http://multizilla.mozdev.org/update.html?version=";
    var mzVersionInfo = gBrowser.mVersion + "&revision=" + gBrowser.mVersionRev;
    var gbVersionInfo = "";
  
    if (gMultiZilla._multiZillaToolbar.mToolboxActivated) { // GoogleBox installed?
      var googleBox = document.getElementById("internalToolBox");
      // Pre GoogleBox v1.4a versions had no version info, that's why we check for 'undefined'
      var googleBoxVersion = (googleBox.mVersion == undefined) ? 0
                                                               : googleBox.mVersion;
      var googleBoxRevision = (googleBox.mVersionRev == undefined) ? ""
                                                                   : googleBox.mVersionRev;
      gbVersionInfo = "&gb-version=" + googleBoxVersion + "&gb-revision=" + googleBoxRevision;
    }
    url += mzVersionInfo + gbVersionInfo;
  
    if (gBrowser.mProduct)
      url += "&product=" + gBrowser.mProduct.toLowerCase();
  
    gBrowser.addOrGetTab(url);
  },

  /* Formerly known as: mzCheckXPInstallPermission() */
  checkXPInstallPermission: function(aURL, aBrowser)
  {
    var browser = (aBrowser == undefined) ? gBrowser.mCurrentBrowser
                                          : aBrowser;
    var pm = Components.classes["@mozilla.org/permissionmanager;1"]
                       .getService(Components.interfaces.nsIPermissionManager);
    var nsIIOService = Components.classes["@mozilla.org/network/io-service;1"]
                                 .getService(Components.interfaces.nsIIOService);
    var uri = nsIIOService.newURI(aURL, null, null);
    var permission = pm.testPermission(uri, "install");
  
    if (gMultiZilla.prefs.readBoolean(gMultiZilla._observer_XPInstall._XPInstallWhiteListPrefString, true)) {
      if (permission == pm.ALLOW_ACTION)
        return true;
    }
    else if (permission != pm.DENY_ACTION) {
      return true;
    }
    browser.lastBlockedXPInstallURL = aURL;
    return false;
  },

  defineShortURL: function(aURI)
  {
    try {
      var i = aURI.indexOf(":");
    } 
    catch(ex) {
      return aURI;
    }
    if (i != -1) {
      while(!this.isAlphaNum(aURI[i]))
        i++;
      return aURI.substring(i, aURI.length);
    }
    return aURI;
  },

  /* Formerly known as: mzPlaySound() */
  playSound: function(aSoundURL)
  {
    var _nsISound = Components.classes["@mozilla.org/sound;1"]
                              .createInstance(Components.interfaces.nsISound);
  
    if (!aSoundURL)
      _nsISound.beep();
    else  {
      if (aSoundURL.indexOf("file://") == -1) {
        try {
          _nsISound.playSystemSound(aSoundURL);
        } catch (ex) {
          _nsISound.beep();
        }
      }
      else {
        var uri = Components.classes["@mozilla.org/network/standard-url;1"]
                            .createInstance(Components.interfaces.nsIURL);
        uri.spec = aSoundURL;
        var file = uri.file;
  
        if (file.exists)
          _nsISound.play(uri);
      }
    }
  },

  playSoundForBlocked: function(aContentType, aEvent)
  {
    var prefBranch = "multizilla.content-blocker.";
    var addEntryResult = this.addBlockedContentEntry(aEvent);
  
    if (aEvent.isXULElement) {
      if (addEntryResult == -1 ||
          gMultiZilla.prefs.readBoolean(prefBranch + "sound_enabled.chrome", false) == false)
        return; // no sound for this XULElement (favicon)
    }
    var shouldPlaySound = gMultiZilla.prefs.readInteger(prefBranch + "sound_enabled." + aContentType, 0);
  
    if (shouldPlaySound) {
      // Only play sound for new entries when sound filter is activated
      if (shouldPlaySound == 1 && addEntryResult == false)
        return; // No, bail out early for this duplicated entry
      var soundURL = gMultiZilla.prefs.readChar(prefBranch + "sound_url." + aContentType, "");
      this.playSound(soundURL);
    }
  },

  /* XXX: Formely known as: mzURLSecurityCheck() */
  doURLSecurityCheck: function(aPrincipal, aURL, aFlags)
  {
    // URL Loading Security Check
    const nsIScriptSecurityManager = Components.interfaces.nsIScriptSecurityManager;
    var secMan = Components.classes["@mozilla.org/scriptsecuritymanager;1"]
                           .getService(nsIScriptSecurityManager);
    var is = Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService);
    var uri = is.newURI(aURL, null, null);
    try {
      secMan.checkLoadURIStrWithPrincipal(aPrincipal, uri, aFlags);
    } catch (ex) {
      throw "MultiZilla: Load of " + aURL + " denied.";
    }
  },

  spoofReferrer: function(url, referrer)
  {
    var spoofURL = "http://ref.spoof.org/refspoof.cgi"; // default fake referrer
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                              .getService(Components.interfaces.nsIIOService);
    var spoofing = gMultiZilla.prefs.readInteger("multizilla.refspoof.spoofing", 1);
  
    switch (spoofing) {
      case 1: // send original referrer
        if (referrer)
          referrer = this.getDocumentReferrer(document);
        break;
      case 2: // send url as referrer
        spoofURL = url;
        try {
          referrer = spoofURL ? ioService.newURI(spoofURL, null, null) : null;
        } catch(ex) {
        }
        break;
      default: // send fake referrer
        try {
          /* spoofURL = pref.getComplexValue("multizilla.refspoof.spoofurl",
                                          Components.interfaces.nsIPrefLocalizedString).data; */
          spoofURL = gMultiZilla.prefs.readString("multizilla.refspoof.spoofurl", spoofURL);
        } catch (ex) {
          dump("\nError: " + ex);
        }
        referrer = ioService.newURI(spoofURL, null, null);
    }
    return(referrer);
  },

  useFirstPage: function() // XXX: We don't seem to use this anymore?
  {
   var uriToLoad = null;
    if ("arguments" in window && window.arguments.length >= 1 && window.arguments[0])
      uriToLoad = window.arguments[0];
    if (gMultiZilla.prefs.readBoolean("multizilla.tabs.blank-tab-first", true)) {
      if ((gMultiZilla.prefs.readInteger("browser.startup.page", 1) == 1 && 
          getHomePage() == "about:blank" || getHomePage() == "") || uriToLoad == "about:blank" || 
          gMultiZilla.prefs.readInteger("browser.startup.page", 0) == 3 && uriToLoad == "about:blank")
        return true;
    }
    return false;
  },

  togglePermission: function(aEvent)
  {
    var property = aEvent.target.id.replace(/-\w*/, '');
    var tabbrowser = getBrowser();
    var value = (aEvent.target.getAttribute('blocked') == 'true') ? 'false' : 'true';
    tabbrowser.toggleDocShellProperty(tabbrowser.mCurrentTab, property, value);
    aEvent.target.setAttribute('blocked', value);
    tabbrowser.reloadTab(tabbrowser.mCurrentTab, false);
  },

  validateThisPage: function(aURL)
  {
    if (aURL) {
      var IOService = Components.classes["@mozilla.org/network/io-service;1"]
                                .getService(Components.interfaces.nsIIOService);
      if (!IOService)
         return;
      var scheme = "";
  
      try {
        scheme = IOService.extractScheme(aURL, {schemeStartPos:0}, {schemeEndPos:0});
      } catch(ex) { 
        alert(gMultiZilla.utility.getTextFromBundle('errorText'));
      }
      if (/https?/.test(scheme)) {
        var checkURL = "http://validator.w3.org/check?uri=" + aURL + 
                       "&charset=%28detect+automatically%29&doctype=Inline";
        getBrowser().addTab(checkURL);
      }
      else
        alert(gMultiZilla.utility.getTextFromBundle("unsupportedProtocolForValidator", [scheme]));
    }
  }
};

/***
  * .prefs: various JavaScript functions to initialize, read and write the
  *         MultiZilla preferences, at startup and on users request.
  */

gMultiZilla.prefs =
{
  _prefService: null,
  _prefBranch: null,
  _prefErrors : 0,
  _newPrefsAdded : 0,

  init: function()
  {
    this._prefService = Components.classes["@mozilla.org/preferences-service;1"]
                                  .getService(Components.interfaces.nsIPrefService);
    this._prefBranch = this._prefService.getBranch("multizilla.");

    if (pref == undefined) // This happens when you close the Tab Managers parent window(s)!
      var pref;

    if (!pref) { // Make sure 'pref' is initialized (this seems to fail in Netscape 7)
      pref = this._prefService.getBranch(null);
    }
  },

  initPreferences: function()
  {
    this.init();

    if (this.readBoolean("multizilla.prefs.checked", false))
      return; // Bail out if already checked. This should improve our startup time

    var preferences = [ //XXX: SQLite me!
        ["bookmark.close-remaining", "bool", false],
        ["bookmark.folders", "int", 5],
        ["bookmark.groups", "int", 5],
        ["bookmark.loadInBackground", "bool", false],
        ["bookmark.menuitem.allow-middleclick", "bool", true],
        ["bookmark.menuitem.middleclick-closepopups", "bool", true],
        ["bookmark.open-tabgroup.confirmation", "bool", true],
        ["bookmark.open-tabgroup-as-windows.confirmation", "bool", true],
        ["bookmark.open-tabgroup-in-window.confirmation", "bool", true],
        ["bookmark.tabgroup.behavior", "int", 2], // 0 = overwrite - 1 = new tabs - 2 = ask me
        ["browsersession.restore.behavior", "int", 3],
        ["browsersession.restore.tabs.behavior", "int", 0],
        ["browsersession.restore.notify", "bool", false],
        ["browsersession.save.behavior", "int", 3],
        ["browsersession.history-items", "int", 0],
        ["browsersession.skip.navigator.startup", "bool", true],
        ["browsersession.skip.confirm-single-window", "bool", false],
        ["browsersession.skip.save-single-tab", "bool", false],
        ["browsersession.window.restore-behavior", "int", 0],
        ["browser.docShell.allowPlugins", "bool", true],
        ["browser.docShell.allowJavascript", "bool", true],
        ["browser.docShell.allowMetaRedirects", "bool", true],
        ["browser.docShell.allowSubframes", "bool", true],
        ["browser.docShell.allowImages", "bool", true],
        ["check.domain-names", "bool", true],
        ["check.webtools", "bool", true],  
        ["confirm.reopen-tabs", "bool", true],
        ["confirm.quit", "int", 11],
        ["confirm.quit-session", "bool", false],
        ["content-blocker.enabled", "bool", false],
        ["content-blocker.check-favicons", "bool", true],
        ["content-blocker.behavior.image", "int", 0], // 0 = disabled - 1 = Accept image - 2 = Accept image - 4 = Block image 8 = Ask me before accepting image
        ["content-blocker.behavior.object", "int", 0],
        ["content-blocker.behavior.script", "int", 0],
        ["content-blocker.behavior.frame", "int", 0],
        ["content-blocker.behavior.document", "int", 0],
        ["content-blocker.behavior.meta", "int", 0],
        ["content-blocker.sound_enabled.chrome", "bool", false],
        ["content-blocker.sound_enabled.image", "int", 0], // 0 = don't play sound 1 = play sound for the first blocked image only 2 = play sound for all blocked images
        ["content-blocker.sound_enabled.object", "int", 0],
        ["content-blocker.sound_enabled.script", "int", 0],
        ["content-blocker.sound_enabled.frame", "int", 0],
        ["content-blocker.sound_enabled.document", "int", 0],
        ["content-blocker.sound_enabled.meta", "int", 0],
        ["content-blocker.sound_url.image", "string", ""],
        ["content-blocker.sound_url.object", "string", ""],
        ["content-blocker.sound_url.script", "string", ""],
        ["content-blocker.sound_url.frame", "string", ""],
        ["content-blocker.sound_url.document", "string", ""],
        ["content-blocker.sound_url.meta", "string", ""],
        ["content-blocker.log_enabled.chrome", "bool", false],
        ["content-blocker.log_enabled.image", "int", 1], // 0 = logging disabled - 1 = log '/multizilla.mozdev.' - 2 = log 'http://multizilla.mozdev.org/graphics/hj.gif'
        ["content-blocker.log_enabled.object", "int", 1],
        ["content-blocker.log_enabled.script", "int", 1],
        ["content-blocker.log_enabled.frame", "int", 1],
        ["content-blocker.log_enabled.document", "int", 1],
        ["content-blocker.log_enabled.meta", "int", 1],
        ["content-blocker.java-enabled", "bool", pref.getBoolPref("security.enable_java")],
        ["content-blocker.trim-behavior.image", "int", 7], // 1 = trim third level domain (www) from url - 2 = trim top level domain (see code snip) - 4 = add leading '/'
        ["content-blocker.trim-behavior.object", "int", 7],
        ["content-blocker.trim-behavior.script", "int", 7],
        ["content-blocker.trim-behavior.frame", "int", 7],
        ["content-blocker.trim-behavior.document", "int", 7],
        ["content-blocker.trim-behavior.meta", "int", 7],
        ["content-blocker.pattern.for.image", "string", ""],  
        ["content-blocker.pattern.for.object", "string", ""],
        ["content-blocker.pattern.for.script", "string", ""],
        ["content-blocker.pattern.for.frame", "string", ""],
        ["content-blocker.pattern.for.document", "string", ""],
        ["content-blocker.pattern.for.meta", "string", ""],
        ["document.context-menu.behavior", "int", 1],
        ["dnd.loadInBackground", "bool", false],
        ["dnd.tabstripdrops.selected", "bool", true],
        ["feed-viewer.enabled", "bool", true],
        ["feeds.refresh.debug_messages", "bool", false],
        ["feeds.refresh.interval", "int", 5],
        ["feeds.refresh.lastcheck", "int", 0],
        ["feeds.refresh.limit", "int", 5],
        ["feeds.refresh.report-errors", "bool", false],
        ["feeds.refresh.header-errors", "string", ""],
        ["forced.autocomplete", "bool", true],
        ["forced.autocomplete.onsubmit-remove", "bool", false],
        ["fullscreen.enabled", "bool", true],
        ["fullscreen.enable.menubar", "bool", false],
        ["fullscreen.enable.personaltoolbar", "bool", false],
        ["fullscreen.enable.status-bar", "bool", true],
        ["fullscreen.first-browserwindow-only", "bool", true],
        ["fullscreen.disable.chromeless", "bool", true],
        ["fullscreen.enable.linktoolbar", "bool", false],
        ["fullscreen.re-enable-chrome", "bool", true],
        ["fullscreen.adaptive-mode.enabled", "bool", false],
        ["fullscreen.adaptive-mode.target-width", "int", 800],
        ["fullscreen.adaptive-mode.target-position", "int", 1],
        ["fullscreen.adaptive-mode.widths", "string", "640,800,1024,1280"],
        ["general.useragent.override", "string", ""],
        ["blocked-popup.show-info-message", "bool", true],
        ["blocked-xpinstall.sound_enabled", "bool", true],
        ["blocked-xpinstall.sound_url", "string", ""],
        ["info-messages.show-close-botton", "bool", true],
        ["info-messages.middle-click-close", "bool", false],
        ["integrated.toolbar", "bool", false],
        ["integrated.toolbar-notify", "bool", true],
        ["integrated.toolbar.hide-all-other-buttons", "bool", false],
        ["mac-ctrl-left-right.behavior", "bool", false],
        ["managers.confirm.quit", "int", 6],
        ["mouse-middle-click.close", "bool", true],
        ["mouse-double-click.close", "bool", true],
        ["newtabfor.bookmarks", "bool", false],
        ["newtabfor.bookmark-manager", "bool", false],
        ["newtabfor.ctrl-n", "bool", false],
        ["newtabfor.locationbar", "bool", false],
        ["newtabfor.mailnews", "bool", false],
        ["newtabfor.personal-toolbar", "bool", false],
        ["newtabfor.sidebar", "bool", false],
        ["newtabfor.view-source", "bool", true],
        ["newtabfor.websearch", "bool", true],
        ["newtabfor.windowopen", "bool", true],
        ["preferred.fornewtabs", "bool", false],
        ["preferred.location", "string", "http://multizilla.mozdev.org/index.html"],
        ["preference-manager.load-delay", "int", 0],
        ["prefs.use-preference-manager", "bool", true],
        ["perm-manager.default-manager", "string", "cookieManager"],
        ["privacy.clear-on-exit.diskcache", "bool", false],
        ["privacy.clear-on-exit.locationbar", "bool", false],
        ["privacy.clear-on-exit.history", "bool", false],
        ["privacy.clear-on-exit.cookies", "bool", false],
        ["privacy.clear-on-exit.downloads", "bool", false],
        ["privacy.clear-on-exit.formdata", "bool", false],
        ["privacy.clear-on-exit.passwords", "bool", false],
        ["privacy.clear-on-exit.confirm", "bool", false],
        ["privacy.clear-on-exit.idn-history", "bool", false],
        ["prevent.double", "bool", true],
        ["refspoof.spoofing", "int", 1],
        ["refspoof.spoofurl", "url", "http://ref.spoof.org/refspoof.cgi"],
        ["remove.form-target", "bool", false],
        ["reopen-tabmenu.display-icons", "int", 2],
        ["reopen-tabmenu.display-maxitems", "int", 15],
        ["session-manager.confirm.property-changes", "bool", true],
        ["session-manager.selectedtab-styling", "string", "bold"],
        ["session-manager.tabselectionmenu-autoclose", "bool", true],
        ["showbutton.privacymanager", "bool", true],
        // Buttons on the navigation toolbar
        ["showbutton.qpmenu", "bool", false],
        ["showbutton.clearlocbar", "bool", false],
        // Buttons on the MultiZilla toolbar
        ["showbutton.mzmenubutton", "bool", true],
        ["showbutton.newtab", "bool", true],
        ["showbutton.tabpos", "bool", true],
        ["showbutton.tabstrip", "bool", true],
        ["showbutton.tabmanager", "bool", true],
        ["showbutton.quickprefmenu", "bool", true],
        ["showbutton.update", "bool", true],
        ["showbutton.jsconsole", "bool", true],
        ["showbutton.inspector", "bool", true],
        ["showbutton.permmanager", "bool", true],
        ["showbutton.sessionmanager", "bool", true],
        ["showbutton.closeThis", "bool", true],
        ["showbutton.closeOther", "bool", true],
        ["showbutton.closeAll", "bool", true],
        ["sidebar.feed-panel.info-popup.show", "bool", true],
        ["sidebar.feed-panel.info-header.length", "int", 250],
        ["sidebar.feed-panel.info-header.show-link", "bool", true],
        ["sidebar.feed-panel.last-channel", "string", ""],
        ["sidebar.feed-panel.open-after-add", "bool", true],
        // XXX: the following two can be removed after the source has been changed???
        ["sidebar.feed-panel.last-refresh", "int", (new Date().getTime())/1000],
        ["sidebar.feed-panel.skip-refresh", "int", 300],
        ["statusbar.display-newFeedContent-button", "bool", true],
        ["statusbar.display-blocked-windows-button", "bool", true],
        ["statusbar.display-allowImages-button", "bool", true],
        ["statusbar.display-allowJavascript-button", "bool", true],
        ["statusbar.display-allowMetaRedirects-button", "bool", true],
        ["statusbar.display-allowPlugins-button", "bool", true],
        ["statusbar.display-allowSubframes-button", "bool", true],
        ["updates.enabled", "bool", true],
        ["updates.check-after-ms", "int", 10000],
        ["updates.frequency", "int", 0],
        ["updates.last_checked", "int", 0],
        ["tabbar.closebutton", "bool", false],
        ["tabbar.newbutton", "bool", false],
        ["tabbar.middle-click-opentab", "int", 0],
        ["tabbar.double-click-opentab", "bool", true],
        ["tab-manager.confirm.close-tabs", "bool", true],
        ["tab-manager.confirm.close-windows", "bool", true],
        ["tab-manager.confirm.skip-close-windows", "bool", true],
        ["tab-manager.selectedtab-styling", "string", "bold"],
        ["tabstrip.middle-click-action", "int", 0],
        ["tabs.autoselect-delay", "int", 500],
        ["tabs.automatic-selection", "int", 0],
        ["tabs.blank-tab-first", "bool", true],
        ["tabs.confirm-close", "bool", true],
        ["tabs.confirm-close-other", "bool", true],
        ["tabs.ctrl-click.url-to-clipboard", "bool", false],
        ["tabs.cropping", "int", 2],
        ["tabs.display-popup-icon", "bool", true],
        ["tabs.duplicate-permissions", "bool", false],
        ["tabs.focus-order", "int", 0],
        ["tabs.groupmark", "string", ""],
        ["tabs.iconclose", "bool", true],
        ["tabs.init-permissions", "bool", false],
        ["tabs.insertstyle", "int", 1],
        ["tabs.last-tab.behavior", "int", 0], // 0 = load "about:blank" - 1 = load "about:blank" and hide tab bar - 2 = close window & quit browser
        ["tabs.load-failure.behavior", "int", 2],
        ["tabs.mintabwidth", "int", 25],
        ["tabs.meta-refresh", "int", 0],
        ["tabs.ontop", "bool", true],
        ["tabs.open-order", "int", 0],
        ["tabs.remove-popup-icon-delay", "int", 4000],
        ["tabs.status-leds", "bool", false],
        ["tabs.tabwidth", "int", 175],
        ["tabs.title", "bool", true],
        ["tabs.label.color", "string", ""], 
        ["tabs.label.backgroundcolor", "string", ""], 
        ["tabs.label.selected.color", "string", ""],
        ["tabs.label.selected.backgroundcolor", "string", ""],
        ["tabs.label.selected.usebold", "bool", true],
        ["uam.show-about-page", "bool", true],
        ["uam.close-about-page", "bool", true],
        ["uam.show-help-onstartup", "bool", true],
        ["uam.prevent-delete-current", "bool", true],
        ["uam.set-default-useragent-onbrowserexit", "bool", true],
        ["uam.sync-with-http-useragent", "bool", true],
        ["window-open.redirector", "bool", false],
        ["window-open.behavior", "int", 2],
        ["window-open.override-features", "bool", false],
        ["window-open.features", "int", 2047],
        ["window-open.display-failures", "bool", false],
        ["window-open.javascript-in-tab", "bool", false],
        ["windows.classic-style", "bool",  /Mac/.test(navigator.platform)],
        ["windows.linux-style", "bool", false],
        ["windows.mac-style", "bool", false]
    ];

    for each(p in preferences)
      this.readMyPref("multizilla." + p[0], p[1], p[2]);

    preferences = [];
    var mPrefChecked = (this._newPrefsAdded == 0);
    this.writeBoolean("multizilla.prefs.checked", mPrefChecked);

    if (this._prefErrors)
      gMultiZilla.quickprefs.doAlert(window, "", gMultiZilla.utility.getTextFromBundle("corruptPref"));
  },

  checkVendorPrefs: function()
  {
    const prefBranch = "general.useragent.";
    var MultiZillaVersion = getBrowser().mVersion+getBrowser().mVersionRev;
    var vendor = this.readString(prefBranch + "vendor", "MultiZilla");
    var vendorSub = this.readString(prefBranch + "vendorSub", MultiZillaVersion);

    if (vendor != "MultiZilla")
      this.writeString(prefBranch + "vendor", "MultiZilla");
    if (vendorSub != MultiZillaVersion)
      this.writeString(prefBranch + "vendorSub", MultiZillaVersion);

    this.checkGeneralUserAgentMiscPref();
    this.checkGeneralUserAgentExtraPrefs();

    var xpinstallConfirmPref = "xpinstall.dialog.confirm";
    // Do we need to enable MultiZilla's XPInstall confirmation window?
    if (this.readBoolean("multizilla." + xpinstallConfirmPref, true)) {
      // Yes, but only if it hasn't been set yet!
      if (!pref.prefHasUserValue(xpinstallConfirmPref))
        this.writeString(xpinstallConfirmPref, "chrome://multiviews/content/xpinstall/xpinstallConfirm.xul");
    }
  },

  checkGeneralUserAgentMiscPref: function()
  {
    const prefID = "general.useragent.misc";

    try {
      if (pref.prefHasUserValue(prefID))
        pref.clearUserPref(prefID);
    } catch(ex) {
    }
  },

  checkGeneralUserAgentExtraPrefs: function()
  {
    const prefBranch = "general.useragent.extra.";
    var browserIDs = ["Mozilla", "SeaMonkey", "Firefox", "Simple"];
    var prefValue, product;
    // Clear manually added prefs, simply because it might prevent update.html from working
    for (i in browserIDs) {
      var prefID = prefBranch + browserIDs[i].toLowerCase();

      if (pref.prefHasUserValue(prefID))
        pref.clearUserPref(prefID);
      // Check for product specification pref
      try {
        prefValue = pref.getComplexValue(prefID, Components.interfaces.nsISupportsString);

        if (prefValue) {
          getBrowser().mProduct = browserIDs[i];
          return; // We have what we need, bail out
        }
      } catch(ex) {}
    }
    getBrowser().mProduct = "undetermined";
  },

  readMyPref: function(aPrefIdentifier, aPrefType, aDefaultValue)
  {
    this.init();
    var prefValue;

    switch(aPrefType)
    {
      case "bool": 
      {
        try {
          prefValue = pref.getBoolPref(aPrefIdentifier); 
        } catch(ex) {
          this.fixWrongPrefSetting(aPrefType, aPrefIdentifier, aDefaultValue);
          return Boolean(aDefaultValue);
        }
        return Boolean(prefValue);
      }
      case "int":
      {
        try {
          prefValue = pref.getIntPref(aPrefIdentifier);
        } catch(ex) {
          this.fixWrongPrefSetting(aPrefType, aPrefIdentifier, aDefaultValue);
          return aDefaultValue;
        }
        return prefValue;
      }
      case "string":
      {
        try {
          prefValue = pref.getComplexValue(aPrefIdentifier, Components.interfaces.nsISupportsString);
        } catch(ex) {
          this.fixWrongPrefSetting(aPrefType, aPrefIdentifier, aDefaultValue);
          return aDefaultValue;
        }
        return prefValue.data.length ? prefValue.data : aDefaultValue;
      }
      case "char":
      {
        try {
          prefValue = pref.getCharPref(aPrefIdentifier);
        } catch(ex) {
          this.fixWrongPrefSetting(aPrefType, aPrefIdentifier, aDefaultValue);
        }
	    return prefValue ? prefValue : aDefaultValue;
      }
      case "url":
      {
        try {
          prefValue = pref.getComplexValue(aPrefIdentifier, Components.interfaces.nsIPrefLocalizedString);
        } catch(ex) {
          this.fixWrongPrefSetting(aPrefType, aPrefIdentifier, aDefaultValue);
          return (aDefaultValue);
        }
        return prefValue.data.length ? prefValue.data : aDefaultValue;
      }
      default:
      {
        dump('this.prefs.readMyPref(): Unsupported pref type "'+aPrefType+'" used for: '+aPrefIdentifier);
      }
    }
    return (aDefaultValue);
  },

  readBoolean: function(aPrefIdentifier, aDefaultValue)
  {
    return this.readMyPref(aPrefIdentifier, "bool", aDefaultValue);
  },

  readInteger: function(aPrefIdentifier, aDefaultValue)
  {
    return this.readMyPref(aPrefIdentifier, "int", aDefaultValue);
  },

  readString: function(aPrefIdentifier, aDefaultValue)
  {
    return this.readMyPref(aPrefIdentifier, "string", aDefaultValue);
  },

  readChar: function(aPrefIdentifier, aDefaultValue)
  {
    return this.readMyPref(aPrefIdentifier, "char", aDefaultValue);
  },

  fixWrongPrefSetting: function(aPrefType, aPrefIdentifier, aDefaultValue) 
  {
    // dump("\nError: prefType = "+aPrefType+"\naPrefIdentifier = "+aPrefIdentifier+"\nprefValue = "+aDefaultValue);

    if (pref.prefHasUserValue(aPrefIdentifier)) {
      pref.clearUserPref(aPrefIdentifier);
      // dump("\nError: prefType = "+aPrefType+"\naPrefIdentifier = "+aPrefIdentifier+"\nprefValue = "+aDefaultValue);
      this._prefErrors++;
    }
    this.writePref(aPrefType, aPrefIdentifier, aDefaultValue);
    this._newPrefsAdded++;
  },

  writePref: function(aPrefType, aPrefIdentifier, aPrefValue)
  {
    this.init();

    try {
      switch (aPrefType) {
        case "bool":
        {
          pref.setBoolPref(aPrefIdentifier, aPrefValue);
          break;
        }
        case "int":
        {
          pref.setIntPref(aPrefIdentifier, aPrefValue);
          break;
        }
        case "string":
        {
          var str;
          str = Components.classes["@mozilla.org/supports-string;1"]
                          .createInstance(Components.interfaces.nsISupportsString);
          str.data = aPrefValue;
          pref.setComplexValue(aPrefIdentifier, Components.interfaces.nsISupportsString, str);
          break;
        }
        case "url":
        {
          pref.setComplexValue(aPrefIdentifier, Components.interfaces.nsIPrefLocalizedString, aPrefValue);
          break;
        }
        case "char":
        {
          pref.setCharPref(aPrefIdentifier, aPrefValue);
          break;
        }
        default:
        {
          dump('writePref(): Unsupported pref type "'+aPrefType+'" used for: '+aPrefIdentifier);
        }
      }
    }
    catch (ex) { // die silently 
    }
  },

  writeBoolean: function(aPrefIdentifier, aPrefValue)
  {
    this.writePref("bool", aPrefIdentifier, aPrefValue);
  },

  writeInteger: function(aPrefIdentifier, aPrefValue)
  {
    this.writePref("int", aPrefIdentifier, aPrefValue);
  },

  writeString: function(aPrefIdentifier, aPrefValue)
  {
    this.writePref("string", aPrefIdentifier, aPrefValue);
  },

  writeChar: function(aPrefIdentifier, aPrefValue)
  {
    this.writePref("char", aPrefIdentifier, aPrefValue);
  },

  toggleBoolPref: function(aPrefIdentifier, aPrefType, aDefaultValue)
  {
    var prefValue = this.readMyPref(aPrefIdentifier, aPrefType, aDefaultValue);
    this.writePref(aPrefType, aPrefIdentifier, !prefValue);
  }

};


/***
  * .chrome: mainly a collection of JavaScript functions to initialize the UI
  *          at startup, and functions to control things like button and toolbar
  *          visibility, including all functions to initialize/refresh context
  *          menus.
  */

gMultiZilla.chrome =
{
  _prefs: gMultiZilla.prefs,

  initGoMenu: function()
  {
    var mainMenu = document.getElementById("main-menubar");
    var goMenu = mainMenu.childNodes[3];
    goMenu.setAttribute("id", "menu_Go");
    var goPopup = goMenu.firstChild;
  
    if (goPopup.childNodes[6].localName == "menuseparator") {
      // Move menu item/separator
      goPopup.appendChild(goPopup.childNodes[5]);
      goPopup.appendChild(goPopup.childNodes[5]);
    }
    // Replace default image manager with MultiZilla's Permission Manager
    var menuitem = document.getElementById("AllowImages");
    menuitem.nextSibling.nextSibling.setAttribute("oncommand", "gMultiZilla.managers.openPermissionManager('','ImagesPrefTab','ImagesPermissionsTab');");
    // Replace default cookie manager with MultiZilla's Permission Manager
    menuitem = document.getElementById("AllowCookies");
    menuitem.nextSibling.nextSibling.setAttribute("oncommand", "gMultiZilla.managers.openPermissionManager('','StoragePrefTab','CookiesTab');");
  
    if (!gBrowser)
      gBrowser = getBrowser();
  
    var prefIdentifier = "multizilla.statusbar.display-";
  
    function _inner(aProperty) {
      var prefValue = gMultiZilla.prefs.readBoolean((prefIdentifier + aProperty + "-button"), true);
      var button = document.getElementById(aProperty + "-button");
      button.setAttribute("show", prefValue ? "true" : "false");
    }
    for (i in gBrowser.mDocShellPropertyNames) {
      _inner(gBrowser.mDocShellPropertyNames[i]);
    }
    _inner("blocked-windows");
    _inner("newFeedContent");
  },
  
  updateGoMenuPopup: function(aPopupMenu)
  {
    var numberOfTabs = gBrowser.mTabs.length;
    var disabled = (numberOfTabs == 1);
    var menuItems = aPopupMenu.getElementsByAttribute("tbattr", "tabbrowser-multiple");
  
    for (var i = 0; i < menuItems.length; i++)
      menuItems[i].setAttribute("disabled", disabled);
  
    if (!disabled) {
      document.getElementById("gotoFirstTab").disabled = (gBrowser.mTabs[0] == gBrowser.mCurrentTab);
      document.getElementById("gotoLastTab").disabled = (gBrowser.mTabs[(gBrowser.mTabs.length-1)] == gBrowser.mCurrentTab);
    }
  },

  initInstalledToolbox: function()
  {
    // GoogleBox support: added for MultiZilla's GoogleBox
    gMultiZilla._multiZillaToolbar = document.getElementById("MultiZillaToolbar");
    gMultiZilla._multiZillaToolbar.updateButtons("multizilla.showbutton.");
    var toolboxEnabled = gMultiZilla.prefs.readBoolean("multizilla.integrated.toolbar", false);
    gMultiZilla._multiZillaToolbar.mzToolboxSetVisibility(toolboxEnabled);
    var showInternalToolbox = document.getElementById("cmd_viewIntegratedToolBox");
  
    if (toolboxEnabled) {
      if (showInternalToolbox.getAttribute("checked") == "true") {
        document.getElementById("internalToolBox").removeAttribute("hidden");
        gMultiZilla._multiZillaToolbar.refreshMultiZillaButtons();
      }
    }
    else {
      showInternalToolbox.setAttribute("checked", "false");
      setTimeout(gMultiZilla.googlebox.toggleOrEnable, 2500, true);
    }
  },
  
  initOrdinals: function()
  {
    var nodeList = document.getAnonymousNodes(getBrowser().mTabContainer);
    var isMac = /Mac/.test(navigator.platform);
    var isMacClassic = (isMac && nodeList.length == 1);
  
    if (isMacClassic) {
      // http://lxr.mozilla.org/seamonkey/source/themes/classic/global/mac/globalBindings.xml
      nodeList = nodeList[0].childNodes[1].firstChild.childNodes;
    }
    for (var i = 0; i < nodeList.length; i++) {
      nodeList[i].ordinal = i;
      if (nodeList[i].localName == "hbox") {
        nodeList[i].setAttribute("class", "multizilla-tabs");
        nodeList[i].firstChild.ordinal = 0;
        nodeList[i].lastChild.ordinal = 9999999;
      }
    }
    if (nodeList.length > 1)
      nodeList[nodeList.length-(1+isMac)].ordinal = 9999999;
  },

  initViewMenu: function(aMenupopup)
  {
    // XXX: Total HACK to remove the original menu items
    var i = 0;
    var removal = aMenupopup.getElementsByAttribute("observes", "cmd_viewpersonaltoolbar"); 
  
    gMultiZilla.googlebox.initMenuitem("GoogleBoxShowHideMenuitem");

    if (removal.length) {
      for (i = 0; i < removal.length; i++) {
        if (!removal[i].hasAttribute("acceltext"))
          aMenupopup.removeChild(removal[i]);
      }
    }
    removal = aMenupopup.getElementsByAttribute("observes", "cmd_viewnavbar");
  
    if (removal.length) {
      for (i = 0; i < removal.length; i++) {
        if (removal[i].getAttribute('id') != "navbarButtonPopup")
          aMenupopup.removeChild(removal[i]);
      }
    }
    gMultiZilla.tabs.initToolbarsMenu();
  },

  initButtonContextMenu: function(aContextMenu, aID)
  {
    // NT = Navigator Toolbar and PT = Personal Toolbar
    var buttonIDs = (aID == "NT") ? [ "skip", "skip", "back-button", "forward-button", 
                                      "reload-button", "stop-button", "skip", 
                                      "qpmenu-tbbutton", "clearlocbar-tbbutton", 
                                      "skip", "go-button", "search-button", "print-button" ]
                                  : [ "skip", "skip", "home-button", "bookmarks-button", 
                                      "qpmenu-button" ];
  
    for (var i = 0; i < buttonIDs.length; i++) {
      if (buttonIDs[i] != "skip") {
        // dump("\nbuttonIDs: " + buttonIDs[i]);
        var button = document.getElementById(buttonIDs[i]);
  
        if (button) {
          var hidden = button.getAttribute("hidden");
          aContextMenu.childNodes[i].setAttribute("checked", (hidden == "true") ? "false" : "true");
        }
        // else
          // dump("\nbutton not found: " + buttonIDs[i]);
      }
    }
    return;
  },

  /* We don't seem to be using this anymore? */
  initTabbarButtonTooltip: function(aLabelNode, aCondition, aConditionTrueText, aConditionFalseText)
  {
    aLabelNode.value = (aCondition) ? toolbarButton.showTabbarTooltiptext
                                    : toolbarButton.hideTabbarTooltiptext;
  },

  askToDisableButton: function(aEvent)
  {
    aEvent.stopPropagation();
  
    var button = aEvent.target;
    var buttonId = button.getAttribute("id");
  
    if (buttonId) {
      var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                    .getService(Components.interfaces.nsIPromptService);
  
      if (promptService.confirm(window,
                                gMultiZilla.utility.getTextFromBundle("disableButtonTitle"),
                                gMultiZilla.utility.getTextFromBundle("disableButtonText"))) {
        var index = buttonId.indexOf("-");
        var buttonName = buttonId.substring(0, index);
        var observerId = button.getAttribute("observes");
  
        if (observerId) {
           button.setAttribute("hidden", "true");
           /* var observer = document.getElementById(observerId);
           alert(observer); */
           // observer.setAttribute("disabled", true);
        }
        if (buttonId == "qpmenu-tbbutton" || buttonId == "clearlocbar-tbbutton")
          this._prefs.writeBoolean("multizilla.showbutton." + buttonName, false);
        else
          this._prefs.writeBoolean("browser.toolbars.showbutton." + buttonName, false);
      }
    }
  },

  doHomeButtonClick: function(aEvent)
  {
    /* BrowserHome() in navigator.js has been updated so now we can use:
    if (aEvent.button == 2)
      this.askToDisableButton(aEvent);
    else 
      BrowserHome(aEvent); */
    if (aEvent.button == 1) {
      var homePage = getHomePage();
      // We don't have to open a new tab for a group
      if (homePage.length == 1) { // single home page set?
        getBrowser().addTab(null, null, null, true);
      }
      BrowserHome();
    }
    else if (aEvent.button == 2)
      this.askToDisableButton(aEvent);
  },

  doButtonUpdate: function(aEvent)
  {
    var buttonId = ('getAttribute' in aEvent) ? aEvent.getAttribute("buttonid")
                                              : aEvent.originalTarget.getAttribute("buttonid");
    var index = buttonId.indexOf("-");
    var buttonName = buttonId.substring(0, index);
  
    if (buttonId == "qpmenu-tbbutton" || buttonId == "clearlocbar-tbbutton")
      this._prefs.toggleBoolPref("multizilla.showbutton." + buttonName, "bool", true);
    else
      this._prefs.toggleBoolPref("browser.toolbars.showbutton." + buttonName, "bool", true);
  },

  overrideContextMenu: function(aEvent)
  {
    if (aEvent.button == 2) {
      var contextMenu = aEvent.target.getAttribute("context-menu");
      var overrideContextPopup = document.getElementById("buttonContextPopup");
      var defaultContextPopup = document.getElementById(contextMenu);
      // Note that attribute popup="" is needed in order to open the context menu at the desired position!
      if (aEvent.ctrlKey) {
        if (defaultContextPopup)
          defaultContextPopup.hidePopup();
        overrideContextPopup.showPopup(overrideContextPopup, aEvent.clientX+2,
                                       aEvent.clientY+2, "popup", "at_pointer");
      }
      else if (contextMenu) {
        overrideContextPopup.hidePopup();
        defaultContextPopup.showPopup(defaultContextPopup, aEvent.clientX+2,
                                      aEvent.clientY+2, "popup", "at_pointer");
      }
    }
  },

  /* XXX: Formerly known as: FillSelectedHistoryMenu() */
  refreshHistoryMenu: function(aParent, aMenu, aTab)
  {
    var sessionHistory = getWebNavigation().sessionHistory;
    var end;
    var j;
    var entry;
  
    if (aTab) {
      var browser = aTab.linkedBrowser;
      sessionHistory = browser.sessionHistory;
    }
    var count = sessionHistory.count;
    var index = sessionHistory.index;
  
    deleteHistoryItems(aParent);
  
    if (aMenu == "global" && count == 0)
      return;
  
    switch (aMenu)
      {
        case "go":
          aParent.lastChild.hidden = (count == 0);
          end = count > MAX_HISTORY_MENU_ITEMS ? count - MAX_HISTORY_MENU_ITEMS : 0;
  
          for (j = count - 1; j >= end; j--) {
            entry = sessionHistory.getEntryAtIndex(j, false);
            if (entry)
              createRadioMenuItem(aParent, j, entry.title, j==index);
          }
        break;
  
        case "tab":
          aParent.lastChild.hidden = (count > 0);
          end = count > MAX_HISTORY_MENU_ITEMS ? count - MAX_HISTORY_MENU_ITEMS : 0;
  
          for (j = count - 1; j >= end; j--) {
            entry = sessionHistory.getEntryAtIndex(j, false);
            if (entry)
              createRadioMenuItem(aParent, j, entry.title, j==index);
          }
        break;
  
        /* case "global":
          var historyTree = document.getElementById("globalHistoryTree");
          var historyIndex = count = historyTree.treeBoxObject.view.rowCount;
          aParent.childNodes[10].hidden = (count == 0);
  
          if (count > MAX_HISTORY_MENU_ITEMS)
            count = MAX_HISTORY_MENU_ITEMS;
          if (count == 0)
           return;
  
          var builder = historyTree.builder.QueryInterface(Components.interfaces.nsIXULTreeBuilder);
          var nameResource = RDF.GetResource("http://home.netscape.com/NC-rdf#Name");
  
          for (j = count-1; j >= 0; j--) {
            var res = builder.getResourceAtIndex(j);
            var url = res.Value;
            var titleRes = historyTree.database.GetTarget(res, nameResource, true);
  
            if (titleRes) {
              var title = titleRes.QueryInterface(Components.interfaces.nsIRDFLiteral);
              createRadioMenuItem(aParent, (historyIndex-j), title.Value, (j+historyIndex)==index);
            }
          }
        break; */
      }
  },

  toggleTabbarVisibility: function()
  {
    // dump("\ntoggleTabbarVisibility()");
    var tabBarDisplayed = gBrowser.getStripVisibility();
    // dump("\ntoggleTabbarVisibility - tabBarDisplayed: " + tabBarDisplayed);
    var classname = tabBarDisplayed ? "mz-toolbar-hide-button" : "mz-toolbar-show-button";
    document.getElementById("tabstrip-tbbutton").className = classname;
    showHideTabbar();
    var switchTabBarMenuItem = document.getElementById("switchTabBarMenuItem");
    tabBarDisplayed = gBrowser.getStripVisibility();
  
    if (tabBarDisplayed)
      switchTabBarMenuItem.removeAttribute("disabled");
    else 
      switchTabBarMenuItem.setAttribute("disabled", "true");
  
    // this._prefs.writeBoolean("browser.tabs.autoHide", !tabBarDisplayed);
  },

  /* XXX: Formerly known as: mzToggleToolbar() */
  toggleToolbar: function(aToolbarID, aBroadcasterID)
  {
    var targetToolbar = document.getElementById(aToolbarID);
    var broadcaster = document.getElementById(aBroadcasterID);
  
    if (targetToolbar) {
      var toolbarIsHidden = (targetToolbar.getAttribute("hidden") == "true");
      // Semi Full Screen Mode enabled?
      var semiFullScreenModeEnabled = this._prefs.readBoolean("multizilla.fullscreen.adaptive", true);
      var windowElement = document.getElementsByTagName("window")[0];
      var semiFullScreenModeActive = (semiFullScreenModeEnabled && windowElement.hasAttribute("hidechrome"));
  
      if (window.fullScreen || semiFullScreenModeActive) {
        targetToolbar.setAttribute("fullscreentoolbar", "true");
        var toolbarIsCollapsed = (targetToolbar.getAttribute("moz-collapsed") == "true");
   
        if (toolbarIsHidden || toolbarIsCollapsed) {
          if (toolbarIsHidden)
            targetToolbar.setAttribute("hidden", "false");
  
          targetToolbar.removeAttribute("moz-collapsed");
        }
        else
          targetToolbar.setAttribute("moz-collapsed", "true");
  
        if (broadcaster)
          broadcaster.setAttribute("checked", toolbarIsCollapsed ? "false" : "true");
      }
      else {
        if (targetToolbar.hasAttribute("moz-collapsed"))
          targetToolbar.removeAttribute("moz-collapsed");
  
        targetToolbar.setAttribute("hidden", toolbarIsHidden ? "false" : "true");
  
        if (broadcaster)
          broadcaster.setAttribute("checked", toolbarIsHidden ? "false" : "true");
  
        document.persist(aToolbarID, "hidden");
        document.persist(aBroadcasterID, "checked");
      }
    }
  },

  initHistoryTooltip: function(aMenuItem)
  {
    var tooltip = document.getElementById("UndoTabHistoryTooltip");
    var tooltipBox = tooltip.firstChild;
    // Remove all previously added <label>'s
    while (tooltipBox.childNodes.length)
      tooltipBox.removeChild(tooltipBox.lastChild);
  
    var index = aMenuItem.getAttribute("value") || -1;
  
    if (index > -1) {
      var savedData = gBrowser.savedBrowsers[index];
      var sessionHistory = savedData.history;
      var count = sessionHistory.count;
  
      if (count) {
        var end = (count > 15) ? (count - 15) : 0;
  
        for (i = count - 1; i >= end; i--) {
          var entry = sessionHistory.getEntryAtIndex(i, false);
          entry = entry.QueryInterface(Components.interfaces.nsISHEntry);
          var label = document.createElement("label");
          label.setAttribute("value", (count == 1) ? entry.URI.spec : entry.title);
          tooltipBox.appendChild(label);
        }
        aMenuItem.parentNode.setAttribute("tooltip", "UndoTabHistoryTooltip");
      }
      /* if (sessionHistory.count == 1) {
        var canvas = tooltip.lastChild.firstChild;
        var browser = savedData.browser;
        // var win = browser.contentWindow;
        var win = gBrowser.selectedBrowser.contentWindow;
        var h = win.innerHeight;
        var ctx;
  
        try {
          ctx = canvas.getContext("2d");
        } catch(ex) {
          return;
        }
        var w = win.innerWidth;
        var canvasW = 300;
        var canvasH = Math.round(canvasW * h / w);
        canvas.width = canvasW;
        canvas.height = canvasH;
        canvas.parentNode.hidden = false;
        var backgroundColor = "#FFFFFF";
  
        if (browser.contentDocument instanceof ImageDocument &&
            !(browser.contentDocument.imageRequest.imageStatus & Components.interfaces.imgIRequest.STATUS_ERROR)) {
          ctx.fillStyle = String(backgroundColor);
          ctx.fillRect(0, 0, canvasW, canvasH);
          var img = browser.contentDocument.body.firstChild;
          var ratio = img.naturalHeight / img.naturalWidth;
  
          if (img.naturalHeight < canvasH && img.naturalWidth < canvasW)
            ctx.drawImage(img, 0, 0, img.naturalWidth, img.naturalHeight);
          else if (ratio * canvasW > canvasH)
            ctx.drawImage(img, 0, 0, canvasH / ratio, canvasH);
          else
            ctx.drawImage(img, 0, 0, canvasW, ratio * canvasW);
        }
        else {
          ctx.save();
          ctx.scale(canvasW / w, canvasH / h);
          ctx.drawWindow(win, win.pageXOffset, win.pageYOffset, w, h, backgroundColor);
          ctx.restore();
        }
      } */
      return;
    }
    aMenuItem.parentNode.removeAttribute("tooltip");
  },
  
  showHistoryTooltip: function(aTooltipBox)
  {
    return (aTooltipBox.childNodes.length > 0);
  }
};


/***
  * .tabs: mainly a collection of JavaScript functions to control tab related 
  *        features like: import, export and what not.
  */

gMultiZilla.tabs =
{
  _prefs: gMultiZilla.prefs,
  _nsIWindowDataSource: Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                  .getService(Components.interfaces.nsIWindowDataSource),
  _nsIPromptService: Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                               .getService(Components.interfaces.nsIPromptService),
  _nsIIOService: Components.classes["@mozilla.org/network/io-service;1"]
                           .getService(Components.interfaces.nsIIOService),

  initTabContextMenu: function()
  {
    var popupNode = document.popupNode;
  
    if (popupNode && popupNode.localName == "toolbarbutton")
      popupNode = getBrowser().selectedTab;
    gMultiZilla.googlebox.initMenuitem("GoogleBoxToolbarsMenuitem");
  
    if (popupNode.hasAttribute("customLabel")) {
      document.getElementById("setStickyMenuItem").setAttribute("hidden", "true");
      document.getElementById("clearStickyMenuItem").removeAttribute("hidden");
    }
    else {
      document.getElementById("clearStickyMenuItem").setAttribute("hidden", "true");
      document.getElementById("setStickyMenuItem").removeAttribute("hidden");
    }
    this.initToolbarsMenu();
  },
  
  initToolbarsMenu: function()
  {
    // Update Menu/View/Show-Hide to reflect the current state of the Tab Bar
    // We need this because we now use observes in our Toolbars menu
    var tabbarMenuItem = document.getElementById("menuitem_showhide_tabbar");
    var tabBarDisplayed = gBrowser.getStripVisibility();
    tabbarMenuItem.setAttribute("checked", tabBarDisplayed);
    // Overwrite the disabled state in updateToolbarStates() in navigator.js
    // This allows the user to hide/show the Tab Bar, at any time
    tabbarMenuItem.removeAttribute("disabled");
  },
  
  initTabImportMenu: function(importMenu)
  {
    var navigatorWindows = 0;
    var item = importMenu.childNodes[1];
    var windowManagerDS = this._nsIWindowDataSource;
  
    while ((item = item.nextSibling)) {
      var windowID = item.getAttribute("id");
      var win = windowManagerDS.getWindowForResource(windowID);
  
      if (win && win == window) {
        item.setAttribute("checked", "true");
        item.setAttribute("disabled", "true");
      }
      else {
        if (gMultiZilla.utility.isNavigatorWindow(win))
          navigatorWindows++;
        else
          item.setAttribute("disabled", "true");
      }
    }
    importMenu.firstChild.setAttribute("disabled", (navigatorWindows < 2) ? "true"
                                                                          : "false");
  },

  initTabExportMenu: function(aParent) 
  {
    // var panelContainer = gBrowser.mPanelContainer;
    var numberOfTabs = gBrowser.mTabContainer.childNodes.length;
  
    deleteHistoryItems(aParent); // XXX: this depends on sessionHistoryUI.js !!!
  
    for (var i = 0; i < numberOfTabs; i++) {
      var tab = gBrowser.mTabs[i];
      // var tabURL = panelContainer.childNodes[tab.ordinal].currentURI.spec; // this might fail for busy tabs!
      var url = tab.linkedBrowser.currentURI.spec; // this might fail for busy tabs!
      var menuitem = document.createElement("menuitem");
      menuitem.setAttribute("label", tab.label);
      menuitem.setAttribute("url", url);
      // set index based on tab.ordinal, to make deleteHistoryItems() work,
          // and we also use this attribute, as ordinal, in function exportTab!
      menuitem.setAttribute("index", i); // tab.ordinal);
      // The following code is copied from method initRestoreTabMenu in tabbrowser.xml
      menuitem.setAttribute("class", "menuitem-iconic bookmark-item"); // Display default bookmarks icon
      var iconURL = tab.getAttribute("image");
  
      if (iconURL && iconURL.match(/^http/i)) {
        if (gMultiZilla.utility.isImageCached(iconURL)) { 
          menuitem.setAttribute("class", "menuitem-iconic");
          menuitem.setAttribute("src", iconURL);
        }
      }
      aParent.appendChild(menuitem);
  
      if (url == "about:blank")
        menuitem.setAttribute("disabled", "true");
    }
    aParent.firstChild.setAttribute("disabled", (aParent.childNodes.length == 3) ? "true" : "false");
  },

  importAllTabs: function(aMenuitem)
  {
    while ((aMenuitem = aMenuitem.nextSibling))
      this.importTabsByWindowId(aMenuitem.getAttribute("id"));
  },
  
  exportTabs: function(aMenuitem)
  {
    // do we have to export all tabs to new windows?
    if (aMenuitem.id == "ExportAllTabs") {
      aMenuitem = aMenuitem.nextSibling;
  
      while ((aMenuitem = aMenuitem.nextSibling))
        this.exportTab(aMenuitem);
    }
    else // no, just export the selected tab to a new window
      this.exportTab(aMenuitem);

  },
  
  exportTab: function(aMenuitem)
  {
    // we skip the disabled 'about:blank' tabs
    if (!aMenuitem.hasAttribute("disabled")) {
      var windowID = openWindowAndReturnWindowID();
      // get ordinal from attribute index because that's what it is ;)
      var index = Number(aMenuitem.getAttribute("index"));
      var tab = gBrowser.mTabs[index];
      // call function moveTabToWindow, because that also handles history
      this.moveTabToWindow(tab, windowID, false);
    }
  },

  importTabsByWindowId: function(aWindowId)
  {
    var windowManagerDS = this._nsIWindowDataSource;
    var sourceWindow = windowManagerDS.getWindowForResource(aWindowId);
    var targetWindow = window;
  
    if (!sourceWindow || sourceWindow == targetWindow ||
        !gMultiZilla.utility.isNavigatorWindow(sourceWindow)) 
      return;
  
    var sourceTabbrowser = sourceWindow.getBrowser();
    var targetTabbrowser = targetWindow.getBrowser();
    var numberOfTabs = sourceTabbrowser.mTabContainer.childNodes.length;
  
    for (var i = 0; i < numberOfTabs; i++) {
      var url = sourceTabbrowser.browsers[i].currentURI.spec;
  
      if (url && url != "about:blank") {
        var blankBrowser = this.checkForBlankTab(targetTabbrowser); // Try to locate a new/unused blank tab
  
        if (blankBrowser)
          blankBrowser.loadURI(url);
        else 
          targetTabbrowser.addTab(url);
      }
    }
  },

  hideTab: function(aTab)
  {
    var clsTab = "close";

    if (aTab.hasAttribute(clsTab))
      aTab.removeAttribute(clsTab);
    else
      aTab.setAttribute(clsTab, "true");
  },

  /* XXX: Formely know as: loadFirstTabPage() */
  loadInitialURL: function(aTab)
  {
    dump("\nloadInitialURL: " + aTab);
    if (aTab == null)
      aTab = gBrowser.selectedTab;
  
    var url = aTab.getAttribute("firstPage");
    var browser = aTab.linkedBrowser;
  
    if (browser && url != browser.currentURI.spec)
      browser.loadURI(url);
  },
  
  closeTabOrWindow: function(aTab)
  {
    var tab = (aTab == undefined) ? gBrowser.mCurrentTab : aTab;
    var tabBrowser = getBrowser();
    var tabs = tabBrowser.mTabs.length;
    // 0 = load "about:blank" / 1 = load "about:blank" and hide Tab Bar / 2 = close window & quit browser
    var lastTabBehavior = this._prefs.readInteger("multizilla.tabs.last-tab.behavior", 0);
    var aboutBlank = (tab.linkedBrowser.currentURI.spec == "about:blank");
  
    if (tabs > 1)
      tabBrowser.removeTab(tab);
    else {
      // This is the last tab, so what do we need to do?
      switch(lastTabBehavior) {
        case 0:
        case 1:
          tabBrowser.removeTab(tab);
          // Do we need to hide the Tab Bar?
          if (lastTabBehavior == 1 && tabBrowser.getStripVisibility())
            gMultiZilla.chrome.toggleTabbarVisibility();
          break;
        case 2:
          if (lastTabBehavior == 2 && confirmCloseWS(1))
            BrowserCloseWindow();
      }
    }
  },
  
  preventDuplicate: function(aURL, useFirstPage, aBrowser, skipReload)
  {
    if (this._prefs.readBoolean("multizilla.prevent.double", true) && aURL.substring(0,10) != "javascript") {
      var tabHref = null;
      var baseHref = aURL.replace(/#.*/, "");
      var tabBrowser = (aBrowser == undefined) ? getBrowser() : aBrowser;
  
      if (baseHref[baseHref.length-1] == '/')
        baseHref = baseHref.substring(0, baseHref.length-1);
  
      for (var i = 0; i < tabBrowser.mTabs.length; i++) {
        if (tabBrowser.mTabs[i].hasAttribute("busy") || useFirstPage)
          tabHref = tabBrowser.mTabs[i].getAttribute("firstPage");
        else { 
          /*** 
            * 'tabHref' is 'about:blank' when we're restoring a previously saved browser 
            * session, because we need to open new tabs first, before we can modify the 
            * DocShell properties for the corresponding browser!
            */
          tabHref = tabBrowser.mTabs[i].linkedBrowser.webNavigation.currentURI.spec;
        }
        tabHref = tabHref.replace(/#.*/, "");
  
        if (tabHref[tabHref.length-1] == '/')
          tabHref = tabHref.substring(0, tabHref.length-1);
        var tab = tabBrowser.mTabs[i];
  
        if (tab && baseHref == tabHref) {
          if (tabBrowser.selectedTab != tab)
            tabBrowser.selectedTab = tab;
  
          var browser = tab.linkedBrowser;
  
          if (aURL.indexOf("#") >= 0) {
            /***
              * Anchors like: http://www.foobar.org#123 -> http://www.foobar.org#456 must 
              * use method loadURI() to point the page to the new anchor (a number in this case)
              */
            if (aURL != browser.webNavigation.currentURI.spec) {
              browser.loadURI(aURL);
              browser.reloadWithFlags(Components.interfaces.nsIWebNavigation.LOAD_FLAGS_NONE);
            }
            else 
              browser.reloadWithFlags(Components.interfaces.nsIWebNavigation.LOAD_FLAGS_IS_LINK); // LOAD_FLAGS_IS_REFRESH);
          }
          else if (!skipReload) {
            if (!tab.hasAttribute("busy"))
              browser.reloadWithFlags(Components.interfaces.nsIWebNavigation.LOAD_FLAGS_NONE);
          }
          return tab;
        }
      }
    }
    return null;
  },
  
  checkForBlankTab: function(aBrowser)
  {
    if (this._prefs.readBoolean("multizilla.tabs.blank-tab-first", true)) {
      var page = "about:blank";
      aBrowser = aBrowser ? aBrowser : getBrowser();
  
      for (var i = 0; i < aBrowser.mTabs.length; i++) {
        if (aBrowser.mTabs[i].hasAttribute("busy"))
          page = aBrowser.mTabs[i].getAttribute("firstpage");
        else 
          page = aBrowser.mTabs[i].linkedBrowser.currentURI.spec;
  
        if (page == "about:blank")
          return aBrowser.mTabs[i].linkedBrowser;
      }
    }
    return null;
  },
  
  openNewTabOrNewWindow: function()
  {
    if (this._prefs.readBoolean("multizilla.newtabfor.ctrl-n", true)) {
      var loadInBackground = gMultiZilla.prefs.readBoolean("browser.tabs.loadInBackground", true);
      gBrowser.addPreferredTab(!loadInBackground);
      setTimeout("gURLBar.focus();", 0);
    }
    else
      OpenBrowserWindow();
  },
  
  initMoveTabMenu: function(aID) 
  {
    var windowManagerDS = this._nsIWindowDataSource;
    var item = document.getElementById(aID);
  
    while ((item = item.nextSibling)) {
      var url = item.getAttribute("id");
      var win = windowManagerDS.getWindowForResource(url);

      if (win && win == window) {
        item.setAttribute("checked", "true");
        item.setAttribute("disabled", "true");
      }
      else if (!gMultiZilla.utility.isNavigatorWindow(win))
        item.setAttribute("disabled", "true");
    }
  },
  
  moveTabToWindow: function(aTab, aWindowID, aRemoveTabFlag, aInsertFlag)
  {
    var windowManagerDS = this._nsIWindowDataSource;
    var win = windowManagerDS.getWindowForResource(aWindowID);
  
    if (win && gMultiZilla.utility.isNavigatorWindow()) {
      var aURL = gBrowser.getBrowserForTab(aTab).currentURI.spec;
  
      if (aURL) {
        var tabBrowser = win.getBrowser();
        // Add a new blank tab and use it as our targetTab
        var targetTab = tabBrowser.addOrInsertTab(null, null, null, true, aInsertFlag, "duplicateTab");
        // 'tabBrowser.setReloadInterval()' fails without the next line!
        // tabBrowser.selectedTab = targetTab;
        // Get firstPage attribute
        var firstPage = aTab.getAttribute("firstPage");
        /***
          * function relaunchBrowserWindows in quickPrefsMenu.js in one of the 
          * callers of this function, and it checks the 'firstPage' attribute 
          * to remove blank/unused tabs, so we have to verify this attribute!
          */
        if (firstPage == "about:blank")
          firstPage = aURL;
        // Match first page attribute, used for menu item "Tab Home" (Shift+Home)
        targetTab.setAttribute("firstPage", firstPage);
        // Check for a custom label on the source tab
        if (aTab.hasAttribute("customLabel"))
          tabBrowser.setCustomLabel(targetTab, aTab.getAttribute("label")); // Match that label
        // Match reload interval
        tabBrowser.setReloadInterval(targetTab, aTab.mReloadInterval);
        // Match docShell properties
        var sourceBrowser = aTab.linkedBrowser;
        var targetBrowser = targetTab.linkedBrowser;
        tabBrowser.copyDocShellProperties(sourceBrowser, targetBrowser);
        // Match session history
        var tabHistory = gBrowser.getTabHistory(aTab);
        tabBrowser.setTabHistory(targetTab, tabHistory);
        targetBrowser.reloadWithFlags(nsIWebNavigation.LOAD_FLAGS_IS_REFRESH);
        // Do we have to remove the source tab?
        if (aRemoveTabFlag)
          gBrowser.removeTab(aTab);
      }
    }
  },
  
  moveTabToNewWindow: function(aTab)
  {
    var browser = aTab.linkedBrowser;
  
    if (!browser)
      return;
  
    var aURL = browser.currentURI.spec;
  
    if (!aURL)
      return;
    else if (aURL == "about:blank" && browser.sessionHistory.count < 1) {
      var promptService = this._nsIPromptService;
      promptService.alert(window, gMultiZilla.utility.getTextFromBundle("mzWizard"),
                          gMultiZilla.utility.getTextFromBundle("moveBlankTab"));
      return;
    }
    else {
      var tabBrowser = openWindowAndReturnBrowser();
      // Select targetTab
      var targetTab = tabBrowser.mTabs[0];
      // Match first page attribute, used for menu item "Tab Home" (Shift+Home)
      targetTab.setAttribute("firstPage", aTab.getAttribute("firstPage"));
      // Check for a custom label on the source tab
      if (aTab.hasAttribute("customLabel"))
        tabBrowser.setCustomLabel(targetTab, aTab.getAttribute("label")); // Match that label
      // Match reload interval
      tabBrowser.setReloadInterval(targetTab, aTab.mReloadInterval);
      // Match docShell properties
      tabBrowser.copyDocShellProperties(browser, tabBrowser);
      // Match session history
      var tabHistory = gBrowser.getTabHistory(aTab);
      tabBrowser.setTabHistory(targetTab, tabHistory);
      var targetBrowser = targetTab.linkedBrowser;
      // targetBrowser.reloadWithFlags(nsIWebNavigation.LOAD_FLAGS_IS_REFRESH);
      // It's a move so remove the source tab
      gBrowser.removeTab(aTab);
    }
  },
  
  doFKey: function(aFunctionKey)
  {
    var focusedElement = document.commandDispatcher.focusedElement;
  
    if (focusedElement instanceof HTMLTextAreaElement ||
        focusedElement instanceof HTMLInputElement 
        /* || focusedElement instanceof HTMLSelectElement */) {
      return;
    }
    aFunctionKey--;
  
    if (aFunctionKey > -1 && aFunctionKey < gBrowser.mTabs.length)
      gBrowser.selectedTab = gBrowser.mTabs[aFunctionKey];
  },
  
  addNewTabFromClipboard: function()
  {
    var aURL = gMultiZilla.utility.getLocationFromClipboard();

    if (aURL)
      gBrowser.selectedTab = gBrowser.addTab(aURL);
  },
  
  doReload: function(aEvent, reloadAll)
  {
    aEvent.stopPropagation();
  
    if (aEvent.ctrlKey || reloadAll)
      gBrowser.reloadAllTabs(aEvent.shiftKey ? true : false);
    else if (aEvent.shiftKey)
      BrowserReloadSkipCache();
    else
      BrowserReload();
  },
  
  doStop: function(aEvent, stopAll)
  {
    aEvent.stopPropagation();
  
    if (aEvent.ctrlKey || stopAll)
      gBrowser.stopAllTabs();
    else 
      BrowserStop();
  },

  /* XXX: Formely knowsn as: mzGetNumberOfBlankTabs() */
  getNumberOfBlankTabs: function(aTabBrowser)
  {
    var blankTabs = 0;
  
    for (var i = 0; i < aTabBrowser.mTabs.length; i++) {
      var tab = aTabBrowser.mTabs[i];
      var browser = aTabBrowser.mTabs[i].linkedBrowser;
  
      if (tab.hasAttribute("busy")) {
        if (tab.getAttribute("firstPage") == "about:blank")
          blankTabs++;
      }
      else if (browser.currentURI.spec == "about:blank")
        blankTabs++;
    }
    return blankTabs;
  },
  
  areAllTabsBlank: function(aTabBrowser)
  {
    if (!aTabBrowser)
      aTabBrowser = getBrowser();
  
    var numberOfTabs = aTabBrowser.mTabs.length;
    var numberOfBlankTabs = this.getNumberOfBlankTabs(aTabBrowser);
    return (numberOfTabs == numberOfBlankTabs);
  },
  
  handleNewTabLoadError: function(aBrowser, aLocation)
  {
    // setTimeout(handleTabLoadError, 500, aBrowser, aLocation);
    this._handleTabLoadError(aBrowser, aLocation);
  },
  
  _handleTabLoadError: function(aBrowser, aLocation)
  {
    /* aURL = aLocation.replace(/\/$/, ""); // trim trailing /
    var firstPage = aBrowser.pendingTab.getAttribute("firstPage").replace(/\/$/, ""); // trim trailing /
    // dump("\nfirstPage: " + firstPage + "\naURL: " + aURL);
  
    if (firstPage != aURL) { // We need this check for 0x80450002 (load aborted) errors!
      aBrowser.pendingTab = null;
    }
    else { */
      var failureAction = 0;
      const failurePrefString = "multizilla.tabs.load-failure.behavior";
  
      failureAction = this._prefs.readInteger(failurePrefString, 2);
  
      if (failureAction == 1) {
        gBrowser.removeTab(aBrowser.pendingTab);
        this.hideInBrowserHistory(aLocation);
        dump("\nWarning: tab closed, automatically, because it failed to load!");
      }
      else if (failureAction == 2)
      {
        var aEvent = window.document.createEvent('Events');
        aEvent.initEvent("DOMWillOpenModalDialog", false, false);
        aBrowser.contentDocument.defaultView.dispatchEvent(aEvent);
  
        var dialogTitle = gMultiZilla.utility.getTextFromBundle("tabFailureDialogTitle");
        var dialogQuestion = gMultiZilla.utility.getTextFromBundle("tabFailureDialogText");
        var dialogCheckbox = {value:false};
        var dialogCheckboxText = gMultiZilla.utility.getTextFromBundle("tabFailureDialogCheckboxText");
        var dialogRetryButton = gMultiZilla.utility.getTextFromBundle("RetryButton");
        var dialogCloseButton = gMultiZilla.utility.getTextFromBundle("CloseButton");
        var dialogLeaveOpenButton = gMultiZilla.utility.getTextFromBundle("LeaveOpenButton");
        var promptService = this._nsIPromptService;
        var result = promptService.confirmEx(window, dialogTitle, dialogQuestion, promptService.BUTTON_POS_0_DEFAULT +
                                             (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_IS_STRING) + 
                                             (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_IS_STRING) +
                                             (promptService.BUTTON_POS_2 * promptService.BUTTON_TITLE_IS_STRING),
                                             dialogRetryButton,
                                             dialogLeaveOpenButton,
                                             dialogCloseButton,
                                             dialogCheckboxText,
                                             dialogCheckbox);
  
        aEvent.initEvent("DOMModalDialogClosed", false, false);
        aBrowser.contentDocument.defaultView.dispatchEvent(aEvent);
  
        if (result == 0) { // Retry button
          aBrowser.pendingTab.removeAttribute("notfound");
          aBrowser.pendingTab = null;
          aBrowser.loadURI(aLocation); // aURL);
        }
        else if (result == 1) // Leave Open button selected (or ESC/X)
          aBrowser.pendingTab = null;
        else { // 2 = Close button selected
          gBrowser.removeTab(aBrowser.pendingTab);
          this.hideInBrowserHistory(aLocation);
          dump("\nWarning: tab closed, after prompt, because it failed to load!");
        }
        if (dialogCheckbox.value)
          this._prefs.writeInteger(failurePrefString, (!result));
      } // else if (failureAction == 2) {
    // } // else {
  },
  
  /* XXX: Formely knows as: mzHideInBrowserHistory() */
  hideInBrowserHistory: function(aURI)
  {
    if (typeof(aURI) == "string")
      aURI = this._nsIIOService.newURI(aURI, null, null);

    var globalHistory;
    // Builds 20040210 and up (see also mozilla bug 224829) 
    if ("@mozilla.org/browser/global-history;2" in Components.classes)
      globalHistory = Components.classes["@mozilla.org/browser/global-history;2"];
    else 
      globalHistory = Components.classes["@mozilla.org/browser/global-history;1"];
  
    globalHistory.getService(Components.interfaces.nsIBrowserHistory).hidePage(aURI);
  },

  /* XXX: Formely knows as: mzOpenSearchInTab() */
  openSearchInTab: function()
  {
    if (this._prefs.readBoolean("multizilla.newtabfor.websearch", true))
      return gBrowser.addTab(null, null, null, true, null, null, null, true);
    return null;
  },

  /* XXX: Formely knows as: mzOpenSearch() */
  openSearch: function(aSearchStr)
  {
    /* See also: http://lxr.mozilla.org/mozilla1.7/source/xpfe/browser/resources/content/navigator.js#999
       and:      http://lxr.mozilla.org/mozilla/source/xpfe/browser/resources/content/navigator.js#1034 */
    var newWindowFlag = (this.openSearchInTab() == null);
    var powerSearch;
  
    try {
      pref.getIntPref("browser.search.powermode");
      powerSearch = true;
    } catch(ex) {
      powerSearch = false;
    }
    if (powerSearch)
      OpenSearch("internet", false, aSearchStr, newWindowFlag);
    else 
      OpenSearch("internet", aSearchStr, newWindowFlag);
  },
  
  /* XXX: Formerly known as: mzOpenTabOrWindow() */
  openTabOrWindow: function(aEvent)
  {
    if (aEvent.button == 1 || aEvent.ctrlKey) {
      if (aEvent.shiftKey)
        window.open();
      else
        OpenBrowserWindow();
    }
    else if (aEvent.button == 0) {
      var loadInBackground = this._prefs.readBoolean("browser.tabs.loadInBackground", true);
  
      if (aEvent.shiftKey)
        loadInBackground = !loadInBackground;
  
      gBrowser.addPreferredTab(!loadInBackground);
      setTimeout('gURLBar.focus();', 0);
    }
  }
};


/***
  * mzManagers is a JavaScript "class" with helper functions to open MultiZilla Managers
  */

gMultiZilla.managers =
{
  _prefs: gMultiZilla.prefs,
  _utility: gMultiZilla.utility,
  _nsIWindowMediator: Components.classes['@mozilla.org/appshell/window-mediator;1']
                                .getService(Components.interfaces.nsIWindowMediator),

  openPreferenceManager: function(aPanelID, aTabID)
  {
    const _multiZilla = "MainPrefPanel";
    const _prefPath = "chrome://multiviews/content/";
    const usePrefManager = this._prefs.readBoolean("multizilla.prefs.use-preference-manager", true);

    aPanelID = (aPanelID == undefined) ? _multiZilla : aPanelID;
    aTabID = (aTabID == undefined) ? "" : aTabID;

    if (usePrefManager || 
        /(General|Confirmations|Content|Feeds|Popups|Privacy|TabSessions|Tabs|Miscellaneous)PrefPanel/.test(aPanelID)) {
      // Is the Preference Manager already opened?
      var prefWindow = this._utility._getMostRecentWindow("multizilla:preferences", false);

      if (prefWindow)
        prefWindow.focus();
      else {
        var dialogOption = /Mac/.test(navigator.platform) ? "=no": "";
        var winFeatures = "chrome, toolbar, dialog" + dialogOption + ", all"
        // The Preference Manager should be modal on Windows!
        if (/Win/.test(navigator.platform))
          winFeatures += ", dependent, modal";

        window.openDialog(_prefPath + "preference-manager/preferenceManager.xul", "", winFeatures,
                          "", aPanelID, aTabID, this._utility.getPlatformExtras());
      }
    }
    else
      goPreferences(_multiZilla, _prefPath + "preferences/multizillaMainPanel.xul", aPanelID);
  },

  openSessionManager: function(aPrefFlag)
  {
    /***
      * Do a reverse lookup, otherwise we will always end up with 'multizilla:sessionmanager'
      * Note: keep the ID's of tsmIDs[] in sync with the XUL files for the windows!
      */
    var tsmIDs = [ "multizilla:tsm-renameSession", "multizilla:tsm-windowSelector", 
                   "multizilla:tabBehaviorSelection", "multizilla:sessionManager" ];

    for (i in tsmIDs) {
      var managerWin = gMultiZilla.utility._getMostRecentWindow(tsmIDs[i], false);

      if (managerWin) {
        if (i == 3) { // All windows that can open dialog windows
          var enumerator = this._nsIWindowMediator.getEnumerator(null);

          while (enumerator.hasMoreElements()) {
            var win = enumerator.getNext();
            // Is addBookmark.xul open?
            if (win.document.documentElement.id == "newBookmarkDialog") {
              managerWin = win; // Yes, use that window
              break;
            }
            // Is there a confirmation dialog opened?
            if (win.document.documentElement.id == "commonDialog" &&
                win.opener == managerWin) {
              managerWin = win; // Yes, use that window
              break;
            }
          }
        }
        managerWin.focus();
        return;
      }
    }
    var dialogOption = /Mac/.test(navigator.platform) ? "=no": "";
    window.openDialog("chrome://multiviews/content/session-manager/sessionManager.xul",  
                      "", "chrome, dialog" + dialogOption + ", all", (aPrefFlag ? "preferences" : ""),
                      gMultiZilla.utility.getPlatformExtras());
  },

  openUserAgentManager: function(aPrefFlag)
  {
    var managerWin = this._utility._getMostRecentWindow("multizilla:useragent-manager", false);

    if (managerWin) {
      var enumerator = this._nsIWindowMediator.getEnumerator(null);

      while (enumerator.hasMoreElements()) {
        var win = enumerator.getNext();
        // Is there a confirmation dialog opened?
        if (win.document.documentElement.id == "commonDialog" && win.opener == managerWin) {
          managerWin = win; // Yes, use that window
          break;
        }
      }
      managerWin.focus();
      return;
    }
    var dialogOption = /Mac/.test(navigator.platform) ? "=no": "";
    window.openDialog("chrome://multiviews/content/useragent-manager/userAgentManager.xul", 
                      "", "chrome, dialog" + dialogOption + ", all", (aPrefFlag ? "preferences" : ""),
                      this._utility.getPlatformExtras());
  },

  openTabManager: function(aPrefFlag)
  {
    var managerWin = this._utility._getMostRecentWindow("multizilla:tabmanager", false);

    if (managerWin) {
      var enumerator = this._nsIWindowMediator.getEnumerator(null);

      while (enumerator.hasMoreElements()) {
        var win = enumerator.getNext();
        // Is there a confirmation dialog opened?
        // if (win.document.documentElement.id == "commonDialog" && win.opener == dialogWin) {
        if (win.document.documentElement.id == "commonDialog" && 
            managerWin.document.documentElement.id == win.opener.document.documentElement.id)
        {
          managerWin = win; // Yes, use that window
          break;
        }
      }
      managerWin.focus();
      return;
    }
    var dialogOption = /Mac/.test(navigator.platform) ? "=no": "";
    window.openDialog("chrome://multiviews/content/tab-manager/tabManager.xul", 
                      "", "chrome, alwaysRaised, dialog" + dialogOption + ", all",
                      (aPrefFlag ? "preferences" : ""), this._utility.getPlatformExtras());
  },

  openPermissionManager: function(aHostPort, aPanelID, aTabID)
  {
    aHostPort = (aHostPort == undefined) ? "" : aHostPort;
    /***
      * Do a reverse lookup, otherwise we will always end up with 'multizilla:permissionmanager'
      * Note: keep the ID's of pmIDs[] in sync with the XUL files for the windows!
      */
    var pmIDs = ["multizilla:pm-filemanager", "multizilla:pm-prefwindow", 
                 "multizilla:permissionmanager"];

    for (i in pmIDs) {
      var targetWindow = this._utility._getMostRecentWindow(pmIDs[i], false);
      // var permManagerWindow = this._nsIWindowMediator.getMostRecentWindow("multizilla:permissionmanager");

      if (targetWindow) {
        var enumerator = this._nsIWindowMediator.getEnumerator(null);

        while (enumerator.hasMoreElements()) {
          var win = enumerator.getNext();
          // Is there a confirmation dialog opened?
          if (win.document.documentElement.id == "commonDialog" && win.opener == targetWindow) {
            targetWindow = win; // Yes, use that window
            break;
          }
        }
        try {
          targetWindow.focus();
          targetWindow.document.focus();
          targetWindow.documentElement.focus();
        }
        catch(ex) {
          if (targetWindow.filePickerParent) {
            // dump("\nID: " + targetWindow.filePickerParent.document.getElementsByTagName("radiogroup")[0].localName); // .documentElement.id);
            targetWindow.filePickerParent.document.getElementsByTagName("radiogroup")[0].focus();
          }
        }
      }
    }
    if (targetWindow)
      return;

    // if (!aType)
      // aType = this._prefs.readString("multizilla.perm-manager.default-manager", "cookieManager");

    if (aHostPort == undefined) {
      var browser = gBrowser.getBrowserForTab(gBrowser.selectedTab);

      if (browser) {
        var currentURI = browser.docShell.QueryInterface(Components.interfaces.nsIWebNavigation).currentURI;
        try {
          aHostPort = (currentURI.scheme == "file") ? "file:" : currentURI.hostPort;
        } catch(ex) {
          // Die silently
        }
      }
    }
    window.openDialog("chrome://multiviews/content/permission-manager/permissionManager.xul", 
                      "", "chrome, dialog, minimizable, resizable", aHostPort,
                      aPanelID, aTabID, this._utility.getPlatformExtras());
  },

  /* XXX: Formerly known as: toMZJavaScriptConsole() */
  openErrorConsole: function()
  {
    var jsConsoleWindow = this._utility._getMostRecentWindow("global:console", false);
  
    if (jsConsoleWindow) {
      var enumerator = this._nsIWindowMediator.getEnumerator(null);
  
      while (enumerator.hasMoreElements()) {
        var win = enumerator.getNext();
  
        if (win.document.documentElement.id == "jsc-OptionsWindow") {
          jsConsoleWindow = win;
          break;
        }
      }
      jsConsoleWindow.focus();
    }
    else {
      window.openDialog("chrome://multiviews/content/js-console/console.xul",
                        "", "chrome, dialog, all",
                        this._utility.getPlatformExtras());
    }
  }
};



/***
  * .bookmarks: various JavaScript functions to initialize the bookmarks 
  *             template, at startup, and functions to open folders and
  *             groupmarks at users request.
  */

gMultiZilla.bookmarks =
{
  MZ_OPEN_BOOKMARKS_AS_TABS: 0,
  MZ_OPEN_BOOKMARKS_AS_WINDOWS: 1,
  MZ_OPEN_BOOKMARKS_IN_NEW_WINDOW: 2,

  _tabIndex: 0, // added for: 
                // openFolderAsTabGroup(aFolder, aOpenFlag);
                // openBookmarkFolder(element, datasource, tabCount, tabOpen);
                // openBookmarkFolderFromResource(datasource, resource, tabCount, tabOpen);
  _prefs: gMultiZilla.prefs,

  /* XXX: Formerly known as: initializeBookmarksTemplate() */
  _initBookmarksTemplate: function()
  {
    // Bookmark Menu modifications
    var mzPlainFolder = document.getElementById("mzPlainFolder");
    var mzPlainGroup = document.getElementById("mzPlainGroup");
    // Bookmarks Button modifications
    var mzBBPlainFolder = document.getElementById("mzBBPlainFolder");
    var mzBBPlainGroup = document.getElementById("mzBBPlainGroup");
  
    var prefValue = this._prefs.readInteger("multizilla.bookmark.folders", 5);
  
    if (prefValue && prefValue != 16) {
      mzPlainFolder.setAttribute("disabled", "true");
      mzBBPlainFolder.setAttribute("disabled", "true");
    }
    prefValue = this._prefs.readInteger("multizilla.bookmark.groups", 5);
  
    if (prefValue && prefValue != 16) {
      mzPlainGroup.setAttribute("disabled", "true");
      mzBBPlainGroup.setAttribute("disabled", "true");
    }
  },

  /* XXX: Formerly known as: initializePersonalToolbarTemplate() */
  _initPersonalToolbarTemplate: function()
  {
    var mzPTPlainFolder = document.getElementById("mzPTPlainFolder");
    var mzPTPlainGroup = document.getElementById("mzPTPlainGroup");
    var mzPTBPlainFolder = document.getElementById("mzPTBPlainFolder");
    var mzPTBPlainGroup = document.getElementById("mzPTBPlainGroup");
    var prefValue = this._prefs.readInteger("multizilla.bookmark.folders", 5);
  
    if (prefValue && !(prefValue & 16)) {
      mzPTPlainFolder.setAttribute("disabled", "true");
      mzPTBPlainFolder.setAttribute("disabled", "true");
    }
    prefValue = this._prefs.readInteger("multizilla.bookmark.groups", 5);
  
    if (prefValue && !(prefValue & 16)) {
      mzPTPlainGroup.setAttribute("disabled", "true");
      mzPTBPlainGroup.setAttribute("disabled", "true");
    }
  },

  initTemplates: function()
  {
    this._initBookmarksTemplate();
    this._initPersonalToolbarTemplate();

    // select modified template for the Bookmarks menu
    var bookmarksMenu = document.getElementById("BookmarksMenu");
    bookmarksMenu.setAttribute("template", "BookmarksMenuTemplate");
    // bookmarksMenu.childNodes[0].setAttribute("context", "bookmarks-context-menu");
  
    // select modified template for the Bookmarks Button
    var bookmarksButton = document.getElementById("bookmarks-button");
    // bookmarksButton.removeChild(bookmarksButton.firstChild);
    bookmarksButton.setAttribute("template", "BookmarksButtonTemplate");
    // bookmarksButton.childNodes[0].setAttribute("context", "bookmarks-context-menu");
  
    // select modified template for the Bookmarks Overflow Menu
    var bookmarksChevron = document.getElementById("bookmarks-chevron");
    bookmarksChevron.setAttribute("template", "BookmarksButtonTemplate");
    // bookmarksChevron.childNodes[0].setAttribute("context", "bookmarks-context-menu");
  
    // replace default PT template with our modified version
    var originalPTBox = document.getElementById("bookmarks-ptf");
    var newPTBox = document.getElementById("NEW-NC:PersonalToolbarFolder");
    var clonedTemplate = newPTBox.childNodes[1].cloneNode(true);
    originalPTBox.replaceChild(clonedTemplate, originalPTBox.childNodes[1]);
  
    if (!/Mac/.test(navigator.platform)) { // Modify Main Bookmarks menu
      // bookmarksMenu.removeAttribute("oncommand"); XXX: see bug 3324
      // bookmarksMenu.setAttribute("onclick", "OpenBookmarkURL(event.target, this.database, event)");
    }
    // cleanup: remove dummy bookmarks menu item
    var dummyBookmarkPopup = document.getElementById("dummyBookmarkPopup");
    dummyBookmarkPopup.removeChild(newPTBox);
  
    // theme support: added for Mac(OSX)Zilla support
    bookmarksButton.firstChild.setAttribute("class", "bookmarks-menupopup");
    bookmarksChevron.firstChild.setAttribute("class", "bookmarks-menupopup");
    // createContextMenu() must return false for our extra menu items!
    var bookmarksContextMenu = document.getElementById("bookmarks-context-menu");
    bookmarksContextMenu.setAttribute("onpopupshowing", "return BookmarksMenu.createContextMenu(event)");
  },

  /* XXX: Formerly known as: refreshBookmarkMenuPopup() */
  refreshBookmarkMenuPopup: function(aParent, aPrefIdentifier)
  {
    if (!aParent || !aPrefIdentifier)
      return;
  
    var menu = aParent;
    var menuItems = menu.childNodes.length;
    var prefValue = Number(this._prefs.readInteger(aPrefIdentifier, 0));
    var i = 1;
  
    if (menuItems > 5) {
      menu.firstChild.setAttribute("hidden", "true");
  
      for (var b = 1; b < 16; b <<= 1) {
        if (b == 8) // Set As Home Page has been removed!
          continue;
        menu.childNodes[i++].setAttribute("hidden", (prefValue & b) ? "false"
                                                                    : "true");
      }
      menu.childNodes[i].setAttribute("hidden", (prefValue & (b-1)) ? "false"
                                                                    : "true");
  
      if (menu.childNodes[1].getAttribute("hidden") == "false") {
        var bundle = gMultiZilla.utility.getStringBundle();
        var isGroupmark = (aPrefIdentifier == "multizilla.bookmark.groups");
        var labelProperties = (isGroupmark) ? [ "openGroupmarkInExistingTabs",
                                                "openGroupmarkInNewTabs",
                                                "openGroupmarkInSelectedTabs" ]
                                            : [ "openFolderInExistingTabs",
                                                "openFolderInNewTabs",
                                                "openFolderInSelectedTabs" ];
        prefValue = Number(this._prefs.readInteger("multizilla.bookmark.tabgroup.behavior", 2));
        var label = bundle.GetStringFromName(labelProperties[prefValue]);
        menu.childNodes[1].label = label;
      }
    }
    if (menuItems == 5 ||
        menuItems == 6 && menu.childNodes[4].localName == "menuitem") {
      menu.childNodes[0].removeAttribute("hidden");

      for (i = 1; i < menuItems; i++)
        menu.childNodes[i].setAttribute("hidden", "true");
    }
  },

  /* XXX: Formerly known as: mzCloseBookmarkMenu() */
  closeMenu: function(aNode)
  {
    while (aNode.nodeName != "toolbarbutton" && aNode.nodeName != "menubar") {
      aNode = aNode.parentNode;
      if (aNode.nodeName == "menupopup")
        aNode.hidePopup();
    }
    // We need to 'Close" the Bookmarks button on the menubar
    if (aNode.nodeName == "menubar") {
      var menu = document.getElementById("BookmarksMenu");
      menu.setAttribute("_moz-menuactive", false);
    }
  },

  /* XXX: Formerly known as: mzLoadFolderAsGroup() */
  openFolderAsTabGroup: function(aFolder, aOpenFlag)
  {
    var shouldConfirm = true;
    var openBehavior = this._prefs.readInteger("multizilla.bookmark.tabgroup.behavior", 2);
  
    if (aOpenFlag == this.MZ_OPEN_BOOKMARKS_IN_NEW_WINDOW &&
        !this._prefs.readBoolean("multizilla.bookmark.open-tabgroup-in-window.confirmation", true))
      shouldConfirm = false;
    else if (aOpenFlag == this.MZ_OPEN_BOOKMARKS_AS_WINDOWS &&
             !this._prefs.readBoolean("multizilla.bookmark.open-tabgroup-as-windows.confirmation", true))
      shouldConfirm = false;
    else if (aOpenFlag == this.MZ_OPEN_BOOKMARKS_AS_TABS && openBehavior == 2)
      shouldConfirm = false;
  
    if (shouldConfirm && !this.confirmTabGroupOpen(aOpenFlag))
      return;
  
    var url;
    var dupFound = 0;
    var openAsWindows = (aOpenFlag == this.MZ_OPEN_BOOKMARKS_AS_WINDOWS);
    var openInNewWindow = (aOpenFlag == this.MZ_OPEN_BOOKMARKS_IN_NEW_WINDOW);
    var children = aFolder.childNodes;
    var closeRemaining = this._prefs.readBoolean("multizilla.bookmark.close-remaining", false);
    var gBrowser = openInNewWindow ? openWindowAndReturnBrowser() : getBrowser();
    var numberOfTabs = gBrowser.mTabs.length;
  
    try {
      if (!openAsWindows && !openInNewWindow && openBehavior == 2) {
        window.openDialog("chrome://multiviews/content/session-manager/miscellaneous/tabBehaviorSelection.xul", 
                          "", "chrome,modal,centerscreen", getBrowser(),
                          closeRemaining, gMultiZilla.utility.getPlatformExtras());
  
        if (gBrowser.mNewTabs == -1)
          return;
  
        openBehavior = gBrowser.mNewTabs;
        closeRemaining = gBrowser.mRemoveRemaining;
      }
      var newTabs = (openBehavior == 1);
      var startTab = newTabs ? numberOfTabs : 0;
  
      this._tabIndex = 0;
      /* [0] (Empty)
       * [1] Open Folder/Groupmark in Existing Tabs...
       *     Open Folder/Groupmark in New Tabs...
       *     Open Folder/Groupmark in Tabs...
       * [2] Open Folder/Groupmark in Windows..
       * [3] Open Folder/Groupmark in a New Window..
       * [4] menuseparator
       */
  
      for (var i = 5; i < children.length; i++) {
        url = children[i].getAttribute("url");
  
        if (children[i].getAttribute("container") == "true") {
          this.openBookmarkFolder(children[i], document.getElementById("BookmarksMenu").database,
                             numberOfTabs, newTabs, openAsWindows, openInNewWindow, gBrowser);
        }
        else if (url) {
          if (openAsWindows) {
            window.open(url);
          } 
          else if (!gMultiZilla.tabs.preventDuplicate(url)) {
            if (!newTabs && this._tabIndex < numberOfTabs) {
              var browser = gBrowser.mTabs[this._tabIndex].linkedBrowser;
              browser.loadURI(url);
            }
            else
              gBrowser.addOrGetTab(url);
            this._tabIndex++;
          }
          else
            dupFound++;
        }
      }
      // Select the first tab in the group.
      if (!openAsWindows && !openInNewWindow)
        gBrowser.selectedTab = gBrowser.mTabs[startTab];
      if (!newWindows && closeRemaining) {
        if (newTabs) {
          /* startTab = gBrowser.mTabs[startTab].ordinal-1;
          for (i = startTab; i >= 0; i--)
            gBrowser.removeTab(gBrowser.mTabs[i]); */
        }
        else if (dupFound == 0) {
          for (i = numberOfTabs-1; i >= this._tabIndex; i--)
            gBrowser.removeTab(gBrowser.mTabs[i]);
        }
      }
      gBrowser = getBrowser();
    } catch(ex) {
      gBrowser = getBrowser();
    }
  },

  /* XXX: Formerly known as: OpenBookmarkFolder() */
  openBookmarkFolder: function(element, datasource, numberOfTabs, newTabs, newWindows, isChild, aTabbrowser)
  {
    if (!datasource)
      return;

    var id = element.getAttribute("id");
    var resource = RDF.GetResource(id, true);

    this.openBookmarkFolderFromResource(datasource, resource, numberOfTabs, newTabs, newWindows, isChild, aTabbrowser);
  },

  /* XXX: Formerly known as: OpenBookmarkFolderFromResource() */
  openBookmarkFolderFromResource: function(datasource, resource, numberOfTabs, newTabs, newWindows, isChild, aTabbrowser)
  {
    var urlProperty = RDF.GetResource("http://home.netscape.com/NC-rdf#URL");
    var rdfContainer = Components.classes["@mozilla.org/rdf/container;1"]
                                 .getService(Components.interfaces.nsIRDFContainer);
    rdfContainer.Init(datasource, resource);
    var containerChildren = rdfContainer.GetElements();
    var gBrowser = aTabbrowser ? aTabbrowser : getBrowser();
    this._tabIndex = 0;
  
    while (containerChildren.hasMoreElements())
    {
      resource = containerChildren.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      var target = datasource.GetTarget(resource, urlProperty, true);
  
      if (target) {
        var url = target.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
  
        if (url) {
          if (newWindows)
            window.open(url);
          else {
            if (!gMultiZilla.tabs.preventDuplicate(url)) {
              var browser = gMultiZilla.tabs.checkForBlankTab();
  
              if (!isChild && !newTabs && this._tabIndex < numberOfTabs) {
                browser = gBrowser.mTabs[this._tabIndex].linkedBrowser;
                browser.loadURI(url);
              }
              else {
                if (browser && this._prefs.readBoolean("multizilla.tabs.blank-tab-first", true))
                  browser.loadURI(url);
                else
                  gBrowser.addTab(url);
              }
              this._tabIndex++;
            }
          }
        }
      }
      else
        this.openBookmarkFolderFromResource(datasource, resource, numberOfTabs, newTabs, newWindows, isChild, aTabbrowser);
    }
  },

  /* XXX: Formerly known as: mzConfirmOpeningOfBookmarks() */
  confirmTabGroupOpen: function(aOpenFlag, aWindow)
  {
    if (aWindow == undefined)
      aWindow = window;
  
    var confirmationText = "";
    var prefTag = "";
  
    switch(aOpenFlag)
    {
      case this.MZ_OPEN_BOOKMARKS_IN_NEW_WINDOW:
           prefTag = "tabgroup-in-window";
           confirmationText = "confirmOpenBookmarksInNewWindow.label";
           break;
      case this.MZ_OPEN_BOOKMARKS_AS_WINDOWS:
           prefTag = "tabgroup-as-windows";
           confirmationText = "confirmOpenBookmarksAsWindows.label";
           break;
      default: // this.MZ_OPEN_BOOKMARKS_AS_TABS
           prefTag = "tabgroup";
           confirmationText = "confirmOpenBookmarksInTabs.label";
    }
    var prefIdentifier = "multizilla.bookmark.open-" + prefTag + ".confirmation";
    var shouldConfirm = this._prefs.readBoolean(prefIdentifier, false);
  
    if (shouldConfirm) {
      var aDialogTitle = gMultiZilla.utility.getTextFromBundle("generalConfirmAction.title");
      var aDialogText = gMultiZilla.utility.getTextFromBundle(confirmationText);
      var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"].getService();
      promptService = promptService.QueryInterface(Components.interfaces.nsIPromptService);
      return promptService.confirm(aWindow, aDialogTitle, aDialogText);
    }
    return true;
  }  
};


/***
  * .feeds: JavaScript functions for MultiZilla's Feed Reader / Feed Viewer
  */

gMultiZilla.feeds =
{
  _prefs: gMultiZilla.prefs,

  initSidebarBuildViewerPopup: function()
  {
    var popupMenu = document.getElementById("sidebar-panel-viewer-popup");
    popupMenu.database.AddDataSource(RDF.GetDataSource(sidebarObj.datasource_uri));
    popupMenu.builder.rebuild();
  
    var currentPanel = document.getElementById("sidebar-panels").getAttribute("last-selected-panel");
    var hideHeaders = this._prefs.readBoolean("multizilla.sidebar-view.hideheaders", true);
    var hideHeadersText = (hideHeaders) ? gMultiZilla.utility.getTextFromBundle("showSidebarHeaders")
                                        : gMultiZilla.utility.getTextFromBundle("hideSidebarHeaders");

    popupMenu.firstChild.label = hideHeadersText;
  
    for (var i = 2; i < popupMenu.childNodes.length; i++) {
      var menuitem = popupMenu.childNodes.item(i);
  
      if (sb_panel_is_excluded(menuitem))
         menuitem.setAttribute("disabled", "true");
      if (menuitem.id == currentPanel)
        menuitem.setAttribute("checked", "true");
    }
  },

  /* XXX: Formerly known as: mzInitWebFeedsButtonTooltip() */
  initWebFeedsButtonTooltip: function(aNode)
  {
    var count = aNode.parentNode.getAttribute("state");
    aNode.firstChild.firstChild.value = String(count) + " feed(s) have unread articles";
  
    /* description = aNode.firstChild.firstChild;
    var bold = document.createElementNS("http://www.w3.org/1999/xhtml", "b");
    bold.appendChild(document.createTextNode(count));
    description.appendChild(bold);
    description.appendChild(document.createTextNode(" feed(s) have unread articles")); */
  
    const mzWebFeedService = Components.classes["@multizilla.org/webfeed-service;1"]
                                       .createInstance(Components.interfaces.nsIWebFeedService);
    aNode.firstChild.childNodes[1].value = mzWebFeedService.nextUpdateCheck
  },

  toggleSidebarViewPref: function()
  {
    var prefID = "multizilla.sidebar-view.hideheaders";
    var hideHeaders = this._prefs.readBoolean(prefID, true);
    var sidebarPanelsNode = document.getElementById("sidebar-panels");
  
    if (hideHeaders)
      sidebarPanelsNode.removeAttribute("hide-headers");
    else
      sidebarPanelsNode.setAttribute("hide-headers", "true");
  
    this._prefs.writeBoolean(prefID, !hideHeaders);
  },

  /* XXX: Formerly known as: mzOpenWebFeedsSideBarPanel() */
  openWebFeedsSideBarPanel: function(aOpenFlag)
  {
    var wasCollapsed = sidebar_is_collapsed();
    var sidebarIsHidden = sidebar_is_hidden();
    var sbBox = document.getElementById("sidebar-box");
  
    if (wasCollapsed) {
      SidebarExpandCollapse();
  
      if (sbBox.width == 30)
        sbBox.width = 200;
    }
    if (gMustInit)
      sidebar_overlay_init();
  
    var webFeedPanelID = "urn:sidebar:panel:webfeeds";
    var sbPanelsNode = document.getElementById("sidebar-panels");
  
    if (!sbPanelsNode.hasAttribute("hide-headers"))
      toggleSidebarViewPref();
  
    if (SidebarGetLastSelectedPanel() != webFeedPanelID) {
      sbPanelsNode.setAttribute("last-selected-panel", webFeedPanelID);
      /**
       * Function SidebarSelectPanel() normally reads the id property from 
       * the selected menu item in the View menu, but won't be there until 
       * you open it yourself, thus it fails most of the times.  We solved 
       * this problem by creating a menuitem and setting the id property ;)
       */
      var menuitem = document.createElement("menuitem");
      menuitem.id = webFeedPanelID;
      SidebarSelectPanel(menuitem, false, false);
      // Leave the sidebar open afer selecting our panel
      if (!wasCollapsed && !sidebarIsHidden)
        wasCollapsed = true;
    }
    if (!wasCollapsed) {
      if (aOpenFlag == undefined || (aOpenFlag && sidebarIsHidden))
        SidebarShowHide();
    }
  }
};

/***
  * mzPrivacyTools is a JavaScript "class" with the helper functions to 
  * remove the stored 'privacy sensitive data' from memory and/or harddrive.
  */
gMultiZilla.privacy = /* XXX: Formerly known as: mzPrivacyTools */
{
  _nsICacheService : Components.classes["@mozilla.org/network/cache-service;1"]
                               .getService(Components.interfaces.nsICacheService),
  _nsICookieManager : Components.classes["@mozilla.org/cookiemanager;1"]
                                .getService(Components.interfaces.nsICookieManager),
  _nsIPasswordManager : Components.classes["@mozilla.org/passwordmanager;1"]
                                  .getService(Components.interfaces.nsIPasswordManager),
  _nsIDownloadManager : Components.classes["@mozilla.org/download-manager;1"]
                                  .getService(Components.interfaces.nsIDownloadManager),
  _nsIWalletEditor : Components.classes["@mozilla.org/walleteditor/walleteditor-world;1"]
                               .createInstance(Components.interfaces.nsIWalletEditor),

  clearDiskMemCache: function()
  {
    this.clearCache("BOTH");
  },

  clearAllBackTraces: function()
  {
    this.clearGlobalHistory();
    this.clearUrlBarHistory();
    this.clearCache("BOTH");
    gMultiZilla.dnp.purgeIDNHistory();
  },

  clearGlobalHistory: function()
  {
    var globalHistory;
    // Builds 20040210 and up (see also mozilla bug 224829) 
    if ("@mozilla.org/browser/global-history;2" in Components.classes)
      globalHistory = Components.classes["@mozilla.org/browser/global-history;2"];
    else 
      globalHistory = Components.classes["@mozilla.org/browser/global-history;1"];

    globalHistory = globalHistory.getService(Components.interfaces.nsIBrowserHistory);
    globalHistory.removeAllPages();
  },

  clearUrlBarHistory: function()
  {
    // New implementation, this only works in Mozilla builds 20040602 and later
    // see also: https://bugzilla.mozilla.org/show_bug.cgi?id=245009
    gMultiZilla.prefs.writeChar("general.open_location.last_url", "");
    var localStore = RDF.GetDataSource("rdf:local-store");
    var urlBarHist = mzRDFCUtils.MakeSeq(localStore, RDF.GetResource("nc:urlbar-history"));

    for (var i = urlBarHist.GetCount(); i > 0; --i)
      urlBarHist.RemoveElementAt(i, true);
  },

  clearCache: function(aCacheType)
  {
    if (aCacheType == "DISK")
      this._nsICacheService.evictEntries(Components.interfaces.nsICache.STORE_ON_DISK);
    else if (aCacheType == "MEMORY")
      this._nsICacheService.evictEntries(Components.interfaces.nsICache.STORE_IN_MEMORY);
    else { // "BOTH"
      this._nsICacheService.evictEntries(Components.interfaces.nsICache.STORE_ON_DISK);
      this._nsICacheService.evictEntries(Components.interfaces.nsICache.STORE_IN_MEMORY);
    }
  },

  clearAllCookies: function()
  {
    this._nsICookieManager.removeAll();
  },
  ////////////////////////////////////////////////////////////////////////////////
  // This clears all saved passwords. This is a serious action (dataloss at risk) 
  // so we require the users confirmation, before doing anything. There's also a 
  // pref "wallet.crypto" that can be used to encrypt sensitive data (passwords)
  // But, we don't have to check this pref, because mozilla will take care of it!
  ////////////////////////////////////////////////////////////////////////////////
  // http://lxr.mozilla.org/seamonkey/source/netwerk/base/public/nsIPasswordManager.idl
  clearAllPasswords: function()
  {
    var sDecoderRing = Components.classes["@mozilla.org/security/sdr;1"]
                                 .getService(Components.interfaces.nsISecretDecoderRing);
    sDecoderRing.encryptString("abc"); // trigger master password prompt!

    var container = this._nsIPasswordManager.enumerator;
    var items = [];

    while (container.hasMoreElements()) {
      var item = container.getNext().QueryInterface(Components.interfaces.nsIPassword);
      items.push(item);
    }
    for (var i = 0; i < items.length; ++i)
      this._nsIPasswordManager.removeUser(items[i].host, items[i].user);
  },

  clearAllDownloads: function()
  {
    // Construct the datasource from the filename, because I can't find it in the source!
    var datasource = mzGetNewOrCurrentDSByFilename("downloads.rdf", false, null, false);
    var downloads = mzConstructRDFContainer(datasource, "seq", "NC:DownloadsRoot", false).GetElements(); 
    var downloadQueue = new Array();

    while (downloads.hasMoreElements()) {
      var item = downloads.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      downloadQueue.push(item);
    }
    this._nsIDownloadManager.startBatchUpdate();

    for (var i = 0; i < downloadQueue.length; i++) {
      try {
        this._nsIDownloadManager.removeDownload(downloadQueue[i].Value);
      } catch(ex) {
        // dump("\nmzClearAllDownloads() failed to remove download data!");
      }
    }
    this._nsIDownloadManager.endBatchUpdate();
  },

  clearAllFormData: function()
  {
    var list = this._nsIWalletEditor.GetValue();

    if (list.length > 0) { // no data available or database locked!
      var BREAK = list[0];
      this._nsIWalletEditor.SetValue("OK" + BREAK, window);
    }
  },

  clearAllLocalData: function()
  {
    /***
    * This can be used to clear all passwords and form data!
    * var walletServiceInterface = Components.classes['@mozilla.org/wallet/wallet-service;1']
    *                                        .getService(Components.interfaces.nsIWalletService);
    * walletServiceInterface.WALLET_DeleteAll(); 
    */
    this.clearCache("ALL");
    this.clearUrlBarHistory()
    this.clearGlobalHistory();
    this.clearAllCookies();
    this.clearAllDownloads();
    this.clearAllFormData();
    this.clearAllPasswords();
    gMultiZilla.dnp.purgeIDNHistory();
  },

  clearAfterConfirmation: function(aElement)
  {
    switch(aElement.id || aElement.observes) {
      case "ClearDiskMemoryCache":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearMemDiskTitle', 'qpClearMemDiskText')) 
          gMultiZilla.privacy.clearDiskMemCache();
        break;
      case "ClearAllCache":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearAllTitle', 'qpClearAllText')) 
          gMultiZilla.privacy.clearAllBackTraces();
        break;
      case "ClearAllCookies":
        if (gMultiZilla.quickprefs.getConfirmation('qpRemoveAllCookiesTitle', 'qpRemoveAllCookiesText')) 
          gMultiZilla.privacy.clearAllCookies();
        break;
      case "ClearAllForms":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearAllFormDataTitle', 'qpClearAllFormDataText')) 
          gMultiZilla.privacy.clearAllFormData();
        break;
      case "ClearAllPasswords":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearAllPasswordsTitle', 'qpClearAllPasswordsText'))
          gMultiZilla.privacy.clearAllPasswords();
        break;
      case "ClearIDNHistory":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearIDNHistoryTitle', 'qpClearIDNHistoryText')) 
          gMultiZilla.dnp.purgeIDNHistory();
        break;
      case "ClearAllLocalData":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearAllDataTitle', 'qpClearAllDataText'))
          gMultiZilla.privacy.clearAllLocalData();
        break;
      case "ClearLocationCache":
        if (gMultiZilla.quickprefs.getConfirmation('qpClearAllLocationTitle', 'qpClearAllLocationText'))
          gMultiZilla.privacy.clearUrlBarHistory();
        break;
    }
  }
};


/***
  * mzQuickPrefsMenu is a JavaScript "class" with the helper functions used to  
  * initialize and handle all actions from the Quick Pref Menu's / menu items.
  */

gMultiZilla.quickprefs = /* XXX: Formerly known as: mzQuickPrefsMenu */
{
  _prefs: gMultiZilla.prefs,
  _extensionManager: null,
  _nsIPromptService : Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                .getService(Components.interfaces.nsIPromptService),
  _nsIPopupWindowManager : Components.classes["@mozilla.org/PopupWindowManager;1"]
                                     .getService(Components.interfaces.nsIPopupWindowManager),
  _nsIPermissionManager : Components.classes["@mozilla.org/permissionmanager;1"]
                                    .getService(Components.interfaces.nsIPermissionManager),
  _qpmBundle : gMultiZilla.utility.getStringBundle("quickprefs/quickPrefsMenu.properties"),

  showPopup: function(aPopupNode, aEvent, aContextMenu)
  {
    if ('openPopupAtScreen' in aPopupNode)
      aPopupNode.openPopupAtScreen(aEvent.screenX, aEvent.screenY, aContextMenu);
    else
      aPopupNode.showPopup(aPopupNode, aEvent.clientX, aEvent.clientY, "popup");
  },

  onQPMClick: function(aEvent, aPopupID)
  {
    if (aEvent.button == 0) {
      var i, menuitems;
      var popupNode = document.getElementById("quickpref-popup");

      if (aEvent.altKey) {
        this.openPreferenceEditor();
        return false;
      }
      else if (aEvent.metaKey || aEvent.ctrlKey) {
        var cachePopupNode = document.getElementById("quickpref-clearCachePopup");
        this.showPopup(cachePopupNode, aEvent, false);
        return true;
      }
      else if (aEvent.shiftKey) {
        menuitems = popupNode.getElementsByAttribute("display", "menu");

        for (i = 0 ; i < menuitems.length; i++)
          menuitems[i].setAttribute("hidden", "true");

        this.showPopup(popupNode, aEvent, false);
        return false;
      }
      menuitems = popupNode.getElementsByAttribute("display", "menu");

      for (i = 0 ; i < menuitems.length; i++)
        menuitems[i].removeAttribute("hidden");

      this.showPopup(popupNode, aEvent, false);
      return true;
    }
    else if (aEvent.button == 1) 
      gMultiZilla.managers.openPreferenceManager();
    else if (aEvent.button == 2) {
      var context = aEvent.target.getAttribute("context");

      if (context) {
        var contextNode = document.getElementById(context);
        this.showPopup(contextNode, aEvent, true);
        return true;
      }
      else {
        askToDisableButton(aEvent);
        return true;
      }
    }
    return false;
  },

  isMultiZillaRunning: function ()
  {
    var mzToolbar = document.getElementById("MultiZillaToolbar");

    if (mzToolbar)
      return mzToolbar;
    else 
      return null;
  },
  
  initQPMmenu: function(aPopupMenu)
  {
    var nsIXULAppInfo = Components.classes["@mozilla.org/xre/app-info;1"]
                                  .getService(Components.interfaces.nsIXULAppInfo);
    var versionChecker = Components.classes["@mozilla.org/xpcom/version-comparator;1"]
                                   .getService(Components.interfaces.nsIVersionComparator);

    if (versionChecker.compare(nsIXULAppInfo.platformVersion, "1.8.9") > 0) {
      this._extensionManager = Components.classes["@mozilla.org/extensions/manager;1"]
                                         .getService(Components.interfaces.nsIExtensionManager);
      document.getElementById("extensionManagementOld").setAttribute("hidden", "true");
      document.getElementById("extensionManagementNew").removeAttribute("hidden");
    }
    var qpUAMenuPopup = document.getElementById("qpUAMenuPopup");

    if (!qpUAMenuPopup.ref) {
      qpUAMenuPopup.ref="urn:useragent-data:root";
      var uaStore = mzGetNewOrCurrentDSByFilename("useragents.rdf", false, null, false);

      if (uaStore)
        qpUAMenuPopup.database.AddDataSource(uaStore);
    }
    var i = 0;
    var mzToolbar = this.isMultiZillaRunning();
    var menuItems = aPopupMenu.getElementsByAttribute("display", "hidden-tabbar");
    // dump("\nmenuItems.length: " + menuItems.length);
    // Hide the <menuseparator> and the three extra <menuitem>'s
    for (i = 0; i < menuItems.length; i++) {
      menuItems[i].hidden = true;
    }
    if (mzToolbar) {
      var extraItems = 0;
      var mzToolbarHidden = (mzToolbar.getAttribute("hidden") == "true") ||
                            (mzToolbar.getAttribute("moz-collapsed") == "true");
      var googleBoxOnly = (mzToolbar.mTabButtons.collapsed && mzToolbar.mToolButtons.collapsed && 
                           mzToolbar.mCloseButtons.collapsed);
      // Loop from bottom to top and display the extra menu items when the MultiZilla 
      // toolbar is hidden, or when the toolbar button is hidden. The extra menuseparator 
      // will only be diplayed when one of the extra menu items is displayed.
      for (i = (menuItems.length-1); i > -1; i--) {
        if (mzToolbarHidden || googleBoxOnly || mzToolbar.mTabButtons.childNodes[2].hidden || 
            (menuItems[i].id == "qpClearLocalDataMenu" && document.getElementById("privacymanager-tbbutton").hidden)) {
          menuItems[i].hidden = false;
        }
        if (mzToolbarHidden || googleBoxOnly ||mzToolbar.mTabButtons.childNodes[0].hidden || 
            (menuItems[i].id == "qpTabManager" && document.getElementById("tabmanager-tbbutton").hidden)) {
          menuItems[i].hidden = false;
          extraItems++;
        }
        else if (mzToolbarHidden || googleBoxOnly ||mzToolbar.mTabButtons.childNodes[1].hidden ||
                 (menuItems[i].id == "qpSessionManager" && document.getElementById("sessionmanager-tbbutton").hidden)) {
          menuItems[i].hidden = false;
          extraItems++;
        }
        else if (mzToolbarHidden || googleBoxOnly ||mzToolbar.mTabButtons.childNodes[3].hidden || 
                 (menuItems[i].id == "qpPermissionManager" && document.getElementById("permmanager-tbbutton").hidden)) {
          menuItems[i].hidden = false;
          extraItems++;
        }
        else if (extraItems && menuItems[i].localName == "menuseparator")
          menuItems[i].hidden = false;
      }
      // Show proxy menu pref is set, or when the status bar is hidden!
      var statusbarHidden = document.getElementById("status-bar").hidden;
      var hideProxyMenu = statusbarHidden;
      hideProxyMenu = this._prefs.readBoolean("multizilla.qpm.proxymenu.show-always", true);
      document.getElementById("qpProxyMenu").hidden = !hideProxyMenu;
    } 
  },

  initGroupPref: function(aMenuElement)
  {
    var prefGroup = aMenuElement.getAttribute("prefgroup");
    var prefElements = document.getElementsByAttribute("prefgroup", prefGroup);

    if (prefElements.length == 0)
      return;

    var prefType = prefElements[1].getAttribute("preftype");
    var prefDefval = prefElements[1].getAttribute("prefdefval");
    var prefIdentifier = prefElements[1].getAttribute("prefstring");
    var prefValue = this._prefs.readMyPref(prefIdentifier, prefType, 0, true);

    for (var i = 1; i < prefElements.length; i++) {
      if (prefElements[i].getAttribute("prefvalue") == prefValue) {
        prefElements[i].setAttribute("checked", "true");
        // prefElements[i].setAttribute("disabled", "true");
      }
      else {
        prefElements[i].removeAttribute("checked");
        // prefElements[i].removeAttribute("disabled");
      }
    }
  },

  initQuickPrefs: function()
  {
    // dump("\ninitQuickPrefs._prefs: " + this._prefs);
    var prefElements = document.getElementsByAttribute("pref", "true");

    for (var i = 0; i < prefElements.length; i++) {
      var prefId = prefElements[i].id;
      var prefType = prefElements[i].getAttribute("preftype");
      var prefIdentifier = prefElements[i].getAttribute("prefstring");
      var prefDefval = prefElements[i].getAttribute("prefdefval");
      var prefAttribute = prefElements[i].getAttribute("prefattribute");
      var prefReversed = prefElements[i].hasAttribute("prefreversed");

      var prefValue = this._prefs.readMyPref(prefIdentifier, prefType, prefDefval);

      if (prefReversed)
        prefValue = !prefValue;

      if (prefAttribute == "checked") {
        if (prefValue >= 1)
          prefElements[i].setAttribute(prefAttribute, "true");
        else 
          prefElements[i].removeAttribute(prefAttribute);
      }
      else 
        prefElements[i].setAttribute(prefAttribute, prefValue);
    }
  },

  initSelectedQPrefsMenu: function(aElement)
  {
    var permission, url;
    var IDs = new Array();
    var baseIDs = [ "qpmBlock-", "qpmAllow-", "qpmDefault-" ];
    var menuPopupID = aElement.getAttribute("prefgroup");

    if (menuPopupID != "install")
      baseIDs[3] = "qpmSeparatorFor-";

    for (i in baseIDs)
      IDs[i] = baseIDs[i] + menuPopupID;

    if (menuPopupID == "cookies") {
      var managePrivacySettings = document.getElementById("qpm_ManageCookiesPrivacySettings");

      if (this._prefs.readInteger("network.cookie.cookieBehavior", 3) == 3)
        managePrivacySettings.removeAttribute("hidden");
      else 
        managePrivacySettings.setAttribute("hidden", "true");
    }
    var uri = getBrowser().currentURI;
 
    if (gBrowser.selectedTab.hasAttribute("busy"))
      url = gBrowser.mCurrentTab.getAttribute("firstPage");
    else 
      url = gBrowser.mCurrentBrowser.webNavigation.currentURI.spec;

    permission = this._nsIPermissionManager.testPermission(uri, menuPopupID.replace(/s$/, ''));
    // selected menu items should be disabled and display a radio dot character!  
    this.enableElement(IDs[0], url, (permission == nsIPermissionManager.DENY_ACTION),
                       permission != nsIPermissionManager.DENY_ACTION);
    this.enableElement(IDs[1], url, (permission == nsIPermissionManager.ALLOW_ACTION),
                       permission != nsIPermissionManager.ALLOW_ACTION)
    if (IDs.length >= 2 && IDs[2])
      this.enableElement(IDs[2], url, (permission == nsIPermissionManager.UNKNOWN_ACTION), 
                         permission != nsIPermissionManager.UNKNOWN_ACTION);
    if (IDs.length >= 3 && IDs[3])
      this.enableElement(IDs[3], url, false, true);
  },

  enableElement: function(aElementID, aURL, aIsCurrentFlag, aIsEnabledFlag)
  {
    var element = document.getElementById(aElementID);

    if (element) {
      // Do we need to display a radio dot for this element?
      if (aIsCurrentFlag)
        element.setAttribute("currentPermission", "true"); // Yes we do!
      else
        element.removeAttribute("currentPermission"); // No, remove attribute

      // if (aIsEnabledFlag && aURL != "about:blank")
      if (aURL == "about:blank")
        element.setAttribute("disabled", "true");
      else
        element.removeAttribute("disabled");
    }
    else
      throw("Unknown ID (" + aElementID + ") used in function initSelectedQPrefsMenu()");
  },

  permissionUpdate: function(aElement, aURL)
  {
    var action = aElement.getAttribute("for");
    var permission = aElement.getAttribute("permission");

    if (permission == "none")
      permission = nsIPermissionManager.UNKNOWN_ACTION;
    else 
      permission = (aElement.getAttribute("permission") == "true") ? nsIPermissionManager.ALLOW_ACTION
                                                                   : nsIPermissionManager.DENY_ACTION;

    var url = (aURL == undefined) ? gBrowser.currentURI.spec : aURL;
    var isFileScheme = url.match(/(scheme:file|file:)/i);
    url = url.replace(/^\s*([-\w]*:\/+)?/, ""); // trim any leading space and scheme
    var host = isFileScheme ? 'file:' : 'http://' + url;
    const nsIIOService = Components.classes["@mozilla.org/network/io-service;1"]
                                   .getService(Components.interfaces.nsIIOService);
    var uri = nsIIOService.newURI(host, null, null);
    this._nsIPermissionManager.add(uri, action, permission);
    var message = aElement.getAttribute("msg");

    if (message)
      setTimeout(this.doAlert, 0, window, "", message);
  },

  changePref: function(aMenuElement, aMenuGroup, aRefreshFlag)
  {
    var prefType = aMenuElement.getAttribute("preftype");
    var prefIdentifier = aMenuElement.getAttribute("prefstring");
    var prefDefval = aMenuElement.getAttribute("prefdefval");
    var prefAttribute = null;
    var prefArray = new Array();
    var noAccess = "noAccess";
    var allAccess = "allAccess";
	
    if (prefIdentifier != null) {
      // retrieve current value
      var prefValue = this._prefs.readMyPref(prefIdentifier, prefType, prefDefval);

      if (prefType == "bool") // flip flop the pref
        this._prefs.writePref(prefType, prefIdentifier, !prefValue);
      else {
        if (aMenuGroup) // special case, use this value
          prefValue = aMenuElement.getAttribute("prefvalue");
        else { // used for int and string, other might follow
          prefAttribute = aMenuElement.getAttribute("prefattribute");

          if (prefAttribute == "checked") {
            // setup an array with values to use
            var prefValues = eval(aMenuElement.getAttribute("prefvalue"));
            prefArray = prefValues;
            // pick the first or second value, based on the "checked" state
            prefValue = aMenuElement.getAttribute(prefAttribute) ? prefArray[1] : prefArray[0];
          }
          else
            prefValue = aMenuElement.getAttribute(prefAttribute);
        }
        // Ok, lets write this sucker
        this._prefs.writePref(prefType, prefIdentifier, prefValue);
      }
      if (aRefreshFlag) // this pref setting needs a refresh
        getWebNavigation().reload(nsIWebNavigation.LOAD_FLAGS_NONE); // LOAD_FLAGS_IS_REFRESH);
    }
  },

  initUAMenuPopup: function(aMenuPopup)
  {
    aMenuPopup.builder.rebuild();

    var customUA = false;
    var restoreOriginalUA = document.getElementById("qpm-restoreOriginalUA");
    var userAgents = aMenuPopup.childNodes.length; 

    for (var i = 4; i < userAgents; i++) {
      var menuitem = aMenuPopup.childNodes[i];
      var menuitemUserAgent = menuitem.getAttribute("useragent");

      if (menuitem.label == "")
        menuitem.label = menuitemUserAgent;
      if (!customUA && (menuitemUserAgent == navigator.userAgent)) {
        menuitem.setAttribute("checked", "true");
        restoreOriginalUA.removeAttribute("disabled");
        restoreOriginalUA.removeAttribute("checked");
        customUA = true;
      }
    }
    if (!customUA) {
      restoreOriginalUA.setAttribute("disabled", "true");
      restoreOriginalUA.setAttribute("checked", "true");
    }
  },

  setUserAgentOverride: function(aEvent)
  {
    const PREFSTRING = "multizilla.general.useragent.override";
    var menuitem = aEvent.originalTarget;
    var menuitemID = menuitem.getAttribute("id");

    if (menuitemID == "qpm-restoreOriginalUA")
      this._prefs.writeChar(PREFSTRING, "");
    else {
      var useragent = menuitem.getAttribute("useragent");

      if (useragent)
        this._prefs.writeChar(PREFSTRING, useragent);
    }
  },

  initRefSpoofingMenu: function(aElement)
  {
    var refSetting = -1;
    var spoofURL = "http://ref.spoof.org/refspoof.cgi"; // default fake referrer
    var aWindow = gBrowser.contentWindow;
    var aDocument = aWindow.document;

    try { 
      refSetting = pref.getIntPref("network.http.sendRefererHeader");
    } catch (ex) {
    }
    for (var i = 0; i < aElement.childNodes.length; i++) {
      aElement.childNodes[i].setAttribute("disabled", (refSetting == 0) ? "true" : "false");
      aElement.childNodes[i].setAttribute("checked", "false");
    }
    try {
      spoofURL = this._prefs.readString("multizilla.refspoof.spoofurl", spoofURL);
    } catch (ex) {
    }
    aElement.childNodes[6].setAttribute("label", this._qpmBundle.GetStringFromName("fake.label") + " " + spoofURL);
    aElement.childNodes[7].setAttribute("label", aDocument.referrer ?  
                                                 this._qpmBundle.GetStringFromName("currentRef.label") + " " + aDocument.referrer : 
                                                 this._qpmBundle.GetStringFromName("noRef.label"));

    if (refSetting)
      this.initGroupPref(aElement);
  },

  displayMessageByNumber: function(aMessageNumber)
  {
    function delayedStartup() {
      var message = null;

      switch(aMessageNumber)
      {
        case 0: message = gMultiZilla.quickprefs._qpmBundle.GetStringFromName("refSpooferDisabled");
          break;
        case 1: message = gMultiZilla.quickprefs._qpmBundle.GetStringFromName("refSpooferEnabled");
          break;
        case 2: message = gMultiZilla.quickprefs._qpmBundle.GetStringFromName("refSpooferNone");
          break;
        case 3: message = gMultiZilla.quickprefs._qpmBundle.GetStringFromName("illegalFakeReferrer");
          break;
        case 4:
        case 5: var spoofer = this._prefs.readInteger("multizilla.refspoof.spoofing", 1);
                message = (spoofer == 1) ? "refSpooferDisabled" : "refSpooferEnabled";
                gMultiZilla.quickprefs._qpmBundle.GetStringFromName(message);
          break;
      }
      if (message) {
        var dialogTitle = gMultiZilla.quickprefs._qpmBundle.GetStringFromName("fakeReferrerDialogTitle");
        gMultiZilla.quickprefs.doAlert(window, dialogTitle, message);
      }
    }
    setTimeout(delayedStartup, 0);
  },

  manageFakeReferrer: function()
  {
    var bundle = gMultiZilla.quickprefs._qpmBundle;
    var dialogTitle = bundle.GetStringFromName("fakeReferrerDialogTitle");
    var dialogText = bundle.GetStringFromName("fakeReferrerDialogText");
    var fakeRefIsText = bundle.GetStringFromName("fakeRefIsSetTo");
    var fakeReferrer = gMultiZilla.prefs.readMyPref("multizilla.refspoof.spoofurl",
                                                    "url", "http://ref.spoof.org/refspoof.cgi");
    var result = {value:fakeReferrer};

    var ret = gMultiZilla.quickprefs._nsIPromptService.prompt(window, dialogTitle, dialogText, result, null, {value:0});

    if (ret && result.value && result.value != "") {
      gMultiZilla.quickprefs.doAlert(window, dialogTitle, fakeRefIsText+result.value);
      gMultiZilla.prefs.writeString("multizilla.refspoof.spoofurl", result.value);
    }
    else
      gMultiZilla.quickprefs.displayMessageByNumber(3);
  },

  initExtensionsMenu: function(aMenuPopup)
  {
    var enabled = 0;
    var disabled = 0;
    var installed = 0;
    var nameAttribute = "packageName";

    // menuitem, menuitem, menuseparator, menuitem, template
    for (var i = (aMenuPopup.childNodes.length - 1); i > 4; i--) {
      if (aMenuPopup.childNodes[i].hasAttribute("injected"))
        aMenuPopup.removeChild(aMenuPopup.childNodes[i]);
    }
    if (this._extensionManager) { // SeaMonkey v2.x and Mozilla Firefox
      this._extensionManager.sortTypeByProperty(Components.interfaces.nsIUpdateItem.TYPE_ADDON, "name", true);
      var ds = this._extensionManager.datasource;
      // Remove datasource first, or the disable state won't show up!
      aMenuPopup.database.RemoveDataSource(ds);
      aMenuPopup.database.AddDataSource(ds);
      nameAttribute = "name";
    }
    aMenuPopup.builder.rebuild();

    for (var i = 5; i < aMenuPopup.childNodes.length; i++) {
      installed++;
      var packageName = aMenuPopup.childNodes[i].getAttribute(nameAttribute);
      var isDisabled = (aMenuPopup.childNodes[i].getAttribute("isDisabled") == "true");

      if (isDisabled) {
        disabled++;
      }
      else {
        if (packageName != "multiviews")
          enabled++;
        // SeaMonkey v2.0 needs this
        if (!aMenuPopup.childNodes[i].hasAttribute("checked") ||
            aMenuPopup.childNodes[i].getAttribute("checked") != "true")
          aMenuPopup.childNodes[i].setAttribute("checked", "true");
      }
    }
    var disableState = (installed == 1 || enabled == 0 || disabled == installed) ? "true" : "false";
    aMenuPopup.childNodes[0].setAttribute("disabled", disableState); // 'Disable all Extensions'
    disableState = (installed == 1 || disabled == 0 || enabled == installed) ? "true" : "false";
    aMenuPopup.childNodes[1].setAttribute("disabled", disableState); // 'Enable all Extensions'

    var menuseparator = document.createElementNS("http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul", "menuseparator");
    menuseparator.setAttribute("injected", "true");
    aMenuPopup.appendChild(menuseparator);

    var bundle = gMultiZilla.quickprefs._qpmBundle;
    var menuitem = document.createElementNS("http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul", "menuitem");
    menuitem.setAttribute("injected", "true");
    menuitem.setAttribute("label", bundle.GetStringFromName("qpPermissionManager.label"));
    menuitem.setAttribute("accesskey", bundle.GetStringFromName("qpPermissionManager.accesskey"));
    menuitem.setAttribute("acceltext", bundle.GetStringFromName("qpPermissionManager.acceltext"));
    menuitem.setAttribute("oncommand", "event.stopPropagation();gMultiZilla.managers.openPermissionManager('','ExtensionsPrefTab','ExtensionsPermissionsTab');");
    aMenuPopup.appendChild(menuitem);
  },

  /***
    * Helper function for: toggleExtensionEnabledState() and setAllExtensionsTo()
    */
  setExtensionStateTo: function(aPackageID, aTargetState)
  {
    dump("\nsetExtensionStateTo: " +  aTargetState);
    // SeaMonkey v2.x and Mozilla Firefox
    if (this._extensionManager) {
      if (aTargetState)
        this._extensionManager.disableItem(aPackageID);
      else
        this._extensionManager.enableItem(aPackageID);
    }
    else { // Backward compatibility with SeaMonkey v1.x
      var registry = null;

      if ('nsIChromeRegistrySea' in Components.interfaces)
        registry = Components.interfaces.nsIChromeRegistrySea;
      else if ('nsIXULChromeRegistry' in Components.interfaces)
        registry = Components.interfaces.nsIXULChromeRegistry;

      if (registry) {
        var chromeRegistry = Components.classes["@mozilla.org/chrome/chrome-registry;1"]
                                       .getService(registry);
        chromeRegistry.setAllowOverlaysForPackage(aPackageID, !aTargetState);
      }
    }
    if (aPackageID == "multiviews") {
      var prefStrings = ["xpinstall.dialog.confirm",
                         "general.useragent.vendor",
                         "general.useragent.vendorSub"];
      // Are we about to reactivate MultiZilla?
      if (aTargetState) {
        this._prefs.writeString(prefStrings[0], "chrome://multiviews/content/xpinstall/xpinstallConfirm.xul");
        gMultiZilla.prefs.checkVendorPrefs();
      }
      else { // Deactivate MultiZilla
        // Clear MultiZilla's user settings!
        for (i in prefStrings) {
          if (pref.prefHasUserValue(prefStrings[i]))
            pref.clearUserPref(prefStrings[i]);
        }
      }
    }
    else if (aPackageID == "googlebox")
      this._prefs.writeBoolean("multizilla.integrated.toolbar", false);
  },

  toggleExtensionEnabledState: function(aEvent, aCompatibilityFlag)
  {
    var item, menuitem = aEvent.originalTarget;
    var isDisabled = (menuitem.getAttribute("isDisabled") == "true");

    if (aCompatibilityFlag) // SeaMonkey v1.x
      item = menuitem.getAttribute("packageName");
    else {
      item = {};
      item.iconURL = menuitem.getAttribute("iconURL");
      // dump("\nadding addon.iconURL: " + item.iconURL);
      item.name = menuitem.getAttribute("name");
      // dump("\nadding addon.name: " + item.name);
      item.version = menuitem.getAttribute("version");
      // dump("\nadding addon.version: " + item.version);
      item.description = menuitem.getAttribute("description");
      // dump("\nadding addon.description: " + item.description);
      item.addonID = menuitem.getAttribute("addonID");
      // dump("\nadding addon.addonID: " + item.addonID);
      item.disabled = isDisabled;
    }
    dump("\nisDisabled: " + isDisabled);
    gMultiZilla.quickprefs.openExtensionManagment([item], !isDisabled);
  },

  setAllExtensionsTo: function(aMenupopup, aNewState, aCompatibilityFlag)
  {
    dump("\nsetAllExtensionsTo: " + aNewState);
    // aCompatibilityFlag is 'true' for SeaMonkey v1.x only!
    var packages = new Array();

    const DISABLE_STATE = "isDisabled";
    const NAME_ATTRIBUTE = (aCompatibilityFlag) ? "packageName" : "name";

    for (var i = 5; i < aMenupopup.childNodes.length; i++) {
      var menuitem = aMenupopup.childNodes[i];

      if (menuitem.hasAttribute(DISABLE_STATE)) {
        var packageName = menuitem.getAttribute(NAME_ATTRIBUTE);
        var packageID = aCompatibilityFlag ? packageName
                                           : menuitem.getAttribute("addonID");
        dump("\npackageID: " + packageID);
        if (aNewState == true && (packageName == "multiviews" ||
                                  packageID == "multizilla@multizilla.mozdev.org")) {
            continue;
        }
        else {
          var currentState = (menuitem.getAttribute(DISABLE_STATE) == "true");
          dump("\ncurrentState: " + currentState);
          if (currentState != aNewState) {
            if (aCompatibilityFlag) {
              packageName = packageName.toLowerCase();
              packages.push(packageName);
            }
            else {
              // dump("\nSeaMonkey v2.0 " + menuitem.label);
              var item = {};
              item.iconURL = menuitem.getAttribute("iconURL");
              // dump("\nadding addon.iconURL: " + item.iconURL);
              item.name = menuitem.getAttribute("name");
              // dump("\nadding addon.name: " + menuitem.getAttribute("name"));
              item.version = menuitem.getAttribute("version");
              // dump("\nadding addon.version: " + menuitem.getAttribute("version"));
              item.description = menuitem.getAttribute("description");
              // dump("\nadding addon.description: " + menuitem.getAttribute("description"));
              item.addonID = packageID;
              // dump("\nadding addon: " + menuitem.getAttribute("addonID"));
              item.disabled = (menuitem.getAttribute(DISABLE_STATE) == "true");
              packages.push(item);
            }
          }
        }
      }
    }
    if (packages.length > 0)
      gMultiZilla.quickprefs.openExtensionManagment(packages, aNewState);
  },

  disableAllAddons: function(aMenuPopupNode)
  {
    gMultiZilla.quickprefs.setAllExtensionsTo(aMenuPopupNode, true, false);
  },
  
  enableAllAddons: function(aMenuPopupNode)
  {
    gMultiZilla.quickprefs.setAllExtensionsTo(aMenuPopupNode, false, false);
  },

  // Helper function for: toggleExtensionEnabledState() and setAllExtensionsTo()
  openExtensionManagment: function(packages, aNewState)
  {
    var dialogWin = gMultiZilla.utility._getMostRecentWindow("multizilla:extension-management", false);

    if (dialogWin) {
      dialogWin.focus();
      return;
    }
    var winFeatures = "chrome, dialog, centerscreen, resizable, dependent, modal";
    window.openDialog("chrome://multiviews/content/quickprefs/miscellaneous/extensionManagement.xul",
                      "", winFeatures, packages, aNewState,
                      gMultiZilla.utility.getPlatformExtras());
  },

  relaunchBrowserWindows: function(aCleanRestartFlag)
  {
    const _nsIAppStartup = Components.interfaces.nsIAppStartup;  
    // SeaMonkey v2.0 and Mozilla Firefox specific code!
    if ('eRestart' in _nsIAppStartup) {
      var os = Components.classes["@mozilla.org/observer-service;1"]
                         .getService(Components.interfaces.nsIObserverService);
      var cancelQuit = Components.classes["@mozilla.org/supports-PRBool;1"]
                                 .createInstance(Components.interfaces.nsISupportsPRBool);
      os.notifyObservers(cancelQuit, "quit-application-requested", "restart");
  
      if (cancelQuit.data)
        return;
      
      Components.classes["@mozilla.org/toolkit/app-startup;1"].getService(_nsIAppStartup)
                .quit(_nsIAppStartup.eRestart | _nsIAppStartup.eAttemptQuit);
      return;
    }
    var windowManagerDS = Components.classes['@mozilla.org/rdf/datasource;1?name=window-mediator']
                                    .getService(nsIWindowDataSource);
    var currentBrowserWindows = mzGetRealWindowResourceIDList();
    var targetWindow = null;

    if (aCleanRestartFlag) {
      window.open();

      for (i in currentBrowserWindows) {
        targetWindow = windowManagerDS.getWindowForResource(currentBrowserWindows[i]);
        targetWindow.close();
      }
    }
    else {
      var sourceWindow = null;

      for (i in currentBrowserWindows) {
        var targetTabbrowser = openWindowAndReturnBrowser();
        targetWindow = targetTabbrowser.ownerDocument.defaultView;
        sourceWindow = windowManagerDS.getWindowForResource(currentBrowserWindows[i]);
        var sourceTabbrowser = sourceWindow.getBrowser();

        if (!sourceWindow.gMultiZilla.tabs.areAllTabsBlank()) {
          var tabs = sourceTabbrowser.mTabs;
          var selectedIndex, selectedTab = sourceTabbrowser.selectedTab;

          for (var t = (tabs.length-1); t >= 0; t--) {
            var openBrowserWindows = mzGetRealWindowResourceIDList();
            var windowID = openBrowserWindows[openBrowserWindows.length-1]

            if (tabs[t] == selectedTab)
              selectedIndex = t;

            sourceWindow.gMultiZilla.tabs.moveTabToWindow(tabs[t], windowID, true, true);
          }
          tabs = targetTabbrowser.mTabs;

          for (t = 0; t < tabs.length; t++) {
            if (tabs[t].getAttribute("firstPage") == 'about:blank') {
              targetTabbrowser.removeTab(tabs[t]);
            }
          }
          var removedTabs = sourceTabbrowser.mRemovedTabs;

          if (removedTabs.length > 0) {
            for (rIndex in removedTabs) {
              var removedTabData = removedTabs[rIndex];
              var removedTab = new Object();
              removedTab.firstPage          = removedTabData.firstPage;
              removedTab.customLabel        = removedTabData.customLabel;
              removedTab.reloadInterval     = removedTabData.reloadInterval;
              removedTab.allowPlugins       = removedTabData.allowPlugins;
              removedTab.allowJavascript    = removedTabData.allowJavascript;
              removedTab.allowMetaRedirects = removedTabData.allowMetaRedirects;
              removedTab.allowSubframes     = removedTabData.allowSubframes;
              removedTab.allowImages        = removedTabData.allowImages;
              removedTab.history            = removedTabData.history;
              targetTabbrowser.mRemovedTabs[rIndex] = removedTab;
            }
          }
          setTimeout(function selectTab(aTabbrowser, aIndex){aTabbrowser.selectedTab = aTabbrowser.mTabs[aIndex];}, 0, targetTabbrowser, selectedIndex);
        }
        sourceWindow.close();
      }
    }
  },

  openPreferenceEditor: function()
  {
    gBrowser.addOrGetTab("chrome://global/content/config.xul", undefined, undefined, true, true); 
    gBrowser.userTypedValue = ""; 
    SetPageProxyState("invalid", null);
  },

  doAlert: function(aWindow, aTitle, aText)
  {
    gMultiZilla.quickprefs._nsIPromptService.alert(aWindow, aTitle, aText);
  },

  getConfirmation: function(aConfirmationTitle, aConfirmationText, aWindow) // mzConfirmAction
  {
    if (aWindow == undefined)
      aWindow = window;
    // dump("\naConfirmationTitle: " + aConfirmationTitle + "\naConfirmationText: " + aConfirmationText);
    var dialogTitle, dialogText;

    try {
      dialogTitle = this._qpmBundle.GetStringFromName(aConfirmationTitle);
    } catch(ex) {
      dialogTitle = aConfirmationTitle;
    }
    try {
      dialogText = this._qpmBundle.GetStringFromName(aConfirmationText);
    } catch(ex) {
      dialogText = aConfirmationText;
    }
    return this._nsIPromptService.confirm(aWindow, dialogTitle, dialogText);
  },

  initQPMTooltip: function(aTooltipNode)
  {
    // Modify the tooltip for Mac users
    if (/Mac/.test(navigator.platform)) {
      aTooltipNode.firstChild.value = this._qpmBundle.GetStringFromName("qpmCtrlClickTooltipMac");
      aTooltipNode.childNodes[2].value = this._qpmBundle.GetStringFromName("qpmAltClickTooltipMac");
      aTooltipNode.lastChild.value = this._qpmBundle.GetStringFromName("qpmRightClickTooltipMac");
    }
  }
};


/***
  * .googlebox: JavaScript functions to initialize and control various features
  *             of the MultiZilla specific GoogleBox.
  */

gMultiZilla.googlebox =
{
  _prefs: gMultiZilla.prefs,
  _chrome: gMultiZilla.chrome,
  
  initMenuitem: function(aMenuitem)
  {
    var enabled = this._prefs.readBoolean("multizilla.integrated.toolbar", false);
  
    if (!enabled || gMultiZilla._multiZillaToolbar.mPlaceHolder.getAttribute("hidden") == "true" || 
        document.getElementById("MultiZillaToolbar").getAttribute("hidden") == "true")
      document.getElementById(aMenuitem).removeAttribute("checked");
    else
      document.getElementById(aMenuitem).setAttribute("checked", "true");
  },

  /* XXX: Formerly known as: aboutGoogleBox() */
  about: function()
  {
    if (this.isInstalled())
      getBrowser().addOrGetTab("about:multizilla");
  },

  /* XXX: Formerly known as: isGoogleBoxInstalled() */
  isInstalled: function(aStartupFlag)
  {
    // We should also check the menu items in QPrefs Extension Manager!!!
    if (!gMultiZilla._multiZillaToolbar.mToolboxActivated) {
      if (!aStartupFlag)
        gBrowser.addOrGetTab("http://multizilla.mozdev.org/features/googlebox.html");
      return false;
    }
    return true;
  },

  /* XXX: Formerly known as: toggleOrEnableGoogleBox() */
  toggleOrEnable: function(aStartupFlag)
  {
    if (!gMultiZilla.googlebox.isInstalled(aStartupFlag))
      return;
  
    var checkboxText = null;
    var prefString = "multizilla.integrated.toolbar";
    var enabled = this._prefs.readBoolean(prefString, false);
    var notify = this._prefs.readBoolean(prefString + "-notify", true);
  
    if (!notify && aStartupFlag)
      return;
  
    if (!enabled) {
      var promptService = Components.classes['@mozilla.org/embedcomp/prompt-service;1']
                                    .getService(Components.interfaces.nsIPromptService);
  
      var title = gMultiZilla.utility.getTextFromBundle("mzWizard");
      var text = gMultiZilla.utility.getTextFromBundle("enableGoogleBoxText");
      var checkbox = {value:0};
  
      if (aStartupFlag)
        checkboxText = gMultiZilla.utility.getTextFromBundle("rememberDecisionText");
  
      var result = promptService.confirmEx(window, title, text, 
                                          (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) +
                                          (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_NO),
                                           null, null, null, checkboxText, checkbox);
  
      if (checkbox.value) {
        this._prefs.writeBoolean(prefString + "-notify", false);
        return;
      }
      if (result == 1)
        return;
      this._prefs.writeBoolean(prefString, true);
      return;
    }
    this.refreshState(true, false);
    gMultiZilla._multiZillaToolbar.refreshMultiZillaButtons();
  },

  /* XXX: Formerly known as: updateGoogleBoxState() */
  refreshState: function(aState, aSkipFlag)
  {
    if (aState && document.getElementById("MultiZillaToolbar").getAttribute("hidden") == "true") {
      if (document.getElementById("internalToolBox").getAttribute("hidden") != "true") {
        this._chrome.goToggleToolbar("MultiZillaToolbar", "cmd_viewMultiZillaToolbar");
        return;
      }
      this._chrome.goToggleToolbar("MultiZillaToolbar", "cmd_viewMultiZillaToolbar");
    }
    if (!aSkipFlag)
      this._chrome.goToggleToolbar("internalToolBox", "cmd_viewIntegratedToolBox");
  }
};

gMultiZilla.dnp =
{
  _db: null,

  /* Formerly known as: mzCheckForUnicodeHost() 
  checkForUnicodeHost; function(aBrowser, aURL, aForcedDisplayFlag)
  {
    return false;
    if (gMultiZilla.prefs.readBoolean("network.enableIDN", false) == false)
      return false;
    else if (!aForcedDisplayFlag && gMultiZilla.prefs.readBoolean("multizilla.check.idn", true) == false)
      return;
  
    var nsIIOService = Components.classes["@mozilla.org/network/io-service;1"]
                                 .getService(Components.interfaces.nsIIOService);
    var uri;
    var scheme = "";
  
    try {
      uri = nsIIOService.newURI(aURL, null, null);
      scheme = uri.scheme;  
    } catch(ex) {
      // dump("\nerror getting scheme");
    }
    if (scheme && scheme.match(/^http/i) == null) {
      // dump("\nwrong scheme ["+scheme+"], bail out!");
      return false;
    }
    var start = (scheme == "") ? 0 : scheme.length + 3;
    var url = aURL.replace(/^\s*([-\w]*:\/+)?/, "");
    var end = url.indexOf("/");
  
    if (end < start)
      end = aURL.length;
  
    var host = aURL.substring(start, (start + end));
    var encodedURL = encodeURIComponent(host);
    var rawhost = encodedURL.replace(/\%3A/g, ":").replace(/\%2F/g, "/").replace(/\%2E/g, ".");
    // dump("\nhost: " + host + "\nrawhost: " + rawhost);
  
    if (rawhost != host)
      return this.displayIDNWarning(aBrowser, aURL, rawhost, host);
    return false;
  }, */
  
  /* Formerly known as: mzDisplaySSLPrompt() */
  displayPrompt: function(aBrowser, aHostArray, aIsNewHostFlag, aIsIDNFlag, aCertificate)
  {
    aBrowser = (aBrowser == undefined) ? gBrowser.mCurrentBrowser : aBrowser;
  
    var bundle = gMultiZilla.utility.getStringBundle();
    var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                  .getService(Components.interfaces.nsIPromptService);
    var titleID = aIsNewHostFlag ? "newSSLTitle" : "otherSSLTitle";
    var textID = aIsNewHostFlag ? "newSSLText" : "otherSSLText";
    var host = aHostArray[0];
  
    if (aIsIDNFlag) {
      textID += "IDN";
      titleID += "IDN";
      aHostArray = [host];
    }
    var dialogTitle = bundle.GetStringFromName(titleID);
    var notPartOfCertificate = bundle.GetStringFromName("notPartOfCertificate");
    aHostArray.push(aCertificate.commonName ? aCertificate.commonName
                                            : notPartOfCertificate);
    aHostArray.push(aCertificate.organization ? aCertificate.organization
                                              : notPartOfCertificate);
    aHostArray.push(aCertificate.issuerCommonName ? aCertificate.issuerCommonName
                                                  : notPartOfCertificate);
    aHostArray.push(aCertificate.issuerOrganization ? aCertificate.issuerOrganization
                                                    : notPartOfCertificate);
    /* dump("\ncommonName: " + aCertificate.commonName);
    dump("\norganization: " + aCertificate.organization);
    dump("\nissuerCommonName: " + aCertificate.issuerCommonName);
    dump("\nissuerOrganization: " + aCertificate.issuerOrganization); */
    var dialogText = bundle.formatStringFromName(textID, aHostArray, aHostArray.length);
    var checkbox = {value:0};
    var checkboxText = null;
    // checkboxText = bundle.formatStringFromName(textID, aHostArray, aHostArray.length);
    var updateIDNHistory = false;
    var retValue = promptService.confirmEx(window, dialogTitle, dialogText, 
                                           (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) + 
                                           (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_NO),
                                           null, null, null, checkboxText, checkbox);
    if (!retValue) {
      if (aIsNewHostFlag)
        updateIDNHistory = true;
    }
    if (updateIDNHistory) {
      this.doIDNHistory(host, true, true);
    }
    aBrowser.lastSSLHost = host;
  },
  
  /* Formerly known as: mzDisplayIDNWarning() */
  displayIDNWarning: function(aBrowser, aURL, aRawhost, aHost)
  {
    var isNewHost = this.doIDNHistory(aRawhost, false, false);
  
    if (isNewHost == -1)
      return true;
    else if (isNewHost) {
      var aEvent = window.document.createEvent('Events');
      aEvent.initEvent("DOMWillOpenModalDialog", false, false);
      aBrowser = (aBrowser == undefined) ? gBrowser.mCurrentBrowser : aBrowser;
      aBrowser.contentDocument.defaultView.dispatchEvent(aEvent);
  
      var bundle = gMultiZilla.utility.getStringBundle();
      var confirmationTitle = bundle.GetStringFromName("mzWizard");
      var confirmationText = bundle.formatStringFromName("idnHost", [aRawhost, aHost], 2);
      var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                    .getService(Components.interfaces.nsIPromptService);
      var checkbox = {value:0};
  
      if (promptService.confirmEx(window, confirmationTitle, confirmationText, 
                                  (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) + 
                                  (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_NO),
                                  null, null, null, null, checkbox)) {
  
        var tab = gBrowser.getTabForBrowser(aBrowser);
        var firstPage = tab.getAttribute("firstPage");
  
        if (firstPage == aURL)
          tab.setAttribute("removeAfterLoad", "true");
        return true;
      }
      aEvent = window.document.createEvent('Events');
      aEvent.initEvent("DOMModalDialogClosed", false, false);
      aBrowser.contentDocument.defaultView.dispatchEvent(aEvent);
      this.doIDNHistory(aRawhost, true, true);
    }
    return false;
  },

  /* Formerly known as: mzIDNHistory() */
  doIDNHistory: function(aHost, aUpdateFlag, aSkipCheckFlag, aPurgeFlag)
  {
    if (!this._db)
      this._db = mzGetNewOrCurrentDSByFilename("idn-history.rdf", false, null, false);
  
    var entries = mzRDFCUtils.MakeSeq(this._db, RDF.GetResource("urn:idn-hash-keys"));
  
    if (!entries)
      return -1;
    if (!entries.GetElements())
      return -1;
  
    var keys = entries.GetCount();
  
    if (aHost == null && aUpdateFlag == undefined && aSkipCheckFlag == undefined
        && aPurgeFlag == undefined)
      return keys;
  
    var index = 0;
    var newHost = true;
    var retValue = false;
    var sDecoderRing = Components.classes["@mozilla.org/security/sdr;1"]
                                 .getService(Components.interfaces.nsISecretDecoderRing);
  
    if (!aHost && aPurgeFlag) { // do we need to purge IDN history?
      // dump("\nTrigger Master Password Prompt");
      var dummy = sDecoderRing.encryptString("abc"); // force master password prompt to be displayed!
      // dump("\nPurge IDN History ");
      index = keys;
      // dump(index + " entries to be removed");
      for (index; index > 0; index--) {
        entries.RemoveElementAt(index, true);
        // dump("\nremoving entry: " + index);
      }
      newHost = false;
    }
    else if (!aSkipCheckFlag) { // do we have to skip the check?
      dump("\nLookup: " + aHost);
      var idnHistory = entries.GetElements();
  
      while (idnHistory.hasMoreElements()) {
        var entry = idnHistory.getNext();
        // dump("\nindex: " + index);
        if (entry) {
          try {
            entry = entry.QueryInterface(Components.interfaces.nsIRDFLiteral);
            index++;
          } catch(ex) {
            continue;
          }
          var host = sDecoderRing.decryptString(entry.Value);
          dump("\nhost("+index+"): " + host);
          if (host == aHost) {
            if (aPurgeFlag) { // do we need to purge a specified host entry?
              entries.RemoveElementAt(index, true);
              // dump("\naPurge host from IDN history: " + aPurgeFlag);
            }
            newHost = false;
            break;
          }
        }
      }
    }
    if (newHost) { // is this a new host?
      // dump("\nnewHost: " + newHost);
      if (aUpdateFlag) { // do we need to add the new host?
        // dump("\naUpdateFlag: " + aUpdateFlag);
        var hash = sDecoderRing.encryptString(aHost);
        entries.InsertElementAt(RDF.GetLiteral(hash), 1, true);
      }
      retValue = true;
    }
    this._db.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
    return retValue;
  },
  
  /* Formerly known as: mzAddHostToIDNHistory() */
  addHostToIDNHistory: function(aHost)
  {
    if (!aHost)
      return
    this.doIDNHistory(aHost, true, true, false);
  },

  /* Formerly known as: mzRemoveHostFromIDNHistory() */
  removeHostFromIDNHistory: function(aHost)
  {
    if (!aHost)
      return
    this.doDNHistory(aHost, false, false, true);
  },
  
  /* Formerly known as: mzPurgeIDNHistory() */
  purgeIDNHistory: function()
  {
    this.doIDNHistory("", false, false, true);
  }
};


/***
  * this is a specifix class, which we don't want to be part of the global
  * object, mostly for security reasons, but this protects the context menu,
  * which is part of Mozilla back-end code these days, but that don't nag you,
  * and thus you have no idea /when/ the menu was changed.  Which make break
  * websites, and thus is it not a good enough for us.
  */

var multiZillaAntiCMenuHijack = 
{
  _bundle : null,

  init: function()
  {
    _bundle = gMultiZilla.utility.getStringBundle("tabbrowser.properties");
  },

  click: function(aEvent)
  {
    if (aEvent.button == 2 ) {
      var cmCheckPrefValue = 0;

      try {
        cmCheckPrefValue = pref.getIntPref("multizilla.document.context-menu.behavior");
      }
      catch(ex) {
        pref.setIntPref("multizilla.document.context-menu.behavior", cmCheckPrefValue);
      }
      if (cmCheckPrefValue > 0) {
        var documentWrapper = Components.lookupMethod(aEvent.originalTarget,
                                                      "ownerDocument").call(aEvent.originalTarget);
        this.contextMenuCheck(documentWrapper, cmCheckPrefValue, aEvent);
      }
    }
  },

  contextMenuCheck: function(aDocument, aAction, aEvent)
  {
    if (aDocument instanceof HTMLDocument) {
      if (aDocument.location.href.indexOf("chrome:") > -1)
        return;

      var targetElements = new Array();
      var hasModifiedContextMenu = (aAction == 2);
      var sContent = new XPCNativeWrapper(window.content, "frames").frames;

      if (aAction == 1) {
        targetElements = this.checkContextMenu(aDocument, aEvent, targetElements);

        if (targetElements.length)
          hasModifiedContextMenu = true;

        for (var i = 0; i < sContent.frames.length; i++) {
          targetElements = this.checkContextMenu(sContent.frames[i].document, aEvent, targetElements);

          if (targetElements.length) {
            hasModifiedContextMenu = true;
            break;
          }
        }
      }
      if (hasModifiedContextMenu) {
        if (aAction == 1 && !this.confirmReset())
          return;
        targetElements = (targetElements.length == 0) ? ["html", "body", "img"] : targetElements;
        this.resetOnContextMenu(aDocument, targetElements);

        for (i = 0; i < sContent.frames.length; i++) {
          this.resetOnContextMenu(sContent.frames[i].document, targetElements);
        }
      }
    }
  },

  checkContextMenu: function(aDocument, aEvent, aModifiedElementList)
  {
    const RETURN_TRUE = "return true;";
    var element = aDocument.getElementsByTagName("html")[0];

    if (element && "oncontextmenu" in element) {
      if (element.oncontextmenu != undefined) {
        if (aModifiedElementList.indexOf("html") == -1)
          aModifiedElementList.push("html");
      }
    }
    if ("oncontextmenu" in aDocument && aDocument.oncontextmenu != undefined || 
        "onmouseup" in aDocument && aDocument.onmouseup != undefined || 
        "onmousedown" in aDocument && aDocument.onmousedown != undefined)
    {
      if (aModifiedElementList.indexOf("html") == -1)
        aModifiedElementList.push("html");
    }
    element = aDocument.getElementsByTagName("body")[0];

    if (element && element.hasAttribute("oncontextmenu")) {
      if (element.getAttribute("oncontextmenu") != RETURN_TRUE) {
        if (aModifiedElementList.indexOf("body") == -1)
          aModifiedElementList.push("body");
      }
    }
    var targetElements = ["img"];
    var clickedNodeName = aEvent.originalTarget.localName.toLowerCase();

    if (clickedNodeName != "img")
      targetElements.push(clickedNodeName);

    for (i in targetElements) {
      var targetElementName = targetElements[i];
      var elementList = aDocument.getElementsByTagName(targetElementName);

      for (var i =0; i < elementList.length; i++) {
        element = elementList[i];

        if (element && element.hasAttribute("oncontextmenu")) {
          if (element.getAttribute("oncontextmenu") != RETURN_TRUE) {
            if (aModifiedElementList.indexOf(targetElementName) == -1)
              aModifiedElementList.push(targetElementName);
            break;
          }
        }
      }
    }
    return (aModifiedElementList);
  },

  confirmReset: function()
  {
    var checkbox = {value:false};
    var dialogTitle = this._bundle.GetStringFromName("mzWizard");
    var dialogText = this._bundle.GetStringFromName("contextMenuCheckTitle");
    var checkboxText = this._bundle.GetStringFromName("contextMenuCheckCheckbox");
    var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                  .getService(Components.interfaces.nsIPromptService);
    var result = promptService.confirmEx(window, dialogTitle, dialogText, 
                                        (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) +
                                        (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_NO),
                                         null, null, null, checkboxText, checkbox);
    if (result)
      return false;
    else if (checkbox.value)
      pref.setIntPref("multizilla.document.context-menu.behavior", 2);
    return true;
  }, 

  resetOnContextMenu: function(aDocument, aTargetElementList)
  {
    const RETURN_TRUE = "return true;";
    var element;

    for (var t = 0; t < aTargetElementList.length; t++) {
      var targetElement = aTargetElementList[t];

      if (targetElement == "html") {
        element = aDocument.getElementsByTagName(targetElement)[0];
        element.onmouseup = undefined;
        element.onmousedown = undefined;
        element.oncontextmenu = undefined;

        aDocument.onmouseup = undefined;
        aDocument.onmousedown = undefined;
        aDocument.oncontextmenu = undefined;
      }
      else { // body, img, etc
        var elementList = aDocument.getElementsByTagName(targetElement);

        for (var n = 0; n < elementList.length; n++) {
          var element = elementList[n];

          if (element && element.getAttribute("oncontextmenu") != RETURN_TRUE)
            element.setAttribute("oncontextmenu", RETURN_TRUE);
        }
      }
    }
  }
};

gMultiZilla.__proto__ = gMultiZilla.prototype;

addEventListener("load", multiZillaAntiCMenuHijack.init, false);
