/* vim: set sw=8 ts=8 si : */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * See http://www.gnu.org/copyleft/ for details.
 *
 * Written by Katja Socher <katja@linuxfocus.org>
 *        and Guido Socher <guido@linuxfocus.org> 
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h> 
#include <sys/types.h>
#include <sys/time.h> 
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <linux/ppdev.h>
#include <sys/ioctl.h>
#include <termios.h>
#include "robi.h"


/* convert a timeval struct with seconds and microseconds
 * into one long integer with milliseconds */
static long int tv2msec(struct timeval *tv)
{
    long int msec,tmp;
    tmp=tv->tv_usec / 1000; /* keep only milliseconds */
    msec=tv->tv_sec % 1000; /* keep max 1000 seconds */
    msec *= 1000; /* shift left */
    msec += tmp;
    return(msec);
}
/* like printf but exit the program */
static int die(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
	exit(1);
}
/* get one character from stdin 
 * Returns non zero if char was read otherwise zero
 * The arrow keys are mapped as follows:
 * <- = h
 * -> = l
 * v = j
 * ^ = k
 */
int getchoice()
{
	int c;
	char s[20];
	
	if (fgets(s,20,stdin)){
		c=s[0];
		switch (c){
			case 0x1b: /* ESC */
				if (s[1] == 0x5b){
					/* arrow keys are pressed */
					switch (s[2]){
						case 0x41: /*up arrow*/
							c='k';
							break;
						case 0x42: /*down arrow*/
							c='j';
							break;
						case 0x44: /*l arrow*/
							c='h';
							break;
						case 0x43: /*r arrow*/
							c='l';
							break;
						default:
							c=0;
					}
				}else{
					c=0;
				}
				break;
			case ' ':
			case 'h':
			case 'j':
			case 'k':
			case 'l':
			case 'q':
			case 'x':
				break;
			default:
				c=0;
		}
		return(c);
	}
	return(0);
}
	
/* Set the Terminal to Non Canonical mode with echo off
 * or reset the terminal.
 * USAGE: set_terminal(1) for canonical
 */
int set_terminal(int canonical)
{
	static struct termios originalsettings;
	struct termios newsettings;
	static int origok=0; /* set if originalsettings valid */
	if (canonical){
		/* save original settings and set canonical mode*/
		tcgetattr(fileno(stdin),&originalsettings);
		newsettings=originalsettings;
		newsettings.c_lflag &= ~ICANON;
		newsettings.c_lflag &= ~ECHO;
		newsettings.c_cc[VMIN]=0; /* do not block */
		newsettings.c_cc[VTIME]=1; /* 100 ms */
		if (tcsetattr(fileno(stdin),TCSANOW,&newsettings) !=0){
			die("ERROR: could not set terminal attributes on stdin\n");
		}
		origok=1;
	}else{
		if (origok){
			/* restore settings */
			tcsetattr(fileno(stdin),TCSANOW,&originalsettings);
		}
	}
	return(0);
}

/* open /dev/parportX device and claim it.
 * USAGE: fd=robi_claim("/dev/parport0");
 * The return value is a file descriptor used by other
 * functions such as robi_setdata */
int robi_claim(char *dev)
{
	int fd,i;

	fd = open(dev, O_RDWR );
        if (fd < 0) {
                die("ERROR: cannot open device %s\n",dev);
        }
	i=0;
	/* we need exclusive rights as we do not set the control lines*/
	/*ioctl(fd, PPEXCL, &i)&&die("ERROR: request for exclusive rights failed\n");*/
	ioctl(fd, PPCLAIM, &i)&&die("ERROR: could not claim parport. Did you load the modules parport_pc, ppdev and parport? Check with /sbin/lsmod\n");
	return(fd);

}
/* Walk left
 */
int walkleft(int fd)
{
	/* first B legs to ground */
	robi_setdata(fd,LEGBD);
	usleep(400 *1000);
	/* all A legs 1 step */
	robi_setdata(fd, LEGB1 | LEGB3 );
	usleep(1100 *1000);

	/* first A legs to ground, cool B*/
	robi_setdata(fd,LEGAD);
	usleep(400 *1000);
	robi_setdata(fd,0);
	usleep(1000 *1000);
	return(0);
}
/* Walk right
 */
int walkright(int fd)
{
	
	/* first A legs to ground */
	robi_setdata(fd,LEGAD);
	usleep(500 *1000);
	robi_setdata(fd,  LEGA3 | LEGAD);
	usleep(300 *1000);
	/* all A legs 1 step */
	robi_setdata(fd, LEGA1 | LEGA3 );
	usleep(1100 *1000);

	/* first B legs to ground, cool A*/
	robi_setdata(fd,LEGBD);
	usleep(400 *1000);
	robi_setdata(fd,0);
	usleep(1000 *1000);
	return(0);
}
/* Walk with all 3 legs 1 step forward
 */
int walkstraight_a(int fd)
{
	
	/* first A legs to ground */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	/* all A legs 1 step */
	robi_setdata(fd, LEGA1 | LEGA2 | LEGA3 );
	usleep(1000 *1000);

	/* first B legs to ground, cool A*/
	robi_setdata(fd,LEGBD);
	usleep(500 *1000);
	robi_setdata(fd,0);
	usleep(1200 *1000);
	return(0);
}
/* Walk with all 3 legs 1 step forward
 */
int walkstraight_b(int fd)
{
	/* first B legs to ground */
	robi_setdata(fd,LEGBD);
	usleep(400 *1000);
	/* all B legs 1 step */
	robi_setdata(fd,LEGB1 | LEGB2 | LEGB3);
	usleep(1000 *1000);
	/* A down and cool */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	robi_setdata(fd,0);
	usleep(1200 *1000);
	return(0);
}
/* Walk with all 6 legs 1 step forward
 */
int walkstraight(int fd)
{
	
	/* first A legs to ground */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	/* all A legs 1 step */
	robi_setdata(fd, LEGA1 | LEGA2 | LEGA3 );
	usleep(1000 *1000);

	/* first B legs to ground, cool A*/
	robi_setdata(fd,LEGBD);
	usleep(500 *1000);
	robi_setdata(fd,0);
	usleep(1800 *1000);

	/* first B legs to ground */
	robi_setdata(fd,LEGBD);
	usleep(400 *1000);
	/* all B legs 1 step */
	robi_setdata(fd,LEGB1 | LEGB2 | LEGB3);
	usleep(1000 *1000);
	/* A down and cool */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	robi_setdata(fd,0);
	usleep(1000 *1000);
	return(0);
}
/* Walk with all 6 legs 1 step back
 */
int walkback(int fd)
{
	
	/* first A legs to ground */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	/* all B legs 1 step in the air*/
	robi_setdata(fd, LEGB1 | LEGB2 | LEGB3 );
	usleep(500 *1000);

	/* first B legs to ground, cool A*/
	robi_setdata(fd,LEGBD);
	usleep(500 *1000);
	/* all A legs 1 step in the air*/
	robi_setdata(fd,LEGA1 | LEGA2 | LEGA3);
	usleep(500 *1000);
	/* A down and cool */
	robi_setdata(fd,LEGAD);
	usleep(800 *1000);
	robi_setdata(fd,0);
	usleep(1000 *1000);
	return(0);
}
/*---------*/
/* Write a bit pattern to the data lines
 * USAGE: rc=robi_setdata(fd,bitpat);
 * The return value is 0 on success.
 */
int robi_setdata(int fd,unsigned char bitpat)
{
	int rc;

	rc=ioctl(fd, PPWDATA, &bitpat);
	return(rc);
}
/*---------*/
